/* 100to110.C */

/* Converts SMB v1.00 message base to v1.10 (new index record field)		*/

#include "smblib.h"

typedef struct {

	ushort	to; 			// 16-bit CRC of recipient name (lower case)
	ushort	from;			// 16-bit CRC of sender name (lower case)
	ushort	subj;			// 16-bit CRC of subject (lower case, w/o RE:)
	ushort	attr;			// attributes (read, permanent, etc.)
	ulong	offset; 		// offset into header file
	ulong	number; 		// number of message (1 based)

	} oldidxrec_t;


/****************************************************************************/
/* Fills msg->idx with message index based on msg->hdr.number				*/
/* OR if msg->hdr.number is 0, based on msg->offset (physical offset).		*/
/* if msg.hdr.number does not equal 0, then msg->offset is filled too.		*/
/* Either msg->hdr.number or msg->offset must be initialized before 		*/
/* calling this function													*/
/* Returns 1 if message number wasn't found, 0 if it was                    */
/****************************************************************************/
int smb_getoldmsgidx(smbmsg_t *msg)
{
	oldidxrec_t idx;
	ulong	 l,length,total,bot,top;

length=filelength(fileno(sid_fp));
if(!length)
	return(1);
total=length/sizeof(oldidxrec_t);
if(!total)
	return(1);

bot=0;
top=total;
l=total/2; /* Start at middle index */
while(1) {
	fseek(sid_fp,l*sizeof(oldidxrec_t),SEEK_SET);
	if(!fread(&idx,sizeof(oldidxrec_t),1,sid_fp))
		return(1);
	if(bot==top-1 && idx.number!=msg->hdr.number)
        return(1);
	if(idx.number>msg->hdr.number) {
		top=l;
		l=bot+((top-bot)/2);
		continue; }
	if(idx.number<msg->hdr.number) {
		bot=l;
		l=top-((top-bot)/2);
		continue; }
	break; }
msg->idx.to=idx.to;
msg->idx.from=idx.from;
msg->idx.subj=idx.subj;
msg->idx.attr=idx.attr;
msg->idx.number=idx.number;
msg->idx.offset=idx.offset;
msg->offset=l;
return(0);
}

/****************************************************************************/
/* Read header information into 'msg' structure                             */
/* msg->idx.offset must be set before calling this function 				*/
/* Must call smb_freemsgmem() to free memory allocated for var len strs 	*/
/* Returns 0 on success, non-zero if error									*/
/****************************************************************************/
int smb_getoldmsghdr(smbmsg_t *msg)
{
	ushort	i;
	ulong	l,offset;
	idxrec_t idx;

fseek(shd_fp,msg->idx.offset,SEEK_SET);
idx=msg->idx;
offset=msg->offset;
memset(msg,0,sizeof(smbmsg_t));
msg->idx=idx;
msg->offset=offset;
if(!fread(&msg->hdr,sizeof(msghdr_t),1,shd_fp))
	return(-1);
if(memcmp(msg->hdr.id,"SHD\x1a",4))
	return(-2);
if(msg->hdr.version<0x100)
	return(-9);
l=sizeof(msghdr_t);
if(msg->hdr.total_dfields && (msg->dfield
	=(dfield_t *)MALLOC(sizeof(dfield_t)*msg->hdr.total_dfields))==NULL) {
	smb_freemsgmem(*msg);
	return(-3); }
i=0;
while(i<msg->hdr.total_dfields && l<msg->hdr.length) {
	if(!fread(&msg->dfield[i],sizeof(dfield_t),1,shd_fp)) {
		smb_freemsgmem(*msg);
		return(-4); }
	i++;
	l+=sizeof(dfield_t); }
if(i<msg->hdr.total_dfields) {
	smb_freemsgmem(*msg);
	return(-8); }

while(l<msg->hdr.length) {
	i=msg->total_hfields;
	if((msg->hfield_dat=(void **)REALLOC(msg->hfield_dat,sizeof(void *)*(i+1)))
		==NULL) {
		smb_freemsgmem(*msg);
		return(-3); }
	if((msg->hfield=(hfield_t *)REALLOC(msg->hfield
		,sizeof(hfield_t)*(i+1)))==NULL) {
		smb_freemsgmem(*msg);
		return(-3); }
	msg->total_hfields++;
	if(!fread(&msg->hfield[i],sizeof(hfield_t),1,shd_fp)) {
		smb_freemsgmem(*msg);
		return(-5); }
	l+=sizeof(hfield_t);
	if((msg->hfield_dat[i]=(char *)MALLOC(msg->hfield[i].length+1))
		==NULL) {			/* Allocate 1 extra for NULL terminator */
		smb_freemsgmem(*msg);  /* or 0 length field */
		return(-3); }
	memset(msg->hfield_dat[i],0,msg->hfield[i].length+1);  /* init to NULL */
	if(msg->hfield[i].length
		&& !fread(msg->hfield_dat[i],msg->hfield[i].length,1,shd_fp)) {
		smb_freemsgmem(*msg);
		return(-6); }
	switch(msg->hfield[i].type) {	/* convenience variables */
		case SENDER:
			msg->from=msg->hfield_dat[i];
			break;
		case SENDERAGENT:
			msg->from_agent=*(ushort *)msg->hfield_dat[i];
            break;
		case SENDEREXT:
			msg->from_ext=msg->hfield_dat[i];
			break;
		case SENDERNETTYPE:
			msg->from_net.type=*(ushort *)msg->hfield_dat[i];
            break;
		case SENDERNETADDR:
			msg->from_net.addr=msg->hfield_dat[i];
            break;
		case REPLYTO:
			msg->replyto=msg->hfield_dat[i];
            break;
		case REPLYTOEXT:
			msg->replyto_ext=msg->hfield_dat[i];
			break;
		case REPLYTOAGENT:
			msg->replyto_agent=*(ushort *)msg->hfield_dat[i];
            break;
		case REPLYTONETTYPE:
			msg->replyto_net.type=*(ushort *)msg->hfield_dat[i];
            break;
		case REPLYTONETADDR:
			msg->replyto_net.addr=msg->hfield_dat[i];
            break;
		case RECIPIENT:
			msg->to=msg->hfield_dat[i];
            break;
		case RECIPIENTEXT:
			msg->to_ext=msg->hfield_dat[i];
			break;
		case RECIPIENTAGENT:
			msg->to_agent=*(ushort *)msg->hfield_dat[i];
            break;
		case RECIPIENTNETTYPE:
			msg->to_net.type=*(ushort *)msg->hfield_dat[i];
            break;
		case RECIPIENTNETADDR:
			msg->to_net.addr=msg->hfield_dat[i];
            break;
		case SUBJECT:
			msg->subj=msg->hfield_dat[i];
			break; }
	l+=msg->hfield[i].length; }

if(!msg->from || !msg->to || !msg->subj) {
	smb_freemsgmem(*msg);
	return(-7); }
return(0);
}


/****************************************************************************/
/* Open a message base of name 'smb_file'                                   */
/* Opens files for READing messages or updating message indices only        */
/****************************************************************************/
int smb_oldopen()
{
    int file;
    char str[128];

sprintf(str,"%s.SDT",smb_file);
if((file=open(str,O_RDWR|O_CREAT|O_BINARY|O_DENYNONE,S_IWRITE|S_IREAD))==-1
	|| (sdt_fp=fdopen(file,"r+b"))==NULL)
	return(1);
setvbuf(sdt_fp,NULL,_IOFBF,2*1024);

sprintf(str,"%s.SHD",smb_file);
if((file=open(str,O_RDWR|O_CREAT|O_BINARY|O_DENYNONE,S_IWRITE|S_IREAD))==-1
	|| (shd_fp=fdopen(file,"r+b"))==NULL)
	return(2);
setvbuf(shd_fp,NULL,_IOFBF,2*1024);

sprintf(str,"%s.SID",smb_file);
if((file=open(str,O_RDWR|O_CREAT|O_BINARY|O_DENYNONE,S_IWRITE|S_IREAD))==-1
	|| (sid_fp=fdopen(file,"r+b"))==NULL)
	return(3);
setvbuf(sid_fp,NULL,_IOFBF,2*1024);

return(0);
}


int main(int argc, char **argv)
{
	char	*p;
	int 	x,i;
	ulong	size,l,length;
	FILE	*sid_tmp;
	idxrec_t idx;
	smbmsg_t msg;
	smbhdr_t hdr;
	smbstatus_t status;

printf("\nSMB v1.00 to v1.10 conversion utility\n");

if(argc<2) {
	printf("\nusage: 100to110 <filespec.SHD>\n");
	return(1); }

for(x=1;x<argc;x++) {

	strcpy(smb_file,argv[x]);
	p=strrchr(smb_file,'.');
	if(p) *p=0;
	strupr(smb_file);

	printf("\nConverting %s\r\n",smb_file);

	if((i=smb_oldopen())!=0) {
		printf("smb_open returned %d\n",i);
		continue; }

	if(filelength(fileno(shd_fp))<sizeof(smbhdr_t)) {
		printf("Empty\n");
		smb_close();
		continue; }

	if((i=smb_locksmbhdr(10))!=0) {
		smb_close();
		printf("smb_locksmbhdr returned %d\n",i);
		continue; }

	fread(&hdr,sizeof(smbhdr_t),1,shd_fp);
    hdr.version=SMB_VERSION;
    fseek(shd_fp,0L,SEEK_SET);
    fwrite(&hdr,sizeof(smbhdr_t),1,shd_fp);

	if((i=smb_getstatus(&status))!=0) {
		smb_unlocksmbhdr();
		smb_close();
		printf("smb_getstatus returned %d\n",i);
		continue; }

	sid_tmp=tmpfile();

	length=filelength(fileno(shd_fp));

	for(l=status.header_offset;l<length;l+=size) {
		printf("\r%2u%%  ",l ? (long)(100.0/((float)length/l)) : 0);
		msg.idx.offset=l;
		if((i=smb_lockmsghdr(msg,10))!=0) {
			printf("\n(%06lX) smb_lockmsghdr returned %d\n",l,i);
			size=SHD_BLOCK_LEN;
			continue; }
		if((i=smb_getoldmsghdr(&msg))!=0) {
			smb_unlockmsghdr(msg);
			printf("\n(%06lX) smb_getmsghdr returned %d\n",l,i);
			size=SHD_BLOCK_LEN;
			continue; }
		printf("#%-5lu (%06lX) %-25.25s ",msg.hdr.number,l,msg.from);
		size=smb_getmsghdrlen(msg);
		while(size%SHD_BLOCK_LEN)
			size++;

		msg.hdr.version=SMB_VERSION;
		if((i=smb_putmsghdr(msg))!=0) {
			smb_unlockmsghdr(msg);
			printf("\n(%06lX) smb_putmsghdr returned %d\n",l,i);
			continue; }

		smb_unlockmsghdr(msg);
		smb_freemsgmem(msg);

		if(msg.hdr.attr&MSG_DELETE) {
			printf("Deleted\n");
			continue; }
		if(smb_getoldmsgidx(&msg)) {
			printf("Not found in index\n");
			continue; }
		msg.idx.time=msg.hdr.when_imported.time;
		fseek(sid_tmp,msg.offset*sizeof(idxrec_t),SEEK_SET);
		fwrite(&msg.idx,sizeof(idxrec_t),1,sid_tmp); }

	printf("\n\nCreating new index...\n");
	rewind(sid_fp);
	chsize(fileno(sid_fp),0L);
	fseek(sid_tmp,0L,SEEK_SET);
	l=0;
	while(1) {
		printf("%lu\r",l++);
		if(!fread(&idx,sizeof(idxrec_t),1,sid_tmp))
			break;
		fwrite(&idx,sizeof(idxrec_t),1,sid_fp); }
	printf("\r%79s\r100%%\n","");
	fclose(sid_tmp);
	smb_close();
	}
return(0);
}
