/*
 *   Copyright 1992, 1993, 1994 John Melton (G0ORX/N6LYT)
 *              All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
	xpb.c

	eXperimental Pacsat Broadcast Protocal Receiver
	 -           -      -

	This software was written to provide a receive only capability
	for collecting files and directories from satellites running
	the Pacsat Protocols running on the Linux Operating System.

	This program has been run using the 1.0 version of the
	Linux kernel with the patches from Alan Cox to provide AX.25
	encapsulation of SLIP.

	The TNC must be setup for KISS.

	John Melton
	G0ORX, N6LYT

	4 Charlwoods Close
	Copthorne
	West Sussex
	RH10 3QZ
	England

	INTERNET:	g0orx@amsat.org
			n6lyt@amsat.org
			john@images.demon.co.uk
			J.D.Melton@slh0613.icl.wins.co.uk

	History:
	-------

	0.1	Initial version - no GUI interface.
	0.2	Added X-Windows (OpenView) interface.
	0.3	Added transmit capability for file and hole fill requests.
	0.4	Added ability to receive fragmented file headers for directory
		broadcasts,  and assemble them together (most headers are
		less than the 244 byte packet limit, but some people have
		set their limit to 128,  which causes the headers to be
		fragmented).
	0.5	Reworked hole lists.

	To Do:
	-----

	1.	Needs an interface to the directory listing program
		to allow point and click requests of files and
		directories.
	2.	Automated file and directory requests.
	3.	Directory hole list pruning.
	4.	Solve problem of crash not updating hole files.
		Biggest problem with directory entries being in
		the file, but not removed from hole list. Causes
		duplicates in the files.
	5.	Limit amount read in from old directories.
*/

#define VERSION_STRING "(version 0.5 by g0orx/n6lyt)"

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/font.h>
#include <xview/tty.h>
#include <xview/notice.h>

#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <linux/ax25.h>

#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include "ftl0.h"
#include "xpb.h"
#include "header.h"
#include "crc.h"

Frame frame;
Panel panel;
Panel_item fillFileId;
Tty log;
Font font;

char satelliteId[16];
char myCall[16];

int s_raw;
int s_file;
int s_directory;
struct sockaddr_ax25 dest;
struct sockaddr_ax25 src;

struct sockaddr_in request_addr;
int s_request;

unsigned char buf[MAXBUFFER];
int bufSize;


OPENFILES of[MAXFILES+1];	/* last entry is for the directory file */
int nextof = 0;

unsigned long maxDays = 5;

unsigned lastId = 0;

unsigned long fragmentId = 0;
unsigned long fragmentOffset = 0;
unsigned char fragmentBuffer[1024];

int frames = 0;
int bytes = 0;
int fileFrames = 0;
int fileBytes = 0;
int directoryFrames = 0;
int directoryBytes = 0;
int crcErrors = 0;

/*
 *	Convert a call from the shifted ascii form used in an
 *	AX.25 packet.
 */
int ConvertCall(char *c, char *call)
{
	char *ep=c+6;
	int ct=0;
	while(ct<6)
	{
		if(((*c>>1)&127)==' ')
			break;
		*call = (*c>>1)&127;
		call++;
		ct++;
		c++;
	}
	
	if((*ep&0x1E)!=0)
	{	
		*call = '-';
		call++;
		call += sprintf( call, "%d",(int)(((*ep)>>1)&0x0F));
	}

	*call = '\0';
	
	if(*ep&1)
	{
		return 0;
	}
	return 1;
}

/*
 *	Convert a call to the shifted ascii form used in an
 *	AX.25 packet.
 */
int MakeAddress(char *name,struct sockaddr_ax25 *sax)
{
	int ct=0;
	int ssid=0;
	unsigned char *p=name;
	while(ct<6)
	{
		if(islower(*p))
			*p=toupper(*p);
		if(!isalnum(*p))
		{
			printf("Invalid symbol in callsign.\n");
			return -1;
		}
		
		sax->sax25_call.ax25_call[ct]=(*p<<1);
		p++;
		ct++;
		if(*p=='-' || *p==0)
			break;
	}
	while(ct<6)
	{
		sax->sax25_call.ax25_call[ct]=' '<<1;
		ct++;
	}
	if(*p==0)
		ssid=0;
	else
	{
		p++;
		if(sscanf(p,"%d",&ssid)!=1 || ssid<0 || ssid>15)
		{
			printf("SSID must follow '-' and be numeric in the range 0-15.\n");
			return -1;
		}
	}
	sax->sax25_call.ax25_call[6]=((ssid+'0')<<1)&0x1E;
	sax->sax25_family=AF_AX25;
	return 0;
}

/*
 * Hole list maintenance
 */
int LoadHoleList( int f, char * fileName )
{
	FILE *holeFile;
	HOLE *hole;
	HOLE *prevHole;
	char temp[80];
	int i;

	holeFile = fopen( fileName, "r" );
	if( holeFile != (FILE *)0 )
	{

		fscanf( holeFile, "%d %s %s %s", &(of[f].hdrSeen), temp, temp, temp );
		fscanf( holeFile, "%d %s %s %s", &(of[f].fileSize), temp, temp, temp );
		fscanf( holeFile, "%d %s", &(of[f].holes), temp );

		prevHole = (HOLE *)0;
		for( i=0; i<of[f].holes; i++ )
		{
			hole = malloc( sizeof( HOLE ) );
			hole->nextHole = (HOLE *)0;
			fscanf( holeFile, "%u, %u", &(hole->start), &(hole->finish) );
			if( prevHole == (HOLE *)0 )
				of[f].firstHole = hole;
			else
				prevHole->nextHole = hole;
			prevHole = hole;
		}
	
		fclose( holeFile );
		return( 1 );
	}
	else
	{
		/* create an empty hole list */
		of[f].firstHole = malloc( sizeof( HOLE ) );
		of[f].firstHole->start = 0;
		of[f].firstHole->finish = 0xFFFFFFFF;
		of[f].firstHole->nextHole = (HOLE *)0;
		of[f].holes = 1;
		return( 0 );
	}

}

void LoadDirectoryHoleList( )
{
	time_t t;

	if( LoadHoleList( MAXFILES, "pfhdir.hol" )  == 0 )
	{
		/* use file size as file number - start at 1 */
		of[MAXFILES].fileSize=1;
	}

	/* prune back the list */
	time( &t );
	t = t - (60*60*24*maxDays);

	UpdateHoleList( MAXFILES, (unsigned)0, (unsigned)t );
}

void SaveHoleList( int f, char * fileName )
{
	FILE *holeFile;
	HOLE *hole;

	holeFile = fopen( fileName, "w" );
	if( holeFile == (FILE *)NULL )
	{
		perror( fileName );
		exit( 1 );
	}
	
	fprintf( holeFile, "%d pfh header received\n", of[f].hdrSeen );
	fprintf( holeFile, "%d pfh file length\n", of[f].fileSize );
	fprintf( holeFile, "%d holes\n", of[f].holes );

	hole = of[f].firstHole;
	while( hole != (HOLE *)0 )
	{
		fprintf( holeFile, "%u, %u\n", hole->start, hole->finish );
		hole = hole->nextHole;
	}

	fclose( holeFile );
}

int UpdateHoleList( int f, unsigned start, unsigned finish )
{
	HOLE *h, *p, *n;
	int ret = 0;

	p = (HOLE *)0;
	h = of[f].firstHole;
	while( h != (HOLE *)0 )
	{
		/* see if this is a candidate */
		if( (start <= h->finish) && (finish >= h->start) )
		{
			/* look for the simple ones first */
			if( start<=h->start )
			{
				/* will remove from the beginning */
				if( finish >= h->finish )
				{
					/* will remove all of it */
					if( p == (HOLE *)0 )
						of[f].firstHole = h->nextHole;
					else
						p->nextHole = h->nextHole;
					of[f].holes--;
					n = h->nextHole;
					free( h );
					h = n;
					ret = 1;
				}
				else
				{
					/* remove the front part */
					h->start = finish+1;
					p = h;
					h = h->nextHole;
					ret = 1;
				}
			}
			else if( finish >= h->finish )
			{
				/* will remove the end */
				h->finish = start-1;
				p = h;
				h = h->nextHole;
				ret = 1;
			}
			else
			{
				/* remove the middle */
				n = malloc( sizeof( HOLE ) );
				n->start = finish+1;
				n->finish = h->finish;
				n->nextHole = h->nextHole;

				/* change the current hole for the first part */
				h->finish = start-1;
				h->nextHole = n;

				/* one more hole */
				of[f].holes++;
				p = h;
				h = h->nextHole;
				ret = 1;
			}
		}
		else
		{
			p = h;
			h = h->nextHole;
		}

		/* stop when we get too far */
		if( h != (HOLE *)0 )
		{
			if( finish < h->start )
				break;
		}
	}

	return ret;
}

/*
 * Message file maintenance
 */
void LoadFile( int fileId, int f )
{
	char holeName[80];
	char fileName[80];

	/* save this file id */
	of[f].fileId = fileId;

	/* load the hole file and open the data file */
	sprintf( fileName, "%x.act", of[f].fileId );
	sprintf( holeName, "%x.hol", of[f].fileId );
	if( LoadHoleList( f, holeName ) )
		of[f].file = open( fileName, O_RDWR, 0660 );
	else
		of[f].file = open( fileName, O_CREAT | O_RDWR, 0660 );
	if( of[f].file == -1 )
	{
		perror( fileName );
		exit( 1 );
	}
}

void SaveFile( int f )
{
	HOLE *hole;
	HOLE *next;
	char fileName[80];


	if( of[f].file != 0 )
	{
		printf( "saving file %x\n", of[f].fileId );

		/* close the data file */
		close( of[f].file );

		/* write out the hole list */
		sprintf( fileName, "%x.hol", of[f].fileId );
		SaveHoleList( f, fileName );

		/* clean up the file table */
		of[f].fileId = 0;
		of[f].file = 0;
		of[f].hdrSeen = 0;
		of[f].fileSize = 0;
		of[f].holes = 0;

		/* free the hole list */
		hole = of[f].firstHole;
		while( hole != (HOLE *)0 )
		{
			next = hole->nextHole;
			free( hole );
			hole = next;
		}
		of[f].firstHole = (HOLE *)0;
	}
}

void CheckDownloaded( int f )
{
	int i;
	int headerSize;
	HEADER *hdr;
	HOLE *h;
	unsigned char *buf;
	char oldName[80];
	char newName[80];
	char holeName[80];

	/* see if we have the header */
	if( of[f].hdrSeen == 0 )
	{
		h = of[f].firstHole;
		if( h->start != 0 )
		{
			/* read the header into a buffer */
			buf = malloc( h->start );
			if( buf != (unsigned char *)0 )
			{
				lseek( of[f].file, 0, 0 );
				read( of[f].file, buf, h->start );
				hdr = ExtractHeader( buf, h->start, &headerSize );
				if( hdr != (HEADER *)0 )
				{
					of[f].hdrSeen = 1;
					of[f].fileSize = hdr->fileSize;
					UpdateHoleList( f, (unsigned)of[f].fileSize, (unsigned)0xFFFFFFFF );
					free( hdr );
				}
				free( buf );
			}
		}
	}

	/* see if we now have the complete file */
	if( (of[f].holes == 0) && (of[f].hdrSeen == 1) )
	{
		unsigned long fileId;

		/* close the current file */
		fileId = of[f].fileId;
		SaveFile( f );

		/* rename the file */
		sprintf( oldName, "%x.act", fileId );
		sprintf( newName, "%x.dl", fileId );
		rename( oldName, newName );

		/* remove the hole file */
		sprintf( holeName, "%x.hol", fileId );
		unlink( holeName );

		/* let the user know */
		printf( "%x downloaded\n", fileId );
	}
}

void BroadcastFile( unsigned char *buffer, int length )
{
	FILEHEADER *fh;
	unsigned char *data;
	unsigned dataOffset;
	unsigned dataLength;
	char fileName[80];
	int f;
	int i;

	/* crc validation */
	if( CheckCRC( buffer, length ) != 0 )
	{
		crcErrors++;
		return;
	}

	/* point to the header */
	fh = (FILEHEADER *)buffer;

	/* point to the data */
	data = buffer + sizeof( FILEHEADER );
	dataOffset = fh->wOffset + (fh->nOffsetHigh<<16);
	dataLength = length - sizeof( FILEHEADER ) - CRCLENGTH;
	
	/* inform the user */
	if( fh->fileId != lastId )
	{
		printf( "Heard message %X\n", fh->fileId );
		lastId = fh->fileId;
	}

	/* see if the file already downloaded */
	sprintf( fileName, "%x.dl", fh->fileId );
	if( (f = open( fileName, O_RDONLY )) != -1 )
	{
		close( f );
		printf( "%x already downloaded\n", fh->fileId );
		return;
	}

	/* see if the file is in the current list of files */
	for( i=0; i<MAXFILES; i++ )
		if( of[i].fileId == fh->fileId )
			break;

	/* if it is not there then we must load it */
	if( i==MAXFILES )
	{
		/* save current file if used */
		SaveFile( nextof );

		/* now use this slot */
		LoadFile( fh->fileId, nextof );
		
		/* setup i for this entry */
		i = nextof;

		/* make sure we round robin the open files */
		nextof = (nextof+1) % MAXFILES;
	}

	if( UpdateHoleList( i, (unsigned)dataOffset, (unsigned)(dataOffset+dataLength-1) ) )
	{
		/* write the data */
		lseek( of[i].file, dataOffset, 0 );
		write( of[i].file, data, dataLength );

		/* see if it has all been downloaded */
		CheckDownloaded( i );
	}
}

/*
 *	Directory Maintenance
 */
void WriteDirectory( unsigned char *data, int dataLength )
{
	char fileName[80];
	int fileSize;
	int i;

	sprintf( fileName, "pb__%04d.pfh", of[MAXFILES].fileSize );

	/* write the data to the end of the current file */
	of[MAXFILES].file = open( fileName, O_RDWR | O_CREAT, 0660 );
	if( of[MAXFILES].file != -1 )
	{
		fileSize = lseek( of[MAXFILES].file, 0, 2 );
		write( of[MAXFILES].file, data, dataLength );
		close( of[MAXFILES].file );
		of[MAXFILES].file = 0;
		fileSize += dataLength;

		/* limit file size to 20000 bytes */
		if( fileSize > 20000 )
			of[MAXFILES].fileSize++;
	}
	else
		perror( fileName );
}

void BroadcastDirectory( unsigned char *buffer, int length )
{
	DIRHEADER *dh;
	int headerSize;
	HEADER *hdr;
	unsigned char *data;
	int dataLength;
	int i;

	/* crc validation */
	if( CheckCRC( buffer, length ) != 0 )
	{
		crcErrors++;
		return;
	}

	/* point to the header */
	dh = (DIRHEADER *)buffer;

	/* check for fragmented directory header */
	if( dh->offset != 0 )
	{
		/* see if this is the next fragment */
		if( (fragmentId == dh->fileId) && (dh->offset == fragmentOffset) )
		{
			/* append this fragment */
			data = buffer + sizeof( DIRHEADER );
			dataLength = length - sizeof( DIRHEADER ) - CRCLENGTH;
			for( i=0; i<dataLength; i++ )
				fragmentBuffer[fragmentOffset+i] = data[i];
			fragmentOffset += dataLength;
			if( (dh->flags & LASTBYTEFLAG) != LASTBYTEFLAG )
			{
				data = fragmentBuffer;
				dataLength = fragmentOffset;
			}
			else
				return;
		}
		else
		{
			fragmentId = 0;
			fragmentOffset = 0;
			return;
		}
	}
	else if( (dh->flags & LASTBYTEFLAG) != LASTBYTEFLAG )
	{
		/* copy the fragment */
		data = buffer + sizeof( DIRHEADER );
		dataLength = length - sizeof( DIRHEADER ) - CRCLENGTH;
		for( i=0; i<dataLength; i++ )
			fragmentBuffer[i] = data[i];

		/* start a new fragment */
		fragmentId = dh->fileId;
		fragmentOffset = dataLength;

		return;
	}
	else
	{
		/* its all there -  point to the data */
		data = buffer + sizeof( DIRHEADER );
		dataLength = length - sizeof( DIRHEADER ) - CRCLENGTH;
	}

	if( (dh->flags & LASTBYTEFLAG) == LASTBYTEFLAG )
	{
		/* reset fragmentation */
		fragmentId = 0;
		fragmentOffset = 0;

		/* try to extract the header */
		hdr = ExtractHeader( data, dataLength, &headerSize );
		if( hdr != (HEADER *)0 )
		{
			/* inform the user */
			if( strlen( hdr->title ) == 0 )
				printf( "dir: %x: from:%s to:%s title:%s\n",
					hdr->fileId,
					hdr->source,
					hdr->destination,
					hdr->fileName );
			else
				printf( "dir: %x: from:%s to:%s title:%s\n",
					hdr->fileId,
					hdr->source,
					hdr->destination,
					hdr->title );
		
			free( hdr );

			/* see if the hole list is loaded */
			if( of[MAXFILES].firstHole == (HOLE *)0 )
				LoadDirectoryHoleList( );

			/* update the directory  if needed */
			if( UpdateHoleList( MAXFILES, (unsigned)dh->tOld, (unsigned)dh->tNew ) )
				WriteDirectory( data, dataLength );
		}
		else
			fprintf( stderr, "**Bad directory header for %x\n", dh->fileId );
	}


}

/*
 *	decode a received frame.
 */
void ProcessFrame ( )
{
	int n;
	int via;
	unsigned char protocol;
	char toCall[10];
	char fromCall[10];
	char viaCall[10];

	frames++;
	bytes+=bufSize;

	/* check that frame is a kiss data frame */
	/* ignore control frames - should not happen */
	n = 0;
	if( (buf[n]&0x0F) == 0 )
	{
		n++;

		/* decode the to/from address */
		/* dont expect via address, but saves last if any */
		via = ConvertCall( buf+n, toCall );
		n+=7;
		via=ConvertCall( buf+n, fromCall );
		n+=7;
		while( via )
		{
			via = ConvertCall( buf+n, viaCall );
			n+=7;
		}

		/* check for a UI frame */
		if( (buf[n]&0xEF) == 0003 )
		{
			n++;
			protocol = buf[n++];

			/* see if the frame is a broadcast frame */	
			if( strcmp( toCall, "QST-1" ) == 0 )
			{
				switch( protocol )
				{
				case PID_FILE:
					fileFrames++;
					fileBytes+=bufSize;
					BroadcastFile( buf+n,  bufSize-n );
					break;
				case PID_DIRECTORY:
					directoryFrames++;
					directoryBytes+=bufSize;
					BroadcastDirectory( buf+n, bufSize-n );
					break;
				default:
					break;
				}
			}
			else if( strcmp( toCall, "PBLIST" ) == 0
				|| strcmp( toCall, "BBSTAT" ) == 0
				|| strcmp( toCall, "QST" ) == 0
				|| strcmp( toCall, myCall ) == 0 )
			{
				buf[bufSize] = '\0';
				printf( "%s\n", buf+n );
			}
		}
	}

}

/*
 *	callback function when a frame is received.
 */
Notify_value GetFrame( Notify_client client, int s )
{
	bufSize = recv( s, buf, MAXBUFFER, 0 );
	if( bufSize == -1 )
	{
		perror( "recv" );
		return NOTIFY_DONE;
	}

	ProcessFrame( );
	return NOTIFY_DONE;
}

/*
 *	callback function when a another app calls us
 */
Notify_value FillRequest( Notify_client client, int s )
{
	unsigned long fileId;
	char temp[16];
	void FillFile( void );
	void FillDirectory( void );

	if( recv( s, (char *)&fileId, sizeof( fileId ), 0 ) == sizeof( fileId ) )
	{
		if( fileId == 0 )
			FillDirectory( );
		else
		{
			sprintf( temp, "%x", fileId );
			xv_set( fillFileId, PANEL_VALUE, temp, NULL );
			FillFile( );
		}
	}
	else
		perror( "recv - FillRequest" );
}

/*
 *	the user wants to exit this program
 */
void QuitButton( )
{
	int	i;

	/* save any open files */
	for( i=0; i<MAXFILES; i++ )
		SaveFile( i );

	/* save the directory hole list */
	if( of[MAXFILES].firstHole != (HOLE *)0 )
		SaveHoleList( MAXFILES, "pfhdir.hol" );

	/* print some stats */
	fprintf( stderr, "\nReceived:\n" );
	fprintf( stderr, "    totalFrames=%d\n", frames );
	fprintf( stderr, "    totalBytes=%d\n", bytes );
	fprintf( stderr, "    directoryFrames=%d\n", directoryFrames );
	fprintf( stderr, "    directoryBytes=%d\n", directoryBytes );
	fprintf( stderr, "    fileFrames=%d\n", fileFrames );
	fprintf( stderr, "    fileBytes=%d\n", fileBytes );
	fprintf( stderr, "    crcErrors=%d\n", crcErrors );

	/* close the sockets */
	close( s_raw );
	close( s_file );
	close( s_directory );

	/* get out */
	xv_destroy_safe( frame );
}

/*
 *	Construct a fill request for the directory.
 */
void FillDirectory( )
{
	DIRREQUESTHEADER *request;
	DIRHOLEPAIR *hole;
	HOLE *h;
	char fileName[80];
	unsigned char *buffer;
	int f;
	int i;

	/* see if the hole list is loaded */
	if( of[MAXFILES].firstHole == (HOLE *)0 )
		LoadDirectoryHoleList( );

	/* construct a hole list request */
	buffer = (unsigned char *)malloc( 256 );
	request = (DIRREQUESTHEADER *)buffer;

	request->flags = 0x00 | VERSION | 0x10;
	request->blockSize = 244;

	hole = (DIRHOLEPAIR *)&buffer[sizeof( DIRREQUESTHEADER )];
	h = of[MAXFILES].firstHole;
	for( i=0; (i<10) && (i<of[MAXFILES].holes); i++ )
	{
		hole->startTime = h->start;
		hole->endTime = h->finish;
		h = h->nextHole;
		hole++;
	}

	if( sendto( s_directory, buffer,
		    sizeof( DIRREQUESTHEADER ) + (i*sizeof( DIRHOLEPAIR )),
		    0, (struct sockaddr *)&dest, sizeof( dest ) ) == -1 )
		perror( "send" );
	printf( "FillDirectory: fill %d holes.\n", i );

	free( buffer );

}

/*
 *	construct a fill request for a file
 */
void FillFile( )
{
	char id[9];
	unsigned int fileId;
	char fileName[80];
	unsigned char *buffer;
	REQUESTHEADER *request;
	HOLEPAIR *hole;
	HOLE *h;
	int length;
	int f;
	int i;

	strcpy( id, (char *)xv_get( fillFileId, PANEL_VALUE, NULL ) );
	if( strlen( id ) == 0 )
		return;

	sscanf( id, "%x", &fileId );
	if( id == 0 )
		return;

	/* see if the file already downloaded */
	sprintf( fileName, "%x.dl", fileId );
	if( (f = open( fileName, O_RDONLY )) != -1 )
	{
		close( f );
		printf( "%x is already downloaded\n", fileId );
		return;
	}

	/* see if the file is in the current list of files */
	for( f=0; f<MAXFILES; f++ )
		if( of[f].fileId == fileId )
			break;

	if( f == MAXFILES )
	{
		/* not there - must save if this is in use */
		SaveFile( nextof );

		/* now use this slot */
		LoadFile( fileId, nextof );

		/* make sure we round robin the files */
		f = nextof;
		nextof = (nextof+1) % MAXFILES;
	}

	buffer = (unsigned char *)malloc( 256 );
	request = (REQUESTHEADER *)buffer;

	if( (of[f].holes == 1)
		&& (of[f].firstHole->start == 0)
		&& (of[f].firstHole->finish == 0xFFFFFFFF) )
	{
		request->flags = 0x00 | VERSION | 0x10;
		request->fileId = fileId;
		request->blockSize = 244;

		if( sendto( s_file, buffer, sizeof( REQUESTHEADER ), 0,
		    (struct sockaddr *)&dest, sizeof( dest ) ) == -1 )
			perror( "send" );
		printf( "FillFile: %x send file.\n", fileId );
	}
	else
	{
		request->flags = 0x02 | VERSION | 0x10;
		request->fileId = fileId;
		request->blockSize = 244;

		hole = (HOLEPAIR *)&buffer[sizeof( REQUESTHEADER )];
		h = of[f].firstHole;
		for( i=0; (i<10) && (i<of[f].holes); i++ )
		{
			hole->offset = h->start;
			hole->offset_msb = h->start >> 16;
			hole->length = 1 + (h->finish - h->start);
			h = h->nextHole;
			hole++;
		}

		if( sendto( s_file, buffer,
			    sizeof( REQUESTHEADER ) + (i*sizeof( HOLEPAIR)), 0,
			    (struct sockaddr *)&dest, sizeof( dest ) ) == -1 )
			perror( "send" );
		printf( "FillFile: %x fill %d holes.\n", fileId, i );
		
	}

	free( buffer );
}

void main( int argc, char ** argv )
{
	int n;
	char title[80];

	strcpy( satelliteId , getenv( "SATELLITE" ) );
	if( strlen( satelliteId ) == 0 )
	{
		printf( "SATELLITE environment variable not set\n" );
		exit( 1 );
	}
	strcat( satelliteId, "-11" );

	strcpy( myCall , getenv( "MYCALL" ) );
	if( strlen( myCall ) == 0 )
	{
		printf( "MYCALL environment variable not set\n" );
		exit( 1 );
	}
	MakeAddress( myCall, &src );
	MakeAddress( satelliteId, &dest );

	if( strlen( getenv( "MAXDAYS" ) ) != 0 )
	{
		maxDays = atoi( getenv( "MAXDAYS" ) );
		if( maxDays <= 0 )
			maxDays = 5;
	}

	sprintf( title, "xpb:%s %s", satelliteId, VERSION_STRING );

	xv_init( XV_INIT_ARGS, argc, argv, NULL );

	frame = (Frame)xv_create( (Frame)NULL, FRAME, FRAME_LABEL, title, NULL );

	font = (Xv_Font)xv_create( frame, FONT,
			FONT_FAMILY, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
			FONT_SIZE, 12,
			NULL );

	panel = (Panel)xv_create( frame, PANEL,
			PANEL_LAYOUT, PANEL_HORIZONTAL,
			XV_FONT, font,
			NULL );

	log = (Tty)xv_create( frame, TTY,
			WIN_COLUMNS, 80,
			WIN_ROWS, 10,
			TTY_ARGV, TTY_ARGV_DO_NOT_FORK,
			NULL );

	xv_set( panel, WIN_WIDTH, 800, NULL );

	(void)xv_create( panel, PANEL_BUTTON,
			PANEL_LABEL_STRING, "Quit",
			PANEL_NOTIFY_PROC, QuitButton,
			NULL );

	(void)xv_create( panel, PANEL_BUTTON,
			PANEL_LABEL_STRING, "Fill Directory",
			PANEL_NOTIFY_PROC, FillDirectory,
			NULL );

	(void)xv_create( panel, PANEL_BUTTON,
			PANEL_LABEL_STRING, "Fill File",
			PANEL_NOTIFY_PROC, FillFile,
			NULL );

	fillFileId = (Panel_item)xv_create( panel, PANEL_TEXT,
			PANEL_VALUE_DISPLAY_LENGTH, 8,
			PANEL_VALUE_STORED_LENGTH, 8,
			PANEL_VALUE, "",
			NULL );

	window_fit_height( panel );
	xv_set( log, WIN_BELOW, panel, NULL );
	window_fit_height( frame );

	dup2( (int)xv_get( log, TTY_TTY_FD ), 0 );
	dup2( (int)xv_get( log, TTY_TTY_FD ), 1 );

	/* initialize the open files structure */
	for( n=0; n<MAXFILES; n++ )
	{
		of[n].fileId = 0;
		of[n].file = 0;
		of[n].hdrSeen = 0;
		of[n].fileSize = 0;
		of[n].holes = 0;
		of[n].firstHole = (HOLE *)0;
	}

	/* open up the raw socket to receive all packets */
	s_raw=socket( AF_INET, SOCK_PACKET, htons( 2 ) );
	if( s_raw==-1 )
	{
		perror( "socket" );
		exit( 1 );
	}

	/* open up the AX25 datagram socket to send file requests */
	s_file = socket( AF_AX25, SOCK_RAW, PID_FILE );
	if( s_file == -1 )
	{
		perror( "socket" );
		exit( 1 );
	}

	if( bind( s_file, (struct sockaddr *)&src, sizeof( src ) ) == -1 )
	{
		perror( "bind" );
		printf( "Check that you have run axattach and ifconfig\n" );
		exit( 1 );
	}

	/* open up the AX25 datagram socket to send directory requests */
	s_directory = socket( AF_AX25, SOCK_RAW, PID_DIRECTORY );
	if( s_directory == -1 )
	{
		perror( "socket" );
		exit( 1 );
	}

	if( bind( s_directory, (struct sockaddr *)&src, sizeof( src ) ) == -1 )
	{
		perror( "bind" );
		exit( 1 );
	}

	/* we want to be notified whenever a frame is received on the raw socket */
	notify_set_input_func( (Notify_client)panel, GetFrame, s_raw );

	/* open up a udp socket to receive file fill requests from other apps */
	memset( (char *)&request_addr, 0, sizeof( request_addr ) );
	request_addr.sin_family = AF_INET;
	request_addr.sin_addr.s_addr = INADDR_ANY;
	request_addr.sin_port = htons( 5100 );

	s_request = socket( AF_INET, SOCK_DGRAM, 0 );
	if( s_request == -1 )
	{
		perror( "socket - s_request" );
	}
	else
		notify_set_input_func( (Notify_client)panel, FillRequest, s_request );
		
	if( bind( s_request, (struct sockaddr *)&request_addr, sizeof( request_addr ) ) == -1 )
	{
		perror( "bind - s_request" );
	}
	xv_main_loop( frame );

	exit( 0 );
}
