/******************************************************************************
 *
 *    SUPDATE  --  SHARE compatible update utility.
 *       
 *    SUPDATE copies files from the source directory to the destination
 *    directory if:
 *
 *       1) the file exists in both the source directory and the
 *          destination directory
 *       
 *       and
 *
 *       2) the file in the source directory is newer than the copy
 *          in the destination directory.
 *
 *    What makes SUPDATE different from other utilities of this kind is
 *    that:
 *    
 *       1) it opens the files in SHARE compatible mode so it works
 *          under multitaskers and on networked machines
 *
 *       and
 *
 *       2) if a file (in either the source or the destination directory)
 *          is locked by another process, SUPDATE will try 5 times to lock
 *          the file and complete the update.
 *
 *
 *    Syntax:
 *
 *       SUPDATE [source] destination
 *
 *    The source and destination can be directories or drives.
 *    The source may include a file mask.
 *
 *    If no files were updated, SUPDATE sets ERRORLEVEL to 0
 *
 *    If SUPDATE successfully updates all of the files in the destination
 *    directory that need to be updated SUPDATE sets ERRORLEVEL to 1
 *    and exits.  If one or more files could not be updated (because they
 *    remain locked by another process) SUPDATE sets ERRORLEVEL to 2
 *    on exit.
 *
 *    The value of ERRORLEVEL can be checked in a batch file and used
 *    to control what happens if SUPDATE is unable to successfully update
 *    all of the files.
 *
 ******************************************************************************
*/

/******************************************************************************
 *
 *    SUPDATE.C
 *
 *    Copyright 1989 by Ken Brown
 *
 *    Permission is given to use this code provided that this copyright
 *    message is not removed.
 *
 *    Modified 3/29/1994 Jason Olasky
 *
 ******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <setjmp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/utime.h>
#include <bios.h>
#include <dos.h>
#include <io.h>
#include <fcntl.h>
#include <share.h>
#include <direct.h>
#include <time.h>

#define TRUE   1
#define FALSE  0

#define BUFFER_LENGTH   4096

/*
 *    This structure will be used to build a list of files in
 *    the current directory.
 *
*/

struct FILELIST {
   char              FileName[13];
   time_t            TimeStamp;
   int               UpdateFlag;
   struct FILELIST   *NextFile;
   } *FileList, *FilePtr;

char  DestDir[81],
      FullPath[81],
      FileBuffer[BUFFER_LENGTH];

char  *DefFileMask = "*.*";

int newdrive, curdrive, UpdateCount = 0;
char curdir[_MAX_DIR], drive [_MAX_DRIVE], dir[_MAX_DIR],
   fname[_MAX_FNAME], ext[_MAX_EXT], newdir[_MAX_DIR];

void Usage (void);
int CopyFile (char *Source, char *Dest);
int LockFile (char *FileName, int Oflag, int Shflag);
void BuildFileList (char *FileMask);
void Sleep (int Seconds);



void main (int argc, char *argv[])

{

int FoundLockedFile, Counter;
char FileMask[13];
struct stat StatBuffer;
struct utimbuf TimeBuffer;

printf ("SUPDATE, SHARE compatible update utility\n"
        "Copyright 1989, Ken Brown\n"
        "Modified 1994, Jason Olasky\n\n");

if (_osmajor < 3)
   {
   printf ("Requires DOS 3.0 or later\n");
   exit (2);
   }

/* At least one argument, the destination directory, is required */
if (argc < 2)
   Usage ();

/* If only one argument is given use the default file mask (*.*) */
if (argc == 2)
   {
   strcpy (FileMask, DefFileMask);

   /* The directory string can be no more than 66 chars in length */
   if (strlen (argv[1]) > 66)
      Usage ();
   strcpy (DestDir, argv[1]);
   }
else
   {
   curdrive = _getdrive ();
   _getdcwd (curdrive, curdir, _MAX_DIR);
   _splitpath (argv[1], drive, dir, fname, ext);

   if (drive[0] != '\0')
      {
      if (_chdrive (toupper (drive[0]) - 'A' +1) != 0)
         {
         printf ("bad drive: %c\n", toupper (argv [1][0]));
         exit (2);
         }
      if (dir[0] == '\0')
         strcpy (dir, "\\");
      if ((strlen (fname)) > 0 & (strlen (ext) == 0))
         {
         strcat (dir, fname);
         fname[0] = '\0';
         }
      strcpy (newdir, drive);
      strcat (newdir, dir);
      if ((newdir[strlen (newdir) - 1] == '\\') & (strlen (newdir) > 3))
         newdir[strlen (newdir) -1] = '\0';
      if (_chdir (newdir) != 0)
         {
         printf ("bad dir: %s\n", newdir );
         exit (2);
         }
      }

   strcpy (FileMask, fname);
   strcat (FileMask, ext);
   if (FileMask[0] == '\0')
      strcpy (FileMask, DefFileMask);

   /* The directory string can be no more than 66 chars in length */
   if (strlen (argv[2]) > 66)
      Usage ();
   strcpy (DestDir, argv[2]);
   }

/* Add a trailing backslash to the destination directory if one
 * is not already present. */

if (DestDir[strlen (DestDir) - 1] != '\\')
   strcat (DestDir,"\\");

/* Build a linked list of all files in the current directory matching
 * the FileMask.  If no matching files are found then BuildFileList()
 * will print an message and call exit(). */

BuildFileList (FileMask);

/* Traverse the list of files in the current directory looking for
 * copies of these files in the destination directory.  If a copy
 * is found in the destination directory check the timestamp to see
 * if it needs to be updated. */

FilePtr = FileList;
while (FilePtr != NULL)
   {
   strcpy (FullPath, DestDir);
   strcat (FullPath, FilePtr->FileName);

   /* See if the file exists in the destination directory. */
   if (stat (FullPath, &StatBuffer) != 0)
      FilePtr->UpdateFlag = FALSE;
   else
      {

      /* If so see if the copy in the current directory is newer. */
      if (FilePtr->TimeStamp > StatBuffer.st_mtime)
         {
         FilePtr->UpdateFlag = TRUE;
         ++UpdateCount;
         }
      else
         FilePtr->UpdateFlag = FALSE;
      }
   FilePtr = FilePtr->NextFile;
   }

/* At this point we have a list of all of the files which need to
 * be updated (all files for which UpdateFlag is set to TRUE).
 * Traverse the list again and try to copy the files to the
 * destination directory.  CopyFile() will print a message if the
 * file is currently locked and return a non-zero value to let
 * us know that the copy failed. */

FilePtr = FileList;
while (FilePtr != NULL)
   {
   if (FilePtr->UpdateFlag == TRUE)
      {
      strcpy (FullPath, DestDir);
      strcat (FullPath, FilePtr->FileName);
      if (CopyFile (FilePtr->FileName, FullPath) == 0)
         {

         /* Set the timestamp in the destination directory to
          * the timestamp in the current directory. */
         TimeBuffer.modtime = FilePtr->TimeStamp;
         TimeBuffer.actime = FilePtr->TimeStamp;
         utime (FullPath, &TimeBuffer);
         FilePtr->UpdateFlag = FALSE;
         }
      }
   FilePtr = FilePtr->NextFile;
   }


/* If all of the files were copied successfully on the first
 * try we are done!  If not then loop through the list 5 (an arbitrary
 * number) times to allow time for other processes to release
 * their locks on the files. */

for (Counter = 0; Counter < 5; Counter++)
   {

   /* Set a flag which will tell us if we need to keep trying */
   FoundLockedFile = FALSE;

   /* Traverse the list */
   FilePtr = FileList;
   while (FilePtr != NULL)
      {
      if (FilePtr->UpdateFlag == TRUE)
         {
         strcpy (FullPath, DestDir);
         strcat (FullPath, FilePtr->FileName);

         /* Try the copy */
         if (CopyFile (FilePtr->FileName, FullPath) == 0)
            {
            TimeBuffer.modtime = FilePtr->TimeStamp;
            TimeBuffer.actime = FilePtr->TimeStamp;
            utime (FullPath, &TimeBuffer);
            FilePtr->UpdateFlag = FALSE;
            }
         else
            FoundLockedFile = TRUE;
         }
      FilePtr = FilePtr->NextFile;
      }

   /* If there are still locked files sleep for a few seconds to
    * give the other process time to release its locks. */

   if (FoundLockedFile == TRUE)
      Sleep (5);
   else
      break;
   }

_chdrive (curdrive);
_chdir (curdir);
printf ("SUPDATE, %d files updated\n", UpdateCount);

/* Set the proper ERRORLEVEL value to signal success or failure. */
if (FoundLockedFile == TRUE)
   exit (2);
else if (UpdateCount > 0)
   exit (1);
else
   exit (0);
}



void Usage (void)

{
printf ("syntax: supdate [source] dest\n");
exit (2);
}



/*  SHARE compatible file copy routine.  Locks the source file in
 *  DENY_WRITE mode for the duration of the copy.  Locks the destination
 *  in DENY_READ_WRITE mode so that no one else can touch it until
 *  the copy is complete. */

int CopyFile (char *Source, char *Dest)

{
int   CharsRead, CharsWritten, FileHandle1, FileHandle2;

if ((FileHandle1 = LockFile (Source, O_RDONLY|O_BINARY, SH_DENYWR)) == -1)
   {
   printf ("File %s in use by another user, cannot be copied\n",Source);
   return -1;
   }

/*
 * This call to LockFile() assumes that the file already exists and
 * that it is to be overwritten (O_TRUNC) by the new file.
 */

if ((FileHandle2 = LockFile (Dest, O_WRONLY|O_BINARY|O_TRUNC, SH_DENYRW)) == -1)
   {
   /* Be sure to unlock the source if the destination is currently locked */
   close (FileHandle1);
   printf ("Destination file %s in use by another user,\ncannot be overwritten\n",Dest);
   return -1;
   }

printf ("Copying %s to %s\n", Source, Dest);

/* Copy the file.  If the destination disk does not have space
 * to hold the file report that the disk is full and exit.  I
 * would prefer to preserve the old copy if this happens but
 * that code is left as an exercise for the reader <grin>. */

while ((CharsRead = read (FileHandle1, FileBuffer, BUFFER_LENGTH)) != 0)
   {
   CharsWritten = write (FileHandle2, FileBuffer, CharsRead);
   if (CharsWritten != CharsRead)
   {
      printf ("Destination disk is full\n");
      close (FileHandle1);
      close (FileHandle2);
      exit (1);
      }
   }

close (FileHandle1);
close (FileHandle2);

/* Signal success */

return 0;
}



int LockFile (char *FileName, int Oflag, int Shflag)

{
int   FileHandle;

if ((FileHandle = sopen (FileName, Oflag, Shflag)) >= 0)
   return FileHandle;
else
   return -1;
}



/* BuildFileList() uses the _dos_findfirst() and _dos_findnext() routines
 * to build a linked list of all files in the current directory which
 * match FileMask.
 *
 * _dos_findfirst() is from the Microsoft C library.  There is a Turbo C
 * equivalent but I don't know the name. */

void BuildFileList (char *FileMask)
{
struct find_t FileInfo;
struct stat StatBuffer;

if (_dos_findfirst (FileMask, _A_NORMAL|_A_RDONLY, &FileInfo) == 0)
   {
   do
      {
      if (FileList == NULL)
         {
         FileList = calloc (sizeof (struct FILELIST), 1);

         /* This is not likely to happen but you never know. */
         if (FileList == NULL)
            {
            printf ("Out of memory\n");
            exit (2);
            }
         FilePtr = FileList;
         }
      else
         {
         FilePtr->NextFile = calloc (sizeof (struct FILELIST), 1);
         if (FilePtr->NextFile == NULL)
            {
            printf ("Out of memory\n");
            exit (2);
            }
         FilePtr = FilePtr->NextFile;
         }

      /* Make a copy of the file name from the find_t structure. */
      strcpy (FilePtr->FileName, FileInfo.name);

      /* This requires an explanation.  Yes I know that the find_t
       * structure contains the timestamp.  The problem is that we
       * are going to use the stat() call later to find the timestamp
       * of the file in the destination directory.  stat() returns
       * the time in seconds since 1/1/1970 (the UNIX way).
       * The find_t structure contains the date and time in the packed
       * DOS format (as it is stored in the directory).  Rather than
       * do the conversion from one format to the other we just waste
       * a system call here and get the timestamp in the stat()
       * format. */

      if (stat (FilePtr->FileName, &StatBuffer) != 0)
         {
         printf ("Internal error\n");
         exit (2);
         }
      else
         FilePtr->TimeStamp = StatBuffer.st_mtime;

      } while (_dos_findnext (&FileInfo) == 0);
   }
else
   {
   printf ("No matching files found\n");

   /* Maybe this should be exit (2) but no matching files means no
    * work needs to be done.  This is not the same as being unable
    * to copy files which need to be updated but are locked. */
   exit (0);
   }
}



void Sleep (int Seconds)

{
long  StartTime, CurrentTime, EndTime;

time (&StartTime);
CurrentTime = StartTime;
EndTime = StartTime + (long) Seconds;

while (CurrentTime < EndTime)
   time (&CurrentTime);
}
