/******************************************************************************
* Module    :   Post --- Create a multi-file, uuencoded USENET News posting
*               from a binary file.
*
* Author    :   John W. M. Stevens
*
* Notes     :   The Subject: line format has been designed to be easily
*               parsed.  This format is the most efficient to use.
******************************************************************************/

#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>

#include    "post.h"

/*********************************** Globals *********************************/
int     MsDosFileNms = 1;
char    *Version = "POST V2.0.1";
/*****************************************************************************/

/*-----------------------------------------------------------------------------
| Routine   :   ShowSynopsis() --- Show a summary of the command line
|               syntax.
-----------------------------------------------------------------------------*/

static  char    *Synopsis[] =
{
"--- Create Multi-Part Postings from a Binary File.\n\n",
"Options:\n",
"    -c[-]           Calculate a 32 bit CCITT CRC for the binary file.\n",
"    -d <file>       Name of description file.\n",
"    -e \"command\"    Execute a command for each part file.\n",
"    -f[-]           Modify file names to be 8 and 3.\n",
"    -h <file>       Name of USENET header information file.\n",
"    -i \"c=string\"   Macro string value.\n",
"    -r              Repost flag.  This is an attempted repost.\n",
"    -t \"string\"     For pictures only, type of picture.\n",
"    -v              Show version number and quit.\n",
NULL
};

static
void    ShowSynopsis(void)
{
    register    int     i;

    printf("%s ", Version);
    for (i = 0; Synopsis[i]; i++)
        printf( Synopsis[i] );
}

/*-----------------------------------------------------------------------------
| Routine   :   CmdLineParse() --- Parse the command line.
|
| Inputs    :   argc    - Number of command line arguments.
|               argv    - List of command line argument strings.
|               SwList  - Switch list.
-----------------------------------------------------------------------------*/

static
void    CmdLineParse(int        argc,
                     char       **argv,
                     char       *SwList)
{
    register    int         i;
     auto        char        c;
    auto        char        *ArgPtr;
    auto        char        *SwPtr;
    auto        char        SwChar;
    auto        BIN_DATA    BinData;
    auto        HDR_TYPE    *HdrPtr;
    auto        FILE        *HdrFl;
    auto        FILE        *DescFl;
    static      char        CmdBfr[BFR_SZ];
    extern      HDR_TYPE    HdrLines[];

    /*  Initialize the binary data structure.   */
    *BinData.FlName    = '\0';
    *BinData.PictType  = '\0';
    BinData.RepostFlag = 0;
    BinData.CalcCRC    = 1;
    *CmdBfr = '\0';

    /*  Scan entire command line.   */
    DescFl = stdin;
    HdrFl = NULL;
    for (i = 1; i < argc; i++)
    {
        /*  Get a pointer to the argument.  */
        ArgPtr = argv[i];

        /*  Is this a switch?   */
        if (*ArgPtr == '-')
        {
            /*  Get switch character.   */
            if ((SwChar = *++ArgPtr) == '\0')
            {
                /*  There is no character after the switch marker,
                *   so declare an error.
                */
                ShowSynopsis();
                exit( 1 );
            }
            else if ((SwPtr = strchr(SwList, SwChar)) == NULL)
            {
                /*  Error, this is evidently an illegal switch
                *   character.
                */
                ShowSynopsis();
                exit( 1 );
            }
            else if (SwPtr[1] == '>')
            {
                /*  Get parameter string.  Parameter string can
                *   follow immediately after the switch character,
                *   or it can be the next command line string.
                */
                if (*++ArgPtr == '\0')
                {
                    /*  Next command line parameter is switch
                    *   parameter string.
                    */
                    if (i + 1 < argc)
                        ArgPtr = argv[++i];
                    else
                    {
                        ShowSynopsis();
                        exit( 1 );
                    }
                }
            }
            else
                ArgPtr++;
        }
        else
            SwChar = ' ';

        /*  Have argument processed.    */
        switch ( SwChar )
        {
        case 'c':
            BinData.CalcCRC = (*ArgPtr != '-');
            break;
        case 'd':
            if (*ArgPtr == '-')
                DescFl = stdin;
            else if ((DescFl = fopen(ArgPtr, TXT_READ)) == NULL)
            {
                fprintf(stderr,
                        "%s %d : Error - Could not open file '%s' for "
                        "reading.\n",
                        __FILE__,
                        __LINE__,
                        ArgPtr);
                exit( 1 );
            }
            break;
        case 'e':
            /*  Get the string to execute.  */
            if ( *ArgPtr )
                strcpy(CmdBfr, ArgPtr);
            else
            {
                fprintf(stderr,
                        "%s %d : Error - Missing command string to execute.\n",
                        __FILE__,
                        __LINE__);
                exit( 1 );
            }
            break;
        case 'f':
            MsDosFileNms = (*ArgPtr != '-');
            break;
        case 'h':
            if (*ArgPtr == '-')
                HdrFl = NULL;
            else if ((HdrFl = fopen(ArgPtr, TXT_READ)) == NULL)
            {
                fprintf(stderr,
                        "%s %d : Error - Could not open file '%s' for "
                        "reading.\n",
                        __FILE__,
                        __LINE__,
                        ArgPtr);
                exit( 1 );
            }
            break;
        case 'i':
            /*  Check for bad macro string. */
            if (ArgPtr[0] == '\0' ||
                ArgPtr[1] != '='  ||
                ArgPtr[2] == '\0')
            {
                fprintf(stderr,
                        "%s %d : Error - bad macro string '%s'\n"
                        __FILE__,
                        __LINE__,
                         ArgPtr);
                exit( 1 );
            }

            /*  Get the Macro tag from the first character. */
            c = *ArgPtr;

            /*  Seek for appropriate macro tag in list, and
            *   if found, copy data into buffer.
            */
            for (HdrPtr = HdrLines; HdrPtr->MacroTag; HdrPtr++)
            {
                /*  Found it yet?   */
                if (HdrPtr->MacroTag == c)
                {
                    strcpy(HdrPtr->Bfr, ArgPtr + 2);
                    break;
                }
            }
            break;
        case 'r':
            BinData.RepostFlag = (*ArgPtr != '-');
            break;
        case 't':
            /*  Get the string to execute.  */
            if ( *ArgPtr )
                strcpy(BinData.PictType, ArgPtr);
            else
            {
                fprintf(stderr,
                        "%s %d : Error - Missing picture type string.\n",
                        __FILE__,
                        __LINE__);
                exit( 1 );
            }
            break;
        case 'v':
            printf("%s\n", Version);
            exit( 0 );
        case ' ':
            /*  UUencode and split. */
            strcpy(BinData.FlName, ArgPtr);
            Split(HdrFl, DescFl, &BinData, CmdBfr);

            /*  Initialize the binary data structure.   */
            *BinData.FlName    = '\0';
            *BinData.PictType  = '\0';
            break;
        default:
            ShowSynopsis();
            exit( 1 );
        }
    }
}

int     main(int    argc,
             char   **argv)
{
    /*  Parse command line parameters.  */
    if (argc >= 2)
        CmdLineParse(argc, argv, "cd>e>fh>i>rt>v");
    else
        ShowSynopsis();
    return( 0 );
}
