/******************************************************************************
* Module    :   NNTP Client for posting to an NNTP news server.
*
* Author    :   John W. M. Stevens
******************************************************************************/

#include    <stdio.h>
#include    <stdlib.h>
#include    <fcntl.h>
#include    <string.h>
#include    <time.h>

#include    <sys/errno.h>
#include    <sys/param.h>
#include    <sys/file.h>
#include    <sys/ioctl.h>
#include    <sys/socket.h>
#include    <sys/un.h>
#include    <netinet/in.h>
#include    <netdb.h>

#define MAX_BFR     513

#define YES 1
#define NO  0

typedef enum    {
    ST_LINE,
    ST_NORM,
    ST_CR
} STATES;

FILE    *ErrFile = stderr;
char    *Version = "SNDNEWS V1.0.0";
extern int  errno;

char    Bfr[MAX_BFR];

/*-----------------------------------------------------------------------------
| Routine   :   WrtSrvr() --- Write a buffer to the server.
|
| Inputs    :   SockDesc    - Socket descriptor.
|               Bfr         - Buffer to read into.
|               Len         - Length of message in buffer.
-----------------------------------------------------------------------------*/

void    WrtSrvr(int     SockDesc,
                char    *Bfr,
                int     Len)
{
    auto    int     ret;

    /*  Write buffer to server. */
    if ((ret = write(SockDesc, Bfr, Len)) <= 0)
    {
        fprintf(stderr,
                "%s %d - Error in writing to news server.\n",
                __FILE__,
                __LINE__);
        fprintf(stderr,
                "\tReturn = %d, Error number = %d\n",
                ret,
                errno);
        exit( 1 );
    }
}

/*-----------------------------------------------------------------------------
| Routine   :   RdSrvr() --- Read a buffer from the server.
|
| Inputs    :   SockDesc    - Socket descriptor.
|               Bfr         - Buffer to read into.
|               MaxLen      - Maximum length of buffer.
-----------------------------------------------------------------------------*/

int     RdSrvr(int  SockDesc,
               char *Bfr,
               int  MaxLen)
{
    auto        int     ret;

    /*  Get the buffer full.    */
    if ((ret = read(SockDesc, Bfr, MaxLen - 1)) < 0)
    {
        fprintf(stderr,
                "%s %d - Error in reading from news server.\n",
                __FILE__,
                __LINE__);
        fprintf(stderr,
                "\tReturn = %d, Error number = %d\n",
                ret,
                errno);
        exit( 1 );
    }
    Bfr[ret] = '\0';
    return( ret );
}

/*-----------------------------------------------------------------------------
| Routine   :   GetPortNo() --- This routine converts a character string
|               to a port number.  It looks up the service by name, and if
|               there is none, then it converts the string to a number with
|               scanf.
|
| Inputs    :   PortName    - Name of port to convert.
|
| Returns   :   Returns the TCP port number for this server.
-----------------------------------------------------------------------------*/

int GetPortNo(char  *PortName)
{
    auto    int     PortNo;
    struct  servent *Service;

    /*  If no port name string, return no port number.  */
    if (PortName == NULL)
        return 0;

    /*  Get server by name and make sure that the protocol
    *   matches as well.
    */
    Service = getservbyname(PortName, "tcp");

    /*  Return either the port number, or 0 for no port number. */
    if (Service != NULL)
        return( Service->s_port );
    else if (sscanf(PortName, "%i", &PortNo) != 1)
        return 0;
    else
        return( htons( PortNo ) );
}

/*-----------------------------------------------------------------------------
| Routine   :   GetHostAddr() --- Convert the host name to an address.
|
| Inputs    :   HostName    - Host name to convert.
|               Addr        - Host address.
|
| Returns   :   Returns 0 for error, 1 for address being returned.
-----------------------------------------------------------------------------*/

int GetHostAddr(char            *HostName,
                struct  in_addr *Addr)
{
    auto        int     len;
    struct      hostent *Host;
    auto        int     count;
    unsigned    int     a1, a2, a3, a4;

    /*  Get a pointer to the host name data.    */
    Host = gethostbyname( HostName );
    if (Host != NULL)
        bcopy(Host->h_addr, Addr, Host->h_length);
    else
    {
        /*  Convert the string representation of the internet
        *   address into bytes.
        */
        count = sscanf(HostName,
                       "%i.%i.%i.%i%n",
                       &a1, &a2, &a3, &a4, &len);
        if (count != 4 || HostName[len] != 0)
            return( 0 );

        /*  Insert the bytes of the internet address into the
        *   return address structure.  I suspect that this is highly
        *   machine specific code.
        */
        Addr->S_un.S_un_b.s_b1 = a1;
        Addr->S_un.S_un_b.s_b2 = a2;
        Addr->S_un.S_un_b.s_b3 = a3;
        Addr->S_un.S_un_b.s_b4 = a4;
    }

    /*  Return no errors occurred.  */
    return( 1 );
}

/*-----------------------------------------------------------------------------
| Routine   :   SetupSocket() --- Set up the socket.
|
| Inputs    :   HostName    - Name of host.
|               PortName    - Name of port.
-----------------------------------------------------------------------------*/

int SetupSocket(char    *HostName,
                char    *PortName)
{
    auto    int         SockDesc;   /*  Socket handle (descriptor).     */
    struct  sockaddr    SockAddr;   /*  Socket address structure.       */
    struct  sockaddr_in *InAddr;    /*  Internet address structure.     */
    struct  in_addr     *AdrPtr;

    /*  Get the address of the server.  */
    InAddr=(struct sockaddr_in *) &SockAddr;
    InAddr->sin_family = AF_INET;
    if (! GetHostAddr(HostName, &InAddr->sin_addr))
    {
        fprintf(stderr,
                "%s %d : Error - Could not convert ",
                __FILE__,
                __LINE__);
        fprintf(stderr,
                "'%s' to a host address.\n",
                HostName);
        exit( 1 );
    }
    AdrPtr = (struct in_addr *) (&InAddr->sin_addr);
    fprintf(stderr,
            "Host '%s', address = %u.%u.%u.%u\n",
            HostName,
            AdrPtr->S_un.S_un_b.s_b1,
            AdrPtr->S_un.S_un_b.s_b2,
            AdrPtr->S_un.S_un_b.s_b3,
            AdrPtr->S_un.S_un_b.s_b4);

    /*  Convert a service name to a port number.    */
    InAddr->sin_port = GetPortNo( PortName );
    if (InAddr->sin_port == 0)
    {
        fprintf(stderr,
                "%s %d : Error - bogus port number '%s'.\n",
                __FILE__,
                __LINE__,
                PortName);
        exit( 1 );
    }
    fprintf(stderr,
            "NNTP port number = %d\n",
            InAddr->sin_port);

    /*  Attempt to get a socket descriptor. */
    SockDesc = socket(AF_INET, SOCK_STREAM, 0);
    if (SockDesc < 0)
    {
        perror("opening stream socket");
        exit( 1 );
    }

    /*  Connect to the server.  */
    if (connect(SockDesc , &SockAddr, sizeof( SockAddr )) < 0)
    {
        /*  Close the old socket descriptor.    */
        perror("connecting");
        fprintf(stderr,
                "%s %d : Error - could not connect to news server.\n",
                __FILE__,
                __LINE__);
        exit( 1 );
    }

    /*  Return the socket descriptor.   */
    return( SockDesc );
}

/*-----------------------------------------------------------------------------
| Routine   :   PostArt() --- Post and article.
-----------------------------------------------------------------------------*/

void    PostArt(int     SockDesc,
                char    *FileNm)
{
    register    int     i;
    auto        FILE    *ArtFp;
     auto        int     Status;

    /*  Open the file.  */
    if ((ArtFp = fopen(FileNm, "r")) == NULL)
    {
        fprintf(stderr,
                "%s %d : Error - could not open file '%s' to read.\n",
                __FILE__,
                __LINE__,
                FileNm);
        return;
    }

    /*  Send posting command.   */
    sprintf(Bfr, "POST\r\n");
    WrtSrvr(SockDesc, Bfr, strlen( Bfr ));

    /*  Get server response.    */
    RdSrvr(SockDesc, Bfr, MAX_BFR);
    fprintf(stderr, "%s\n", Bfr);
    sscanf(Bfr, "%d", &Status);

    /*  Check status.   */
    switch ( Status )
    {
    case 440:
        fprintf(stderr,
                "%s %d : Error - posting not allowed.\n",
                __FILE__,
                __LINE__);
        break;
    case 340:
        break;
    default:
        fprintf(stderr,
                "%s %d : Error - %d.\n",
                __FILE__,
                __LINE__,
                Status);
    }

    /*  Get lines from the file.    */
    while ( fgets(Bfr, MAX_BFR - 2, ArtFp) )
    {
        /*  Check for special case and fix. */
        if (Bfr[0] == '.')
        {
            /*  Move string up to make room.    */
            for (i = strlen( Bfr ) + 1;
                 i > 0;
                 i--
                )
                Bfr[i] = Bfr[i - 1];
            Bfr[0] = '.';
        }

        /*  Strip line termination and then put it back.    */
        for (i = strlen( Bfr );
             i > 0 && (Bfr[i - 1] == '\r' || Bfr[i - 1] == '\n');
             i--
            )
            ;
        Bfr[i++] = '\r';
        Bfr[i++] = '\n';
        Bfr[i] = '\0';

         /*  Send line.  */
        WrtSrvr(SockDesc, Bfr, i);
    }

    /*  Close article file. */
    fclose( ArtFp );

    /*  Send end of article line indication.    */
    strcpy(Bfr, ".\r\n");
    WrtSrvr(SockDesc, Bfr, strlen( Bfr ));

    /*  Get server response.    */
    RdSrvr(SockDesc, Bfr, MAX_BFR);
    fprintf(stderr, "%s\n", Bfr);
    sscanf(Bfr, "%d", &Status);

    /*  Check status.   */
    switch ( Status )
    {
    case 441:
        fprintf(stderr,
                "%s %d : Error - Posting failed.\n",
                __FILE__,
                __LINE__);
        break;
    case 240:
        break;
    default:
        fprintf(stderr,
                "%s %d : Error - %d.\n",
                __FILE__,
                __LINE__,
                Status);
    }
}

void    main(int    argc,
             char   **argv)
{
    auto        int             SockDesc;
    auto        FILE            *KillFp;

    /*  Check the argument count.   */
    if (argc < 3)
    {
        /*  Bad argument count, give synopsis.  */
        fprintf(stderr,
                "Usage : %s <server> <file to post>\n",
                argv[0]);
        exit( 1 );
    }

    /*  Attempt to set up the socket.   */
    SockDesc = SetupSocket(argv[1], "nntp");

    /*  Read the sign on message and print it.  */
    fprintf(stderr, "%s\n", Version);
    RdSrvr(SockDesc, Bfr, MAX_BFR);
    fprintf(stderr, "%s", Bfr);

    /*  Post the file.    */
    PostArt(SockDesc, argv[2]);

    /*  Quit server.    */
    strcpy(Bfr, "QUIT\r\n");
    WrtSrvr(SockDesc, Bfr, strlen( Bfr ));
    RdSrvr(SockDesc, Bfr, MAX_BFR);
    fprintf(stderr, "%s", Bfr);

    /*  Close the socket descriptor and exit with no errors.    */
    close( SockDesc );
    exit( 0 );
}
