/******************************************************************************
* Module    :   File Names --- File name utilities.
*
* Author    :   John W. M. Stevens
******************************************************************************/

#include    <stdio.h>
#include    <stdlib.h>
#include    <ctype.h>
#include    <string.h>

#include    "post.h"
#include    "sets.h"

/*  Define maximum base and extension file name string lengths.
*   Currently, as MS-DOS is the most restrictive, names will tested
*   for fit into MS-DOS length restrictions.
*/
#define MAX_FILE            8
#define MAX_EXT             3
#define EXT_SEP_CHAR        '.'

/*  Define path and volume delimiting characters for extracting a base
*   name from a path name for the executing system.
*/
#if defined(SYSTEM_MS_DOS)

#define PATH_SEP            '\\'
#define PATH_SEP2           '/'
#define VOL_SEP             ':'

#elif   defined(SYSTEM_UNIX)

#define PATH_SEP            '/'
#define PATH_SEP2           '\0'
#define VOL_SEP             '\0'

#elif   defined(SYSTEM_AMIGA)

#define PATH_SEP            '/'
#define PATH_SEP2           '\0'
#define VOL_SEP             ':'

#endif

/*-----------------------------------------------------------------------------
| Routine   :   StripExt() --- Strip off the extension.
|
| Inputs    :   FileName    - File name to remove extension from.
| Outputs   :   OutFlNm     - File name with extension removed.
-----------------------------------------------------------------------------*/

void    StripExt(char   *FlName,
                 char   *OutFlNm)
{
    register    int     i;

    /*  Check characters in string. */
    for (i = strlen( FlName ); i; i--)
    {
        /*  Look for first occurence of a character that indicates
        *   the begining of the file name.
        */
        if (FlName[i - 1] == PATH_SEP  ||
            FlName[i - 1] == PATH_SEP2 ||
            FlName[i - 1] == VOL_SEP
           )
        {
            break;
        }
    }

    /*  Now scan back forward to find the first occurence of the
    *   EXT_SEP_CHAR in the string.
    */
    strcpy(OutFlNm, FlName + i);
    for (i = 0;
         OutFlNm[i] && OutFlNm[i] != EXT_SEP_CHAR;
         i++)
        ;
    OutFlNm[i] = '\0';
}

/*-----------------------------------------------------------------------------
| Routine   :   BaseName() --- Get just the file name.
|
| Inputs    :   FileName    - Full path name.
| Outputs   :   BaseName    - Base file name.
-----------------------------------------------------------------------------*/

void    BaseName(char   *FlName,
                 char   *BaseName)
{
    register    int     i;

    /*  Check characters in string. */
    for (i = strlen( FlName ); i; i--)
        if (FlName[i - 1] == PATH_SEP  ||
            FlName[i - 1] == PATH_SEP2 ||
            FlName[i - 1] == VOL_SEP
           )
        {
            strcpy(BaseName, FlName + i);
            return;
        }

    /*  Copy whole file name to base file name string.  */
    strcpy(BaseName, FlName);
}

/*-----------------------------------------------------------------------------
| Routine   :   ChkFlNmLen() --- Check the file name length.
|
| Inputs    :   FileName    - File name.
-----------------------------------------------------------------------------*/

static
int     ChkFlNmLen(char     *FileNm)
{
    register    int     i;

    /*  Count file characters.  */
    for (i = 0;
         FileNm[i] && FileNm[i] != EXT_SEP_CHAR;
         i++)
        ;

    /*  Return length of part before extension separator character.   */
    return( i );
}

/*-----------------------------------------------------------------------------
| Routine   :   MsDosFlNm() --- Convert to lower case and make sure that this
|               file name conforms to MS-DOS restrictions/limitations and
|               USENET posting standards.
|
| Inputs    :   FlName  - File name.
|               Exten   - Pointer to extension string.
| Outputs   :   OutName - Modified file name.
-----------------------------------------------------------------------------*/

int     ModifyFlNm(char         *FlName,
                   BIN_TYPES    FileType,
                   char         *Exten,
                   char         *OutName)
{
    register    int     i;
    auto        int     len;
    auto        int     ChopFlag;
    auto        char    *cp;
    auto        char    *tp;
    auto        char    c;
    auto        SET     StandSet;
    auto        SET     ThrowSet;
    auto        SET     NoSet;
    auto        SET     AlphaSet;

    /*  Strings for character sets. */
    static      char    *Stand = "a-z0-9_.-";
    static      char    *Throw = "aeiouy_-";
    static      char    *Numbers = "0-9";
    static      char    *Alpha = "a-z";

    /*  Create character sets.  */
    tp = Stand;
    CrtSet(&tp, StandSet);
    tp = Throw;
    CrtSet(&tp, ThrowSet);
    tp = Numbers;
    CrtSet(&tp, NoSet);
    tp = Alpha;
    CrtSet(&tp, AlphaSet);

    /*  Convert string to lower case and strip out illegal characters.  */
    for (cp = OutName, tp = FlName; *tp; tp++)
    {
        /*  First, convert to lower case.   */
        c = (char) tolower( *tp );

        /*  Next, check for standards conformity.   */
        if ( InSet(StandSet, c) )
            *cp++ = c;
    }
    *cp = *tp;

    /*  If we have more than 8 characters in the file name, then shorten
    *   the file name.
    */
    if ((len = ChkFlNmLen( OutName )) > 8)
    {
        /*  Shorten by removing vowels and separator characters.    */
        for (tp = cp = OutName, i = len;
             *tp && *tp != EXT_SEP_CHAR && i > 8;
             tp++)
        {
            /*  If this is not a throw away character, save it back
            *   in the string.
            */
            if (! InSet(ThrowSet, *tp))
            {
                *cp++ = *tp;
                continue;
            }
            else if (tp == OutName)
            {
                *cp++ = *tp;
                continue;
            }

            /*  Decrement the file name size.   */
            i--;
        }
        strcpy(cp, tp);
    }
    else
        goto ShortEnough;

    /*  If we still have more than 8 characters in the file name, chop
    *   off leading zeros in any numeric strings in the file name.
    */
    if ((len = ChkFlNmLen( OutName )) > 8)
    {
        /*  Loop through string.    */
        for (tp = cp = OutName, i = len;
             *tp && *tp != EXT_SEP_CHAR && i > 8;
             tp++)
        {
            /*  If this is a numeric sub-string, strip any leading
            *   zeros.
            */
            if (tp[1] && tp[1] != '.' &&
                InSet(NoSet, *tp) && InSet(NoSet, tp[1]) &&
                *tp == '0')
            {
                i--;
                continue;
            }

            /*  Copy character in.  */
            *cp++ = *tp;
        }
        strcpy(cp, tp);
    }
    else
        goto ShortEnough;

    /*  If we still have more than 8 characters in the file name, chop
    *   alphabetic strings short by throwing away excess characters.
    */
    while ((len = ChkFlNmLen( OutName )) > 8)
    {
        /*  Loop through string.    */
        for (tp = cp = OutName, ChopFlag = 0, i = len;
             *tp && *tp != EXT_SEP_CHAR && i > 8;
             tp++)
        {
            /*  Do not throw away numeric characters at this point. */
            if ( InSet(NoSet, *tp) )
            {
                *cp++ = *tp;
                continue;
            }

            /*  Throw away the trailing character in an alphabetic
            *   substring.
            */
            if (*tp && tp[1] != EXT_SEP_CHAR &&
                InSet(AlphaSet, *tp) && InSet(AlphaSet, tp[1]))
            {
                *cp++ = *tp;
                continue;
            }

            /*  Yup, we are chopping out a character.   */
            i--;
            ChopFlag = 1;
        }
        strcpy(cp, tp);

        /*  If not even one character is chopped out in a pass
        *   through the file name list, drop out and issue an
        *   error.
        */
        if (ChopFlag == 0)
        {
            fprintf(stderr,
                    "%s %d : Error - Could not modify file name to be "
                    "MS-DOS conformant.\n",
                    __FILE__,
                    __LINE__,
                    FlName);
            return( 1 );
        }
    }

    /*  Done with file name, now append extension.  */
ShortEnough:
    for (tp = OutName;
         *tp && *tp != EXT_SEP_CHAR;
         tp++)
        ;
    if (*tp == EXT_SEP_CHAR)
    {
        /*  If the file type is recognizable, then append conformant
        *   type, otherwise, chop extension to three characters.
        */
        if (FileType == UNKNOWN_TYPE)
        {
            /*  Move four characters across (including '.') and we
            *   are done.
            */
            for (i = 0, tp++;
                 *tp && *tp != EXT_SEP_CHAR && i < MAX_EXT;
                 i++, tp++)
                ;
            *tp = '\0';
        }
        else
            strcpy(tp, Exten);
    }

    /*  Return no error.    */
    return( 0 );
}
