**************************************************************************
* STRLIB.S | A library of character and string handling routines.	 *
* ---------------------------------------------------------------------- *
* Written by Robert Birmingham [73637,1120] and Bill Aycock [76703,4061] *
* LTOA routine written by Carl Barron [75066,3204]
* Release date: still in progress as of 10/11/90...			 *
**************************************************************************

*****************************
*atol: ascii (decimal) string to unsigned long
*in:	a0.l=pointer to ascii string
*out:	d0.l=unsigned long
*calls: find_any, instr, strcmp
*stack use: 44 bytes
*notes: Conversion starts at first decimal digit in string and ends at the
*  first non-decimal digit, up to 10 digits. If the converted value would
*  be >$ffffffff, routine returns a zero and the overflow (V) flag is set.
*****************************
atol:	movem.l	d1-d2/a0-a2,-(sp)	;save regs
	lea	.decdig,a1		;look for 1st decimal digit
	moveq	#0,d0			;starting at 1st byte
	bsr	find_any
	tst.l	d0			;any dec digits?
	bpl.s	.get			;yup, continue
	moveq	#0,d0			;nope, return 0
	bra.s	.done
.get:	lea	0(a0,d0),a0		;point to 1st dec digit
	lea	.work,a2		;point to work string
	exg	a0,a1			;set dectbl ptr for validation
	clr.b	d2			;init digit counter
.more:	move.b	(a1)+,d0		;get a digit
	beq.s	.cvt			;string end, finish up
	tst.b	d2			;need to chk for leading 0s?
	bne.s	.cont			;nope, continue
	cmpi.b	#'0',d0			;leading zero?
	beq.s	.more			;yup, get next digit
.cont:	move.b	d0,d1			;save digit
	bsr	instr			;legal decimal digit?
	bmi.s	.cvt			;nope, end of decimal string!
	move.b	d1,(a2)+		;add to work string
	addq.b	#1,d2			;increment digit counter
	cmpi.b	#10,d2			;done max (10 digits)?
	bne.s	.more			;nope, get another
.cvt:	clr.b	(a2)			;terminate work string
	cmpi.b	#10,d2			;string is 10 digits?
	bne.s	.ok			;nope, within range
	lea	.work,a0		;see if too big/out of range
	lea	.max,a1
	bsr	strcmp
	beq.s	.ffs			;is max value!
	bmi.s	.ok			;ok, within range
	moveq	#0,d0			;too big!! return 0 and
	ori	#2,ccr			;set OVERFLOW flag (bit 2)
	bra.s	.done			;and quit.
.ffs:	move.l	#$ffffffff,d0		;shortcut, return max value
	bra.s	.done
.ok:	lea	.work,a0		;point to decimal string
	moveq	#0,d0			;init accumulator
.again:	moveq	#0,d1			;init digit
	move.b	(a0)+,d1		;get a digit
	beq.s	.done			;end of string, all done!
	subi.b	#'0',d1			;cvt ascii to value
	move.l	d0,d2			;save result for multiply...
	asl.l	#3,d0			;mult by 8...
	asl.l	#1,d2			;and by 2...
	add.l	d2,d0			;giving mult-by-10 value
	add.l	d1,d0			;add new digit
	bra.s	.again			;go back for next digit
.done:	movem.l	(sp)+,d1-d2/a0-a2	;restore regs
	rts
.decdig:	dc.b	"0123456789",0
.max:	dc.b	"4294967295",0
.work:	dc.b	"0000000000",0

*****************************
*atosl: ascii (decimal) string to signed long
*in:	a0.l = pointer to ascii string
*out:	d0.l = signed long value
*	all others unchanged
*calls: atol, find_any
*stack use: 48 bytes
*notes: For a negative number, the leading minus sign must immediately
*  precede any decimal digits. Conversion starts at first decimal digit
*  and ends at first non-decimal digit, up to 10 digits.
*The overflow (V) flag is set upon return if value is out of range... also:
*  If absolute value > $ffffffff, routine returns 0.L.
*  If absolute value > $efffffff, routine returns long UNsigned value.
*****************************
atosl:	bsr	atol		;cvt string to unsigned long
	bvs.s	.ex		;if already too big, signal overflow
	tst.l	d0		;is long value negative?
	bpl.s	.ok		;nope, check for "-" sign
	ori.b	#2,ccr		;set overflow flag (bit 1)
	bra.s	.ex		;and return (long value in d0)
.ok:	movem.l	d1/a1,-(sp)	;save regs
	move.l	d0,d1		;save long value
	beq.s	.done		;if value=0, skip tests
	lea	.decdig,a1	;look for 1st decimal digit
	moveq	#0,d0		;starting at 1st char
	bsr	find_any	;will always be at least 1 digit here
	tst.w	d0		;digits start at 1st character?
	beq.s	.done		;yup, must be positive
	cmpi.b	#'-',-1(a0,d0)	;preceding char is a minus sign?
	bne.s	.done		;nope, it's positive!
.neg:	neg.l	d1		;negate it, leading minus sign!
.done:	move.l	d1,d0		;put result in d0
	movem.l	(sp)+,d1/a1	;restore regs
.ex:	rts
.decdig:	dc.b	"0123456789",0

*****************************
*btol: convert a binary string (up to 32 binary digits) to a long
*starts at 1st "1" in string, stops at 1st non-0-or-1 after that
*in:	a0.l = pointer to binary string
*out:	d0.l = long value
*	no other changes
*calls: instr
*stack use: 20 bytes
*****************************
btol:	movem.l d1-d2/a0,-(sp)	;save regs
	move.w	#'1',d0		;look for 1st "1" digit
	bsr	instr
	tst.w	d0		;any there?
	bpl.s	.start		;yep, do it!
	moveq	#0,d0		;nope, return a 0
	bra.s	.done
.start:	lea	0(a0,d0),a0	;point to string start
	moveq	#0,d0		;init result
	moveq	#0,d1		;init digit counter
.get:	move.b	(a0)+,d2	;get a digit
	cmpi.b	#'0',d2		;a zero?
	beq.s	.cont		;yup, cont
	cmpi.b	#'1',d2		;a one?
	bne.s	.done		;nope, all done!
.cont:	asl.l	#1,d0		;shift result up 1 bit
	subi.b	#'0',d2		;make ascii to value
	beq.s	.zero		;if zero, don't set new bit
	ori.l	#1,d0		;set bit, must be 1
.zero:	addq.b	#1,d1		;incr digit counter
	cmpi.b	#32,d1		;done 32 bits?
	bne.s	.get		;nope, get another bit
.done:	movem.l (sp)+,d1-d2/a0	;restore regs
	rts

*****************************
*char_out: print a character to screen using Bconout
*in:	d0.w = the character to output
*out:	no changes
*calls: none
*stack use: 32 bytes
*****************************
char_out: movem.l d0-d3/a0-a3,-(sp)	; save the necessary registers
	move.w	d0,-(sp)		; push character to print
	move.w	#CON,-(sp)		; push console device number
	move.w	#Bconout,-(sp)		; Bconout function number
	trap	#BIOS			; call BIOS
	addq.l	#6,sp			; clean up the stack
	movem.l (sp)+,d0-d3/a0-a3	; restore the saved registers
	rts				; back to wherever...

*****************************
*clear_screen: clear the screen using VT52 escape sequence
*in:	none
*out:	no changes
*calls: char_out
*stack use: 38 bytes
*****************************
clear_screen: move.w  d0,-(sp)	; save register
	move.w	#27,d0		; send a VT-52	ESC E
	bsr	char_out	; This clears the...
	move.w	#'E',d0 	; screen and homes...
	bsr	char_out	; the text cursor
	move.w	(sp)+,d0	; restore register
	rts

*****************************
*cmp_wild: string compare, case sensitive, with wildcards
*in:	a0.l = address of string 1
*	a1.l = address of string 2 (the one with wildcards)
*	d0.b = wildcard character (e.g. "?") or zero for no wildcard
*out:	d0.l = -1 if string 1 precedes string 2
*	d0.l =  0 if string 1 matches string 2
*	d0.l = +1 if string 1 follows string 2
*	no other changes
*calls: none
*stack use: 16 bytes
*****************************
cmp_wild:	movem.l d1-d2/a0-a1,-(sp)	;save regs
.get:	move.b	(a0)+,d1	;get char from string 1
	move.b	(a1)+,d2	;get char from string 2
	bne.s	.not0s		;string 2 not ended
	tst.b	d1		;string 1 ended?
	bne.s	.not0s		;nope, check a char
	moveq	#0,d0		;both end, return "equal"
	bra.s	.quit		;and exit.
.not0s: tst.b	d1		;string 1 ended?
	bne.s	.nots1		;nope, check string 2
	moveq	#-1,d0		;yep, string 1 precedes (shorter)
	bra.s	.quit		;return to caller
.nots1: tst.b	d2		;string 2 ended?
	bne.s	.nots2		;nope, gotta compare chars
	moveq	#1,d0		;yep, string 2 precedes (shorter)
	bra.s	.quit		;return to caller
.nots2: cmp.b	d1,d2		;chars match?
	beq.s	.get		;yup, compare next chars
	cmp.b	d0,d2		;this char of string 2 is wild?
	beq.s	.get		;yep, goto next char
.nowc:	moveq	#-1,d0		;signal str 1 precedes
	cmp.b	d1,d2		;no match, which is first?
	bgt.s	.quit		;str 2 follows so skip reset
	moveq	#1,d0		;signal str 2 precedes
.quit:	movem.l (sp)+,d1-d2/a0-a1	;restore regs
	rts

*****************************
*cmp_wild_nc: string compare, NOT case sensitive, with wildcards
*in:	a0.l = address of string 1
*	a1.l = address of string 2 (the one with wildcards)
*	d0.b = wildcard character (e.g. "?") or zero for no wildcard
*out:	d0.l = -1 if string 1 precedes string 2
*	d0.l =  0 if string 1 matches string 2
*	d0.l = +1 if string 1 follows string 2
*	no other changes
*calls: to_upper
*stack use: 20 bytes
*****************************
cmp_wild_nc:	movem.l d1-d2/a0-a1,-(sp)	;save regs
.get:	move.b	(a0)+,d1	;get char from string 1
	move.b	(a1)+,d2	;get char from string 2
	bne.s	.not0s		;string 2 not ended
	tst.b	d1		;string 1 ended?
	bne.s	.not0s		;nope, check a char
	moveq	#0,d0		;both end, return "equal"
	bra.s	.quit		;and exit.
.not0s: tst.b	d1		;string 1 ended?
	bne.s	.nots1		;nope, check string 2
	moveq	#-1,d0		;yep, string 1 precedes (shorter)
	bra.s	.quit		;return to caller
.nots1: tst.b	d2		;string 2 ended?
	bne.s	.nots2		;nope, gotta compare chars
	moveq	#1,d0		;yep, string 2 precedes (shorter)
	bra.s	.quit		;return to caller
.nots2: exg	d0,d1		;wildcard to d1, str 1 char to d0
	bsr	to_upper	;make str 1 char upper
	exg	d0,d2		;str 1 char to d2, str 2 char to d0
	bsr	to_upper	;make str 2 char upper
	exg	d0,d1		;wildcard back to d0, str 2 char in d1
	cmp.b	d1,d2		;chars match?
	beq.s	.get		;yup, compare next position in strings
	cmp.b	d0,d1		;this char of string 2 is wild?
	beq.s	.get		;yep, goto next char
.nowc:	moveq	#-1,d0		;signal str 1 precedes
	cmp.b	d2,d1		;no match, which is first?
	bgt.s	.quit		;str 2 follows so skip reset
	moveq	#1,d0		;signal str 2 precedes
.quit:	movem.l (sp)+,d1-d2/a0-a1	;restore regs
	rts

*****************************
*cr_lf: output a CR/LF sequence to screen using Bconout
*in:	none
*out:	no changes
*calls: char_out
*stack use: 40 bytes
*****************************
cr_lf:	move.l	d0,-(sp)	; save D0
	move.w	#13,d0		; send a...
	bsr	char_out	; Carriage Return
	move.w	#10,d0		; send a...
	bsr	char_out	; Line Feed
	move.l	(sp)+,d0	; restore D0
	rts

*****************************
*del_all: delete all (char)s in a string
*in:	a0.l = address of string
*	d0.b = char to remove from string
*out:	no changes
*calls: del_char, instr
*stack use: 32 bytes
*****************************
del_all:	movem.l d0-d1/a0,-(sp)	;save regs
	move.b	d0,d1		;save char to remove
	beq.s	.exit		;can't remove zeros!!!
.again: move.b	d1,d0		;char to look for
	bsr	instr		;any (char)s there?
	tst.w	d0		;check return
	bmi.s	.exit		;no more, all done!
	lea	0(a0,d0),a0	;point to char to delete
	clr.w	d0		;zero offset from ptr
	bsr	del_char	;delete it
	tst.b	(a0)		;end of string?
	bne.s	.again		;nope, look for next instance
.exit:	movem.l (sp)+,d0-d1/a0	;restore regs
	rts

*****************************
*del_char: deletes one char from a string
*in:	a0.l = string address
*	d0.w = offset of char to delete
*out:	a0.l = string address
*	d0.l = 0 if ok, or -1 if error
*calls: strlen
*stack use: 16 bytes
*****************************
del_char:	movem.l d1/a0-a1,-(sp)	;save registers
	move.w	d0,d1		;save offset in d1
	bsr	strlen		;get length into d0
	cmp.w	d0,d1		;offset > str length?
	blt.s	.go		;nope, do it!
	moveq	#-1,d0		;signal error
	bra.s	.quit		;and exit
.go:	moveq	#0,d0		;signal ok
	lea	0(a0,d1),a1	;ptr to char to delete
	lea	1(a1),a0	;ptr to following char
.more:	move.b	(a0)+,(a1)+	;move chars down one
	bne.s	.more		;quit after final zero	
.quit:	movem.l (sp)+,d1/a0-a1	;restore registers
	rts

*****************************
*del_leading: remove leading (char)s
*in:	a0.l = address of string
*	d0.b = char to delete from start of string
*out:	no changes
*calls: del_substr, skip_past
*stack use: 36 bytes
*****************************
del_leading:	movem.l d0-d1/a1,-(sp)	;save regs
	move.b	d0,.chr 	;save char to remove
	beq.s	.exit		;can't remove zeros!!!
	clr.w	d0		;set offset=0 for skip_past
	lea	.chr,a1 	;list of chrs to skip
	bsr	skip_past	;find 1st chr NOT to kill
	tst.w	d0		;chop out to where?
	beq.s	.exit		;none to delete, exit
	bpl.s	.some		;only delete some chars
	clr.b	(a0)		;delete all!!
	bra.s	.exit		;& return
.some:	move.l	d0,d1		;size of block to delete
	clr.w	d0		;offset zero
	bsr	del_substr	;remove the block & exit
.exit:	movem.l (sp)+,d0-d1/a1	;restore regs
	rts
.chr:	dc.b	0,0		;'string' of single char

*****************************
*del_substr: remove a substring from a string
*in:	a0.l = string address
*	d0.w = offset of substring to delete
*	d1.w = size of substring to delete
*out:	d0.l = 0 if ok, or -1 if error
*	no other changes
*calls: strlen
*stack use: 20 bytes
*notes: If specified size is too large, string will simply be truncated.
*****************************
del_substr:	movem.l d2/a0-a2,-(sp)	;save registers
	move.w	d0,d2		;save offset in d2
	bsr	strlen		;get str length into d0
	cmp.w	d0,d2		;offset > str length?
	blt.s	.go		;nope, do it!
	moveq	#-1,d0		;signal error
	bra.s	.quit		;and return
.go:	lea	0(a0,d2),a1	;ptr to char to delete
	lea	0(a0,d0),a0	;ptr to term.zero
	lea	0(a1,d1),a2	;ptr to 1st char after block
	cmpa.l	a0,a2		;block extends to end?
	blt.s	.ok		;nope, do it!
	clr.b	(a1)		;truncate!
	bra.s	.yes		;signal ok and return
.ok:	move.b	(a2)+,(a1)+	;move chars down one
	bne.s	.ok		;quit after final zero	
.yes:	moveq	#0,d0		;signal we did something
.quit:	movem.l (sp)+,d2/a0-a2	;restore registers
	rts

*****************************
*del_trailing: remove trailing (char)s
*in:	a0.l = address of string
*	d0.b = char to delete from end of string
*out:	no changes
*calls: strlen
*stack use: 16 bytes
*****************************
del_trailing:	movem.l d0-d1/a1,-(sp)	;save regs
	move.b	d0,d1		;save char to remove
	beq.s	.exit		;can't remove zeros!!!
	bsr	strlen		;find string length
	lea	0(a0,d0),a1	;make ptr to end of string
.again: cmp.b	-(a1),d1	;decrement & check for delete
	beq.s	.again		;keep going until no match
	clr.b	1(a1)		;new terminating zero
.exit:	movem.l (sp)+,d0-d1/a1	;restore regs
	rts

*****************************
*find_any: find 1st char in string, starting at offset, that IS in 2nd string
*in:	a0.l = address of string to search
*	a1.l = address of string with chars to look for
*	d0.w = offset in string at which to start search
*out:	d0.w = offset of 1st char in list, or -1 if none are in list
*	no other changes
*calls: instr, strlen
*stack use: 20 bytes
*notes: If either string is null, find_any returns -1.L.
*	If offset is larger than string size, returns -1.L.
*****************************
find_any:	movem.l d1/a0-a1,-(sp)	;save regs
	move.w	d0,d1		;save offset
	bsr	strlen		;size of string 1
	bne.s	.s1ok		;not 0, go on
.fail:	moveq	#-1,d0		;signal failure
	bra.s	.quit		;& return
.s1ok:	cmp.w	d0,d1		;offset in string too large?
	bge.s	.fail		;yup, quit!
	exg	a0,a1		;swap string addresses! a0=list!
	bsr	strlen		;size of string 2 (char.list)
	beq.s	.fail		;signal failure, empty list!
.get:	move.b	0(a1,d1),d0	;get char from string1 into d0
	beq.s	.fail		;end of string, signal failure
.go:	bsr	instr		;is d0 char in char.list?
	bpl.s	.yup		;yup, this is it!
	addq.w	#1,d1		;increment list index
	bra.s	.get		;try next char
.yup:	move.w	d1,d0		;nope, this is the offset!
.quit:	movem.l (sp)+,d1/a0-a1	;restore regs
	rts

*****************************
*find_char: find a char in a string
*in:	a0.l = string address
*	d0.b = character to find
*	d1.w = offset from which to start looking
*out:	d0.w = offset at which char was found, or -1 if not found
*	no other changes
*calls: strlen
*stack use: 16 bytes
*notes: If character to find is zero, or offset to start search is
*  greater than string length, routine will return -1.L.
*****************************
find_char:	movem.l d1-d2/a1,-(sp)	;save regs
	tst.b	d0		;search for zero?
	beq.s	.fail		;yup, quit
	move.b	d0,d2		;save char...
	bsr	strlen		;find length
	cmp.w	d0,d1		;offset too big?
	blt.s	.ok		;nope- go do it
.fail:	moveq	#-1,d0		;signal failure
	bra.s	.quit		;restore regs & rts
.ok:	lea	0(a0,d1),a1	;start looking here
.next:	tst.b	(a1)		;end of string?
	beq.s	.fail		;bummer, quit
	cmp.b	(a1)+,d2	;is this it?
	bne.s	.next		;nope, check next
.found: suba.l	a0,a1		;yes! get offset+1,
	move.l	a1,d0		;stick in d0 for return,
	subq.w	#1,d0		;fix from post-increment,
.quit:	movem.l (sp)+,d1-d2/a1	;restore regs & return
	rts

*****************************
*find_last: find LAST char in string that IS in a 2nd string
*in:	a0.l = address of string to search
*	a1.l = address of string with chars to find
*out:	d0.w = offset of last char in list, or -1 if none are in list
*	no other changes
*calls: instr, strlen
*stack use: 20 bytes
*****************************
find_last:	movem.l d1/a0-a1,-(sp)	;save regs
	bsr	strlen		;size of string to search
	tst.w	d0		;any chars in string?
	bne.s	.ok1		;not 0, continue
.fail:	moveq	#-1,d0		;signal failure
	bra.s	.quit		;and return
.ok1:	move.w	d0,d1		;save length of string 1
	exg	a0,a1		;swap string pointers!!!
	bsr	strlen		;size of character list
	tst.w	d0		;any chars in string?
	bne.s	.get		;not 0, continue
	bra.s	.fail		;signal failure & return
.get:	move.b	0(a1,d1),d0	;get char from string
	bsr	instr		;is it in list?
	bge.s	.yes		;yup, this is it!
	dbra	d1,.get 	;go back one char
	bra.s	.fail		;fail if last chr was 1st in string!
.yes:	move.w	d1,d0		;return offset
.quit:	movem.l (sp)+,d1/a0-a1	;restore regs
	rts

*****************************
*find_substr: find a substring in a string, starting at nth char
*in:	a0.l = pointer to string to search
*	a1.l = pointer to substring to look for
*	d0.w = offset in string to start search
*out:	d0.w = offset of substring in string, or -1.L if not found
*	no other changes
*calls: find_char, strlen
*stack use: 48 bytes
*notes: If substring to find is null, routine always returns 0.
*****************************
find_substr:	movem.l	d1-d3/a0-a3,-(sp)	;save regs
	move.w	d0,d1		;save start offset
	exg	a0,a1		;find substr length 1st
	bsr	strlen
	tst.w	d0		;null substring?
	beq.s	.ex		;yes! return the 0!
	exg	a0,a1		;swap 'em back
	bsr	strlen		;now chk string length
	tst.w	d0		;null string?
	beq.s	.fail		;yes! fail, return -1!
	move.b	(a1)+,d2	;save 1st char of substring, pt to 2nd
.find:	move.b	d2,d0		;look for 1st char of substring
	bsr	find_char	;(offset in d1 not changed)
	tst.w	d0		;found one?
	bpl.s	.chk		;yup, check rest!
.fail:	moveq	#-1,d0		;nope, signal failure!
	bra.s	.ex		;and quit.
.chk:	move.b	d0,d1		;save offset of found byte
	movea.l	a1,a2		;temp ptr to substr bytes
	lea	1(a0,d1),a3	;temp ptr to string
.again:	move.b	(a2)+,d3	;get sub byte, chk for end
	bne.s	.nope		;not end, keep comparing
	move.l	d1,d0		;found it! tell offset
	bra.s	.ex		;& exit.
.nope:	cmp.b	(a3)+,d3	;this byte matches?
	beq.s	.again		;yep, try next byte
.next:	addq.l	#1,d1		;no match, try next offset
	bra.s	.find
.ex:	movem.l	(sp)+,d1-d3/a0-a3	;restore regs
	rts

*****************************
*find_word: find start offset of specified word in a string
*in:	a0.l = address of the string
*	d0.w = the number of the word to get (starting at 1 for 1st word)
*out:	d0.w = offset from a0 of the start of the extracted word, or
*	d0.l = -1 if there are not enough words in the string
*	no other changes
*calls: is_whitespace
*stack use: 32 bytes
*notes: See is_whitespace for list of word delimiters.
*****************************
find_word:	movem.l d1-d2/a0-a1,-(sp)	;save regs
	tst.w	d0		;get what word?
	bge.s	.ok		;ok, reasonable value
.fail:	moveq	#-1,d0		;signal failure
	bra.s	.quit		;& return
.ok:	movea.l a0,a1		;save original pointer
	move.w	d0,d1		;save desired word #
	clr.w	d2		;init word counter
.get1:	move.b	(a0)+,d0	;get a char
	beq.s	.fail		;yikes, end of string!
	bsr	is_whitespace	;word delimiter?
	bne.s	.get1		;skip all whitespace
	addq.w	#1,d2		;must have a word start!
	cmp.w	d2,d1		;the one we want?
	beq.s	.found		;yup!
.get2:	move.b	(a0)+,d0	;nope, skip this one
	beq.s	.fail		;yikes, end!
	bsr	is_whitespace	;to end of word yet?
	bne.s	.get1		;yup, skip whitespace again
	bra.s	.get2		;nope, skip this word
.found: suba.l	a1,a0		;sub original from current -> size
	move.w	a0,d0		;return size!
	subq.w	#1,d0		;correct for postincrement
.quit:	movem.l (sp)+,d1-d2/a0-a1	;restore regs
	rts

*****************************
*get_key: wait for a keypress and return its value
*in:	none
*out:	d0.l = long scancode for the keypress
*calls: none
*stack use: 28 bytes
*****************************
get_key: movem.l d1-d3/a0-a3,-(sp)	; save the necessary registers
	move.w	#Crawcin,-(sp)		; push Crawcin function #
	trap	#GEMDOS 		; call GEMDOS
	addq.l	#2,sp			; clean up the stack
	movem.l (sp)+,d1-d3/a0-a3	; restore all saved registers
	rts

*****************************
*get_word: extract a word from a string
*in:	a0.l = address of the string
*	a1.l = address at which to store extracted word
*	d0.w = the number of the word to get (starting at 1 for 1st word)
*out:	d0.l = -1 if there are not enough words in the string, else 0
*	no other changes
*calls: find_word, is_whitespace
*stack use: 48 bytes
*notes: See is_whitespace for list of word delimiters.
*****************************
get_word:	movem.l d1/a0-a1,-(sp)	;save regs
	bsr	find_word	;find its starting position
	bmi.s	.quit		;doesn't exist! return the -1
	lea	0(a0,d0),a0	;point to start of word
.loop:	move.b	(a0)+,d0	;get a char
	beq.s	.end		;end of string, that's it!
	move.b	d0,d1		;save the char
	bsr	is_whitespace	;delimiter?
	bne.s	.end		;yup, end of word!
	move.b	d1,(a1)+	;copy char to target
	bra.s	.loop		;now get another one
.end:	clr.b	(a1)		;that's it, terminate string!
	moveq	#0,d0		;signal success
.quit:	movem.l (sp)+,d1/a0-a1	;restore regs
	rts

*****************************
* goto_xy: set the position of the text cursor
*in:	d0.w = X position of cursor (0-79+)
*	d1.w = Y position of cursor (0-24+)
*out:	no changes
*calls: char_out
*stack use: 44 bytes
*****************************
goto_xy: move.l  d0,-(sp)	; save my registers...
	move.l	d1,-(sp)

	addi.w	#32,d0		; correct X position...
	move.w	d0,-(sp)	; and save it!
	addi.w	#32,d1		; correct Y position...
	move.w	d1,-(sp)	; and save it!

	move.w	#27,d0		; send ESCAPE code...
	bsr	char_out
	move.w	#'Y',d0 	; send POSITION code...
	bsr	char_out

	move.w	(sp)+,d0	; pull the Y position and send it...
	bsr	char_out
	move.w	(sp)+,d0	; pull the X position and send it...
	bsr	char_out

	move.l	(sp)+,d1	; restore the registers an leave!
	move.l	(sp)+,d0
	rts

*****************************
*hide_cursor: hide the text cursor using VT52 escape sequence
*in:	none
*out:	no changes
*calls: char_out
*stack use: 40 bytes
*****************************
hide_cursor: move.l d0,-(sp)	; save the register I use
	move.w	#27,d0		; send an ASCII <ESC>...
	bsr	char_out	; to the console
	move.w	#'f',d0 	; send VT-52 code to...
	bsr	char_out	; hide the text cursor
	move.l	(sp)+,d0	; restore saved D0
	rts			; and return

*****************************
*hide_mouse: hide the mouse cursor using LINE-A call
*in:	none
*out:	no changes
*calls: none
*stack use: 60 bytes
*****************************
hide_mouse: movem.l d0-d7/a0-a6,-(sp)
	dc.w	$A00A
	movem.l (sp)+,d0-d7/a0-a6
	rts

*****************************
*htol: convert a hex string (up to 8 hex digits) to a long
*starts at 1st hex digit in string, stops at 1st non-hex-digit after that
*in:	a0.l = pointer to hex string
*out:	d0.l = long value
*	no other changes
*calls: find_any, instr, to_upper
*stack use: 48 bytes
*****************************
htol:	movem.l d1-d3/a0-a2,-(sp)	;save regs
	moveq	#0,d1			;init result
	lea	.hexdig,a1		;look for hex digits
	moveq	#0,d0			;start at 1st char
	bsr	find_any		;find 'em
	tst.w	d0			;any there?
	bmi.s	.done			;nope, quit
	lea	0(a0,d0),a2		;point to string start
	clr.b	d2			;init digit counter
	move.l	a1,a0			;point to hex dig list
.more:	move.b	(a2)+,d0		;get digit
	move.b	d0,d3			;save for if hex digit
	bsr	instr			;is hex digit?
	bmi.s	.done			;nope, all done!
	move.b	d3,d0			;get digit back
	bsr	to_upper		;make uppercase
	subi.b	#'0',d0 		;cvt to number
	cmpi.b	#9,d0			;0-9?
	ble.s	.next			;yup, skip a-f correction
	subi.b	#7,d0			;correct for a-f
.next:	asl.l	#4,d1			;make room for new digit
	or.b	d0,d1			;insert new digit in result
	addq.b	#1,d2			;incr digit counter
	cmpi.b	#8,d2			;done 8 digits?
	bne.s	.more			;nope, go for next one
.done:	move.l	d1,d0			;put result in position
	movem.l (sp)+,d1-d3/a0-a2	;restore regs
	rts
.hexdig:	dc.b	"0123456789AaBbCcDdEeFf",0

*****************************
*instr: find first occurrence of a character in a string
*in:	a0.l = address of the string to search
*	d0.b = the character to search for
*out:	d0.w = the position of the character in the string, or
*	d0.w = -1: if character was not found in the string
*calls: none
*stack use: 4 bytes
*****************************
instr:	move.l	a1,-(sp)	; save value in a1
	movea.l a0,a1		; make a working copy of the string address

.scan:	tst.b	(a1)		; have we reached the end of the string?
	beq.s	.eos		; yes, then leave!
	cmp.b	(a1)+,d0	; else, have we found our character?
	beq.s	.found		; yes, then stop looking
	bra.s	.scan		; otherwise, try again

.found: suba.l	a0,a1		; get number of characters we scanned
	move.l	a1,d0		; put difference in d0 and...
	subq.w	#1,d0		; adjust because of last post increment
	move.l	(sp)+,a1	; restore a1
	rts			; and we're done!

.eos:	move.w	#-1,d0		; -1 means the character wasn't found
	move.l	(sp)+,a1	; restore a1
	rts			; get out of here!

*****************************
*ins_char: insert a character in a string
*in:	a0.l = pointer to string
*	d0.b = character to insert in string
*	d1.w = offset in string at which to insert character
*out:	no changes
*calls: strlen
*stack use: 28 bytes
*notes: Be sure there's enough room to insert a character!
*  If offset is > (length of string + 1), nothing is done.
*  Also, nothing is done if string is max length already (32766 chars).
*****************************
ins_char:	movem.l	d0-d3/a0-a1,-(sp)	;save regs
	tst.w	d1			;neg offset?
	bmi.s	.done			;tsk tsk, can't have that!
	move.b	d0,d2			;save char to insert
	bsr	strlen			;find string length
	cmpi.w	#32766,d0		;at max length already?
	beq.s	.done			;yup, abort
	move.l	d0,d3			;save length for later
	addq.w	#1,d0			;add 1 for new char
	cmp.w	d0,d1			;new position ok?
	bge.s	.done			;nope, abort! (bgE!)
	sub.w	d1,d0			;find #chars we need to shift
	lea	1(a0,d3),a0		;point to end of string
	lea	1(a0),a1		;point to end + 1
.more:	move.b	-(a0),-(a1)		;move last char
	dbra	d0,.more		;until all done
	move.b	d2,(a1)			;insert the char!
.done:	movem.l	(sp)+,d0-d3/a0-a1	;restore regs
	rts

*****************************
*ins_substr: insert a substring into a string
*notes:	be sure there's enough room to insert the string!
*in:	a0.l = pointer to string
*	a1.l = pointer to substring to insert in string
*	d0.w = offset in string at which to insert substring
*out:	d0.l = -1 if substring was not inserted or if offset is negative
*calls: strcat, strlen
*stack use: 40 bytes
*notes: Substring is not inserted if offset is > (length of string + 1), or
*  if the string would be longer than the max length (32766 chars).
*****************************
ins_substr:	movem.l	d1-d3/a0-a3,-(sp)	;save regs
	tst.w	d0			;neg offset?
	bmi.s	.abort			;tsk tsk, can't have that!
	move.w	d0,d1			;save offset
	movea.l	a0,a2			;save string ptr
	bsr	strlen			;get string length
	cmp.w	d0,d1			;insert at end of string?
	bne.s	.go			;nope, continue
	bsr	strcat			;yup, use concat routine!
	bra.s	.done			;restore regs & quit
.go:	move.l	d0,d2			;save string length
	cmp.w	d1,d0			;out of range?
	blt.s	.abort			;yup, abort!
	movea.l	a1,a0			;shift ptr to...
	bsr	strlen			;get substr len
	move.l	d0,d3			;save substr len
	add.w	d2,d0			;look at strlen + substr len
	bmi.s	.abort			;too big, abort!
	beq.s	.abort			;just barely too big, abort!
	cmpi.w	#32766,d0		;right at max?
	bgt.s	.abort			;over max, abort!
*now we know new string isn't too long (phew!)
	move.w	d2,d0			;retrieve str len
	sub.w	d1,d0			;find #chars we need to shift
	subq.w	#1,d0			;fix #chrs for dbra loop test
	lea	0(a2,d2),a0		;point to end of string
	lea	0(a0,d3),a3		;point to end + substr len
.more:	move.b	-(a0),-(a3)		;move last char
	dbra	d0,.more		;until all done
	lea	0(a2,d1),a0		;point to position for substr
.more2:	move.b	(a1)+,(a0)+		;insert a char
	tst.b	(a1)			;at end of substring?
	bne.s	.more2			;nope, copy another chr
	moveq	#0,d0			;signal ok!
	bra.s	.done			;yup, all done!
.abort:	moveq	#-1,d0			;signal failure
.done	movem.l	(sp)+,d1-d3/a0-a3	;restore regs
	rts

*****************************
*is_alpha: find if a char is alphabetic (also find case if so)
*in:	d0.b = char
*out:	d0.l = +1 if char is uppercase alphabetic
*	d0.l =  0 if char is not alphabetic
*	d0.l = -1 if char is lowercase alphabetic
*calls: none
*stack use: 0 bytes
*****************************
is_alpha:	cmpi.b	#'a',d0 	;above lower bound?
	blt.s	.chkup		;nope!
	cmpi.b	#'z',d0 	;below upper bound?
	bgt.s	.chkup		;nope!
	moveq	#-1,d0		;lowercase!
	rts
.chkup: cmpi.b	#'A',d0 	;above lower bound?
	blt.s	.fail		;nope!
	cmpi.b	#'Z',d0 	;below upper bound?
	bgt.s	.fail		;nope!
	moveq	#1,d0		;uppercase!
	rts
.fail:	moveq	#0,d0		;signal not alpha
	rts

*****************************
*is_digit: find if a char is a decimal digit
*in:	d0.b = char
*out:	d0.l = +1 if char IS a decimal digit
*	d0.l =  0 if char is NOT a decimal digit
*calls: none
*stack use: 0 bytes
*****************************
is_digit:	cmpi.b	#'0',d0 	;above lower bound?
	blt.s	.fail		;nope!
	cmpi.b	#'9',d0 	;below upper bound?
	bgt.s	.fail		;nope!
	moveq	#1,d0		;yep, it's a digit!
	rts
.fail:	moveq	#0,d0		;signal not a digit
	rts

*****************************
*is_whitespace: find if a char is whitespace (blank, tab, cr, lf, ff)
*in:	d0.b = char
*out:	d0.l = +1 if char IS whitespace
*	d0.l =  0 if char is NOT whitespace
*calls: instr
*stack use: 12 bytes
*****************************
is_whitespace:	movem.l a0,-(sp)	;save regs
	lea	.white,a0	;point to whitespace definition
	bsr	instr		;see if char is in the string
	bmi.s	.nope		;nope...
	moveq	#1,d0		;signal YES
	bra.s	.quit		;& return
.nope:	moveq	#0,d0		;signal NO & return
.quit:	movem.l (sp)+,a0	;restore regs
	rts
.white: dc.b	32,9,13,10,12,0 	;whitespace definition!

*****************************
*itoa: convert an UNsigned integer to a string
*in:	a0.l = address of string to hold converted integer
*	d0.w = unsigned integer value to convert
*out:	no changes
*calls: none
*stack use: 16 bytes
*notes: Improved from version in Programming the 68000 - Williams
*****************************
itoa:	tst.w	d0		; zero as a special case
	bne.s	.1		; skip if not zero
	move.b	#$30,(a0)+	; move "0"
	clr.b	(a0)		; null terminate
	subq.l	#1,a0		; restore pointer
	rts			; return...

.1:	movem.l d0-d1/a0-a1,-(sp)	; save the registers

	lea	.scratch,a1	; get address of scratch string
	addq.l	#5,a1		; go to end of the scratch string area
	clr.b	(a1)		; make sure it is null terminated
	move.w	#4,d1		; initialize count register

.loop:	andi.l	#$ffff,d0	; clear high word
	divu	#10,d0		; Divide by 10 to isolate a digit
	swap	d0		; digit is now in low-word (remainder)
	addi.b	#'0',d0		; convert to ASCII
	move.b	d0,-(a1)	; move isolated value to string area
	swap	d0		; swap register to restore it to normal
	dbra	d1,.loop	; keep looping until d1 equals 0

	lea	.scratch,a1	; reload address of scratch area
.loop2: cmp.b	#'0',(a1)	; scan for a digit other than '0'
	bne.s	.done		; I found one!
	addq.l	#1,a1		; next character please
	bra.s	.loop2		; try again

.done:	move.b	(a1)+,(a0)+	; shove it
	bne.s	.done		; continue until null terminator is reached

	movem.l (sp)+,d0-d1/a0-a1	; restore the registers
	rts

.scratch: dc.b	 "00000000",0
	even

*****************************
*itob: convert an integer to a binary string
*in:	a0.l = pointer to destination string
*	d0.w = integer value to convert
*out:	no changes
*calls: none
*stack use: 4 bytes
*****************************
itob:	move.w	d1,-(sp)	; save the registers I use...
	move.w	d2,-(sp)

	lea	17(a0),a0	; go to the end of the binary string
	move.b	#0,-(a0)	; make sure it's null terminated!

	move.w	#15,d2		; # of loops (characters in string - 1)
.loop:	move.w	d0,d1		; make working copy of value to convert
	andi.w	#1,d1		; mask off all but the low bit
	addi.b	#'0',d1 	; make it an ASCII "0" or "1"
	move.b	d1,-(a0)	; put it in the string area
	ror.w	#1,d0		; move the next bit into position
	dbra	d2,.loop	; and continue until all 16 bits are done.

	move.w	(sp)+,d2	; restore the registers I used...
	move.w	(sp)+,d1
	rts

*****************************
*itoh: convert an integer to a hex string
*in:	a0.l = pointer to destination string
*	d0.w = integer value to convert
*out:	no changes
*calls: none
*stack use: 12 bytes
*****************************
itoh:	movem.l d1-d2/a1,-(sp)	; save the registers which change

	lea	5(a0),a0	; go to the end of the hex string
	move.b	#0,-(a0)	; make sure it's null terminated!

	lea	.hextab,a1	; get the hex conversion table address

	moveq.l #3,d2		; # of loops (char's in string - 1)
.loop:	move.w	d0,d1		; make a copy of D0
	andi.w	#$000f,d1	; mask off all but low nybble for index
	move.b	0(a1,d1),-(a0)	; copy the hex char into the string
	ror.w	#4,d0		; rotate next nybble into place
	dbra	d2,.loop	; and loop until finished

	movem.l (sp)+,d1-d2/a1	; restore the registers
	rts			; and leave!

.hextab: dc.b	 "0123456789ABCDEF"

*****************************
*itoo: convert an integer to an octal string
*in:	a0.l = pointer to destination string
*	d0.w = integer value to convert
*out:	no changes
*calls: none
*stack use: 8 bytes
*****************************
itoo:	move.l	d0,-(sp)	; save the registers I use!
	move.l	d2,-(sp)

	lea	7(a0),a0	; go to the end of the octal string
	move.b	#0,-(a0)	; make sure it's null terminated!

	moveq.l #5,d2		; # of loops (characters in string -1)
.loop:	move.w	d0,d1		; make working copy of value to convert
	andi.w	#7,d1		; mask off all but the lower 3 bits
	addi.b	#'0',d1 	; convert to ASCII char from "0" to "7"
	move.b	d1,-(a0)	; put the character in the string area
	lsr.w	#3,d0		; shift next 3 bits into position
	dbra	d2,.loop	; and continue for all 6 octal digits

	move.l (sp)+,d2	; restore the registers I used!
	move.l (sp)+,d0
	rts

*****************************
*kbshift: set or retrieve state of system keyboard shift bits
*in:	d0.w = new states for shift bits, or -1 to retrieve shift bits
*out:	d0.b = keyboard shift bit states (if retrieved)
*	d0.b = previous shift bit states (if changed)
*calls: none
*stack use: 28 bytes
*notes: The bit assignments are:
*  0: Right shift key
*  1: Left shift key
*  2: Control key
*  3: Alternate Key
*  4: Caps Lock key
*  5: Right mouse button (Clr/Home)
*  6: Left mouse button (Insert)
*  7: reserved
*****************************
kbshift: movem.l d1-d3/a0-a3,-(sp)
	move.w	d0,-(sp)	; pass get/set shift bits flag/value
	move.w	#Kbshift,-(sp)	; get/set the keyboard...
	trap	#BIOS		; shift keys bits
	addq.l	#4,sp		; clean up the stack
	movem.l (sp)+,d1-d3/a0-a3
	rts

*****************************
*keystat: find current console (keyboard) input status (Bconstat)
*in:	none
*out:	d0.w = Current console input status
*calls: none
*stack use: 28 bytes
*****************************
keystat: movem.l d1-d3/a0-a3,-(sp)
	move.w	#CON,-(sp)	; get the current...
	move.w	#Bconstat,-(sp) ; console input...
	trap	#BIOS		; status.
	addq.l	#4,sp		; clean up the stack
	movem.l (sp)+,d1-d3/a0-a3
	rts			; get out of here!

*****************************
*ltoa: unsigned long 32 bit to null-terminated ascii string
*in:	d0.l = unsigned long number
*	a0.l = pointer to resulting ascii string
*out:	no changes
*calls: none
*stack use: 60 bytes
*notes: This routine is by Carl Barron [75066,3204], Oct 1990
*---------------------------
*notes: position independent code 94 bytes long ,three divides/digit.
*MODULUS EQU 65536 ... 
*(10*A+B)*65536+(10*C+D) ... 10*(A*65536+C)+(B*65536+D) ... B*65536+D=10*E+F
*REMAINDER = F ... QUOTIENT = A*65536+C+E
*equates used in original routine:
*  workreg equr d0      abs    equr d1      data_a   equr d2
*  data_b  equr d3      data_c equr d4      data_d   equr d5
*  pq      equr d6      pr     equr d7      workarea equ -20
*****************************
ltoa:	link	a6,#-20		;allocate work area on stack
	movem.l	d0-d7/a0-a1,-(sp)	;save ALL registers we use!
	lea	-20(a6),a1	;a1- current char in temp str
	clr.b	(a1)+		;clear indicate end of string
.ok:	swap	d0
	move.w	d0,d0		;is number only 16 bits or less
	beq.s	.xshrt		;branch if so
	swap	d0		;restore partial quotient
	move.l	d0,d1		;save current
	clr.w	d0		;do the high division
	swap	d0
	divu	#10,d0
	move.w	d0,d2		;high divide quotient to data_a
	swap	d0
	move.w	d0,d3		;high divide remainder to data_b
	moveq	#0,d0
	move.w	d1,d0
	divu	#10,d0		;do the low division
	move.w	d0,d4		;low quotient to data_c
	swap	d0
	move.w	d0,d5		;low remainder to data_d
	move.w	d2,d0		;create long quotient
	swap	d0
	move.w	d4,d0
	move.l	d0,d6
	move.w	d3,d0		;create long remainder
	swap	d0
	move.w	d5,d0
	divu	#10,d0		;divide by 10
	swap	d0
	move.w	d0,d7		;short remainder is result
	clr.w	d0		;create unsigned long number
	swap	d0			
	add.l	d0,d6		;add this to the long quotient
	add.w	#'0',d7		;convert remainder to ascii
	move.b	d7,(a1)+	;and save result.
	move.l	d6,d0		;move partial quotient to d0
	bra.s	.ok		;continue long processs.
.xshrt:	swap	d0		;don't need longs any more...
.xshr1:	divu	#10,d0		;remaining work can be done with shorts
	swap	d0		;and straight hardware divides.
	add.b	#'0',d0		;convert remainder to ascii
	move.b	d0,(a1)+	;save it.
	clr.w	d0		;convert quoutient to unsigned long
	swap	d0
	bne.s	.xshr1		;loop until quotient equals zero
.cp:	move.b	-(a1),(a0)+	;reverse copy the string to its destination
	bne.s	.cp
	movem.l	(sp)+,d0-d7/a0-a1	;restore and return
	unlk	a6
.ex:	rts

*****************************
*ltob: convert an unsigned long to a binary string
*in:	a0.l = pointer to destination string
*	d0.l = long integer value to convert
*calls: none
*stack use: 8 bytes
*****************************
ltob:	move.l	d1,-(sp)	; save the registers I use...
	move.l	d2,-(sp)

	lea	33(a0),a0	; go to the end of the binary string
	move.b	#0,-(a0)	; make sure it's null terminated!

	move.w	#31,d2		; # of loops (characters in string - 1)
.loop:	move.l	d0,d1		; make working copy of value to convert
	andi.l	#1,d1		; mask off all but the low bit
	addi.b	#'0',d1 	; make it an ASCII "0" or "1"
	move.b	d1,-(a0)	; put it in the string area
	ror.l	#1,d0		; move the next bit into position
	dbra	d2,.loop	; and continue until all 16 bits are done.

	move.l	(sp)+,d2	; restore the registers I used...
	move.l	(sp)+,d1
	rts

*****************************
*ltoh: convert an unsigned long to a hex string
*in:	a0.l = pointer to destination string
*	d0.l = long integer value to convert
*calls: none
*stack use: 12 bytes
*****************************
ltoh:	movem.l d1-d2/a1,-(sp)	; save the registers which change

	lea	9(a0),a0	; go to the end of the hex string
	move.b	#0,-(a0)	; make sure it's null terminated!

	lea	.hextab,a1	; get the hex conversion table address

	moveq.l #7,d2		; # of loops (char's in string - 1)
.loop:	move.l	d0,d1		; make a copy of D0
	andi.l	#$0000000f,d1	; mask off all but low nybble for index
	move.b	0(a1,d1),-(a0)	; copy the hex char into the string
	ror.l	#4,d0		; rotate next nybble into place
	dbra	d2,.loop	; and loop until finished

	movem.l (sp)+,d1-d2/a1	; restore the registers
	rts			; and leave!

.hextab: dc.b	 "0123456789ABCDEF"

*****************************
*ltoo: convert an unsigned long to an octal string
*in:	a0.l = pointer to destination string
*	d0.l = long integer value to convert
*calls: none
*stack use: 8 bytes
*****************************
ltoo:	move.l	d0,-(sp)	; save the registers I use!
	move.l	d2,-(sp)

	lea	12(a0),a0	; go to the end of the octal string
	move.b	#0,-(a0)	; make sure it's null terminated!

	moveq.l #10,d2		; # of loops (characters in string -1)
.loop:	move.l	d0,d1		; make working copy of value to convert
	andi.l	#7,d1		; mask off all but the lower 3 bits
	addi.b	#'0',d1 	; convert to ASCII char from "0" to "7"
	move.b	d1,-(a0)	; put the character in the string area
	lsr.l	#3,d0		; shift next 3 bits into position
	dbra	d2,.loop	; and continue for all 11 octal digits

	move.l	(sp)+,d2	; restore the registers I used!
	move.l	(sp)+,d0
	rts

*****************************
*make_fn: make a filespec from separate path and filename strings
*  such as would be returned from a file selector call
*in:	a0.l = pointer to path string
*	a1.l = pointer to filename string
*	a2.l = pointer to full filespec output string
*out:	no changes
*calls: strlen
*stack use: 20 bytes
*****************************
make_fn:	movem.l d0/a0-a2,-(sp)	;save regs
	bsr	strlen		;path how long?
	tst.w	d0		;if it's zero,
	beq.s	.more2		;go right to name
	move.l	a2,d0		;else save target pointer
.more1: move.b	(a0)+,(a2)+	;copy a byte of path
	bne.s	.more1		;go until zero copied
.down1: subq.l	#1,a2		;move back 1 char in path
	cmpi.b	#$5C,(a2)	;is backslash?
	beq.s	.inc1		;yep, do filename
	cmpi.b	#':',(a2)	;is colon (for drive)?
	beq.s	.inc1		;yup, do filename
	cmpa.l	d0,a2		;more chars in path??
	bne.s	.down1		;yep, try next one!
.inc1:	addq.l	#1,a2		;incr 1 for start of name
.more2: move.b	(a1)+,(a2)+	;now copy file name
	bne.s	.more2		;go until zero copied
	movem.l (sp)+,d0/a0-a2	;restore regs
	rts

*****************************
*mid_get: extract a substring from a string (mid$ function)
*in:	a0.l = address of source string
*	a1.l = address of destination string
*	d0.w = offset of 1st char to get
*	d1.w = size of substring to get
*out:	no changes
*calls: strlen
*stack use: 24 bytes
*****************************
mid_get:	movem.l d0-d2/a0-a1,-(sp)	;save regs
	tst.w	d1		;substr size too small?
	ble.s	.quit		;if so quit now
	subq.w	#1,d1		;fix size for dbeq below
	move.w	d0,d2		;save offset
	bsr	strlen		;get length of source
	cmp.w	d0,d2		;offset too high?
	blt.s	.ok		;nope, go do it
	clr.b	(a1)		;null output string (but not error?)
	bra.s	.quit		;tha's all!
.ok:	lea	(a0,d2),a0	;ptr to start of block
.more:	move.b	(a0)+,(a1)+	;copy a byte...
	dbeq	d1,.more	;copy to max size or to end
	beq.s	.quit		;if string end, don't add zero
	clr.b	(a1)		;terminate string
.quit:	movem.l (sp)+,d0-d2/a0-a1	;restore regs
	rts

*****************************
*mid_set: change a substring in a string (mid$ statement)
*in:	a0.l = address of source string
*	a1.l = address of destination string
*	d0.w = offset of 1st char of destination to change
*out:	no changes
*calls: strlen
*stack use: 24 bytes
*notes: This routine does not change size of destination string!
*****************************
mid_set:	movem.l d0-d2/a0-a1,-(sp)	;save regs
	move.w	d0,d1		;save offset
	bsr	strlen		;get source size
	ble.s	.quit		;if too small, quit now
	move.w	d0,d2		;save source size
	exg	a0,a1		;swap source & dest pointers
	bsr	strlen		;get dest length
	sub.w	d1,d0		;subtr. offset from dest size
	ble.s	.quit		;available target length <= zero!
	cmp.w	d0,d2		;copy less of target avail, source length
	bge.s	.go		;use d0 (available target area size)
	move.w	d2,d0		;use source length
.go:	subq.w	#1,d0		;correct for dbra (end at -1)
	lea	(a0,d1),a0	;point to 1st target char
.more:	move.b	(a1)+,(a0)+	;copy a safe byte
	dbra	d0,.more	;copy next char (we know how many)
.quit:	movem.l (sp)+,d0-d2/a0-a1	;restore regs
	rts

*****************************
*num_words: count number of words in a string
*in:	a0.l = address of string
*out:	d0.w = number of words in the string
*	no other changes
*calls: is_whitespace
*stack use: 12 bytes
*notes: See is_whitespace for list of word delimiters.
*****************************
num_words:	move.l a0,-(sp)	;save regs
	move.l d1,-(sp)
	clr.w	d1		;init word counter
.get1:	move.b	(a0)+,d0	;get a char
	beq.s	.quit		;end, all counted!
	bsr	is_whitespace	;word delimiter?
	bne.s	.get1		;skip all whitespace
	addq.w	#1,d1		;must have a word start!
.get2:	move.b	(a0)+,d0	;nope, skip this one
	beq.s	.quit		;end, all counted!
	bsr	is_whitespace	;to end of word yet?
	bne.s	.get1		;yup, skip whitespace again
	bra.s	.get2		;nope, skip this word
.quit:	move.w	d1,d0		;return the # we counted
	move.l (sp)+,d1		;restore regs
	move.l (sp)+,a0
	rts

*****************************
*otol: convert an octal string (up to 11 octal digits) to a long
*in:	a0.l = pointer to octal string
*out:	d0.l = long value
*	no other changes
*calls: find_any, instr
*stack use: 48 bytes
*notes: Conversion starts at the first octal digit (0..7) in string, stops
*  at the first non-octal-digit after that (up to 11 octal digits).
*  If overflow (i.e., 11 digits, 1st is >3) occurs, the CARRY (C) flag
*  will be SET upon exit, and the result will be the rest of the value
*  (i.e., the 32 low bits of a 33-bit value).
*****************************
otol:	movem.l d1-d3/a0-a2,-(sp)	;save regs
	moveq	#0,d1			;init result
	lea	.octdig,a1		;look for octal digits
	moveq	#0,d0			;start at 1st char
	bsr	find_any		;find 'em
	tst.w	d0			;any there?
	bmi.s	.done			;nope, quit
	lea	0(a0,d0),a2		;point to string start
	clr.b	d2			;init digit counter
	move.l	a1,a0			;point to oct dig list
.more:	move.b	(a2)+,d0		;get digit
	move.b	d0,d3			;save for if oct digit
	bsr	instr			;is oct digit?
	bmi.s	.done			;nope, all done!
	move.b	d3,d0			;get digit back
	subi.b	#'0',d0 		;cvt to number
	asl.l	#3,d1			;make room for new digit
*notes: carry bit gets hi bit, so if overflow then C flag is SET!
	or.b	d0,d1			;insert new digit in result
	addq.b	#1,d2			;incr digit counter
	cmpi.b	#10,d2			;done 10 digits?
	bne.s	.more			;nope, go for next one
.done:	move.l	d1,d0			;put result in position
	movem.l (sp)+,d1-d3/a0-a2	;restore regs
	rts
.octdig:	dc.b	"01234567",0

*****************************
*print_string: print a string to screen
*in:	a0.l = address of string to print to screen
*out:	no changes
*calls: char_out
*stack use: 42 bytes
*****************************
print_string:	move.l  a0,-(sp)	; save any regs that will change
	move.w  d0,-(sp)

.loop:	clr.w	d0		; make sure high byte of word is clear
	move.b	(a0)+,d0	; get first or next character
	beq.s	.last		; if null then stop printing
	bsr	char_out	; otherwise print the character
	bra.s	.loop		; keep looping

.last:	move.w	(sp)+,d0	; restore the registers
	move.l	(sp)+,a0
	rts

*****************************
* restore_xy: restore the saved X and Y position of the text cursor
*** NOTE: save_xy must have been called previously! ***
*in:	none
*out:	no changes
*calls: char_out
*stack use: 40 bytes
*****************************
restore_xy: move.l  d0,-(sp)	; save register
	move.w	#27,d0		; send an ASCII <ESC>...
	bsr	char_out	; to the console
	move.w	#'k',d0 	; send VT-52 code to...
	bsr	char_out	; restore the cursor position
	move.l	(sp)+,d0	; restore register
	rts			; and return

*****************************
*reverse_off: turns the VT-52 reverse text mode OFF
*in:	none
*out:	no changes
*calls: char_out
*stack use: 38 bytes
*****************************
reverse_off: move.w  d0,-(sp)
	move.w	#27,d0
	bsr	char_out
	move.w	#'q',d0
	bsr	char_out
	move.w	(sp)+,d0  
	rts

*****************************
*reverse_on: turns the VT-52 reverse text mode ON
*in:	none
*out:	no changes
*calls: char_out
*stack use: 38 bytes
*****************************
reverse_on: move.w  d0,-(sp)
	move.w	#27,d0
	bsr	char_out
	move.w	#'p',d0
	bsr	char_out
	move.w	(sp)+,d0  
	rts

*****************************
*save_xy: save the X and Y position of the text cursor
*** NOTE: this must be called before calling restore_xy! ***
*in:	none
*out:	no changes
*calls: char_out
*stack use: 40 bytes
*****************************
save_xy: move.l  d0,-(sp)
	move.w	#27,d0		; send an ASCII <ESC>...
	bsr	char_out	; to the console
	move.w	#'j',d0 	; send VT-52 code to...
	bsr	char_out	; save the cursor position
	move.l	(sp)+,d0
	rts			; and return

*****************************
*show_cursor: shows (unhides) the text cursor using VT52 codes
*in:	none
*out:	no changes
*calls: char_out
*stack use: 40 bytes
*****************************
show_cursor: move.l  d0,-(sp)
	move.w	#27,d0		; send an ASCII <ESC>...
	bsr	char_out	; to the console
	move.w	#'e',d0 	; send VT-52 code to...
	bsr	char_out	; show the text cursor
	move.l	(sp)+,d0
	rts			; and return

*****************************
*show_mouse: show (unhide) the mouse pointer using LINE-A call
*in:	none
*out:	no changes
*calls: none
*stack use: 60 bytes
*****************************
show_mouse:	movem.l d0-d7/a0-a6,-(sp)
	dc.w	$A009
	movem.l (sp)+,d0-d7/a0-a6
	rts

*****************************
*sitoa: convert a SIGNED integer (-32767..32767) to an ascii string
*in:	d0.w = signed integer value to convert
*	a0.l = address of string to hold converted integer
*out:	no changes
*calls: none
*stack use: 16 bytes
*notes: Improved from version in Programming the 68000 - Williams
*****************************
sitoa:	tst.w	d0		; zero as a special case
	bne.s	.1		; skip if not zero
	move.b	#$30,(a0)+	; move "0"
	clr.b	(a0)		; null terminate
	subq.l	#1,a0		; restore pointer
	rts			; return...

.1:	movem.l d0-d2/a0-a1,-(sp)	; save the registers

	lea	.scratch,a1	; get address of scratch string
	addq.l	#5,a1		; go to end of the scratch string area
	clr.b	(a1)		; make sure it is null terminated
	move.w	#4,d1		; initialize count register

	move.b	#'+',d2 	; Assume positive flag
	tst.w	d0		; Negative?
	bpl.s	.loop		; If not negative then use the '+' flag
	move.b	#'-',d2 	; otherwise make it a negative flag
	neg.w	d0		; convert number to positive value

.loop:	ext.l	d0		; extend register to 32 bits
	divs	#10,d0		; Divide by 10 to isolate a digit
	swap	d0		; digit is now in low-word (remainder)
	add.b	#'0',d0		; convert to ASCII
	move.b	d0,-(a1)	; move isolated value to string area
	swap	d0		; swap register to restore it to normal
	dbra	d1,.loop	; keep looping until d1 equals 0

	lea	.scratch,a1	; reload address of scratch area
.loop2: cmp.b	#'0',(a1)	; scan for a digit other than '0'
	bne.s	.done		; I found one!
	addq.l	#1,a1		; next character please
	bra.s	.loop2		; try again

.done:	cmp.b	#'-',d2 	; place sign in string address passed
	bne.s	.final		; only if it's '-'
	move.b	d2,(a0)+	; move it and point to the next position

.final: move.b	(a1)+,(a0)+	; shove it
	bne.s	.final		; continue until null terminator is reached

	movem.l (sp)+,d0-d2/a0-a1	; restore the registers
	rts

.scratch: dc.b	 "00000000",0
	even

*****************************
*skip_past: find 1st char of string, starting at nth char, that
*	is NOT in a second string
*in:	a0.l = address of string to search
*	a1.l = address of second string (with list of chars to skip)
*	d0.w = offset in 1st string to start looking
*out:	d0.w = offset of 1st different char in string1, else
*	d0.l = -1 if all chars in string1 are in string2
*	no other changes
*calls: instr, strlen
*stack use: 20 bytes
*notes: If string 1 is null, skip_past returns -1.L.
*	If string 2 is null, skip_past returns 0.W.
*	If offset is larger than string size, returns -1.L.
*****************************
skip_past:	movem.l d1/a0-a1,-(sp)	;save regs
	move.w	d0,d1		;save offset
	bsr	strlen		;size of string 1
	bne.s	.s1ok		;not 0, go on
.fail:	moveq	#-1,d0		;signal no chrs not in string
	bra.s	.quit		;& return
.s1ok:	cmp.w	d0,d1		;offset > str length?
	bge.s	.fail		;yup, quit!
	exg	a0,a1		;swap string addresses!!!
	bsr	strlen		;size of string 2 (char.list)
	bne.s	.get		;not 0, go on
	clr.w	d0		;1st char is not in null string
	bra.s	.quit		;& return
.get:	move.b	0(a1,d1),d0	;get char from string1
	bne.s	.go		;not end of string
	bra.s	.fail		;return -1
.go:	bsr	instr		;is d0 char in char.list?
	bmi.s	.nope		;nope, this is it!
	addq.w	#1,d1		;increment list index
	bra.s	.get		;try next char
.nope:	move.w	d1,d0		;nope, this is the offset!
.quit:	movem.l (sp)+,d1/a0-a1	;restore regs
	rts

*****************************
*sltoa: signed long to ascii (decimal) string
*in:	a0.l = pointer to output string area
*	d0.l = signed long value to convert
*out:	no changes
*calls: ltoa
*stack use: 72 bytes
*****************************
sltoa:	move.l	a0,-(sp)	;save regs
	move.l	d0,-(sp)
	tst.l	d0		;is number negative?
	bpl.s	.pos		;nope, do normal unsigned routine
	move.b	#'-',(a0)+	;yup, start with a - sign
	neg.l	d0		;then negate value (i.e. make it +)
.pos:	bsr	ltoa		;call unsigned routine
	move.l	(sp)+,d0	;restore regs & quit
	move.l	(sp)+,a0
.ex:	rts

*****************************
*sprintf: construct a formatted output string
*NOTE: This is similar but _not identical_ to the ANSI C sprintf function!
*in:	a0.l = address of string in which to store results
*	a1.l = address of format string
*	d0.? = first parameter
*	d1.? = second parameter
*	...	...
*	d7.? = eighth parameter
*out:	no changes
*calls: atol, del_leading, del_substr, instr, is_digit, ins_char, itoa,
*  itob, itoh, itoo, ltoa, ltob, ltoh, ltoo, sitoa, sltoa, strlen, to_upper
*stack use: 104 bytes
*memory required: 1332 bytes
*----------------------------
*The sprintf subroutine builds a null-terminated output string based on a
*'format string'. The format string contains both normal characters to be
*copied directly to the output string, plus special formatting instructions
*which are used to convert the input parameters into ASCII strings which
*are then copied to the output string. Embedded formatting instructions all
*begin with a percent ('%') character, and are of the form:
*
*            "%[attributes][minimum][.precision]type"   where:
*
*  attributes: zero or more of these characters, in any order:
*
*    - = left-justify value in the field (right-justify is the default)
*	(only valid if minimum field width is specified)
*    0 = use zero fill instead of blank fill for a number (dDuUbBoOhHxX)
*	(only valid if minimum field width is specified)
*    / = remove leading zeros from a non-decimal number (bBoOhHxX)
*	(leading zeros are always removed from decimal numbers)
*    + = precede a positive signed decimal with a + sign (d or D only!)
*	(the + takes up one position in the field!)
*    space = precede a positive signed decimal with a blank (d or D only!)
*	(the blank takes up one position in the field!)
*	NOTE: the preceding + sign format takes precedence!
*    > = floating sign: a preceding plus, minus, or blank should be output
*	immediately in front of a number's digits (the default is that the
*	preceding char is output in the first position of the output field)
*	(only valid if minimum field width is specified)
*
*  minimum: (optional) a number (1 to 99) indicating the minimum width for
*	output (sprintf will output AT LEAST this number of characters)
*
*  precision: (optional) a period followed by a number (1 to 99) indicating
*	the maximum width for output data. For strings, this is the maximum
*	number of characters from the string to be output; any extra
*	characters in the string are ignored. For numeric data types
*	(bBoOdDuUhHxX), output will be the LAST n digits: e.g. with a
*	format "%.2d" and value 1234 the result is "34", or with a
*	format "%.2H" and value $abcd the result is "CD".
*
*  type: one of these format conversion characters:
*
*	-----------FORMAT CHARACTER-----------	---PARM SIZE---
*	% = a real "%" character		(no parameter)
*	c or C = a character (cannot be zero!)	byte (1 byte)
*	s or S = a string			long (4 bytes)
*	d = signed integer in decimal form	word (2 bytes)
*	D = signed long in decimal form		long (4 bytes)
*	u = unsigned integer in decimal form	word (2 bytes)
*	U = unsigned long in decimal form	long (4 bytes)
*	h or x = unsigned integer in hex form	word (2 bytes)
*	H or X = unsigned long in hex form	long (4 bytes)
*	o = unsigned integer in octal form	word (2 bytes)
*	O = unsigned long in octal form		long (4 bytes)
*	b = unsigned integer in binary form	word (2 bytes)
*	B = unsigned long in binary form	long (4 bytes)
*	e or E = end-of-line (CR/LF) characters	(no parameter)
*	n or N = end-of-line (CR/LF) characters	(no parameter)
*
*----------------------------
*Notes: Only 8 parameters can be passed but you can have more than 8 format
*  characters since some ("%eEnN" chars) don't require parameters. If your
*  format string requires more than eight parameters, SPRINTF will
*  'recycle' the 1st parameter for the 9th, 2nd parm for the 10th, etc.
*Make sure the parameters you pass agree with the data types your format
*  string specifies, or you'll get some odd results!
*MAKE SURE THE OUTPUT STRING HAS ENOUGH ROOM FOR YOUR RESULTS or you'll end
*  up overwriting memory (and most likely crashing your system)!
*For string types, the parameter is the address of the string.
*For all other types, the parameter is a VALUE (see size chart above).
*----------------------------
*IMPORTANT NOTE: if minimum width or precision is specified, at most 99
*  characters can be output for a format item. If a value greater than 99
*  is specified, the value 99 will be used for that value. A value of zero
*  will be ignored (i.e., it's the same as not specifying a value).
*Attributes, minimum widths, and precisions are ignored for format types
*  where these options are not valid.
*For string types: if NEITHER a minimum width NOR a precision is specified,
*  the entire string will be copied to output. If EITHER IS specified,
*  then at most 99 bytes of the string can be output. If a precision is
*  specified, the first [precision] characters of the string are used.
*  If both are specified, the first [precision] characters are output in a
*  field that is [minimum] spaces wide.
*If you try to use an illegal format character, sprintf will simply output
*  the character; the % and special formatting instructions are lost.
*Any character in the format string which is not part of a format command
*  is copied to the output string as is.
*Floating point numbers are not supported.
*****************************
sprintf:	movem.l	a0-a6,-(sp)	;save addr regs on stack
	movem.l	d0-d7,.parms		;save data regs in memory
	movea.l	a0,a6		;save results pointer
	movea.l	a1,a5		;save format string pointer
	clr.w	.poff		;init (clear) PARM OFFSET

*MAIN LOOP START, reading from format string
.main:	move.b	(a5)+,d0	;get a char from format string
	beq.s	.done		;end of string, finish up!
	cmpi.b	#'%',d0		;start formatted output?
	beq.s	.attr		;yes, start decoding!
	move.b	d0,(a6)+	;nope, output char & go on
	bra.s	.main		;go back to format string

*the exit routine is HERE, not at end of code!
.done:	clr.b	(a6)		;terminate output string
	movem.l	(sp)+,a0-a6	;restore address registers
	movem.l	.parms,d0-d7	;restore data registers
	rts

*check for attributes... (note: bit7 used for str/char type signal!)
.attr:	clr.b	.fspec		;erase all old special flags
.atr0:	move.b	(a5)+,d0	;get possible instruction character
	beq.s	.done		;end of string, all done!
	cmpi.b	#'/',d0		;remove leading 0s?
	bne.s	.atr1		;nope, check next bit
	bset.b	#0,.fspec	;yep! set flag & try for another
	bra.s	.atr0
.atr1:	cmpi.b	#'+',d0		;precede pos dec #s with +?
	bne.s	.atr2		;nope, check next bit
	bset.b	#1,.fspec	;yep! set flag & try for another
	bra.s	.atr0
.atr2:	cmpi.b	#'-',d0		;left-justify?
	bne.s	.atr3		;nope, check next bit
	bset.b	#2,.fspec	;yep! set flag & try for another
	bra.s	.atr0
.atr3:	cmpi.b	#'0',d0		;zero-fill?
	bne.s	.atr4		;nope, check next bit
	bset.b	#3,.fspec	;yep! set flag & try for another
	bra.s	.atr0
.atr4:	cmpi.b	#' ',d0		;precede pos.dec# with space char?
	bne.s	.atr5		;nope, check next bit
	bset.b	#4,.fspec	;yep! set flag & try for another
	bra.s	.atr0
.atr5:	cmpi.b	#'>',d0		;floating sign character?
	bne.s	.atr6		;nope, check next bit
	bset.b	#5,.fspec	;yep! set flag & try for another
	bra.s	.atr0
.atr6:	subq.l	#1,a5		;not an attribute, fix pointer & go on

*check for minimum width spec...
.min:	move.w	#0,.fmin	;init min size spec
	move.b	(a5),d0		;get next char WITHOUT incr ptr!
	bsr	is_digit	;check if field size specification
	beq.s	.max		;nope, chk for max field size spec
.min1:	move.w	#99,.fmin	;assume maximum value
	movea.l	a5,a0		;get and save requested field size
	bsr	atol
	cmpi.w	#99,d0		;too big?
	bge.s	.min2		;yup, use 99 already there
	move.w	d0,.fmin
.min2:	move.b	(a5)+,d0	;now skip past field size spec
	beq	.done		;end of string, all done!
	bsr	is_digit
	bne.s	.min2
	subq.l	#1,a5		;fix ptr to 1st chr past length

*check for precision spec...
.max:	move.w	#0,.fmax	;init preciaion
	move.b	(a5),d0		;get next char WITHOUT incr ptr!
	cmpi.b	#'.',d0		;period preceding precision?
	bne.s	.fmt		;nope, go look for format character now!
	addq.l	#1,a5		;skip the period,
	move.b	(a5),d0		;get possible field size char W/O incr ptr
	bsr	is_digit	;check if field size specification
	beq.s	.fmt		;nope, chk for legal fmt char
.max1:	move.w	#99,.fmax	;assume maximum value
	movea.l	a5,a0		;get and save requested field size
	bsr	atol
	cmpi.w	#99,d0		;too big?
	bge.s	.max2		;yup, use the 99 already there
	move.w	d0,.fmax
.max2:	move.b	(a5)+,d0	;now skip past field size spec
	beq	.done		;end of string, all done!
	bsr	is_digit	;still part of size spec?
	bne.s	.max2
	subq.l	#1,a5		;fix ptr to 1st chr past length

*check for a legal format character...
.fmt:	move.b	(a5)+,d0	;get format character
	beq	.done		;end of string, all done!
	move.b	d0,.fchar	;save fmt char for later checks
	move.b	d0,d1		;save original and convert
	bsr	to_upper	; to uppercase version
	move.b	d0,.fchup	;and save that too
	move.b	d1,d0		;get original fchar again
	lea	.legal,a0	;point to list of legal chars
	bsr	instr		;see if char is in the list
	bpl.s	.which		;yup, legal so see which one
	move.b	.fchar,(a6)+	;NOT LEGAL, just copy char to output
	bra	.main		;go back for next part of fmt string!

*first check for non-parm-using chars (%eEnN) before getting a parm...
.which:	cmpi.b	#'%',.fchar	;percent?*************************
	bne.s	.e
	move.b	#'%',(a6)+
	bra	.main

.e:	cmpi.b	#'E',.fchup	;E=end of line/crlf?**************
	bne.s	.n
.eol:	move.b	#13,(a6)+
	move.b	#10,(a6)+
	bra	.main

.n:	cmpi.b	#'N',.fchup	;N=newline=eol=crlf?**************
	bne.s	.getp
	bra.s	.eol		;use routine above

*all other format chars use a parameter so get it now...
.getp:	lea	.parms,a4	;point to saved parms
	move.w	.poff,d7	;and get current offset
	move.l	0(a4,d7),d0	;get the parameter into d0
	addq.w	#4,d7		;increment parm offset
	cmpi.w	#32,d7		;used all 8 parms?
	bne.s	.gpex		;nope, not yet
	clr.w	d7		;reset to start of list!
.gpex:	move.w	d7,.poff	;save offset for next time

*chk for output type and put ascii conversion into output field...
	lea	.field,a0	;make pointer to output work area!
	move.b	.fchar,d1	;retrieve format type for easy checking

	cmpi.b	#'b',d1		;binary int?**********************
	bne.s	.bb		;nope, check next legal fmt.char
	bsr	itob		;yup, get string value into .field
	bra	.ffix		;then do the formatting fixup
.bb:	cmpi.b	#'B',d1		;binary long?*********************
	bne.s	.c		;(same comments as above)
	bsr	ltob
	bra	.ffix
.c:	cmpi.b	#'C',.fchup	;character?***********************
	bne.s	.d
	bset.b	#7,.fspec	;signal alpha data type for formatting
	move.b	d0,(a0)+
	clr.b	(a0)
	bra	.ffix
.d:	cmpi.b	#'d',d1		;signed int?**********************
	bne.s	.dd
	bsr	sitoa
	bra	.ffix
.dd:	cmpi.b	#'D',d1		;signed long?*********************
	bne.s	.h
	bsr	sltoa
	bra	.ffix
.h:	cmpi.b	#'h',d1		;hex int?*************************
	bne.s	.hh
	bsr	itoh
	bra	.ffix
.hh:	cmpi.b	#'H',d1		;hex long?************************
	bne.s	.o
	bsr	ltoh
	bra	.ffix
.o:	cmpi.b	#'o',d1		;octal int?***********************
	bne.s	.oo
	bsr	itoo
	bra	.ffix
.oo:	cmpi.b	#'O',d1		;octal long?**********************
	bne.s	.u
	bsr	ltoo
	bra.s	.ffix
.u:	cmpi.b	#'u',d1		;unsigned int?********************
	bne.s	.uu
	bsr	itoa
	bra.s	.ffix
.uu:	cmpi.b	#'U',d1		;unsigned long?*******************
	bne.s	.x
	bsr	ltoa
	bra.s	.ffix
.x:	cmpi.b	#'x',d1		;hex int?*************************
	bne.s	.xx
	bsr	itoh
	bra.s	.ffix
.xx:	cmpi.b	#'X',d1		;hex long?************************
	bne.s	.s
	bsr	ltoh
	bra.s	.ffix
.s:	cmpi.b	#'S',.fchup	;string?**************************
	bne.s	.none
	bset.b	#7,.fspec	;signal alpha data type for formatting
	movea.l	d0,a1		;make pointer to string in a_1_
	tst.w	.fmin		;min width specified?
	bne.s	.s_fix		;>0 so go do the prep
	tst.w	.fmax		;precision specified?
	bne.s	.s_fix		;>0 so go do the prep
.s_cpy:	move.b	(a1)+,(a6)+	;neither: copy whole string...
	bne.s	.s_cpy		;up through terminating zero...
	subq.l	#1,a6		;then fix output pointer.
	bra	.main		;go back for next part of fmt string!
.s_fix:	move.w	#98,d0		;copy up to 99 bytes of string
.s_fi1:	move.b	(a1)+,(a0)+	;copy byte from string to .field
	dbeq	d0,.s_fi1	;go until end of str _OR_ 99 copied
	clr.b	(a0)		;make sure it's terminated
	bra.s	.ffix		;then go to fixup routine
.none:	move.b	.fchar,(a6)+	;ILLEGAL!*************************
	bra	.main		;go back for next part of fmt string!

*precision: if s, truncate, or if number, save last (precision) digits only
.ffix:	tst.w	.fmax		;precision specified?
	beq.s	.ff0		;nope, go on to next section
	move.b	.fchup,d0	;get uppercase format type
	cmpi.b	#'S',d0		;string type?
	bne.s	.ffpr1		;nope, chk for number type
	move.w	.fmax,d0	;yup, truncate as needed!
	lea	.field,a0
	clr.b	0(a0,d0)
	bra.s	.ff0		;and go on to next section
.ffpr1:	lea	.nums,a0	;number type?
	bsr	instr
	bmi.s	.ff0		;nope, skip precision stuff
	lea	.field,a0	;yup! point to number string
	cmpi.b	#'-',(a0)	;preceding minus sign?
	bne.s	.ffpr2		;nope, no correction needed
	addq.l	#1,a0		;skip the minus sign
.ffpr2:	bsr	strlen		;see how long
	sub.w	.fmax,d0	;see how many digits to delete
	ble.s	.ff0		;already small enough, go on!
	move.w	d0,d1		;say how many to delete...
	clr.w	d0		;from offset zero...
	bsr	del_substr	;and do it!

*delete leading zeros from numbers...
.ff0:	btst.b	#0,.fspec	;bit0=delete ldg zeros (dDuUhHoObB)
	beq.s	.ffmt		;nope, do other formatting stuff
	lea	.nums,a0	;legal fmt.char for this?
	move.b	.fchar,d0
	bsr	instr
	bmi.s	.ffmt		;nope, do other formatting stuff
	lea	.field,a0	;yup, trash 'em!
	move.b	#'0',d0
	bsr	del_leading
	tst.b	(a0)		;make sure there's a digit (if zero)!
	bne.s	.ffmt		;ok, there is one, we're safe
	move.b	#'0',(a0)+	;woops! add back a zero!
	clr.b	(a0)

*setup, left/right justification...
.ffmt:	move.b	#' ',.rjchr	;assume blank fill
	btst.b	#3,.fspec	;use zero fill?
	beq.s	.ffn1		;nope, continue
	move.b	#'0',.rjchr	;yup, save zero char to use for fill
.ffn1:	tst.w	.fmin		;if minimum width not specified,
	beq.s	.ffpre		; then skip this section
	lea	.field,a0	;get size of string
	bsr	strlen
	move.w	d0,d1		;save length for left-justify
	sub.w	.fmin,d0	;need to add any chars?
	bge.s	.ffpre		;nope, go finish up!
	neg.w	d0		;make a positive # of chars to add
.ffjL:	btst.b	#2,.fspec	;left-justify bit set?
	beq.s	.ffjR		;nope, go do setup for R-justify
	lea	0(a0,d1),a0	;point to end of string for L-justify
	bra.s	.ffjgo		;go insert!
.ffjR:	btst.b	#7,.fspec	;alphabetic type?
	bne.s	.ffjgo		;yup, skip this test!
	cmpi.b	#'-',(a0)	;leading minus (and R-justify)?
	bne.s	.ffjgo		;nope, skip this adjustment
*	only skip adjustment if NO 0-fill AND floating sign
	btst.b	#3,.fspec	;zero fill?
	bne.s	.ffjfx		;yes, do the adjustment!
	btst.b	#5,.fspec	;floating minus sign wanted?
	bne.s	.ffjgo		;yep, skip this adjustment
.ffjfx:	addq.l	#1,a0		;point past minus...
.ffjgo:	subi.w	#1,d0		;fix counter for dbra loop
	move.w	d0,d2		;and move it out of the way
	move.b	#' ',d0		;insert blanks...
	clr.w	d1		;at our (fixed) pointer
	btst.b	#7,.fspec	;alphabetic type?
	bne.s	.ffjin		;yup, continue
	btst.b	#2,.fspec	;left-justify?
	bne.s	.ffjin		;yup, continue
	move.b	.rjchr,d0	;use blank or zero per instructions
.ffjin:	bsr	ins_char
	dbra	d2,.ffjin	;do as many as needed, then continue

*preceding plus/blank section...
.ffpre:	cmpi.b	#'D',.fchup	;signed decimal type?
	bne	.cpy		;nope, skip this!
	clr.b	d2		;assume NO preceding char
	btst.b	#4,.fspec	;use preceding blank?
	beq.s	.ffp0		;nope, check for preceding plus
	move.b	#' ',d2		;yup, use a blank!
.ffp0:	btst.b	#1,.fspec	;use preceding plus?
	beq.s	.ffp1		;nope, continue checks
	move.b	#'+',d2		;save plus to use (override blank)
.ffp1:	tst.b	d2		;any char there to insert?
	beq.s	.cpy		;nope, skip this!
	lea	.field,a0	;point to string
	move.b	#'-',d0		;is there a minus sign?
	bsr	instr
	bpl.s	.cpy		;yup, skip all this!
	btst.b	#5,.fspec	;floating sign?
	bne.s	.prefl		;yup, go do other routine
*	non-floating sign routine...
	move.b	(a0),d1		;grab first char in string
	cmpi.b	#'0',d1		;leading zero?
	bne.s	.pre0		;nope, do blank check
	move.b	d2,(a0)		;ok, safe to replace with plus/blank!
	bra.s	.cpy		;and go to next section
.pre0:	cmpi.b	#' ',d1		;leading blank?
	bne.s	.ffp3		;nope, will need to insert char
	move.b	d2,(a0)		;yup, replace with plus/blank!
	bra.s	.cpy		;and go to next section
*	floating sign routine...
.prefl:	lea	.blnks,a1	;see if we can skip any blanks
	moveq	#0,d0		;start at 1st char
	bsr	skip_past	;will ALWAYS be one non-blank char
	tst.w	d0		;any starting blanks?
	beq.s	.pre1		;nope, go to next test
	move.b	d2,-1(a0,d0)	;yup, replace last w/sign
	bra.s	.cpy		;...and go finish up!
.pre1:	cmpi.b	#'0',(a0)	;1st character is a zero?
	bne.s	.ffp3		;nope, go insert sign!
	move.b	1(a0),d3	;1st is '0'. get following chr now
	beq.s	.ffp3		;null, insert sign!
	cmpi.b	#' ',d3		;blank follows?
	beq.s	.ffp3		;yup, insert sign!
	move.b	d2,(a0)		;ok to replace with sign!
	bra.s	.cpy		;and then finish up!
*	insert sign char routine...
.ffp3:	move.b	d2,d0		;move desired char into place
	clr.w	d1		;insert it at start
	bsr	ins_char	;then continue
	bsr	strlen		;now kill last char if it's a blank
	lea	-1(a0,d0),a0	;point to last char of string
	cmpi.b	#' ',(a0)	;is it a blank?
	bne.s	.cpy		;nope, can't delete it!
	clr.b	(a0)		;yup, delete it, it's not needed!

*copy output field to actual output string...
.cpy:	lea	.field,a0	;point to results
.cpy1:	move.b	(a0)+,(a6)+	;copy to output
	bne.s	.cpy1		;up through term.zero
	subq.l	#1,a6		;then fix output pointer

	bra	.main		;go back to loop top for more input!

*=================================================*
*======   END OF ROUTINE'S MAIN LOOP CODE   ======*
*====== ROUTINE EXIT IS AT TOP OF MAIN LOOP ======*
*=================================================*
*======    sprintf routine local data...    ======*
*=================================================*
.parms:	dcb.l	8,0	;parm (data) register save area
.poff:	dc.w	1	;offset in parm save area
.blnks:	dc.b	" ",0	;blank for skip-past routine
.spchr:	dc.b	" -+",0	;special format preceding-sign chars
.nums:	dc.b	"dDuUhHoObB",0			;number format chars
.legal:	dc.b	"%BCDEHNOSUXbcdehnosux",0	;legal format chars
.field:	dcb.b	100,0	;temporary field setup area
.fchar:	dc.b	1	;format character (type)
.fchup:	dc.b	1	;format char in UPPERCASE
.fspec:	dc.b	1	;special format flag bits
.rjchr:	dc.b	1	;fill char for numerics ("0" or " ")
.fmin:	dc.w	1	;specified min field size
.fmax:	dc.w	1	;specified precision
*=================================================*
*================ end of sprintf! ================*
*=================================================*

*****************************
*strcat: concatenate two null-terminated strings
*in:	a0.l = address of first string (destination)
*	a1.l = address of second string
*out:	no changes
*calls: none
*stack use: 8 bytes
*notes: Make sure destination string is large enough!!!
*****************************
strcat: move.l	a0,-(sp)	; save the necessary registers...
	move.l	a1,-(sp)

.1:	tst.b	(a0)+		; is this the end of the string?
	bne.s	.1		; no?, then keep trying
	subq.l	#1,a0		; eos was found, adjust pointer
 
.2:	move.b	(a1)+,(a0)+	; move string 2 to end of string 1
	bne.s	.2		; loop until null is reached in string 2

	move.l	(sp)+,a1	; restore the saved registers...
	move.l	(sp)+,a0
	rts

*****************************
*strcmp: compare two null terminated strings
*in:	a0.l = address of string #1
*	a1.l = address of string #2
*out:	d0.w < 0: if string #1 < string #2
*	d0.w = 0: if string #1 = string #2
*	d0.w > 0: if string #1 > string #2
*	no other changes
*calls: none
*stack use: 8 bytes
*****************************
strcmp: move.l	a0,-(sp)	; save the registers I use...
	move.l	a1,-(sp)

.cmp:	move.b	(a0)+,d0	; load character from string1
	beq.s	.eos		; found null terminator of string1
	sub.b	(a1)+,d0	; get the difference between the chars
	beq.s	.cmp		; if zero, then the bytes were the same

.diff:	ext.w	d0		; sign extend result to a word
	move.l	(sp)+,a1	; restore the registers...
	move.l	(sp)+,a0
	rts			; return to the calling routine

.eos:	clr.w	d0		; clear D0 for equality
	move.l	(sp)+,a1	; restore the registers...
	move.l	(sp)+,a0
	rts			; return to the calling routine

*****************************
*streq: find if two strings are identical (exactly equal)
*in:	a0.l = address of first null-terminated string
*	a1.l = address of second null-terminated string
*out:	d0.l = 0 if strings are identical
*	d0.l = -1 if strings are not identical
*	no other changes
*calls: none
*stack use: 8 bytes
*****************************
streq:	move.l	a0,-(sp)	;save regs
	move.l	a1,-(sp)
.get0:	move.b	(a0)+,d0	;get char from 1st string
	cmp.b	(a1)+,d0	;compare to 2nd string char
	beq.s	.same		;if same, see if end of strings
	moveq	#-1,d0		;different, show failure
	bra.s	.ex
.same:	tst.b	d0		;are we at end of string(s)?
	bne.s	.get0		;nope, compare next chars
	moveq	#0,d0		;yup! identical strings!
.ex:	movea.l	(sp)+,a1	;restore regs
	movea.l	(sp)+,a0
	rts

*****************************
*strlen: find the length of a null terminated string (up to 32767)
*in:	a0.l = address of null terminated string
*out:	d0.w = length of string pointed to by a0
*	no other changes
*calls: none
*stack use: 0 bytes
*****************************
strlen: move.l	a1,d0		; save value of a1

	movea.l a0,a1		; copy string address to a1
.1:	tst.b	(a1)+		; have we reached the end yet?
	bne.s	.1		; no?, keep looking!

	suba.l	a0,a1		; a1 = a1 - a0 (length+1)
	exg	a1,d0		; swap d0 (old a1) and a1 (length+1)
	subq.w	#1,d0		; correct length value
	rts			; and return!

*****************************
*sub_cmp: compare two substrings, with wildcard, case sensitive or not
*in:	a0.l = address of string1
*	a1.l = address of string2 (the one with wildcards)
*	d0.w = offset in string1 to start comparison
*	d1.w = offset in string2 to start comparison
*	d2.w = size of substring to compare, or 0 for rest of string
*	d3.b = <string2> wildcard character (or 0 for no wildcard)
*	d4.b = 0 if not case sensitive, or non-zero if case sensitive
*out:	d0.l = -1 if string1 substring precedes string2 substring
*	d0.l =  0 if substrings match
*	d0.l = +1 if string1 substring follows string2 substring
*	no other changes
*calls: strlen, to_upper
*stack use: 24 bytes
*notes: All characters past the specified size are ignored... for example,
*  if string1="hello world", string2="hell", offset=0, size=4, then the
*  routine would return a zero because the substrings match.
*Also, if an offset would point past the end of a string, it counts as if
*  points to the end of the string.
*****************************
sub_cmp:	movem.l	d1-d2/d5/a0-a1,-(sp)	;save regs
	move.w	d0,d5		;save str1 offset!
	exg	a0,a1		;chk str2 length 1st
	bsr	strlen
	cmp.w	d0,d1		;offset past end of string?
	blt.s	.chk2		;nope
	move.w	d0,d1		;yup, use end of string!
.chk2:	exg	a0,a1		;chk str1 length now
	bsr	strlen
	cmp.w	d0,d5		;offset past end of string?
	blt.s	.go		;nope
	move.w	d0,d5		;yup, use end of string!
.go:	lea	0(a0,d5),a0	;point to substring1
	lea	0(a1,d1),a1	;point to substring2
.next:	move.b	(a0)+,d0	;get str1 char
	move.b	(a1)+,d1	;get str2 char
	cmp.b	d0,d1		;match?
	beq.s	.eq		;yup, do match stuff
	tst.b	d0		;str1 end?
	bne.s	.tstd1		;nope, chk str2
.s1pre:	move.l	#-1,d0		;yup, string1 precedes!
	bra.s	.ex
.tstd1:	tst.b	d1		;str2 end?
	bne.s	.cmp		;nope, compare chars
.s2pre:	moveq	#1,d0		;yup, string2 precedes!
	bra.s	.ex
.cmp:	cmp.b	d1,d3		;wildcard?
*********** here: test d0 for wc char too?***********
	beq.s	.eq		;yup, count as a match
	tst.b	d4		;case sensitive?
	bne.s	.cont		;yup
	bsr	to_upper	;make both uppercase for cmp
	exg	d0,d1
	bsr	to_upper
	exg	d0,d1
.cont:	cmp.b	d0,d1		;compare chars
	beq.s	.eq		;if same, look at next
	bgt.s	.s1pre		;signal str1 precedes!
	blt.s	.s2pre		;signal str2 precedes!
.eq:	tst.b	d0		;both strings ended?
	beq.s	.match		;yup, they match!
	subq.w	#1,d2		;decrement size counter
	beq.s	.match		;all same for size specified!
	bra.s	.next		;ok, test next char	
.match:	moveq	#0,d0		;they're the same!
.ex:	movem.l	(sp)+,d1-d2/d5/a0-a1	;restore regs
	rts

*****************************
*to_lower: convert a char to lowercase
*in:	d0.b = a character
*out:	d0.b = char in lowercase (not changed unless it's upper)
*calls: none
*stack use: 0 bytes
*****************************
to_lower:	cmpi.b	#'A',d0 	;above lower bound?
	blt.s	.fail		;nope!
	cmpi.b	#'Z',d0 	;below upper bound?
	bgt.s	.fail		;nope!
	addi.b	#'a'-'A',d0	;ok, fix it!
.fail:	rts

*****************************
*to_upper: convert a char to uppercase
*in:	d0.b = a character
*out:	d0.b = char in uppercase (not changed unless it's lower)
*calls: none
*stack use: 0 bytes
*****************************
to_upper:	cmpi.b	#'a',d0 	;above lower bound?
	blt.s	.fail		;nope!
	cmpi.b	#'z',d0 	;below upper bound?
	bgt.s	.fail		;nope!
	subi.b	#'a'-'A',d0	;ok, fix it!
.fail:	rts

************************** End of STRLIB.S ******************************
