/**********************************************************************
	SOUNDEMO.C	Sound testing program

	This program demonstrates the uses of the sound chip.
**********************************************************************/

/**************************************************
	System Header Files & Constants
**************************************************/

#include	<stdio.h>				/* Standard IO */
#include	<osbind.h>			/* GEMDOS routines */
#include	<gemdefs.h>			/* GEM structures */
#include	<obdefs.h>			/* GEM write modes */

#include	<ctype.h>				/* character macros */

#define	FALSE	0
#define	TRUE		!FALSE


/**************************************************
	GEM Application Overhead
**************************************************/

/* Declare global arrays for VDI. */
typedef	int	WORD;			/* WORD is 16 bits */
WORD		contrl[12],			/* VDI control array */
		intout[128], intin[128],	/* VDI input arrays */
		ptsin[128], ptsout[128];	/* VDI output arrays */

WORD		screen_vhandle,		/* virtual screen workstation */
		screen_phandle,		/* physical screen workstation */
		screen_rez,			/* screen resolution 0,1, or 2 */
		color_screen,			/* flag if color monitor */
		x_max,				/* max x screen coord */
		y_max;				/* max y screen coord */

char		conterm;				/* console status byte */


/**************************************************
	Application Specific Data
**************************************************/

/* Sound register names */
char	*reg_name[] = {
		"Voice A Fine Tune",
		"Voice A Coarse Tune",
		"Voice B Fine Tune",
		"Voice B Coarse Tune",
		"Voice C Fine Tune",
		"Voice C Coarse Tune",
		"Noise Period",
		"Mixer Selection",
		"Voice A Volume",
		"Voice B Volume",
		"Voice C Volume",
		"Envelope Period Fine Tune",
		"Envelope Period Coarse Tune",
		"Envelope Shape/Cycle" };

/* Sound chip instructions. To write data into a sound register, use
*	function Giaccess(dat, reg_num | 0x80) where dat is the data
*	to write and reg_num is the register number. The following
*	defines set the write bit for use with Giaccess() (i.e. AFINE 
*	puts data into register 0). Note: DO NOT write directly into
*	the mixer register, use function mixer() (see text for the
*	explanation.
*/

#define	AFINE	0 | 0x80
#define	ACOARSE	1 | 0x80
#define	BFINE	2 | 0x80
#define	BCOARSE	3 | 0x80
#define	CFINE	4 | 0x80
#define	CCOARSE	5 | 0x80
#define	NOISEPER	6 | 0x80
#define	MIXER	7 | 0x80
#define	AVOL		8 | 0x80
#define	BVOL		9 | 0x80
#define	CVOL		10 | 0x80
#define	ENVFINE	11 | 0x80
#define	ENVCOARSE	12 | 0x80
#define	ENVCYCLE	13 | 0x80


/* Channel output selection (in octal). Use bitwise AND (&) to 
	combine channels, then use the mixer() function. */

#define	TONEA	076
#define	TONEB	075
#define	TONEC	073
#define	NOISEA	067
#define	NOISEB	057
#define	NOISEC	037
#define	ALLOFF	077

char	sound_demo[] = {
	/* play note A (440 Hz) for 2 seconds */
	0, 28, 1, 1, 7, 62, 8, 8, 130, 100, 7, 63,

	/* play chord of A, C#. E for 2 seconds */
	0, 28, 1, 1, 2, 194, 3, 1, 4, 123, 5, 1, 7, 56,
	8, 8, 9, 8, 10, 8, 130, 100, 7, 63,

	/* "Ramping" sound effect */
	0, 0, 1, 1, 7, 62, 8, 8, 128, 1, 129, 0, 1, 255, 7, 63, 255, 0
};


/**************************************************
	GEM-related Functions
**************************************************/

WORD	open_vwork(phys_handle)
WORD	phys_handle;
/**************************************************
Function:	This function opens a virtual workstation.
Input:	phys_handle	= physical workstation handle
Output:	Returns handle of workstation.
**************************************************/
{
WORD	work_in[11],
		work_out[57],
		new_handle;				/* handle of workstation */
int		i;

	for (i = 0; i < 10; i++)			/* set for default values */
		work_in[i] = 1;
	work_in[10] = 2;				/* use raster coords */
	new_handle = phys_handle;		/* use currently open wkstation */
	v_opnvwk(work_in, &new_handle, work_out);
	v_clrwk(new_handle);			/* clear workstation */
	return(new_handle);
}


set_screen_attr()
/**************************************************
Function:	Set global values about screen.
Input:	None. Uses screen_vhandle.
Output:	Sets x_max, y_max, color_screen, and screen_rez.
**************************************************/
{
WORD	work_out[57];

	vq_extnd(screen_vhandle, 0, work_out);
	x_max = work_out[0];
	y_max = work_out[1];
	screen_rez = Getrez();		/* 0 = low, 1 = med, 2 = high */
	color_screen = (screen_rez < 2);	/* mono 2, color 0 or 1 */
}


/**************************************************
	Application Functions
**************************************************/

set_conterm()
/**************************************************
Function:	Sets Atari ST global variable location 0x484
Input:	Variable conterm must be set to new value
Output:	None.
Notes:	This function MUST be called using supervisor mode,
		e.g., Supexec(set_conterm)
***************************************************/
{
	*(char *)0x484 = conterm;
	return;
}


get_conterm()
/***************************************************
Function:	Gets value of Atari ST global variable location 0x484
Input:	None.
Output:	None. Sets variable conterm.
Notes:	This function MUST be called using supervisor mode,
		e.g., Supexec(get_conterm)
***************************************************/
{
	conterm = *(char *)0x484;
	return;
}


set_keyclick(setting)
int	setting;
/**************************************************
Function:	Provide access to key click setting.
Input:	setting	= TRUE to turn on
				  FALSE to turn off
Output:	Returns current setting of key click.
Notes:	Uses functions set_conterm() and get_conterm()
**************************************************/
{
char	save_con;	

	Supexec(get_conterm);
	save_con = conterm;		/* save current settings */
	if (setting)
		conterm = save_con | 1;		/* key click on */
	else
		conterm = save_con & 0xfe;	/* key click off */
	Supexec(set_conterm);
	return(save_con & 1);		/* bit 0 is key click setting */ 
}


wait(ms)
long	ms;
/**************************************************
Function:	Provide delay for sound effects routines.
Input:	ms	= delay in milliseconds.
Output:	None.
Notes:	Because the routine is implemented in C. The delay
	is only approximate. The inner loop takes 1 ms.
**************************************************/
{
long	i;

	for(; ms > 0; ms--)
		for (i = 125; i > 0; i--)
			;
	return;
}


mixer(ch)
int	ch;
/**************************************************
Function:	Sets mixer register in sound chip.
Input:	ch	= integer value to put in register.
Output:	None. Register 7 set.
**************************************************/
{
register	temp;

	temp = Giaccess(0,7);		/* read current value */
	temp = temp | 077;			/* turn off current mixer setting */
	ch = ch | 0300;			/* set bits 6,7 to prevent */
							/* changing of I/O bits */
	Giaccess(temp & ch, MIXER);	/* write new setting */
	return;
}


clear_sound()
/**************************************************
Function:	Sets sound registers to 0.
Input:	None.
Output:	None. Turns off all sound generation.
**************************************************/
{
register	i;

	for(i = 0; i < 7; i++)
		Giaccess(0, i | 0x80);
	mixer(ALLOFF);				/* handle reg 7 separately */
	Giaccess(0, AVOL);
	Giaccess(0, BVOL);
	Giaccess(0, CVOL);
	return;
}


enter_notes()
/**************************************************
Function:	Allows user to enter sound registers from the keyboard.
Input:	None.
Output:	None.
**************************************************/
{
int	reg,						/* input register */
	dat;						/* register data */

	clear_sound();
for(;;)						/* loops until neg reg entered */
{
	v_curhome(screen_vhandle);
	v_eeos(screen_vhandle);
	printf("Current values:\n");
	for(reg = 0; reg <= 13; reg++)	/* display registers */
		printf("%2d> %-30s %4o\n",
			reg, reg_name[reg], (Giaccess(0, reg) & 0377));
	printf("\n");
	
	printf("Enter register number to change\n");
	printf("or a negative value to end: ");
	scanf("%d", &reg);
	if (reg < 0)					/* neg reg ends function */
	{
		clear_sound();
		return;
	}
	printf("Enter data to store: ");
	scanf("%o", &dat);				/* get data */
	
	if (reg == 7)					/* if mixer, user mixer() */
		mixer(dat);
	else if (reg <= 13)				/* else write data */
		Giaccess(dat, reg | 0x80);
	/* ignore all else */
}
}


siren()
/**************************************************
Function:	Sound effect for European siren.
Input:	None.
Output:	None.
**************************************************/
{
	printf("\nPress any key to stop:\n");
	clear_sound();
	mixer(TONEA);				/* set tone output on channel A */
	Giaccess(15, AVOL);			/* set max volume on A */
	do
	{
		Giaccess(254, AFINE);	/* set higher tone on A */
		Giaccess(0, ACOARSE);
		wait(350L);			/* wait 350 ms */
		Giaccess(86, AFINE);	/* set lower tone on A */
		Giaccess(1, ACOARSE);
		wait(350L);			/* wait 350 ms */
	} while (!Cconis());		/* check if any key pressed */
	clear_sound();				/* turn off sound */
	Crawcin();				/* capture key pressed */
	return;
}


gunshot()
/**************************************************
Function:	Sound effect for gunshot.
Input:	None.
Output:	None.
**************************************************/
{
	clear_sound();
	Giaccess(15, NOISEPER);			/* set max noise period */
	mixer(NOISEA & NOISEB & NOISEC);	/* set noise on all channels */
	Giaccess(16, AVOL);				/* set volume to be */
	Giaccess(16, BVOL);				/* controlled by envelope */
	Giaccess(16, CVOL);				/* generator */
	Giaccess(16, ENVCOARSE);			/* set envelope period */
	Giaccess(0, ENVCYCLE);			/* set cycle type */
	return;
}


explosion()
/**************************************************
Function:	Sound effect for explosion.
Input:	None.
Output:	None.
**************************************************/
{
	clear_sound();
	Giaccess(8, NOISEPER);			/* set noise period */
	mixer(NOISEA & NOISEB & NOISEC);	/* noise on all channels */
	Giaccess(16, AVOL);				/* envelope control on */
	Giaccess(16, BVOL);				/* all channels */
	Giaccess(16, CVOL);
	Giaccess(56, ENVCOARSE);			/* envelope period */
	Giaccess(0, ENVCYCLE);			/* cycle type */
	return;
}


laser()
/**************************************************
Function:	Sound effect for laser.
Input:	None.
Output:	None.
**************************************************/
{
register	sweep;

	printf("\nPress any key to stop.\n");
	clear_sound();
	mixer(TONEA);					/* use channel A tone only */
	Giaccess(15, AVOL);				/* set max volume */
	do
	{
		for (sweep = 48; sweep <= 70; sweep++)
		{						/* decreasing tone sweep */
			Giaccess(sweep, AFINE);	/* on channel A */
			wait(3L);				/* 3 ms delay */
		}
	} while (!Cconis());			/* check for keypress */
	Giaccess(0, AVOL);				/* turn off sound */
	Crawcin();					/* capture key */
	return;
}


bomb()
/**************************************************
Function:	Sound effect for bomb.
Input:	None.
Output:	None.
**************************************************/
{
register	sweep;

	clear_sound();
	mixer(TONEA);					/* tone on channel A */
	Giaccess(15, AVOL);				/* max volume */
	for(sweep = 48; sweep <= 160; sweep++)
	{							/* decreasing tone sweep */
		Giaccess(sweep, AFINE);
		wait(25L);
	}
	explosion();					/* end effect */
	return;
}


whistle()
/**************************************************
Function:	Sound effect for whistle.
Input:	None.
Output:	None.
**************************************************/
{
register	sweep;

	clear_sound();
	Giaccess(1, NOISEPER);		/* set noise period */
	mixer(TONEA & NOISEB);		/* set channel output */
	Giaccess(15, AVOL);			/* max vol on A */
	Giaccess(9, BVOL);			/* lesser vol on B */
	for(sweep = 64; sweep >= 32; sweep--)
	{						/* increasing tone sweep */
		Giaccess(sweep, AFINE);
		wait(20L);
	}
	Giaccess(0, AVOL);			/* stop tone sound */
	wait(150L);				/* wait */
	Giaccess(15, AVOL);			/* restart tone sound */
	for(sweep = 80; sweep >= 48; sweep--)
	{						/* increasing tone sweep */
		Giaccess(sweep, AFINE);
		wait(35L);
	}
	for(sweep = 48; sweep <= 104; sweep++)
	{						/* decreasing tone sweep */
		Giaccess(sweep, AFINE);
		wait(15L);
	}
	clear_sound();				/* end sound */
	return;
}


race_car()
/**************************************************
Function:	Sound effect for race car.
Input:	None.
Output:	None.
**************************************************/
{
register	sweepf, 				/* fine register sweep */
		sweepc;				/* coarse register sweep */

	clear_sound();
	Giaccess(6, BCOARSE);		/* set B tone value */
	Giaccess(15, BFINE);
	mixer(TONEA & TONEB);		/* tones on A and B */
	Giaccess(15, AVOL);			/* max volume on A */
	Giaccess(10, BVOL);			/* lower volume on B */
	for(sweepc = 10; sweepc >= 5; sweepc--)
	{						/* increasing tone on A */
		Giaccess(sweepc, ACOARSE);
		for(sweepf = 255; sweepf >= 0; sweepf--)
		{
			Giaccess(sweepf, AFINE);
			wait(2L);
		}
	}
	for(sweepc = 8; sweepc >= 3; sweepc--)
	{						/* increasing on A at higher tone */
		Giaccess(sweepc, ACOARSE);
		for(sweepf = 255; sweepf >= 0; sweepf--)
		{
			Giaccess(sweepf, AFINE);
			wait(4L);
		}
	}
	for(sweepc = 6; sweepc >= 2; sweepc--)
	{						/* increasing on A higher still */
		Giaccess(sweepc, ACOARSE);
		for(sweepf = 255; sweepf >= 0; sweepf--)
		{
			Giaccess(sweepf, AFINE);
			wait(6L);
		}
	}
	clear_sound();				/* end sound effect */
	return;
}


/**************************************************
	Main Program
**************************************************/

main()
{
int	ap_id;					/* application init verify */
int	select;					/* menu selection */

WORD	gr_wchar, gr_hchar,			/* values for VDI handle */ 
	gr_wbox, gr_hbox;

int	key_click;				/* save key clikc setting */


/**************************************************
	Initialize GEM Access
**************************************************/

	ap_id = appl_init();		/* Initialize AES routines */
	if (ap_id < 0)				/* no calls can be made to AES */
	{						/* use GEMDOS */
		Cconws("***> Initialization Error. <***\n");
		Cconws("Press any key to continue.\n");
		Crawcin();
		exit(-1);				/* set exit value to show error */
	}
		
	screen_phandle = 			/* Get handle for screen */
		graf_handle(&gr_wchar, &gr_hchar, &gr_wbox, &gr_hbox);
	screen_vhandle = open_vwork(screen_phandle);
	set_screen_attr();			/* Get screen attributes */
	
/***************************************************
	Application Specific Routines
***************************************************/

/* turn off key click */
	key_click = set_keyclick(FALSE);

	v_enter_cur(screen_vhandle);	/* use alpha mode */
	do
	{
		v_curhome(screen_vhandle);
		v_eeos(screen_vhandle);	/* clear screen */
		printf("\n\n\n\nSound Demonstration Program\n\n\n");
		printf("  1> European Siren\n");
		printf("  2> Gunshot\n");
		printf("  3> Explosion\n");
		printf("  4> Laser\n");
		printf("  5> Falling Bomb\n");
		printf("  6> Whistle\n");
		printf("  7> Race Car\n");
		printf("  8> Enter Your Own\n");
		printf("  9> Dosound() Demo\n");
		printf("  0> Exit Program\n\n");

		do
		{
			printf("Enter your selection: ");
			scanf("%d", &select);

			if (select == 0)
				break;			/* exit loop */

			switch (select)
			{
			case	1:	siren();
					break;
			case 2:	gunshot();
					break;
			case 3:	explosion();
					break;
			case 4:	laser();
					break;
			case 5:	bomb();
					break;
			case 6:	whistle();
					break;
			case 7:	race_car();
					break;
			case 8:	enter_notes();
					break;
			case 9:	Dosound(sound_demo);
					break;
			default:	select = -1;
					break;
			}
		} while (select < 0);
	} while (select != 0);			/* end do loop */


/***************************************************
	Program Clean-up and Exit
***************************************************/
	
/* Wait for keyboard before exiting program */
	set_keyclick(key_click);		/* restore key click */
	v_exit_cur(screen_vhandle);	/* exit alpha mode */
	v_clsvwk(screen_vhandle);	/* close workstation */
	appl_exit();				/* end program */
}
