/**********************************************************************
	RASTER.C	Raster exercising program
	
	This program demonstrates raster operations. Use this program
	to experiment with the raster operations.
**********************************************************************/

/**************************************************
	System Header Files & Constants
**************************************************/

#include	<stdio.h>				/* Standard IO */
#include	<osbind.h>			/* GEMDOS routines */
#include	<gemdefs.h>			/* GEM AES */
#include	<obdefs.h>			/* GEM constants */

#define	FALSE	0
#define	TRUE		!FALSE

/**************************************************
	GEM Application Overhead
**************************************************/

/* Declare global arrays for VDI. */
typedef	int	WORD;			/* WORD is 16 bits */
WORD		contrl[12],			/* VDI control array */
		intout[128], intin[128],	/* VDI input arrays */
		ptsin[128], ptsout[128];	/* VDI output arrays */

WORD		screen_vhandle,		/* virtual screen workstation */
		screen_phandle,		/* physical screen workstation */
		screen_rez,			/* screen resolution 0,1, or 2 */
		color_screen,			/* flag if color monitor */
		x_max,				/* max x screen coord */
		y_max;				/* max y screen coord */

typedef	struct	mfdbstr
{
	char		*addr;			/* address of raster area */
	WORD		wide;			/* width of raster in pixels */
	WORD		high;			/* height of raster in pixels */
	WORD		word_width;		/* width of raster in words */
	WORD		format;			/* standard or device specific */
	WORD		planes;			/* number of planes in raster */
							/* reserved for future use */
	WORD		reserv1, reserv2, reserv3;
} MFDB;


/**************************************************
	Application Specific Data
**************************************************/

MFDB	chexMFDB,
	stripeMFDB,
	blockMFDB,
	scrMFDB,
	tempMFDB;
	
WORD	block[16] = {	0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
				0xFFFF,
			};

WORD	stripe[16] = {	0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
				0xAAAA,
			};

WORD	chex[32] = {	0xCCCC,		/* plane 0 */
				0xCCCC,
				0x3333,
				0x3333,
				0xCCCC,
				0xCCCC,
				0x3333,
				0x3333,
				0xCCCC,
				0xCCCC,
				0x3333,
				0x3333,
				0xCCCC,
				0xCCCC,
				0x3333,
				0x3333
			};				/* plane 1 will be set to 0s */
			
WORD	temp[16][4];				/* allow for 4 planes */


/**************************************************
	GEM-related Functions
**************************************************/

WORD	open_vwork(phys_handle)
WORD	phys_handle;
/**************************************************
Function:	This function opens a virtual workstation.
Input:	phys_handle	= physical workstation handle
Output:	Returns handle of workstation.
**************************************************/
{
WORD	work_in[11],
		work_out[57],
		new_handle;				/* handle of workstation */
int		i;

	for (i = 0; i < 10; i++)
		work_in[i] = 1;
	work_in[10] = 2;
	new_handle = phys_handle;		/* use currently open wkstation */
	v_opnvwk(work_in, &new_handle, work_out);
	return(new_handle);
}


set_screen_attr()
/**************************************************
Function:	Set global values about screen.
Input:	None. Uses screen_vhandle.
Output:	Sets x_max, y_max, color_screen, and screen_rez.
**************************************************/
{
WORD	work_out[57];

	vq_extnd(screen_vhandle, 0, work_out);
	x_max = work_out[0];
	y_max = work_out[1];
	screen_rez = Getrez();		/* 0 = low, 1 = med, 2 = high */
	color_screen = (screen_rez < 2);	/* mono 2, color 0 or 1 */
}


/**************************************************
	Application Functions
**************************************************/

raster_form()
/**************************************************
Function:	Demonstrates different raster formats.
Input:	None. Uses global MFDBs and rasters listed above.
Output:	Sets fields in MFDBs which should be reset before
		doing anything else.
**************************************************/
{
int	i;

/* Checkerboard raster */
	chexMFDB.addr		=	(char *)chex;
	chexMFDB.wide		=	16;
	chexMFDB.high		=	16;
	chexMFDB.word_width	=	1;
	chexMFDB.format	=	1;		/* standard format */
	chexMFDB.planes	=	2;

/* Temporary raster area */
	tempMFDB.addr		=	(char *)temp;
	tempMFDB.wide		=	16;
	tempMFDB.high		=	16;
	tempMFDB.word_width	=	1;
	tempMFDB.format	=	0;		/* will be set by vr_trnfm() */
	tempMFDB.planes	=	2;		/* must be same as source */
	
	vr_trnfm(screen_vhandle, &chexMFDB, &tempMFDB);
	printf("Transform from Standard to Device-specific\n");
	for (i = 0; i < 32; i++)
		printf("%2d: Standard %6x  Device %6x\n",
			i, chex[i], temp[i]);
	Crawcin();

	chexMFDB.format	=	0;		/* set to device-specific */
	vr_trnfm(screen_vhandle, &chexMFDB, &tempMFDB);
	printf("Transform form Device-specific to Standard\n");
	for (i = 0; i < 32; i++)
		printf("%2d: Device %6x  Standard %6x\n",
			i, chex[i], temp[i]);
	Crawcin();
}


set_up_rasters()
/**************************************************
Function:	Sets up global MFDBs to point to rasters.
Input:	None. Uses global MFDBs and rasters listed above.
Output:	Sets appropriate fields in MFDBs.
**************************************************/
{

/* Checkerboard raster */
	chexMFDB.addr		=	(char *)chex;
	chexMFDB.wide		=	16;
	chexMFDB.high		=	16;
	chexMFDB.word_width	=	1;
	chexMFDB.format	=	0;		/* must be device-specific */
	chexMFDB.planes	=	1;		/* monochrome has 1 plane */

/* Stripe raster */
	stripeMFDB.addr		=	(char *)stripe;
	stripeMFDB.wide		=	16;
	stripeMFDB.high		=	16;
	stripeMFDB.word_width	=	1;
	stripeMFDB.format		=	0;
	stripeMFDB.planes		=	1;	/* monochrome has 1 plane */

/* Block raster */
	blockMFDB.addr			=	(char *)block;
	blockMFDB.wide			=	16;
	blockMFDB.high			=	16;
	blockMFDB.word_width	=	1;
	blockMFDB.format		=	0;
	blockMFDB.planes		=	1;	/* monochrome has 1 plane */

/* Screen raster area */
	scrMFDB.addr		=	(char *)Logbase();
	scrMFDB.wide		=	640;
	scrMFDB.high		=	400;
	scrMFDB.word_width	=	40;
	scrMFDB.format		=	0;
	scrMFDB.planes		=	1;		/* monochrome has 1 plane */
	if (screen_rez == 0)			/* low resolution */
	{
		scrMFDB.wide	=	320;
		scrMFDB.high	=	200;
		scrMFDB.word_width = 20;
		scrMFDB.planes =	4;
	}
	else if (screen_rez == 1)		/* medium resolution */
	{
		scrMFDB.high	=	200;
		scrMFDB.planes	=	2;
	}

/* Temporary raster area */
	tempMFDB.addr		=	(char *)temp;
	tempMFDB.wide		=	16;
	tempMFDB.high		=	16;
	tempMFDB.word_width	=	1;
	tempMFDB.format	=	0;
	tempMFDB.planes	=	scrMFDB.planes;	/* same as screen */
}


raster_test()
/**************************************************
Function:	Draw patterns using rasters.
Input:	None. Uses MFDBs and rasters defined above.
Output:	None. No changes to MFDBs and rasters except temp.
**************************************************/
{
WORD	pxy[8],
	color_index[2];
int	pass;					/* use to loop twice */

	color_index[0] = 1;			/* Foreground color value */
	color_index[1] = 0;			/* Background color value */

for (pass = 0; pass < 2; pass++)	/* first pass uses vrt_cpyfm */
{							/* second pass uses vro_cpyfm */
							/* set background pattern */
	v_clrwk(screen_vhandle);
	vsf_interior(screen_vhandle, 2);
	vsf_style(screen_vhandle, 2);
	pxy[0] = pxy[1] = 10;
	pxy[2] = pxy[3] = 150;
	vr_recfl(screen_vhandle, pxy);

/* Save portion of screen in temp */
	pxy[0] = 100;				/* source x1 coord */
	pxy[1] = 100;				/* source y1 coord */
	pxy[2] = 115;				/* source x2 coord */
	pxy[3] = 115;				/* source y2 coord */
	pxy[4] = 0;				/* dest x1 coord */
	pxy[5] = 0;				/* dest y1 coord */
	pxy[6] = 15;				/* dest x2 coord */
	pxy[7] = 15;				/* dest y2 coord */

	vro_cpyfm(screen_vhandle, S_ONLY, pxy, 
		&scrMFDB, &tempMFDB);

/* Draw checkerboard pattern in saved area */
	pxy[0] = pxy[1] = 0;
	pxy[2] = pxy[3] = 15;
	pxy[4] = pxy[5] = 100;
	pxy[6] = pxy[7] = 115;
	if (pass)
		vro_cpyfm(screen_vhandle, S_ONLY, pxy, 
			&chexMFDB, &scrMFDB);
	else
		vrt_cpyfm(screen_vhandle, MD_REPLACE, pxy, 
			&chexMFDB, &scrMFDB, color_index);
	Crawcin();

/* Draw stripe pattern over part of checkerboard */
	pxy[0] = pxy[1] = 0;
	pxy[2] = pxy[3] = 15;
	pxy[4] = pxy[5] = 110;
	pxy[6] = pxy[7] = 125;
	if (pass)
		vro_cpyfm(screen_vhandle, S_ONLY, pxy, 
			&stripeMFDB, &scrMFDB);
	else
		vrt_cpyfm(screen_vhandle, MD_REPLACE, pxy, 
			&stripeMFDB, &scrMFDB, color_index);
	Crawcin();

/* Draw block pattern */
	pxy[0] = pxy[1] = 0;
	pxy[2] = pxy[3] = 15;
	pxy[4] = pxy[5] = 132;
	pxy[6] = pxy[7] = 147;
	if (pass)
		vro_cpyfm(screen_vhandle, S_ONLY, pxy, 
			&blockMFDB, &scrMFDB);
	else
		vrt_cpyfm(screen_vhandle, MD_REPLACE, pxy, 
			&blockMFDB, &scrMFDB, color_index);
	Crawcin();

/* Demonstrate different rectangle sizes with block pattern */
	pxy[0] = pxy[1] = 0;		/* destination wider than source */
	pxy[2] = pxy[3] = 10;
	pxy[4] = pxy[5] = 10;
	pxy[6] = 50;
	pxy[7] = 20;
	if (pass)
		vro_cpyfm(screen_vhandle, S_ONLY, pxy, 
			&blockMFDB, &scrMFDB);
	else
		vrt_cpyfm(screen_vhandle, MD_REPLACE, pxy, 
			&blockMFDB, &scrMFDB, color_index);
	Crawcin();

	pxy[0] = pxy[1] = 0;		/* destination longer than source */
	pxy[2] = pxy[3] = 10;
	pxy[4] = 60;
	pxy[5] = 10;
	pxy[6] = 70;
	pxy[7] = 50;
	if (pass)
		vro_cpyfm(screen_vhandle, S_ONLY, pxy, 
			&blockMFDB, &scrMFDB);
	else
		vrt_cpyfm(screen_vhandle, MD_REPLACE, pxy, 
			&blockMFDB, &scrMFDB, color_index);
	Crawcin();

/* Replace saved portion of screen from temp */
	pxy[0] = pxy[1] = 0;
	pxy[2] = pxy[3] = 15;
	pxy[4] = pxy[5] = 100;
	pxy[6] = pxy[7] = 115;
	vro_cpyfm(screen_vhandle, S_ONLY, pxy, 
		&tempMFDB, &scrMFDB);
	Crawcin();
} /* end pass loop */
}

color_test()
/**************************************************
Function:	Show use of raster copy and writing modes with color.
Input:	None. Uses chex[], chexMFDB, and scrMFDB.
Output:	None.
**************************************************/
{
WORD		pxy[8],
		color_index[2];

	v_clrwk(screen_vhandle);			/* clear screen */
	pxy[0] = 10;	pxy[1] = 10;
	pxy[2] = 300;	pxy[3] = 150;
	vsf_interior(screen_vhandle, 1);	/* solid fill */
	vsf_color(screen_vhandle, 4);		/* use color 4 to fill */
	vr_recfl(screen_vhandle, pxy);	/* draw rectangle */

/* use replace mode */
	pxy[0] = 0;	pxy[1] = 0;
	pxy[2] = 15;	pxy[3] = 15;
	pxy[4] = 50;	pxy[5] = 20;
	pxy[6] = 65;	pxy[7] = 35;
	if (color_screen)
	{
		color_index[0] = 2;			/* color for 1 bits */
		color_index[1] = 3;			/* color for 0 bits */
	}
	else
	{
		color_index[0] = 1;
		color_index[1] = 0;
	}
	vrt_cpyfm(screen_vhandle, MD_REPLACE, pxy,
		&chexMFDB, &scrMFDB, color_index);

/* use transparent mode */
	pxy[4] = 100;	pxy[5] = 20;
	pxy[6] = 115;	pxy[7] = 35;
	vrt_cpyfm(screen_vhandle, MD_TRANS, pxy,
		&chexMFDB, &scrMFDB, color_index);

/* use XOR mode */
	pxy[4] = 150;	pxy[5] = 20;
	pxy[6] = 165;	pxy[7] = 35;
	vrt_cpyfm(screen_vhandle, MD_XOR, pxy,
		&chexMFDB, &scrMFDB, color_index);

/* use reverse transparent mode */
	pxy[4] = 200;	pxy[5] = 20;
	pxy[6] = 215;	pxy[7] = 35;
	vrt_cpyfm(screen_vhandle, MD_ERASE, pxy,
		&chexMFDB, &scrMFDB, color_index);

	return;
}


/**************************************************
	Main Program
**************************************************/

main()
{
int	ap_id;					/* application init verify */

WORD	gr_wchar, gr_hchar,			/* values for VDI handle */ 
	gr_wbox, gr_hbox;

/**************************************************
	Initialize GEM Access
**************************************************/

	ap_id = appl_init();		/* Initialize AES routines */
	if (ap_id < 0)				/* no calls can be made to AES */
	{						/* use GEMDOS */
		Cconws("***> Initialization Error. <***\n");
		Cconws("Press any key to continue.\n");
		Crawcin();
		exit(-1);				/* set exit value to show error */
	}
		
	screen_phandle = 			/* Get handle for screen */
		graf_handle(&gr_wchar, &gr_hchar, &gr_wbox, &gr_hbox);
	screen_vhandle = open_vwork(screen_phandle);
	set_screen_attr();			/* Get screen attributes */
	
/***************************************************
	Application Specific Routines
***************************************************/

	v_clrwk(screen_vhandle);		/* clear workstation */
	raster_form();
	set_up_rasters();
	raster_test();
	color_test();

/***************************************************
	Program Clean-up and Exit
***************************************************/
	
/* Wait for keyboard before exiting program */
	Crawcin();				/* GEMDOS character input */
	v_clsvwk(screen_vhandle);	/* close workstation */
	appl_exit();				/* end program */
}
/***************************************************/
