/**********************************************************************
	MOUSE.C	Mouse demonstration program

	This program shows how to use the mouse, event manager,
	object library, and graphics library.
**********************************************************************/

/**************************************************
	System Header Files & Constants
**************************************************/

#include	<stdio.h>				/* Standard IO */
#include	<osbind.h>			/* GEMDOS routines */
#include	<gemdefs.h>			/* GEM AES */
#include	<obdefs.h>			/* GEM constants */
#include	<errno.h>				/* errno declaration */

#define	FALSE	0
#define	TRUE		!FALSE

/**************************************************
	GEM Application Overhead
**************************************************/

/* Declare global arrays for VDI. */
typedef	int	WORD;			/* WORD is 16 bits */
WORD		contrl[12],			/* VDI control array */
		intout[128], intin[128],	/* VDI input arrays */
		ptsin[128], ptsout[128];	/* VDI output arrays */

WORD		screen_vhandle,		/* virtual screen workstation */
		screen_phandle,		/* physical screen workstation */
		screen_rez,			/* screen resolution 0,1, or 2 */
		color_screen,			/* flag if color monitor */
		x_max,				/* max x screen coord */
		y_max,				/* max y screen coord */
		m_hidden = FALSE;		/* mouse visibility status */

/**************************************************
	Application Specific Data
**************************************************/

#include	"mouse.h"

OBJECT	*tree_addr;			/* address of screen tree */
GRECT	box[4];				/* 4 rectangles on screen */


/**************************************************
	GEM-related Functions
**************************************************/

WORD	open_vwork(phys_handle)
WORD	phys_handle;
/**************************************************
Function:	This function opens a virtual workstation.
Input:	phys_handle	= physical workstation handle
Output:	Returns handle of workstation.
**************************************************/
{
WORD	work_in[11],
		work_out[57],
		new_handle;				/* handle of workstation */
int		i;

	for (i = 0; i < 10; i++)			/* set for default values */
		work_in[i] = 1;
	work_in[10] = 2;				/* use raster coords */
	new_handle = phys_handle;		/* use currently open wkstation */
	v_opnvwk(work_in, &new_handle, work_out);
	return(new_handle);
}


set_screen_attr()
/**************************************************
Function:	Set global values about screen.
Input:	None. Uses screen_vhandle.
Output:	Sets x_max, y_max, color_screen, and screen_rez.
**************************************************/
{
WORD	work_out[57];

	vq_extnd(screen_vhandle, 0, work_out);
	x_max = work_out[0];
	y_max = work_out[1];
	screen_rez = Getrez();		/* 0 = low, 1 = med, 2 = high */
	color_screen = (screen_rez < 2);	/* mono 2, color 0 or 1 */
}


hide_mouse()
/**************************************************
Function:	Make mouse invisible if currently visible.
Input:	None. Uses variable m_hidden.
Output:	Sets m_hidden to TRUE.
**************************************************/
{
	if (!m_hidden)
	{
		graf_mouse(M_OFF, 0x0L);
		m_hidden = TRUE;
	}
}


show_mouse()
/**************************************************
Function:	Make mouse visible if currently invisible.
Input:	None. Uses m_hidden.
Output:	None. Sets m_hidden to FALSE.
**************************************************/
{
	if (m_hidden)
	{
		graf_mouse(M_ON, 0x0L);
		m_hidden = FALSE;
	}
}


load_resource(rfile)
char	*rfile;
/**************************************************
Function:	Load resource file.
Input:	rfile	= string with resource file name.
Output:	Returns TRUE if file loaded, else FALSE.
**************************************************/
{
char	temp[128];

	if (!rsrc_load(rfile))
	{
		sprintf(temp, "[0][Cannot load file %s |Exiting ...][OK]", rfile);
		form_alert(1, temp);
		return(FALSE);
	}
	return(TRUE);
}


/**************************************************
	Application Functions
**************************************************/

initialize()
/**************************************************
Function:	Draw program screen.
Input:	Resource file must be loaded.
Output:	None.
**************************************************/
{
	hide_mouse();
	rsrc_gaddr(0, MAINTREE, &tree_addr);
	v_clrwk(screen_vhandle);
	tree_addr[MAINTREE].ob_y = 20;	/* move down past menu bar */
	tree_addr[MAINTREE].ob_x = 10;
	objc_draw(tree_addr, 0, 10, 0, 0, x_max, y_max);

/* load rectangle coords */
	box[0].g_w = tree_addr[BOX0].ob_width;
	box[0].g_h = tree_addr[BOX0].ob_height;
	objc_offset(tree_addr, BOX0, &box[0].g_x, &box[0].g_y);

	box[1].g_w = tree_addr[BOX1].ob_width;
	box[1].g_h = tree_addr[BOX1].ob_height;
	objc_offset(tree_addr, BOX1, &box[1].g_x, &box[1].g_y);

	box[2].g_w = tree_addr[BOX2].ob_width;
	box[2].g_h = tree_addr[BOX2].ob_height;
	objc_offset(tree_addr, BOX2, &box[2].g_x, &box[2].g_y);

	box[3].g_w = tree_addr[BOX3].ob_width;
	box[3].g_h = tree_addr[BOX3].ob_height;
	objc_offset(tree_addr, BOX3, &box[3].g_x, &box[3].g_y);

	show_mouse();
	return;
}


move_box()
/**************************************************
Function:	Move a box.
Input:	None. Uses MAINTREE and global variables.
Output:	None. Moves box.
**************************************************/
{
WORD	sx, sy,					/* starting coords */
	fx, fy;					/* final coords of box */

	graf_mouse(FLAT_HAND, 0x0L);	/* change mouse form */
	objc_offset(tree_addr, FLEXBOX, &sx, &sy);
	graf_dragbox(tree_addr[FLEXBOX].ob_width,
		tree_addr[FLEXBOX].ob_height,
		sx, sy,
		box[0].g_x, box[0].g_y, box[0].g_w, box[0].g_h,
		&fx, &fy);
	graf_mouse(ARROW, 0x0L);		/* restore form */

/* redraw parent box only to erase current position */
	objc_draw(tree_addr, BOX0, 0, 0, 0, x_max, y_max);
	graf_mbox(tree_addr[FLEXBOX].ob_width,
		tree_addr[FLEXBOX].ob_height,
		sx, sy, fx, fy);

/* set new location relative to parent */
	tree_addr[FLEXBOX].ob_x += (fx - sx);
	tree_addr[FLEXBOX].ob_y += (fy - sy);

/* redraw object */
	objc_draw(tree_addr, FLEXBOX, 10, 0, 0, x_max, y_max);
	return;
}


size_box()
/**************************************************
Function:	Change size of box.
Input:	None. Uses MAINTREE and global variables.
Output:	None. Moves box.
**************************************************/
{
WORD	sx, sy,					/* starting x,y */
	new_width, new_height;		/* ending size */
static WORD	min_w = 0,		/* smallest size of box */
			min_h = 0;

/* set minimum values on first time through */
/* minimum values are initial values of FLEXBOX */
	if (min_w == 0 && min_h == 0)
	{
		min_w = tree_addr[FLEXBOX].ob_width;
		min_h = tree_addr[FLEXBOX].ob_height;
	}

	graf_mouse(POINT_HAND, 0x0L);	/* change mouse form */
	objc_offset(tree_addr, FLEXBOX, &sx, &sy);
	graf_rubberbox(sx, sy, min_w, min_h, &new_width, &new_height);
	graf_mouse(ARROW, 0x0L);		/* restore form */

/* redraw parent box only */
	objc_draw(tree_addr, BOX0, 0, 0, 0, x_max, y_max);

/* test if new size is greater than enclosing box */
	if ((new_width + tree_addr[FLEXBOX].ob_x) > 
			tree_addr[BOX0].ob_width)
	{
		new_width =			/* if so, set to box edge */
			tree_addr[BOX0].ob_width - tree_addr[FLEXBOX].ob_x;
	}
	if ((new_height + tree_addr[FLEXBOX].ob_y) > 
			tree_addr[BOX0].ob_height)
	{
		new_height =			/* if so, set to box edge */
			tree_addr[BOX0].ob_height - tree_addr[FLEXBOX].ob_y;
	}

/* set new width */
	tree_addr[FLEXBOX].ob_width = new_width;
	tree_addr[FLEXBOX].ob_height = new_height;

/* move size box */
	tree_addr[SIZEBOX].ob_x =
		tree_addr[FLEXBOX].ob_width - tree_addr[SIZEBOX].ob_width;
	tree_addr[SIZEBOX].ob_y =
		tree_addr[FLEXBOX].ob_height - tree_addr[SIZEBOX].ob_height;

/* redraw object */
	objc_draw(tree_addr, FLEXBOX, 10, 0, 0, x_max, y_max);
	return;
}


h_slide()
/**************************************************
Function:	Change horizontal slider.
Input:	None. Uses MAINTREE and global variables.
Output:	None. Moves box.
**************************************************/
{
register	long	val;			/* long needed for large widths */

	val = graf_slidebox(tree_addr, HSLIDEIN, HSLIDE, 0);
/* find new x based on ratio: x = width * val / 1000 */
	val *= (tree_addr[HSLIDEIN].ob_width -
		tree_addr[HSLIDE].ob_width);
	tree_addr[HSLIDE].ob_x = val / 1000;
	objc_draw(tree_addr, HSLIDEIN, 10, 0, 0, x_max, y_max);
	return;
}


v_slide()
/**************************************************
Function:	Change vertical slider.
Input:	None. Uses MAINTREE and global variables.
Output:	None. Moves box.
**************************************************/
{
register	long	val;			/* long needed for large heights */

	val = graf_slidebox(tree_addr, VSLIDEIN, VSLIDE, 1);
/* find new y based on ratio: y = height * val / 1000 */
	val *= (tree_addr[VSLIDEIN].ob_height -
		tree_addr[VSLIDE].ob_height);
	tree_addr[VSLIDE].ob_y = (val / 1000);
	objc_draw(tree_addr, VSLIDEIN, 10, 0, 0, x_max, y_max);
	return;
}


watch_box()
/**************************************************
Function:	Change watch box.
Input:	None. Uses MAINTREE and global variables.
Output:	None. Moves box.
**************************************************/
{
/* if mouse released in box then change text */
	if (graf_watchbox(tree_addr, WATCHBOX, SELECTED, NORMAL))
	{
		if (strcmp(tree_addr[WATCHTXT].ob_spec, "Off") == 0)
			strcpy(tree_addr[WATCHTXT].ob_spec, "On ");
		else
			strcpy(tree_addr[WATCHTXT].ob_spec, "Off");
	}

	tree_addr[WATCHBOX].ob_state = NORMAL;
	objc_draw(tree_addr, BOX2, 10, 0, 0, x_max, y_max);
	return;
}


control()
/**************************************************
Function:	Master control function.
Input:	None. Program initialization must be done before
		entering this function.
Output:	None. Returns for normal program termination.
**************************************************/
{
OBJECT	*menu_addr;			/* address for menu */
WORD		mevx, mevy,			/* evnt_multi parameters */
		mevbut,
		keystate,
		keycode,
		mbreturn,
		msg_buf[8];			/* message buffer */
WORD		event,				/* evnt_multi result */
		event_flag,			/* events to look for */
		m1_flag = 0,			/* 0 for entry to rectangle,
								1 for exit to rectangle */
		m1_x, m1_y, m1_w, m1_h,	/* rectangle for M1 event */
		menu_index;			/* keyboard menu title selected */

/* get address of menu */
	rsrc_gaddr(0, MAINMENU, &menu_addr);

/* display menu bar */
	menu_bar(menu_addr, TRUE);

/* wait for a message indicating a menu selection */
	event_flag = (MU_BUTTON | MU_M1 | MU_MESAG);
for(;;)						/* continue loop until quit */
{
	event = evnt_multi( event_flag,
		0,					/* # mouse clicks */
		01,					/* mouse buttons of interest */
		1,					/* button state */
		m1_flag,				/* first rectangle flags */
		box[3].g_x, box[3].g_y,	/* x,y of 1st rectangle */
		box[3].g_w, box[3].g_h,	/* height, width of 1st rect */
		0,					/* second rectangle flags */
		0, 0,				/* x,y of 2nd rect */
		0, 0,				/* w,h of 2nd rect */
		msg_buf,				/* message buffer */
		0, 0,				/* low, high words for timer */
		&mevx, &mevy,			/* x,y of mouse event */
		&mevbut,				/* button state at event */
		&keystate,			/* status of keyboard at event */
		&keycode,				/* keyboard code for key pressed */
		&mbreturn);			/* # times mouse key enter state */

	if (event & MU_MESAG)
	{
		if (msg_buf[0] != MN_SELECTED)	/* not a menu message */
			continue;					/* then ignore */
		if (msg_buf[4] == QUIT)
			break;			/* exit loop */

		switch(msg_buf[4])		/* find object index */
		{
		case	INFO:			/* info selected */
			break;			/* ignore */
		default:
			break;
		}
							/* reset title state */
		menu_tnormal(menu_addr, msg_buf[3], 1);
		menu_bar(menu_addr, TRUE);
	}						/* end message handler */

	if (event & MU_BUTTON)
	{
		if (mevbut & 01)		/* button pressed down  */
		{					/* find where event occurred */
			switch (objc_find(tree_addr, MAINTREE, 10, mevx, mevy))
			{
			case	FLEXBOX:		/* the box itself */
			case FLEXBOX1:		/* 1st text line in box */
			case FLEXBOX2:		/* 2nd text line in box */
				move_box();
				break;

			case SIZEBOX:
				size_box();
				break;

			case HSLIDE:
				h_slide();
				break;

			case VSLIDE:
				v_slide();
				break;

			case WATCHBOX:
				watch_box();
				break;

			default:
				break;
			}
		}
	}						/* end button handler */

	if (event & MU_M1)
	{
		if (!m1_flag)			/* entered rectangle */
		{					/* set to alternate mouse */
			graf_mouse(HOURGLASS, 0x0L);
			m1_flag = 1;		/* wait to exit */
			event_flag = MU_M1;	/* only event */
		}
		else
		{
			graf_mouse(ARROW, 0x0L);
			m1_flag = 0;		/* wait for entry */
			event_flag = (MU_BUTTON | MU_M1 | MU_MESAG);
		}
	}
}							/* end infinite loop */
	return;
}							/* end function */


/**************************************************
	Main Program
**************************************************/

main()
{
int	ap_id;					/* application init verify */

WORD	gr_wchar, gr_hchar,			/* values for VDI handle */ 
	gr_wbox, gr_hbox;

/**************************************************
	Initialize GEM Access
**************************************************/

	ap_id = appl_init();		/* Initialize AES routines */
	if (ap_id < 0)				/* no calls can be made to AES */
	{						/* use GEMDOS */
		Cconws("***> Initialization Error. <***\n");
		Cconws("Press any key to continue.\n");
		Crawcin();
		exit(-1);				/* set exit value to show error */
	}
		
	screen_phandle = 			/* Get handle for screen */
		graf_handle(&gr_wchar, &gr_hchar, &gr_wbox, &gr_hbox);
	screen_vhandle = open_vwork(screen_phandle);
	set_screen_attr();			/* Get screen attributes */

/***************************************************
	Application Specific Routines
***************************************************/

	if (!load_resource("MOUSE.RSC"))	/* no resource file loaded */
		exit(1);

	initialize();
	control();

/***************************************************
	Program Clean-up and Exit
***************************************************/
	
/* Wait for keyboard before exiting program */
	rsrc_free();
	v_clsvwk(screen_vhandle);	/* close workstation */
	appl_exit();				/* end program */
}
/**************************************************/

/**************/
