/**********************************************************************
	LISTER.C	File display program

	This program will display a disk file in text or hexadecimal
	format.
**********************************************************************/

/**************************************************
	System Header Files & Constants
**************************************************/

#include	<stdio.h>				/* Standard IO */
#include	<osbind.h>			/* GEMDOS routines */
#include	<gemdefs.h>			/* GEM AES */
#include	<obdefs.h>			/* GEM constants */
#include	<errno.h>				/* errno declaration */

#define	FALSE	0
#define	TRUE		!FALSE

/* The Megamax compiler was giving an error when the FILE
	type definition was used. To avoid this problem, the 
	FILE type was redefined as type FP. This error occurred
	in this program ONLY.
*/
typedef FILE FP;

/**************************************************
	GEM Application Overhead
**************************************************/

/* Declare global arrays for VDI. */
typedef	int	WORD;			/* WORD is 16 bits */
WORD		contrl[12],			/* VDI control array */
		intout[128], intin[128],	/* VDI input arrays */
		ptsin[128], ptsout[128];	/* VDI output arrays */

WORD		screen_vhandle,		/* virtual screen workstation */
		screen_phandle,		/* physical screen workstation */
		screen_rez,			/* screen resolution 0,1, or 2 */
		color_screen,			/* flag if color monitor */
		x_max,				/* max x screen coord */
		y_max,				/* max y screen coord */
		m_hidden = FALSE;		/* mouse visibility status */

/**************************************************
	Application Specific Data
**************************************************/

#include	"lister.h"
#define	QUIT_KEY	0x1011		/* control-Q to quit */
#define	DUMP_KEY	0x2004		/* control-D for dump file */
#define	TEXT_KEY	0x1414		/* control-T for text file */

#define	LINE_FEED	0x0a
#define	ESCAPE	27
#define	CR		13

int	max_lines = 23;			/* number of lines to print
								before pausing */
char	def_search[32] = 			/* default search path */
		"A:\*.*",
	sel_file[16],				/* file selected */
	file_name[64];				/* full file name to open */

/**************************************************
	GEM-related Functions
**************************************************/

WORD	open_vwork(phys_handle)
WORD	phys_handle;
/**************************************************
Function:	This function opens a virtual workstation.
Input:	phys_handle	= physical workstation handle
Output:	Returns handle of workstation.
**************************************************/
{
WORD	work_in[11],
		work_out[57],
		new_handle;				/* handle of workstation */
int		i;

	for (i = 0; i < 10; i++)			/* set for default values */
		work_in[i] = 1;
	work_in[10] = 2;				/* use raster coords */
	new_handle = phys_handle;		/* use currently open wkstation */
	v_opnvwk(work_in, &new_handle, work_out);
	return(new_handle);
}


set_screen_attr()
/**************************************************
Function:	Set global values about screen.
Input:	None. Uses screen_vhandle.
Output:	Sets x_max, y_max, color_screen, and screen_rez.
**************************************************/
{
WORD	work_out[57];

	vq_extnd(screen_vhandle, 0, work_out);
	x_max = work_out[0];
	y_max = work_out[1];
	screen_rez = Getrez();		/* 0 = low, 1 = med, 2 = high */
	color_screen = (screen_rez < 2);	/* mono 2, color 0 or 1 */
}


hide_mouse()
/**************************************************
Function:	Make mouse invisible if currently visible.
Input:	None. Uses variable m_hidden.
Output:	Sets m_hidden to TRUE.
**************************************************/
{
	if (!m_hidden)
	{
		graf_mouse(M_OFF, 0x0L);
		m_hidden = TRUE;
	}
}


show_mouse()
/**************************************************
Function:	Make mouse visible if currently invisible.
Input:	None. Uses m_hidden.
Output:	None. Sets m_hidden to FALSE.
**************************************************/
{
	if (m_hidden)
	{
		graf_mouse(M_ON, 0x0L);
		m_hidden = FALSE;
	}
}


/**************************************************
	Application Functions
**************************************************/
	
do_dialog(box_index)
WORD	box_index;
/**************************************************
Function:	Display a dialog box.
Input:	box_addr	= index of dialog box
Output:	Returns index of object used for exit.
**************************************************/
{
WORD	xbox, ybox, hbox, wbox;
WORD	smallx, smally, smallw, smallh;
WORD	exit_object;
OBJECT	*box_addr;

/* get address of box */
	rsrc_gaddr(0, box_index, &box_addr);

/* get size and location of a box centered on screen */
	form_center(box_addr, &xbox, &ybox, &wbox, &hbox);
	smallx = xbox + (wbox / 2);
	smally = ybox + (hbox / 2);
	smallw = 0;
	smallh = 0;

/* reserve area on screen for box display */
	form_dial(FMD_START,
		smallx, smally, smallw, smallh,
		xbox, ybox, wbox, hbox);

/* draw an expanding box */
	form_dial(FMD_GROW,
		smallx, smally, smallw, smallh,
		xbox, ybox, wbox, hbox);

/* draw dialog box */
	objc_draw(box_addr, 0, 10, xbox, ybox, wbox, hbox);

/* handle dialog input */
	exit_object = form_do(box_addr, 0);

/* draw a shrinking box */
	form_dial(FMD_SHRINK,
		smallx, smally, smallw, smallh,
		xbox, ybox, wbox, hbox);

/* reserve area on screen for box display */
	form_dial(FMD_FINISH,
		smallx, smally, smallw, smallh,
		xbox, ybox, wbox, hbox);

/* reset exit object state to unselected */
	box_addr[exit_object].ob_state = NORMAL;

	return(exit_object);
}


get_file()
/**************************************************
Function:	Get a file name using file selector.
Input:	None. Uses def_search[] and sel_file[].
Output:	FALSE if cancelled, TRUE otherwise.
		Sets default file search, file selected,
		and file_name[] to contain the full file name.
**************************************************/
{
WORD	exit_button;
char	*temp;
int	i;

	fsel_input(def_search, sel_file, &exit_button);
	if (!exit_button)				/* cancelled */
		return(exit_button);

	strcpy(file_name, def_search);	/* set path */
	i = strlen(file_name);
	temp = file_name + (i - 1);		/* point to last character */
	while ( (*temp != '\\')			/* search for path end */
			&& (*temp != ':')		/* drive id */
			&& (temp >= file_name) )	/* or start of string */
		temp--;
	temp++;						/* move to next position */
	strcpy(temp, sel_file);			/* add file name */
	return(exit_button);
}


FP *open_file(xx)
int xx;
/**************************************************
Function:	Open file specified in file_name[] for read only.
Input:	None. Uses file_name[].
Output:	Returns file descriptor or NULL.
**************************************************/
{
FP	*file_des;

	file_des = fopen(file_name, "br");
	if (file_des == NULL)
		form_error(-errno);		/* errors are negative values */
	return(file_des);
}


end_page()
/**************************************************
Function:	Handle end of page condition.
Input:	None.
Output:	FALSE if abort requested.
		TRUE otherwise.
**************************************************/
{

	Cconws("Press any key to continue or ESC to abort: ");
	if ( (Crawcin() & 0x7f) == ESCAPE)
		return(FALSE);
	Cconout(CR);				/* return to start of line */
	v_eeol(screen_vhandle);		/* erase to end of line */
	return(TRUE);
}


text_display()
/**************************************************
Function:	Display a file in text format.
Input:	None.
Output:	None.
**************************************************/
{
int	line_num,					/* # of lines printed */
	in_char,					/* char from file */
	i;
FP	*file_num;				/* file descriptor */

/* have user select a file */
	if (!get_file())			/* file selection canceled */
		return;

/* open file for reading */
	file_num = open_file();
	if (file_num == NULL)		/* error opening file */
		return;

/* initialize for reading */
	hide_mouse();
	v_clrwk(screen_vhandle);
	v_curhome(screen_vhandle);

	line_num = 0;
	while ((in_char = getc(file_num)) != EOF)
	{
		printf("%c", in_char);
		if (in_char == LINE_FEED)
		{					/* next line */
			printf("\r");		/* insure at start of new line */
			line_num++;
		}
		if (line_num == max_lines)
		{					/* end of page */
			if (!end_page())	/* listing aborted */
				break;
			else
				line_num = 0;
		}
	}						/* end while loop */
	fclose(file_num);
	Cconws("END OF FILE");
	Crawcin();
	v_clrwk(screen_vhandle);
	show_mouse();
}


dump_display()
/**************************************************
Function:	Display a file in hexadecimal format.
Input:	None.
Output:	None.
**************************************************/
{
int	line_num,					/* # of lines printed */
	count,					/* number of chars read */
	i;
FP	*file_num;				/* file descriptor */
char	in_char[16],				/* char from file */
	temp[80];					/* temporary format string */

/* have user select a file */
	if (!get_file())			/* file selection canceled */
		return;

/* open file for reading */
	file_num = open_file();
	if (file_num == NULL)		/* error opening file */
		return;

/* initialize for reading */
	hide_mouse();
	v_clrwk(screen_vhandle);
	v_curhome(screen_vhandle);

	line_num = 0;
	while ((count = fread(in_char, sizeof(char), 16, file_num)) != 0)
	{
		for (i = 0; i < count; i++)	/* format hexadecimal area */
			sprintf(&temp[i*3], " %02x", in_char[i]);
		temp[count*3] = 0;			/* put null at end */
		printf("%-60s", temp);		/* output left justified */
		for (i = 0; i < count; i++)	/* output character area */
			if (in_char[i] < ' ' || in_char[i] > '~')
				printf(".");		/* non-printable char */
			else
				printf("%c", in_char[i]);
		printf("\n");				/* next line */
		line_num++;
		if (line_num == max_lines)	/* end of page */
			if (!end_page())
				break;			/* listing aborted */
			else
				line_num = 0;
	}						/* end while loop */
	fclose(file_num);
	Cconws("END OF FILE");
	Crawcin();
	v_clrwk(screen_vhandle);
	show_mouse();
}


control()
/**************************************************
Function:	Master control function.
Input:	None. Program initialization must be done before
		entering this function.
Output:	None. Returns for normal program termination.
**************************************************/
{
OBJECT	*menu_addr;			/* address for menu */
WORD		mevx, mevy,			/* evnt_multi parameters */
		mevbut,
		keystate,
		keycode,
		mbreturn,
		msg_buf[8];			/* message buffer */
WORD		event,				/* evnt_multi result */
		menu_index;			/* keyboard menu title selected */

/* get address of menu */
	rsrc_gaddr(0, MAINMENU, &menu_addr);

/* display menu bar */
	menu_bar(menu_addr, TRUE);

/* wait for a message indicating a menu selection */
for(;;)						/* continue loop until quit */
{
	event = evnt_multi( (MU_KEYBD | MU_MESAG),
		0,					/* # mouse clicks */
		0,					/* mouse buttons of interest */
		0,					/* button state */
		0,					/* first rectangle flags */
		0, 0,				/* x,y of 1st rectangle */
		0, 0,				/* height, widht of 1st rect */
		0,					/* second rectand flags */
		0, 0,				/* x,y of 2nd rect */
		0, 0,				/* w,h of 2nd rect */
		msg_buf,				/* message buffer */
		0, 0,				/* low, high words for timer */
		&mevx, &mevy,			/* x,y of mouse event */
		&mevbut,					/* button state at event */
		&keystate,			/* status of keyboard at event */
		&keycode,				/* keyboard code for key pressed */
		&mbreturn);			/* # times mouse key enter state */

	if (event & MU_MESAG)
	{
		if (msg_buf[0] != MN_SELECTED)	/* not a menu message */
			continue;					/* then ignore */
		if (msg_buf[4] == QUIT)
			break;			/* exit loop */

		switch(msg_buf[4])		/* find object index */
		{
		case	TEXTFILE:			/* display file as text */
			text_display();
			menu_bar(menu_addr, FALSE);
			break;

		case DUMPFILE:			/* display file in dump */
			dump_display();
			menu_bar(menu_addr, FALSE);
			break;

		case	INFO:			/* display program info */
			do_dialog(INFOBOX);
			break;

		default:
			break;
		}
							/* reset title state */
		menu_tnormal(menu_addr, msg_buf[3], 1);
		menu_bar(menu_addr, TRUE);
	}						/* end message handler */

	if (event & MU_KEYBD)
	{
		if (keycode == QUIT_KEY)
		{
			menu_index = FILE;
			menu_tnormal(menu_addr, menu_index, 0);
			break;
		}

		switch(keycode)
		{
		case	TEXT_KEY:
			menu_index = FILE;
			menu_tnormal(menu_addr, menu_index, 0);
			text_display();
			menu_bar(menu_addr, FALSE);
			break;

		case	DUMP_KEY:
			menu_index = FILE;
			menu_tnormal(menu_addr, menu_index, 0);
			dump_display();
			menu_bar(menu_addr, FALSE);
			break;

		default:
			break;
		}
		menu_tnormal(menu_addr, menu_index, 1);
		menu_bar(menu_addr, TRUE);
	}						/* end keyboard handler */
}							/* end infinite loop */
	return;
}							/* end function */


/**************************************************
	Main Program
**************************************************/

main()
{
int	ap_id;					/* application init verify */

WORD	gr_wchar, gr_hchar,			/* values for VDI handle */ 
	gr_wbox, gr_hbox;

/**************************************************
	Initialize GEM Access
**************************************************/

	ap_id = appl_init();		/* Initialize AES routines */
	if (ap_id < 0)				/* no calls can be made to AES */
	{						/* use GEMDOS */
		Cconws("***> Initialization Error. <***\n");
		Cconws("Press any key to continue.\n");
		Crawcin();
		exit(-1);				/* set exit value to show error */
	}
		
	screen_phandle = 			/* Get handle for screen */
		graf_handle(&gr_wchar, &gr_hchar, &gr_wbox, &gr_hbox);
	screen_vhandle = open_vwork(screen_phandle);
	set_screen_attr();			/* Get screen attributes */

/***************************************************
	Application Specific Routines
***************************************************/

	if (!rsrc_load("LISTER.RSC"))
	{
		form_alert(1, "[0][Cannot file LISTER.RSC file|Exiting ...][OK]");
		exit(1);
	}

	control();

/***************************************************
	Program Clean-up and Exit
***************************************************/
	
/* Wait for keyboard before exiting program */
	rsrc_free();
	v_clsvwk(screen_vhandle);	/* close workstation */
	appl_exit();				/* end program */
}
/**************************************************/

