/* WADGC.C */
/* A simple program to make Doom sprites, wall patches,
   textures and floortiles and put them in a PWAD file
   for inclusion in custom Doom levels.
   Temporary beta version only - no support guaranteed!
   Author: Stefan Gustavson (stefang@isy.liu.se) 1994
*/

#include <stdio.h>
#include <string.h>
#include <search.h>
#include "doompal.h"

unsigned char sortedpal[256][4];

#define T1FILENAME "texture1.raw"
#define T2FILENAME "texture2.raw"
#define PNFILENAME "pnames.raw"


#define square(x) ((x)*(x))
#define TRUE (1)
#define FALSE (0)

int comparecols(col1, col2)
     char *col1, *col2;
{
  return(strncmp(col1, col2, 3));
}

int lookup(pixel, palette, warning)
     unsigned char pixel[4], palette[][4];
     int *warning;
{
  char *p;
  int i,index,err,minerr;

  /* Cyan in the input means transparent like in DMGRAPH */
  if((pixel[0]==0)&&(pixel[1]==255)&&(pixel[2]==255))
    return(-1);
  /* Quick look for a perfect match - this requires a sorted palette */
  p = (char*)bsearch(pixel, palette, 256, 4, comparecols);
  if (p!=NULL) return(p[3]);

  /* If no perfect match is found, issue a warning and */
  /* perform an extensive search for the closest match */
  if(!*warning) /* Warn only once */
    {
      fprintf(stderr, "WARNING: the PPM file uses colors not present\n%s\n",
	      "in the Doom palette. The quantization will take some time.");
      *warning = TRUE;
    }
  minerr=256*256*3;
  for(i=0; i<256; i++)
    {
      err=(square(pixel[0]-palette[i][0])
           + square(pixel[1]-palette[i][1])
           + square(pixel[2]-palette[i][2]));
      if(err<minerr)
        {
          minerr = err;
          index = i;
        }
    }
  return(index);
}


int findpatch(name, nump, pname)
     char *name;
     int nump;
     char pname[][8];
{
  int i;
  i=0;
  while(strncmp(name, pname[i], 8) && (i<nump)) i++;
  if(i==nump)
    {
      fprintf(stderr,"Couldn't find wall patch %s", name);
      exit(-1);
    }
  return(i);
}


readppmfile(ppmfilename, imagedata, imgw, imgh)
     char *ppmfilename;
     short **imagedata;
     int *imgw, *imgh;
{
  FILE *ppmfile;
  char magic[2];
  short *imageptr;
  int i, warning;
  unsigned char pixel[3];
  
  printf("Reading PPM file %s... \n", ppmfilename);
  ppmfile = fopen(ppmfilename, "r");
  if (ppmfile == NULL)
    {
      fprintf(stderr,"Unable to open PPM file %s.\n", ppmfilename);
      exit(-1);
    }
  if(*imagedata != NULL) free(*imagedata);
  fscanf(ppmfile, "%2c", magic);
  if ((magic[0]!='P') || (magic[1]!='6'))
    {
      fprintf(stderr,"Image file %s is not a raw PPM file!\n", ppmfilename);
      exit(-1);
    }
  fscanf(ppmfile, " ");
  while(ungetc(getc(ppmfile),ppmfile) == '#') /* While comment */
    while(getc(ppmfile) != '\n'); /* Read to end-of-line */
  fscanf(ppmfile, " %d %d %*d%*1c", imgw, imgh);
  imageptr = *imagedata = (short *)malloc(*imgw * *imgh * sizeof(short));
  
  warning = FALSE;
  for (i=0; i<(*imgw * *imgh); i++)
    {
      pixel[0]=getc(ppmfile); /* Red */
      pixel[1]=getc(ppmfile); /* Grn */
      pixel[2]=getc(ppmfile); /* Blu */
      *imageptr++ = (short)lookup(pixel, sortedpal, &warning);
    }
  fclose(ppmfile);
}


void rawtodoom(imagedata,imgw,imgh,ox,oy,w,h,x,y, doomimage, reslength)
     short *imagedata;
     int imgw, imgh, ox, oy, w, h, x, y;
     unsigned char *doomimage;
     int *reslength;
{
  int i,j;
  int dataptr, tempptr;
  short pxl;

  doomimage[0]= w & 0xff;
  doomimage[1]= (w & 0xff00) >> 8;
  doomimage[2]= h & 0xff;
  doomimage[3]= (h & 0xff00) >> 8;
  doomimage[4]= x & 0xff;
  doomimage[5]= (x & 0xff00) >> 8;
  doomimage[6]= y & 0xff;
  doomimage[7]= (y & 0xff00) >> 8;
  /* doomimage[8] to doomimage[8+w*4-1] are column indices */
  dataptr=8+w*4;
  for(i=0; i<w; i++)
    {
      doomimage[8+4*i] = dataptr & 0xff;
      doomimage[8+4*i+1] = (dataptr & 0xff00) >> 8;
      doomimage[8+4*i+2] = (dataptr & 0xff0000) >> 16;
      doomimage[8+4*i+3] = (dataptr & 0xff000000) >> 24;

      j=0;
      while(j<h)
	{
	  while((imagedata[(oy+j)*imgw+ox+i]==-1) && (j<h)) j++;
	  if (j>=h) break;
	  /* Write out one post */
	  doomimage[dataptr++]=j & 0xff;
	  tempptr = dataptr++;
	  doomimage[dataptr++]=0; /* One dummy pixel for some reason */
	  while((j<h) && ((pxl=imagedata[(oy+j)*imgw+ox+i]) != -1))
	    {
	      doomimage[dataptr++]=pxl;
	      j++;
	    }
	  doomimage[dataptr++]=0; /* One dummy pixel for some reason */
	  doomimage[tempptr] = dataptr - tempptr - 3;
	}
      /* Write out the end-of-column marker */
      doomimage[dataptr++]=255;
    }
  *reslength = dataptr;
}


void appenddata(wadfile, filename)
     FILE *wadfile;
     char *filename;
{
  FILE *datafile;
  int buf;

  datafile = fopen(filename, "r");
  if(datafile != NULL)
    {
      while (!feof(datafile))
	{
	  buf = getc(datafile);
	  if(!feof(datafile)) putc(buf,wadfile);
	}
      fclose(datafile);
    }
}


int getshort(filep)
     FILE *filep;
{
  short s;
  s = getc(filep) & 0xff;
  s += (getc(filep) & 0xff)<<8;
  return(s);
}

int getlong(filep)
     FILE *filep;
{
  int l;
  l = getc(filep) & 0xff;
  l += (getc(filep) & 0xff) <<8;
  l += (getc(filep) & 0xff) <<16;
  l += (getc(filep) & 0xff) <<24;
  return(l);
}

void putshort(d, filep)
     short d;
     FILE *filep;
{
  putc(d & 0xff, filep);
  putc((d & 0xff00) >> 8, filep);  
}

void putlong(d, filep)
     long d;
     FILE *filep;
{
  putc(d & 0xff, filep);
  putc((d & 0xff00) >> 8, filep);
  putc((d & 0xff0000) >> 16, filep);
  putc((d & 0xff000000) >> 24, filep);
}


void allowcomment(infile)
     FILE *infile;
{
  while (ungetc(getc(infile),infile) == '#')
    while (getc(infile) != '\n');
}


int initpnames(filename, pname)
     char *filename;
     char pname[][8];
{
  FILE *pnfile;
  int n;
  pnfile = fopen(filename, "r");
  if(pnfile==NULL)
    {
      fprintf(stderr, "Unable to open data file %s.\n", filename);
      exit(-1);
    }
  n = getlong(pnfile);
  fread(pname, 8, n, pnfile);
  fclose(pnfile);
  return(n);
}


int inittexture(filename, tsize, reslength, datafile)
     char *filename;
     long tsize[];
     int *reslength;
     FILE *datafile;
{
  FILE *tfile;
  int numtextures, i, j, n;
  char tname[8];

  tfile = fopen(filename, "r");
  if(tfile==NULL)
    {
      fprintf(stderr, "Unable to open data file %s.\n", filename);
      exit(-1);
    }
  numtextures = getlong(tfile);
  for(i=0; i<numtextures; i++)
    getlong(tfile); /* Skip the pointer table - it is rebuilt later */
  for(i=0; i<numtextures; i++)
    {
      fread(tname, 8, 1, tfile);
      fwrite(tname, 8, 1, datafile);
      putshort(getshort(tfile), datafile);
      putshort(getshort(tfile), datafile);
      putshort(getshort(tfile), datafile); /* Width */
      putshort(getshort(tfile), datafile); /* Height */
      putlong(getlong(tfile), datafile);
      putshort(n = getshort(tfile), datafile); /* Numpatches */
      for(j=0; j<n; j++)
	{
	  putshort(getshort(tfile), datafile); /* X offset */
	  putshort(getshort(tfile), datafile); /* Y offset */
	  putshort(getshort(tfile), datafile); /* Patch index */
	  putshort(getshort(tfile), datafile);
	  putshort(getshort(tfile), datafile);
	}
      tsize[i] = 22 + 10 * n;
      *reslength += tsize[i] + 4;
    }
  return numtextures;
}


main(argc, argv)
     int argc;
     char *argv[];
{
  FILE *infile, *wadfile;
  FILE *dirfile1, *dirfile2, *datafile;
  char keyword[20];
  char spritename[20], patchname[20], texturename[20], tilename[20];
  char ppmfilename[80], lastppmfilename[80];
  short *imagedata;
  int imgw, imgh;
  int ox, oy, w, h, x, y;
  unsigned char *doomimage;
  int reslength, ackreslength, numdirentries;
  int i, j;
  int lastsprite;
  char pname[1024][8]; /* For PNAMES index lookup entry */
  int texturepatches, numpatches;
  long t1size[512]; /* Temp buffer for TEXTURE1 sizes */
  long t2size[512]; /* Temp buffer for TEXTURE2 sizes */
  int numt1, numt2, ackpos;
  char entryname[9], entryname8[8];
  int entrysize, entrypos;
  int spritesection, patchsection, texture1section, texture2section;
  int flat1section, flat2section;

  if (argc != 3)
    {
      fprintf(stderr,"Usage: %s infile outfile\n", argv[0]);
      exit(-1);
    }
  infile=fopen(argv[1],"r");
  if(infile==NULL)
    {
      fprintf(stderr,"Unable to open input file %s\n", argv[1]);
      exit(-1);
    }
  wadfile=fopen(argv[2],"w");
  if(wadfile==NULL)
    {
      fprintf(stderr,"Unable to create output file %s\n", argv[2]);
      exit(-1);
    }
  dirfile1=fopen("temp.d1","w"); /* Directory for TEXTURE1&2 and PNAMES */
  if(dirfile1==NULL)
    {
      fprintf(stderr,"Unable to create temporary file temp.d1\n");
      exit(-1);
    }
  dirfile2=fopen("temp.d2","w"); /* Directory from S_START to F_END */
  if(dirfile2==NULL)
    {
      fprintf(stderr,"Unable to create temporary file temp.d2\n");
      exit(-1);
    }
  lastppmfilename[0] = '\0';
  imagedata = NULL;
  /* No dynamic allocation - just make room for a full 320x200 image */
  doomimage = (unsigned char *)malloc(68168*sizeof(char));
  ackreslength = 0;
  numdirentries = 0;

  /* Sort the palette to speed up the image input */
  for(i=0; i<256; i++)
    {
      sortedpal[i][0]=doompal[i][0];
      sortedpal[i][1]=doompal[i][1];
      sortedpal[i][2]=doompal[i][2];
      sortedpal[i][3]=i;
    }      
  qsort((char*)sortedpal, 256, 4, comparecols);

  spritesection = FALSE;
  patchsection = FALSE;
  texture1section = FALSE;
  texture2section = FALSE;
  flat1section = FALSE;
  flat2section = FALSE;

  allowcomment(infile);
  fscanf(infile, " %s ", keyword);

  if(!strcmp(keyword,"S_START"))
    {
      spritesection = TRUE;
      datafile=fopen("temp.s","w");
      if(datafile==NULL)
	{
	  fprintf(stderr,"Unable to create temporary file temp.s\n");
	  exit(-1);
	}
      fprintf(dirfile2, "S_START 0\n");
      numdirentries +=1;
      /* Process sprites until S_END is found */
      lastsprite = 0;
      while(!lastsprite)
	{
	  allowcomment(infile);
	  fscanf(infile, " %s ", spritename);
	  if(!strcmp(spritename,"S_END")) lastsprite = 1;
	  else
	    {
	      printf("Creating sprite %s... \n", spritename);
	      fscanf(infile, " %s @ %d %d W %d H %d X %d Y %d ",
		     ppmfilename, &ox, &oy, &w, &h, &x, &y);
	      allowcomment(infile);
	      /* If the ppm file is not the same as before: read it in */
	      if(strcmp(ppmfilename, lastppmfilename))
		{
		  readppmfile(ppmfilename, &imagedata, &imgw, &imgh);
		  strcpy(lastppmfilename, ppmfilename);
		}
	      /* Clip out the specified region and convert it */
	      rawtodoom(imagedata, imgw, imgh, ox, oy, w, h, x, y,
			doomimage, &reslength);
	      /* Write data to datafile and some info to dirfile2 */
	      for(i=0; i<reslength; i++)
		putc(doomimage[i], datafile);
	      fprintf(dirfile2, "%s %d\n", spritename, reslength);
	      ackreslength += reslength;
	      numdirentries +=1;
	    }
	}
      fprintf(dirfile2, "S_END 0\n");
      numdirentries +=1;
      fclose(datafile);
      fscanf(infile, " %s ", keyword);
    }

  if(!strcmp(keyword,"P_START"))
    {
      patchsection = TRUE;
      /* Initialize PNAMES with all the original entries */
      numpatches = initpnames(PNFILENAME, pname);
      datafile=fopen("temp.p","w");
      if(datafile==NULL)
	{
	  fprintf(stderr,"Unable to create temporary file temp.p\n");
	  exit(-1);
	}
      /* Read patches one by one, updating pname[] and numpatches */
      lastsprite = 0;
      while(!lastsprite)
	{
	  allowcomment(infile);
	  fscanf(infile, " %s ", spritename);
	  if(!strcmp(spritename,"P_END")) lastsprite = 1;
	  else
	    {
	      printf("Creating wall patch %s... \n", spritename);
	      strncpy(pname[numpatches++], spritename, 8);
	      fscanf(infile, " %s @ %d %d W %d H %d ",
		     ppmfilename, &ox, &oy, &w, &h);
	      allowcomment(infile);
	      /* If the ppm file is not the same as before: read it in */
	      if(strcmp(ppmfilename, lastppmfilename))
		{
		  readppmfile(ppmfilename, &imagedata, &imgw, &imgh);
		  strcpy(lastppmfilename, ppmfilename);
		}
	      /* Clip out the specified region and convert it */
	      rawtodoom(imagedata, imgw, imgh, ox, oy, w, h, w/2-1, h,
			doomimage, &reslength);
	      /* Write data to datafile and some info to dirfile2 */
	      for(i=0; i<reslength; i++)
		putc(doomimage[i], datafile);
	      fprintf(dirfile2, "%s %d\n", spritename, reslength);
	      ackreslength += reslength;
	      numdirentries +=1;
	    }
	}
      fclose(datafile);
      allowcomment(infile);
      fscanf(infile, " %s ", keyword);
    }  
  

  numt1 = 0;
  if(!strcmp(keyword,"TEXTURE1_START"))
    {
      texture1section = TRUE;
      datafile=fopen("temp.t1","w");
      reslength = 4;
      /* Init the structure with all the original entries */
      numt1 = inittexture(T1FILENAME, t1size, &reslength, datafile);
      /* Read the textures, updating t1size, numt1 and reslength */
      allowcomment(infile);
      fscanf(infile, " %s ", keyword);
      while(strcmp(keyword, "TEXTURE1_END"))
	{
	  printf("Creating texture %s... \n", keyword);
	  fscanf(infile, " %d %d %d ", &w, &h, &texturepatches);
	  allowcomment(infile);
	  strncpy(entryname8, keyword, 8);
	  fwrite(entryname8, 8, 1, datafile);
	  putshort(0, datafile);
	  putshort(0, datafile);
	  putshort(w, datafile);
	  putshort(h, datafile);
	  putlong(0, datafile);
	  putshort(texturepatches, datafile);
	  for (i=0; i< texturepatches; i++)
	    {
	      fscanf(infile, " %s %d %d ", patchname, &x, &y);
	      allowcomment(infile);
	      j = findpatch(patchname, numpatches, pname);
	      putshort(x, datafile);
	      putshort(y, datafile);
	      putshort(j, datafile);
	      putshort(0, datafile);
	      putshort(0, datafile);
	    }
	  t1size[numt1++] = 22 + 10 * texturepatches;
	  reslength += t1size[numt1-1] + 4;
	  fscanf(infile, " %s ", keyword);
	}
      fprintf(dirfile1, "TEXTURE1 %d\n", reslength);
      numdirentries +=1;
      ackreslength += reslength;
      fclose(datafile);
      allowcomment(infile);
      fscanf(infile, " %s ", keyword);
    }
  
  numt2 = 0;
  if(!strcmp(keyword,"TEXTURE2_START"))
    {
      texture2section = TRUE;
      datafile=fopen("temp.t2","w");
      reslength = 4;
      /* Init the structure with all the original entries */
      numt2 = inittexture(T2FILENAME, t2size, &reslength, datafile);
      /* Read the textures, updating t2size, numt2 and reslength */
      allowcomment(infile);
      fscanf(infile, " %s ", keyword);
      while(strcmp(keyword, "TEXTURE2_END"))
	{
	  printf("Creating texture %s... \n", keyword);
	  fscanf(infile, " %d %d %d ", &w, &h, &texturepatches);
	  allowcomment(infile);
	  strncpy(entryname8, keyword, 8);
	  fwrite(entryname8, 8, 1, datafile);
	  putshort(0, datafile);
	  putshort(0, datafile);
	  putshort(w, datafile);
	  putshort(h, datafile);
	  putlong(0, datafile);
	  putshort(texturepatches, datafile);
	  for (i=0; i< texturepatches; i++)
	    {
	      fscanf(infile, " %s %d %d ", patchname, &x, &y);
	      allowcomment(infile);
	      j = findpatch(patchname, numpatches, pname);
	      putshort(x, datafile);
	      putshort(y, datafile);
	      putshort(j, datafile);
	      putshort(0, datafile);
	      putshort(0, datafile);
	    }
	  t2size[numt1++] = 22 + 10 * texturepatches;
	  reslength += t2size[numt2-1] + 4;
	  fscanf(infile, " %s ", keyword);
	}
      fprintf(dirfile1, "TEXTURE2 %d\n", reslength);
      numdirentries +=1;
      ackreslength += reslength;
      fclose(datafile);
      allowcomment(infile);
      fscanf(infile, " %s ", keyword);
    }
  
  if (patchsection)
    {
      fprintf(dirfile1, "PNAMES %d\n", numpatches*8+4);
      numdirentries +=1;
      ackreslength += (numpatches*8+4);
    }
  fclose(dirfile1);
  
  if(!strcmp(keyword,"F_START"))
    {
      fprintf(dirfile2, "F_START 0\n");
      numdirentries +=1;
      allowcomment(infile);
      fscanf(infile, " %s ", keyword);
      if(!strcmp(keyword,"F1_START"))
	{
	  allowcomment(infile);
	  flat1section = TRUE;
	  fprintf(dirfile2, "F1_START 0\n");
	  numdirentries +=1;
	  datafile=fopen("temp.f1","w");
	  /* Read the floor tiles one by one, writing names to dirfile2 */
	  fscanf(infile, " %s ", keyword);
	  while(strcmp(keyword, "F1_END"))
	    {
	      printf("Creating floor/ceiling %s... \n", keyword);
	      fscanf(infile, " %s @ %d %d ", ppmfilename, &ox, &oy);
	      allowcomment(infile);
	      /* If the ppm file is not the same as before: read it in */
	      if(strcmp(ppmfilename, lastppmfilename))
		{
		  readppmfile(ppmfilename, &imagedata, &imgw, &imgh);
		  strcpy(lastppmfilename, ppmfilename);
		}
	      /* Write out the specified region in raw format */
	      for (i=0; i<64; i++)
		for(j=0; j<64; j++)
		  putc(imagedata[(oy+i)*imgw+ox+j], datafile);
	      /* Write some info to dirfile2 */
	      fprintf(dirfile2, "%s %d\n", keyword, 4096);
	      ackreslength += 4096;
	      numdirentries +=1;
	      fscanf(infile, " %s ", keyword);
	    }
	  fprintf(dirfile2, "F1_END 0\n");
	  numdirentries +=1;
	  fclose(datafile);
	  allowcomment(infile);
	  fscanf(infile, " %s ", keyword);
	}
      if(!strcmp(keyword,"F2_START"))
	{
	  allowcomment(infile);
	  flat2section = TRUE;
	  fprintf(dirfile2, "F2_START 0\n");
	  numdirentries +=1;
	  datafile=fopen("temp.f2","w");
	  /* Read the floor tiles one by one, writing names to dirfile2 */
	  fscanf(infile, " %s ", keyword);
	  while(strcmp(keyword, "F2_END"))
	    {
	      printf("Creating floor/ceiling %s... \n", keyword);
	      fscanf(infile, " %s @ %d %d ", ppmfilename, &ox, &oy);
	      allowcomment(infile);
	      /* If the ppm file is not the same as before: read it in */
	      if(strcmp(ppmfilename, lastppmfilename))
		{
		  readppmfile(ppmfilename, &imagedata, &imgw, &imgh);
		  strcpy(lastppmfilename, ppmfilename);
		}
	      /* Write out the specified region in raw format */
	      for (i=0; i<64; i++)
		for(j=0; j<64; j++)
		  putc(imagedata[(oy+i)*imgw+ox+j], datafile);
	      /* Write some info to dirfile2 */
	      fprintf(dirfile2, "%s %d\n", keyword, 4096);
	      ackreslength += 4096;
	      numdirentries +=1;
	      allowcomment(infile);
	      fscanf(infile, " %s ", keyword);
	    }
	  fprintf(dirfile2, "F2_END 0\n");
	  numdirentries +=1;
	  fclose(datafile);
	  allowcomment(infile);
	  fscanf(infile, " %s ", keyword);
	}
      fprintf(dirfile2, "F_END 0\n");
      numdirentries +=1;
      allowcomment(infile);
      fscanf(infile, " %s ", keyword);
    }
  fclose(infile);
  fclose(dirfile2);
  free(doomimage);

  /* Concatenate all the data into the final PWAD file */
  printf("Creating WAD file %s... \n", argv[2]);

  /* Header */
  fprintf(wadfile,"PWAD");
  putlong(numdirentries, wadfile);
  ackreslength += 12; /* The header is 12 bytes */
  putlong(ackreslength, wadfile);

  /* TEXTURE1 */
  if(texture1section)
    {
      printf("Adding entry TEXTURE1\n");
      putlong(numt1, wadfile);
      ackpos = 4 + numt1*4;
      for(i=0; i<numt1; i++)
	{
	  putlong(ackpos, wadfile);
	  ackpos += t1size[i];
	}
      appenddata(wadfile, "temp.t1");
    }

  /* TEXTURE2 */
  if(texture2section)
    {
      printf("Adding entry TEXTURE2\n");
      putlong(numt2, wadfile);
      ackpos = 4 + numt2*4;
      for(i=0; i<numt2; i++)
	{
	  putlong(ackpos, wadfile);
	  ackpos += t2size[i];
	}
      appenddata(wadfile, "temp.t2");
    }

  /* PNAMES */
  if(patchsection)
    {
      printf("Adding entry PNAMES\n");
      putlong(numpatches, wadfile);
      fwrite(&pname[0][0], 8, numpatches, wadfile);
    }
  
  /* Sprites */
  if(spritesection)
    {
      printf("Adding sprite entries\n");
      appenddata(wadfile, "temp.s");
    }

  /* Wall patches */
  if(patchsection)
    {
      printf("Adding wall patch entries\n");
      appenddata(wadfile, "temp.p");
    }

  /* Floor tiles */
  if (flat1section)
    appenddata(wadfile, "temp.f1");
  if (flat2section)
    appenddata(wadfile, "temp.f2");
  
  /* Now, write the directory by using data from the directory files. */
  entrypos = 12;
  dirfile1 = fopen("temp.d1","r");
  if(dirfile1 != NULL)
    {
      while(!feof(dirfile1))
	{
	  fscanf(dirfile1, " %s %d", entryname, &entrysize);
	  if (!feof(dirfile1))
	    {
	      putlong(entrypos, wadfile);
	      putlong(entrysize, wadfile);
	      strncpy(entryname8, entryname, 8);
	      fwrite(entryname8, 8, 1, wadfile);
	      entrypos += entrysize;
	    }
	}
      fclose(dirfile1);
    }
  dirfile2 = fopen("temp.d2","r");
  if(dirfile2 != NULL)
    {
      while(!feof(dirfile2))
	{
	  fscanf(dirfile2, " %s %d", entryname, &entrysize);
	  if (!feof(dirfile2))
	    {
	      putlong(entrypos, wadfile);
	      putlong(entrysize, wadfile);
	      strncpy(entryname8, entryname, 8);
	      fwrite(entryname8, 8, 1, wadfile);
	      entrypos += entrysize;
	    }
	}
      fclose(dirfile2);
    }

  /* DONE! */
  printf("WAD file %s complete.\n", argv[2]);
  fclose(wadfile);
}





