/*
   DooM PostScript Maps Utility, by Frans P. de Vries.

Derived from:

   Doom Editor Utility, by Brendon Wyber and Raphal Quinet.

   You are allowed to use any parts of this code in another program, as
   long as you give credits to the authors in the documentation and in
   the program itself.  Read the file README for more information.

   This program comes with absolutely no warranty.

   PRINT.C - Print and Analyze routines.
*/


/* the includes */
#include "dmps.h"
#include "levels.h"
#include "things.h"


/* global variables */
				/* if TRUE ('+');        if FALSE ('-')        */
Bool FlagDoublePg = FALSE;	/* print double page;    print normal page     */
Bool FlagQuadplPg = FALSE;	/* print quadruple page; print normal page     */
Bool FlagA4Letter = TRUE;	/* print A4 paper;       print Letter paper    */
Bool FlagBorder   = TRUE;	/* print border;         print no border       */
Bool FlagDecors   = TRUE;	/* print decors;         print no decors       */
Bool FlagEnemies  = TRUE;	/* print enemies;        print no enemies      */
Bool FlagFadeLins = TRUE;	/* fade secret lines;    don't fade sec.lines  */
Bool FlagGoodies  = TRUE;	/* print goodies;        print no goodies      */
Bool FlagLegend   = TRUE;	/* print thing legend;   print no thing legend */
Bool FlagMultPlay = FALSE;	/* print multi things;   print no multi things */
Bool FlagName     = TRUE;	/* print level name;     print no level name   */
Bool FlagPortrait = TRUE;	/* print portrait;       print landscape       */
Bool FlagShdAreas = TRUE;	/* shade secret areas;   don't shade sec.areas */
Bool FlagTeleprts = TRUE;	/* link teleports;       don't link teleports  */
Bool FlagUltraVlc = TRUE;	/* print UV things;      print non-UV things   */
Bool FlagAutoZoom = TRUE;	/* auto-zoom map scale;  default map scale     */

BCINT MapCenterX;		/* X coord of map center */
BCINT MapCenterY;		/* Y coord of map center */


/* map attribute definitions */
#define L_IMPASS 0x01		/* impassible linedef */
#define L_SECRET 0x20		/* secret linedef */
#define L_TELEP1 39		/* one-time teleport linedef */
#define L_TELEPR 97		/* repeatable teleport linedef */
#define L_TELPM1 125		/* one-time monster-only teleport linedef */
#define L_TELPMR 126		/* repeatable monster-only teleport linedef */
#define S_SECRET 9		/* secret sector */
#define T_MULTI  0x10		/* multi-player thing */
#define T_ULTRA  0x04		/* ultra-violence thing */
#define T_NONUV  0x03		/* non ultra-violence thing */


/*
   read level & print its map
*/
void PrintLevel( BCINT episode, BCINT mission)
{
   ReadLevelData( episode, mission);
   /* define map center */
   MapCenterX = (MapMinX + MapMaxX) / 2;
   MapCenterY = (MapMinY + MapMaxY) / 2;

   /* initialize, scale & draw the page */
   InitPage( episode, mission, ( UserLvlNm != NULL ? UserLvlNm : LevelName ));

   InitScale( MapMaxX - MapMinX, MapMaxY - MapMinY,
	      ( UserLvlNm != NULL ? UserLvlNm : LevelName ));
   if (FlagAutoZoom && (FlagName || FlagLegend))
   {
      BCINT n;
      for (n = 0; n < NumVertexes; n++)
	 /* rough & fine check for covering name or legend areas */
	 if (Vertexes[ n].y > MapCenterY)
	    if (CheckScale( Vertexes[ n].x, Vertexes[ n].y))
	       break;
      if (n < NumVertexes)
	 AdjustScale( MapMaxX - MapMinX, MapMaxY - MapMinY);
   }

   PrintMap();
   TermPage();

   /* clean up & free space */
   ForgetLevelData();
}


/*
   draw the actual game map
*/
void PrintMap( void)
{
   BCINT n, m;

   /* draw the shaded secret Sectors if desired */
   if (FlagShdAreas)
   {
      BCINT l, end;
      BCINT lines[ 100];

      InitShades();

      /* find all secret Sectors */
      for (n = 0; n < NumSectors; n++)
	 if (Sectors[ n].special == S_SECRET)
	 {
	    end = 0;

	    /* build list of all LineDefs that make up the Sector */
	    for (m = 0; m < NumLineDefs; m++)
	    {
	       l = SideDefs[ LineDefs[ m].sidedef1].sector;
	       if (l != n && LineDefs[ m].sidedef2 != -1)
		  l = SideDefs[ LineDefs[ m].sidedef2].sector;
	       if (l == n)
		  if (end < 100) /* safety check (E3M5 sector 128 has 46 lines) */
		     lines[ end++] = m;
		  else
		  {
		     printf( "[Secret sector %d has too many lines for shading.]\n", n);
		     break;
		  }
	    }
	    if (m < NumLineDefs) /* extend above safety check */
	       break;

	    /* find LineDef with the most lower-left start Vertex */
	    l = 0;
	    for (m = 1; m < end; m++)
	       if ((Vertexes[ LineDefs[ lines[ l]].start].y > Vertexes[ LineDefs[ lines[ m]].start].y) ||
	           (Vertexes[ LineDefs[ lines[ l]].start].y == Vertexes[ LineDefs[ lines[ m]].start].y &&
		    Vertexes[ LineDefs[ lines[ l]].start].x > Vertexes[ LineDefs[ lines[ m]].start].x))
		  l = m;
	    /* put it at front (makes it most probable that list starts with outer
	       Sector); saves elaborate checks on which Sector contains which others */
	    if (l != 0)
	       exch( lines[ 0], lines[ l]);

	    /* sort list for consecutive Vertices on outer and any inner Sectors
	       (eg. E2M1 sector 3 has 4 inner ones), negating "reverse" LineDefs */
	    for (m = 0; m < end - 1; m++)
	    {
	       for (l = m + 1; l < end; l++)
		  if (lines[ m] >= 0)
		  {
		     if (LineDefs[ lines[ m]].end == LineDefs[ lines[ l]].start)
			break;
		     if (LineDefs[ lines[ m]].end == LineDefs[ lines[ l]].end)
		     {
			lines[ l] = - lines[ l];
			break;
		     }
		  }
		  else /* lines[ m] < 0 */
		  {
		     if (LineDefs[ - lines[ m]].start == LineDefs[ lines[ l]].start)
			break;
		     if (LineDefs[ - lines[ m]].start == LineDefs[ lines[ l]].end)
		     {
			lines[ l] = - lines[ l];
			break;
		     }
		  }

	       /* put next LineDef behind current one */
	       if (l < end)
		  exch( lines[ m + 1], lines[ l]);
	    }

	    /* shade the secret Sector */
	    PrintShade( 'B', Vertexes[ LineDefs[ lines[ 0]].start].x,
			     Vertexes[ LineDefs[ lines[ 0]].start].y);
	    for (m = 0; m < end; m++)
	       if (lines[ m] >= 0)
		  if (LineDefs[ lines[ 0]].start == LineDefs[ lines[ m]].end)
		     break;
		  else
		     PrintShade( 'N', Vertexes[ LineDefs[ lines[ m]].end].x,
				      Vertexes[ LineDefs[ lines[ m]].end].y);
	       else /* lines[ m] < 0 */
		  if (LineDefs[ lines[ 0]].start == LineDefs[ - lines[ m]].start)
		     break;
		  else
		     PrintShade( 'N', Vertexes[ LineDefs[ - lines[ m]].start].x,
				      Vertexes[ LineDefs[ - lines[ m]].start].y);
	    PrintShade ('S', 0, 0);

	    /* un-shade any inner Sectors of the secret one */
	    l = m + 1;
	    while (l < end)
	    {
	       PrintShade( 'B', Vertexes[ LineDefs[ lines[ l]].start].x,
				Vertexes[ LineDefs[ lines[ l]].start].y);
	       m = l;
	       while (m < end)
	       {
	 	  if (lines[ m] >= 0)
		     if (LineDefs[ lines[ l]].start == LineDefs[ lines[ m]].end)
			break;
		     else
			PrintShade( 'N', Vertexes[ LineDefs[ lines[ m]].end].x,
				 	 Vertexes[ LineDefs[ lines[ m]].end].y);
		  else /* lines[ m] < 0 */
		     if (LineDefs[ lines[ l]].start == LineDefs[ - lines[ m]].start)
			break;
		     else
			PrintShade( 'N', Vertexes[ LineDefs[ - lines[ m]].start].x,
				 	 Vertexes[ LineDefs[ - lines[ m]].start].y);
		  m++;
	       }
	       PrintShade ('W', 0, 0);
	       l = m + 1;
	    }
	}
   }

   /* draw the LineDefs to form the map */
   InitLines();

   for (n = 0; n < NumLineDefs; n++)
      if (FlagFadeLins)
      {
	 /* check if Sector on either side of LineDef is secret */
	 m = Sectors[ SideDefs[ LineDefs[ n].sidedef1].sector].special;
	 if (m != S_SECRET && LineDefs[ n].sidedef2 != -1)
	    m = Sectors[ SideDefs[ LineDefs[ n].sidedef2].sector].special;

	 PrintMapLine( Vertexes[ LineDefs[ n].start].x, Vertexes[ LineDefs[ n].start].y,
		       Vertexes[ LineDefs[ n].end].x,   Vertexes[ LineDefs[ n].end].y,
		       (LineDefs[ n].flags & L_IMPASS),
		       (LineDefs[ n].flags & L_SECRET) || m == S_SECRET);
      }
      else
	 PrintMapLine( Vertexes[ LineDefs[ n].start].x, Vertexes[ LineDefs[ n].start].y,
		       Vertexes[ LineDefs[ n].end].x,   Vertexes[ LineDefs[ n].end].y,
		       (LineDefs[ n].flags & L_IMPASS), FALSE);

   /* draw all desired Things */
   InitThings();

   for (n = 0; n < NumThings; n++)
      /* check whether multi-player Thing should be printed */
      if (FlagMultPlay || ! (Things[ n].when & T_MULTI))
	 /* check whether UV or non-UV Thing should be printed */
	 if ((FlagUltraVlc && (Things[ n].when & T_ULTRA)) ||
	     (! FlagUltraVlc && (Things[ n].when & T_NONUV)))
	 {
	    m = Things[ n].type;
	    switch( GetThingClass( m))
	    {
	    case CLASS_START:
	       /* always print Player 1 Start, others only if multi-player Things */
	       if (FlagMultPlay || m == THING_PLAYER1)
		  break;
	       else
		  continue;
	    case CLASS_ENHANCE:
	       if (FlagGoodies)
		  break;
	       else
		  continue;
	    case CLASS_BONUS:
	       if (FlagGoodies)
		  break;
	       else
		  continue;
	    case CLASS_WEAPON:
	       if (FlagGoodies)
		  break;
	       else
		  continue;
	    case CLASS_ENEMY:
	       if (FlagEnemies)
		  break;
	       else
		  continue;
	    case CLASS_DECOR:
	       if (FlagDecors)
		  break;
	       else
		  continue;
	    case CLASS_BARREL:
	       if (FlagDecors)
		  break;
	       else
		  continue;
	    case CLASS_TELEPORT:
	       /* always print Teleport Exits */
	       break;
	    default:
	       break;
	    }

	    PrintMapThing( GetThingClass( m), Things[ n].xpos, Things[ n].ypos,
			   GetThingRadius( m), Things[ n].angle);
	 }

   /* draw the Teleport links if desired */
   if (FlagTeleprts)
   {
      BCINT t;

      InitLinks();

      /* find all (incl. monster-only) teleporting LineDefs */
      for (n = 0; n < NumLineDefs; n++)
	 if (LineDefs[ n].type == L_TELEP1 || LineDefs[ n].type == L_TELEPR ||
	     LineDefs[ n].type == L_TELPM1 || LineDefs[ n].type == L_TELPMR)
	 {
	    /* find associated Sector */
	    for (m = 0; m < NumSectors; m++)
	       if (Sectors[ m].tag == LineDefs[ n].tag)
		  if ((t = FindTeleExit( m)) != NumThings)
		  {
		     /* draw link from LineDef midpoint to teleport exit center */
		     PrintTeleLink( (Vertexes[ LineDefs[ n].start].x + Vertexes[ LineDefs[ n].end].x) / 2,
				    (Vertexes[ LineDefs[ n].start].y + Vertexes[ LineDefs[ n].end].y) / 2,
				    Things[ t].xpos, Things[ t].ypos);
		     break;
		  }

	    if (m == NumSectors) /* safety check */
	       printf( "[No sector tagged %d found with teleport exit.]\n", LineDefs[ n].tag);
	 }
   }
}


/*
   find a Teleport Exit inside the given Sector
*/
BCINT FindTeleExit( BCINT sector)
{
   BCINT n, s, t;
   BCINT ll, lr, ul, ur;         /* lower/upper left/right quadrants */
   BCINT x0, y0, x1, y1, x2, y2; /* teleport exit point & LineDef end points */

   /* find all teleport exits */
   for (t = 0; t < NumThings; t++)
      if (Things[ t].type == THING_TELEPORT)
      {
	 ll = 0; lr = 0; ul = 0; ur = 0;
	 x0 = Things[ t].xpos;
	 y0 = Things[ t].ypos;

	 /* find all LineDefs that make up the Sector */
	 for (n = 0; n < NumLineDefs; n++)
	 {
	    s = SideDefs[ LineDefs[ n].sidedef1].sector;
	    if (s != sector && LineDefs[ n].sidedef2 != -1)
	       s = SideDefs[ LineDefs[ n].sidedef2].sector;
	    if (s == sector)
	    {
	       x1 = Vertexes[ LineDefs[ n].start].x;
	       y1 = Vertexes[ LineDefs[ n].start].y;
	       x2 = Vertexes[ LineDefs[ n].end].x;
	       y2 = Vertexes[ LineDefs[ n].end].y;

	       /* determine position of Vertices relative to teleport exit */
	       /* don't swap < & >= for <= & > : the telelinks inside the mid-east
		  building on MAP13 will be misconnected; all the more indication
		  that this whole algorithm sucks, but it's the only one I know
		  that works at least for all DOOM and most DOOM ][ maps */
	       ll += (x0 >= x1) && (y0 >= y1);
	       lr += (x0 <  x1) && (y0 >= y1);
	       ul += (x0 >= x1) && (y0 <  y1);
	       ur += (x0 <  x1) && (y0 <  y1);
	       ll += (x0 >= x2) && (y0 >= y2);
	       lr += (x0 <  x2) && (y0 >= y2);
	       ul += (x0 >= x2) && (y0 <  y2);
	       ur += (x0 <  x2) && (y0 <  y2);
	    }
	 }
	 /* check all 4 quadrants whether teleport exit is surrounded by LineDefs */
	 if (ll > 0 && lr > 0 && ul > 0 && ur > 0)
	    break;

	 /* check (by diagonal opposites) whether it's worth computing perpendicular
	    intersects; actually, finding the sector's linedefs again might be less
	    efficient than computing the intersects everytime, but this stuff really
	    came in as an afterthought because of 3 teleport links in E3M6 & E3M9 */
	 if (ll * ur > 0 || lr * ul > 0)
	 {
	    /* find all LineDefs that make up the Sector */
	    for (n = 0; n < NumLineDefs; n++)
	    {
	       s = SideDefs[ LineDefs[ n].sidedef1].sector;
	       if (s != sector && LineDefs[ n].sidedef2 != -1)
		  s = SideDefs[ LineDefs[ n].sidedef2].sector;
	       if (s == sector)
	       {
		  BCINT xi, yi;     /* intersect point */
		  BCINT A, B, C, D; /* equation coefficients */
		  long  E, F, det;  /* & determinant */

		  x1 = Vertexes[ LineDefs[ n].start].x;
		  y1 = Vertexes[ LineDefs[ n].start].y;
		  x2 = Vertexes[ LineDefs[ n].end].x;
		  y2 = Vertexes[ LineDefs[ n].end].y;

		  /* a little geometry of similar triangles results in the following
		     relations for computing the intersect point of the LineDef and
		     its perpendicular going through the teleport exit:

			y2 - y1   xi - x0       yi - y1   y2 - yi
			------- = -------  and  ------- = -------
			x2 - x1   y0 - yi       xi - x1   x2 - xi

		     this can be rewritten into 2 equations in xi and yi, which
		     can then easily be solved for:

			(y1 - y2) * xi + (x2 - x1) * yi = x2 * y1 - x1 * y2
			(x2 - x1) * xi + (y2 - y1) * yi = x0 * (x2-x1) - y0 * (y2-y1)
		  */
		  A = y1 - y2; D = -A;
		  B = x2 - x1; C = B;
		  E = x2 * y1 - x1 * y2;
		  F = x0 * C  + y0 * D;
		  if ((det = A * D - B * C) != 0)
		  {
		     xi = ( D * E - C * F ) / det;
		     yi = ( A * F - B * E ) / det;

		     /* check if intersect is outside LineDef */
		     if ((xi < x1 && xi < x2) || (xi > x1 && xi > x2) ||
			 (yi < y1 && yi < y2) || (yi > y1 && yi > y2))
		     {
			xi = x0; /* set dummy intersect not in any quadrant */
			yi = y0;
		     }
		  }
		  else /* avoiding the unlikely but dreaded division by zero... */
		  {
		     xi = x0; /* ditto */
		     yi = y0;
		  }

		  /* determine position of intersect relative to teleport exit */
		  ll += (x0 > xi) && (y0 > yi);
		  lr += (x0 < xi) && (y0 > yi);
		  ul += (x0 > xi) && (y0 < yi);
		  ur += (x0 < xi) && (y0 < yi);
	       }
	    }
	    /* check all 4 quadrants again for surrounding perpendicular intersects */
	    if (ll > 0 && lr > 0 && ul > 0 && ur > 0)
	       break;
	 }
      }
   return t;
}


/*
   display all print flags
*/
void DisplayFlags( void)
{
   printf( "Print flags:\n");
   printf( "-----------\n");
   printf( "%s\n", ( FlagDoublePg ? "+2\tuse double page"
				  : "-2\tuse normal page" ));
   printf( "%s\n", ( FlagQuadplPg ? "+4\tuse quadruple page"
				  : "-4\tuse normal page" ));
   printf( "%s\n", ( FlagA4Letter ? "+A\tuse A4 paper size"
				  : "-A\tuse Letter paper size" ));
   printf( "%s\n", ( FlagBorder   ? "+B\tprint border around map"
				  : "-B\tprint no border around map" ));
   printf( "%s\n", ( FlagDecors   ? "+D\tprint all decorations & barrels"
				  : "-D\tprint no decorations & barrels" ));
   printf( "%s\n", ( FlagEnemies  ? "+E\tprint all enemies"
				  : "-E\tprint no enemies" ));
   printf( "%s\n", ( FlagFadeLins ? "+F\tfade secret lines"
				  : "-F\tdon't fade secret lines" ));
   printf( "%s\n", ( FlagGoodies  ? "+G\tprint all goodies (weapons/bonuses/enhancements)"
				  : "-G\tprint no goodies (weapons/bonuses/enhancements)" ));
   printf( "%s\n", ( FlagLegend   ? "+L\tprint legend above map"
				  : "-L\tprint no legend above map" ));
   printf( "%s\n", ( FlagMultPlay ? "+M\tprint multi-player things"
				  : "-M\tprint no multi-player things" ));
   printf( "%s\n", ( FlagName     ? "+N\tprint name above map"
				  : "-N\tprint no name above map" ));
   printf( "%s\n", ( FlagPortrait ? "+P\tuse Portrait orientation"
				  : "-P\tuse Landscape orientation" ));
   printf( "%s\n", ( FlagShdAreas ? "+S\tshade secret areas"
				  : "-S\tdon't shade secret areas" ));
   printf( "%s\n", ( FlagTeleprts ? "+T\tlink teleports"
				  : "-T\tdon't link teleports" ));
   printf( "%s\n", ( FlagUltraVlc ? "+U\tprint things for Ultra-Violence skill"
				  : "-U\tprint things below Ultra-Violence skill" ));
   printf( "%s\n", ( FlagAutoZoom ? "+Z\tauto-zoom map scale"
				  : "-Z\tdefault map scale" ));
}


/*
   set a print flag
*/
void SetFlag( char flag, char val)
{
   switch( flag)
   {
   case '2':
      FlagDoublePg = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagDoublePg ? "+2\tuse double page"
				     : "-2\tuse normal page" ));
      if (FlagDoublePg && FlagQuadplPg)
	 SetFlag( '4', '-');
      break;
   case '4':
      FlagQuadplPg = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagQuadplPg ? "+4\tuse quadruple page"
				     : "-4\tuse normal page" ));
      if (FlagQuadplPg && FlagDoublePg)
	 SetFlag( '2', '-');
      break;
   case 'A':
      FlagA4Letter = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagA4Letter ? "+A\tuse A4 paper size"
				     : "-A\tuse Letter paper size" ));
      break;
   case 'B':
      FlagBorder   = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagBorder   ? "+B\tprint border around map"
				     : "-B\tprint no border around map" ));
      break;
   case 'D':
      FlagDecors   = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagDecors   ? "+D\tprint all decorations & barrels"
				     : "-D\tprint no decorations & barrels" ));
      break;
   case 'E':
      FlagEnemies  = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagEnemies  ? "+E\tprint all enemies"
				     : "-E\tprint no enemies" ));
      break;
   case 'F':
      FlagFadeLins = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagFadeLins ? "+F\tfade secret lines"
				     : "-F\tdon't fade secret lines" ));
      break;
   case 'G':
      FlagGoodies  = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagGoodies  ? "+G\tprint all goodies (weapons/bonuses/enhancements)"
				     : "-G\tprint no goodies (weapons/bonuses/enhancements)" ));
      break;
   case 'L':
      FlagLegend   = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagLegend   ? "+L\tprint legend above map"
				     : "-L\tprint no legend above map" ));
      break;
   case 'M':
      FlagMultPlay = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagMultPlay ? "+M\tprint multi-player things"
				     : "-M\tprint no multi-player things" ));
      break;
   case 'N':
      FlagName     = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagName     ? "+N\tprint name above map"
				     : "-N\tprint no name above map" ));
      break;
   case 'P':
      FlagPortrait = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagPortrait ? "+P\tuse Portrait orientation"
				     : "-P\tuse Landscape orientation" ));
      break;
   case 'S':
      FlagShdAreas = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagShdAreas ? "+S\tshade secret areas"
				     : "-S\tdon't shade secret areas" ));
      break;
   case 'T':
      FlagTeleprts = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagTeleprts ? "+T\tlink teleports"
				     : "-T\tdon't link teleports" ));
      break;
   case 'U':
      FlagUltraVlc = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagUltraVlc ? "+U\tprint things for Ultra-Violence skill"
				     : "-U\tprint things below Ultra-Violence skill" ));
      break;
   case 'Z':
      FlagAutoZoom = ( val == '+' ? TRUE : FALSE );
      printf( "%s\n", ( FlagAutoZoom ? "+Z\tauto-zoom map scale"
				     : "-Z\tdefault map scale" ));
      break;
   }
}


/* power-ups local array index names */
#define Shotgun  0
#define Cmbatgun 1
#define Chaingun 2
#define RcktLnch 3
#define Plasmagn 4
#define Bfg9000  5
#define Chainsaw 6
#define Berserk  7
#define Backpack 8
#define SecurArm 9
#define CmbatArm 10
#define SoulSphr 11
#define MegaSphr 12
#define BlurArtf 13
#define InvlnArt 14
#define RadSuit  15
#define CompMap  16
#define LiteAmp  17

/* enemies local array index names */
#define Human    0
#define Sergeant 1
#define Commando 2
#define Imp      3
#define Demon    4
#define Spectre  5
#define LostSoul 6
#define CacoDmn  7
#define PainElem 8
#define Mancubus 9
#define Revenant 10
#define Knight   11
#define Baron    12
#define Arachno  13
#define ArchVile 14
#define CyberDmn 15
#define SpiderBs 16
#define Wolf3dSS 17

/*
   read level & analyze its statistics
*/
void AnalyzeLevel( BCINT episode, BCINT mission)
{
   BCINT classes[ 9] = { 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   BCINT powrups[ 18] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   BCINT enemies[ 18] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
   BCINT n, m, secret = 0, telept = 0;

   ReadLevelData( episode, mission);

   /* count Thing classes, Power-Ups types & Enemies types */
   for (n = 0; n < NumThings; n++)
      /* check whether multi-player Thing should be counted */
      if (FlagMultPlay || ! (Things[ n].when & T_MULTI))
	 /* check whether UV or non-UV Thing should be counted */
	 if ((FlagUltraVlc && (Things[ n].when & T_ULTRA)) ||
	     (! FlagUltraVlc && (Things[ n].when & T_NONUV)))
	 {
	    m = Things[ n].type;
	    classes[ GetThingClass( m)]++;

	    if (GetThingClass( m) == CLASS_WEAPON || GetThingClass( m) == CLASS_ENHANCE)
	       switch( m)
	       {
		  case THING_SHOTGUN:	powrups[ Shotgun ]++; break;
		  case THING_COMBATGUN:	powrups[ Cmbatgun]++; break;
		  case THING_CHAINGUN:	powrups[ Chaingun]++; break;
		  case THING_LAUNCHER:	powrups[ RcktLnch]++; break;
		  case THING_PLASMAGUN:	powrups[ Plasmagn]++; break;
		  case THING_BFG9000:	powrups[ Bfg9000 ]++; break;
		  case THING_CHAINSAW:	powrups[ Chainsaw]++; break;
		  case THING_BERSERK:	powrups[ Berserk ]++; break;
		  case THING_BACKPACK:	powrups[ Backpack]++; break;
		  case THING_GREENARMOR:powrups[ SecurArm]++; break;
		  case THING_BLUEARMOR:	powrups[ CmbatArm]++; break;
		  case THING_SOULSPHERE:powrups[ SoulSphr]++; break;
		  case THING_MEGASPHERE:powrups[ MegaSphr]++; break;
		  case THING_BLURSPHERE:powrups[ BlurArtf]++; break;
		  case THING_INVULN:	powrups[ InvlnArt]++; break;
		  case THING_RADSUIT:	powrups[ RadSuit ]++; break;
		  case THING_COMPMAP:	powrups[ CompMap ]++; break;
		  case THING_LITEAMP:	powrups[ LiteAmp ]++; break;
		  default: break;	/* ignore ammo boxes & keys */
	       }

	    if (GetThingClass( m) == CLASS_ENEMY)
	       switch( m)
	       {
		  case THING_HUMAN:	enemies[ Human   ]++; break;
		  case THING_SERGEANT:	enemies[ Sergeant]++; break;
		  case THING_COMMANDO:	enemies[ Commando]++; break;
		  case THING_IMP:	enemies[ Imp     ]++; break;
		  case THING_DEMON:	enemies[ Demon   ]++; break;
		  case THING_SPECTRE:	enemies[ Spectre ]++; break;
		  case THING_LOSTSOUL:	enemies[ LostSoul]++; break;
		  case THING_CACODEMON:	enemies[ CacoDmn ]++; break;
		  case THING_PAINELEM:	enemies[ PainElem]++; break;
		  case THING_MANCUBUS:	enemies[ Mancubus]++; break;
		  case THING_REVENANT:	enemies[ Revenant]++; break;
		  case THING_KNIGHT:	enemies[ Knight  ]++; break;
		  case THING_BARON:	enemies[ Baron   ]++; break;
		  case THING_ARACHNO:	enemies[ Arachno ]++; break;
		  case THING_ARCHVILE:	enemies[ ArchVile]++; break;
		  case THING_CYBERDEMON:enemies[ CyberDmn]++; break;
		  case THING_SPIDERBOSS:enemies[ SpiderBs]++; break;
		  case THING_WOLF3DSS:	enemies[ Wolf3dSS]++; break;
	       }
	 }

   /* count secret Sectors & teleporting LineDefs */
   for (n = 0; n < NumSectors; n++)
      if (Sectors[ n].special == S_SECRET)
	 secret++;
   for (n = 0; n < NumLineDefs; n++)
      if (LineDefs[ n].type == L_TELEP1 || LineDefs[ n].type == L_TELEPR)
	 telept++;

   /* print all statistics */
   if (Commercial) {
      printf( "Statistics of level MAP%02d: %s\n", mission,
	       ( UserLvlNm != NULL ? UserLvlNm : LevelName ));
      printf( "=========================\n\n");
   }
   else {
      printf( "Statistics of level E%dM%d: %s\n", episode, mission,
	       ( UserLvlNm != NULL ? UserLvlNm : LevelName ));
      printf( "========================\n\n");
   }

   printf( "MapDim:\t  MaxX\t  MinX\t SizeX\t  MaxY\t  MinY\t SizeY\n");
   printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
	    MapMaxX, MapMinX, MapMaxX - MapMinX, MapMaxY, MapMinY, MapMaxY - MapMinY);

   printf( "\n");
   printf( "Struct:\tThings\tVertxs\tLinDfs\tTelePt\tSidDfs\tSectrs\tSecret\n");
   printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
	    NumThings, NumVertexes, NumLineDefs, telept, NumSideDefs, NumSectors, secret);

   printf( "\n");
   printf( "Things:\t START\tTLPORT\t BONUS\tWEAPON\tENHANC\t ENEMY\t DECOR\tBARREL\n");
   printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
	    classes[ CLASS_START],   classes[ CLASS_TELEPORT], classes[ CLASS_BONUS], classes[ CLASS_WEAPON],
	    classes[ CLASS_ENHANCE], classes[ CLASS_ENEMY],    classes[ CLASS_DECOR], classes[ CLASS_BARREL]);
   if (classes[ CLASS_UNKNOWN] != 0)
      printf( "Things:\tUNKNOWN\t%6d\n", classes[ CLASS_UNKNOWN]);

   printf( "\n");
   printf( "PowrUps:SHOTGN\tCOMBAT\tCHAING\tRCKTLR\tPLASMA\t BFG9K\tCHNSAW\tBERSRK\tBCKPCK\n");
   printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
	    powrups[ Shotgun], powrups[ Cmbatgun], powrups[ Chaingun], powrups[ RcktLnch], powrups[ Plasmagn],
	    powrups[ Bfg9000], powrups[ Chainsaw], powrups[ Berserk], powrups[ Backpack]);
   printf( "\tGRNARM\tBLUARM\tSOULSP\tMEGASP\tBLURAF\tINVLAF\tRDSUIT\tCMPMAP\tLITAMP\n");
   printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
	    powrups[ SecurArm], powrups[ CmbatArm], powrups[ SoulSphr], powrups[ MegaSphr], powrups[ BlurArtf],
	    powrups[ InvlnArt], powrups[ RadSuit], powrups[ CompMap], powrups[ LiteAmp]);

   printf( "\n");
   printf( "Enemies: HUMAN\t SARGE\tCOMNDO\t   IMP\t DEMON\tSPECTR\tLOSTSL\tCACODM\tPNELEM\n");
   printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
	    enemies[ Human], enemies[ Sergeant], enemies[ Commando], enemies[ Imp], enemies[ Demon],
	    enemies[ Spectre], enemies[ LostSoul], enemies[ CacoDmn], enemies[ PainElem]);
   printf( "\tMANCBS\tREVENT\tKNIGHT\t BARON\tARCHNO\tARVILE\tCYBERD\tSPIDER\tWOLFSS\n");
   printf( "\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\t%6d\n",
	    enemies[ Mancubus], enemies[ Revenant], enemies[ Knight], enemies[ Baron], enemies[ Arachno],
	    enemies[ ArchVile], enemies[ CyberDmn], enemies[ SpiderBs], enemies[ Wolf3dSS]);

   /* clean up & free space */
   ForgetLevelData();
}

/* end of file */
