/*
     Doom WAD level info program by Gerhard Karnik

     Based on the Doom PostScript map generator by James Bonfield.

Description
-----------
This program displays various information about a DOOM level.

Command line options

USAGE: checklev [-dm] [wadfile] [level]

-dm             Display deathmatch items
wadfile         Selects another wad file (default is "DOOM.WAD")
level           Level number (default is all levels), can use "E1M1" or "11"

Example Output:

C:\GAMES\DOOM> checklev doom.wad 11

DOOM.WAD E1M1  (Skill Level 3) Stats

  Total Monsters: 6 (Spider = 0, Cyber = 0, Baron = 0, CacoDemon = 0)
         Weapons: ShotGun 
   Player Starts: 1234, 5 Deathmatch Starts
     Level Exits: 1 normal, 0 secret
    Skill Levels: Implemented


Compiled using Borland C++ 4.0 using 'bcc -mh'

*/

#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <share.h>
#include <fcntl.h>
#include <dos.h>
#include <dir.h>
#include <alloc.h>

unsigned nread;         /* throw away variable for _dos_read() */
char wadfile[80], level[80];

typedef long int4;
typedef short int2;

struct wad_header {
        int4 magic;
        int4 dir_size;
        int4 dir_off;
};

struct directory {
        int4 start;
        int4 length;
        char name[8];
};

typedef struct linedef_ {
        int2 from_vertex;
        int2 to_vertex;
        int2 attrib;
        int2 type;
        int2 trigger;
        int2 sidedef1;
        int2 sidedef2;
} linedef;

/* linedef attrib bits */
#define LI_IMPASS 0x01
#define LI_SECRET 0x20

typedef struct thing_ {
        int2 x;
        int2 y;
        int2 angle;
        int2 type;
        int2 attrib;
} thing;

/* thing 'type' bits */
#define TH_SKILL1  0x01
#define TH_SKILL2  0x01
#define TH_SKILL3  0x02
#define TH_SKILL4  0x04
#define TH_MULTI   0x10

void _ExceptInit(void) {};  /* reduces size of borland C++ 4.0 executable */

struct directory *open_wad(char *file, int *fd, long *size);
long get_index(struct directory *d, long size, char *name, long st);
linedef *read_linedefs(int fd, struct directory *d, long size, long start, long *num);
thing *read_things(int fd, struct directory *d, long size, long start, long *num);
void usage(void);
int doit(int fd, struct directory *dir, long size, int deathmatch);


void usage()
{
        fprintf(stderr, "Doom WAD level info program\n");
        fprintf(stderr, "This program displays various information about a DOOM level.\n");
        fprintf(stderr, "\n");
        fprintf(stderr, "USAGE: checklev [-dm] [wadfile] [level]\n");
        fprintf(stderr, "  -dm             Display deathmatch items\n");
        fprintf(stderr, "  -? or -h        This help screen\n");
        fprintf(stderr, "  wadfile         Selects another wad file \(default is \"DOOM.WAD\"\)\n");
        fprintf(stderr, "  level           Level number \(default is all levels\), can use \"E1M1\" or \"11\"\n");
        exit(-1);
}


int main(int argc, char **argv)
{
        struct directory *d;
        struct ffblk ffblk;
        int fd, i, ep, lv, deathmatch=0;
        long size;
        int done;
        char path[80];

        for(i=1; i<=argc; i++) {
                if(argv[i][0] == '/')        argv[i][0] = '-';
                if(!strcmpi(argv[i], "-DM")) deathmatch = 1;
                if(!strcmpi(argv[i], "-?"))  usage();
                if(!strcmpi(argv[i], "-h"))  usage();
                if(argv[i][0] != '-')        break; /* no more params */
        }

        switch( argc-i ) {
        case 0: /* no more params */
                strcpy(path, "DOOM.WAD");
                strcpy(level, "");
                break;

        case 1: /* only wad name */
                strcpy(path, argv[i]);
                strcpy(level, "");
                break;

        case 2: /* wad name + level */
                strcpy(path, argv[i]);
                strcpy(level, argv[i+1]);

                if(strlen(level) == 2) {        /* convert '11' to 'E1M1' */
                        level[4] = 0;
                        level[3] = level[1];
                        level[2] = 'M';
                        level[1] = level[0];
                        level[0] = 'E';
                }
                break;

        default:
                usage();
        }

        if(strlen(path)>12)
                fprintf(stderr, "WARNING: Can only get WADs in current directory.\n");

        done = findfirst(path, &ffblk, 0);
        if(done) {
                fprintf(stderr, "Failed to open wad file \"%s\".\n", path);
                return -1;
        }

        while(!done) {
                strcpy(wadfile, ffblk.ff_name);

                if((d = open_wad(wadfile, &fd, &size)) != NULL) {

                        if(!level[0]) { /* do all levels */
                            for(ep=1; ep<=3; ep++)
                                for(lv=1; lv<=9; lv++) {
                                    sprintf(level, "E%dM%d", ep, lv);
                                    doit(fd, d, size, deathmatch);
                                }
                            strcpy(level, "");
                        }
                        else
                            if(doit(fd, d, size, deathmatch) == -1)
                                fprintf(stderr, "Unknown level: %s\n", level);

                        _dos_close(fd);
                }

                done = findnext(&ffblk);
        }

        return 0;
}


int doit(int fd, struct directory *dir, long size, int deathmatch)
{
        linedef *linedefs;
        thing *things;
        long numline, numthing;
        long lev_index;
        int i, tot_mons;

        struct {
                int spider;
                int sergeant;
                int cyber;
                int invis;
                int imp;
                int demon;
                int baron;
                int human;
                int beholder;
                int skull;

                int shotgun;
                int chaingun;
                int rocketl;
                int plasmagun;
                int chainsaw;
                int bfg;

                int normal_exit;
                int secret_exit;

                int p1_start;
                int p2_start;
                int p3_start;
                int p4_start;
                int dm_start;

                int skill;
        } stuff = {0};

        /* find level index */
        lev_index = get_index(dir, size, level, 0);
        if(lev_index == -1) return(-1);  /* level not found */

        /* load relevent arrays for this level */

        linedefs = read_linedefs(fd, dir, size, lev_index, &numline);
        things   = read_things(fd, dir, size, lev_index, &numthing);

        printf("%s %s  (Skill Level 3%s) Stats\n\n", wadfile, level,
                deathmatch?" + Deathmatch":"");

        for(i=0; i<numline; i++) {
                switch(linedefs[i].type) {
                case 11:   stuff.normal_exit++; break;
                case 51:   stuff.secret_exit++; break;
                case 52:   stuff.normal_exit++; break;
                default:                        break;
                }
        }

        for(i=0; i<numthing; i++) {

                if((things[i].attrib & 7) != 7) stuff.skill = 1;

                /* filter out skill level 4 & 5 stuff */
                if(!(things[i].attrib & (TH_SKILL2 + TH_SKILL3)))
                        continue;

                /* filter out deathmatch items */
                if((things[i].attrib & TH_MULTI) && !deathmatch)
                        continue;

                switch(things[i].type) {
                case 7:    stuff.spider++;      break;
                case 9:    stuff.sergeant++;    break;
                case 16:   stuff.cyber++;       break;
                case 58:   stuff.invis++;       break;
                case 3001: stuff.imp++;         break;
                case 3002: stuff.demon++;       break;
                case 3003: stuff.baron++;       break;
                case 3004: stuff.human++;       break;
                case 3005: stuff.beholder++;    break;
                case 3006: stuff.skull++;       break;

                case 2001: stuff.shotgun++;     break;
                case 2002: stuff.chaingun++;    break;
                case 2003: stuff.rocketl++;     break;
                case 2004: stuff.plasmagun++;   break;
                case 2005: stuff.chainsaw++;    break;
                case 2006: stuff.bfg++;         break;

                case 1:    stuff.p1_start++;    break;
                case 2:    stuff.p2_start++;    break;
                case 3:    stuff.p3_start++;    break;
                case 4:    stuff.p4_start++;    break;
                case 11:   stuff.dm_start++;    break;

                default:                        break;
                }
        }

        tot_mons = stuff.spider + stuff.sergeant + stuff.cyber + stuff.invis +
                   stuff.imp + stuff.demon + stuff.baron + stuff.human +
                   stuff.beholder + stuff.skull;

        printf("  Total Monsters: %d", tot_mons);
        printf(" (Spider = %d, Cyber = %d, Baron = %d, CacoDemon = %d)\n",
                stuff.spider, stuff.cyber, stuff.baron, stuff.beholder);

        printf("         Weapons: ");
        if(stuff.shotgun)   printf("ShotGun ");
        if(stuff.chaingun)  printf("ChainGun ");
        if(stuff.rocketl)   printf("RocketLauncher ");
        if(stuff.plasmagun) printf("PlasmaGun ");
        if(stuff.chainsaw)  printf("ChainSaw ");
        if(stuff.bfg)       printf("BFG");
        printf("\n");

        printf("   Player Starts: ");
        if(stuff.p1_start)  printf("1");
        if(stuff.p2_start)  printf("2");
        if(stuff.p3_start)  printf("3");
        if(stuff.p4_start)  printf("4");
        printf(", %d Deathmatch Starts\n", stuff.dm_start);

        printf("     Level Exits: %d normal, %d secret\n",
               stuff.normal_exit, stuff.secret_exit);

        printf("    Skill Levels: %sImplemented\n\n", stuff.skill ? "":"NOT ");

        farfree(linedefs);
        farfree(things);

        return(0);
}


struct directory *open_wad(char *file, int *fd, long *size)
{
        struct wad_header h;
        struct directory *d;
        char *s;

        if(_dos_open(file, O_RDONLY, fd) != 0) {
                perror(file);
                _dos_close(*fd);
                return (struct directory *)0;
        }

        _dos_read(*fd, &h, sizeof(h), &nread);

        s = (char *)&h.magic;

        if(s[1] != 'W' || s[2] != 'A' || s[3] != 'D') {
                fprintf(stderr, "Not a WAD file: \"%s\"\n", file);
                _dos_close(*fd);
                return (struct directory *) 0;
        }

        if(NULL == (d = (struct directory *)farcalloc(h.dir_size, sizeof(*d)))) {
                fprintf(stderr, "Could not allocated %ld bytes.\n",
                        h.dir_size * sizeof(*d));
                _dos_close(*fd);
                return (struct directory *) 0;
        }
        *size = h.dir_size;

        lseek(*fd, h.dir_off, SEEK_SET);
        _dos_read(*fd, d, *size * sizeof(*d), &nread);

        return d;
}


long get_index(struct directory *d, long size, char *name, long start)
{
        long i;

        for(i=start; i<size; i++)
                if(_fstrncmp(d[i].name, name, 8) == 0) return i;

        return -1;
}


linedef *read_linedefs(int fd, struct directory *d, long size, long start, long *num)
{
        long index;
        linedef *indexp;
        long i;

        index = get_index(d, size, "LINEDEFS", start);
        indexp = (linedef *) farmalloc(d[index].length);

        if(!indexp) {
                printf("Could not allocate %ld bytes\n",d[index].length);
                exit(-1);
        }
        lseek(fd, d[index].start, SEEK_SET);
        _dos_read(fd, indexp, d[index].length, &nread);
        *num = d[index].length / sizeof(linedef);

        return indexp;
}


thing *read_things(int fd, struct directory *d, long size, long start, long *num)
{
        long index;
        thing *indexp;
        long i;

        index = get_index(d, size, "THINGS", start);
        indexp = (thing *) farmalloc(d[index].length);

        if(!indexp) {
                printf("Could not allocate %ld bytes\n",d[index].length);
                exit(-1);
        }
        lseek(fd, d[index].start, SEEK_SET);
        _dos_read(fd, indexp, d[index].length, &nread);
        *num = d[index].length / sizeof(thing);

        return indexp;
}

