/*
 * FastMtool - Perform interactive exploring of the Mandelbrot set on
 * Sun workstations.
 * Copyright (c) 1989 P{r Emanuelsson, pell@isy.liu.se. All Rights Reserved.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You can receive a copy of the GNU General Public License from the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * See the README for more information.
 */

#include <suntool/sunview.h>
#include <suntool/panel.h>
#include <suntool/canvas.h>

/* Change event_action to event_id(event) for pre 4.0 */
#ifndef event_action
#define event_action event_id
#define oldSunOS
#endif

#ifndef oldSunOS
#include <suntool/alert.h>	/* Only for SunOS 4 */
#include <alloca.h>		/* Cannot find this on 3.5 */
#endif

#include <math.h>

typedef unsigned char Bool;

#define Stacksize 100		/* Dynamic allocation would be cleaner */

struct stack {
  Pixrect *pr;
  double xmin, xmax, ymin, ymax;
} Stack [Stacksize];

static int sp = 0;		/* Stackpointer */

#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define sgn(x) ((x) >= 0 ? 1 : -1)

#define SQRT_2 1.41421356237
#define Nmax 800		/* Maximum image size */

static Bool MSet[Nmax][Nmax];
static double xmin = -2.0, xmax=0.5, ymin = -1.25, ymax=1.25,
              side=2.5, delta, recur;
static int maxiter=20, incr=5, rec=5;
static int start_x, start_y, new_x, new_y; /* For region select */
static Bool selected = FALSE, draw_new_image=FALSE, abort=FALSE;

#define BUTTONFONT "/usr/lib/fonts/fixedwidthfonts/gallant.r.19"
static char *default_font =
  "DEFAULT_FONT=/usr/lib/fonts/fixedwidthfonts/screen.r.13";

static Frame fr;
static Canvas cv;
static Pixwin *pw;
static Pixfont *bf, *pf;
static Panel pn;
static Panel_item recur_p, incr_p, maxiter_p, stop_p;

#ifdef notdef			/* Future enhancement */
static int N=512;		/* Current image size */
#else
#define N 512
#endif

static char Nmax_str[10];

extern int panel_pw_text(), panel_fonthome(); /* Undocumented, but nice */

double MSetDist();
void quit(), stop(), up(), draw(), sel_region();

void main()
{
  (void) putenv(default_font);
  bf = pf_open(BUTTONFONT);
  pf = pf_default();

/* This code is full of strange magic numbers. I apologize for that!!! */

  fr = window_create(0, FRAME,
		     WIN_X, 400,
		     WIN_Y, 150,
		     WIN_SHOW, TRUE,
		     FRAME_LABEL, "FastMtool - Mandelbrot exploring",
		     0);
  pn = window_create(fr, PANEL, 0);

/* The brain-damaged SunView uses the PANEL_MAX_VALUE to determine the
   position of the slider, instead of e.g. always allocate 10 positions
   for this and right-justify or something. This means that I have to
   use delicate (ugly) pixel-positioning to get the sliders to line up. */

  recur_p = panel_create_item(pn, PANEL_SLIDER,
			      PANEL_LABEL_X, 5,
			      PANEL_LABEL_Y, 5,
			      PANEL_VALUE_X, 110,
			      PANEL_VALUE_Y, 5,
			      PANEL_LABEL_STRING, "Recur:",
			      PANEL_VALUE, rec,
			      PANEL_MIN_VALUE, 1,
			      PANEL_MAX_VALUE, 50,
			      0);
  incr_p = panel_create_item(pn, PANEL_SLIDER,
			     PANEL_LABEL_X, 5,
			     PANEL_LABEL_Y, 25,
			     PANEL_VALUE_X, 110,
			     PANEL_VALUE_Y, 25,
			     PANEL_LABEL_STRING, "Increment:",
			     PANEL_VALUE, incr,
			     PANEL_MIN_VALUE, 1,
			     PANEL_MAX_VALUE, 10,
			     0);
  maxiter_p = panel_create_item(pn, PANEL_SLIDER,
				PANEL_LABEL_X, 5,
				PANEL_LABEL_Y, 45,
				PANEL_VALUE_X, 78,
				PANEL_VALUE_Y, 45,
				PANEL_LABEL_STRING, "Maxiter:",
				PANEL_VALUE, maxiter,
				PANEL_MIN_VALUE, 10,
				PANEL_MAX_VALUE, 1000,
				0);
  panel_create_item(pn, PANEL_MESSAGE,
		    PANEL_ITEM_X, 5,
		    PANEL_ITEM_Y, 100,
		    PANEL_LABEL_STRING, "Xmin:",
		    PANEL_LABEL_BOLD, TRUE,
		    0);
  panel_create_item(pn, PANEL_MESSAGE,
		    PANEL_ITEM_X, 200,
		    PANEL_ITEM_Y, 100,
		    PANEL_LABEL_STRING, "Xmax:",
		    PANEL_LABEL_BOLD, TRUE,
		    0);
  panel_create_item(pn, PANEL_MESSAGE,
		    PANEL_ITEM_X, 5,
		    PANEL_ITEM_Y, 120,
		    PANEL_LABEL_STRING, "Ymin:",
		    PANEL_LABEL_BOLD, TRUE,
		    0);
  panel_create_item(pn, PANEL_MESSAGE,
		    PANEL_ITEM_X, 200,
		    PANEL_ITEM_Y, 120,
		    PANEL_LABEL_STRING, "Ymax:",
		    PANEL_LABEL_BOLD, TRUE,
		    0);

#ifdef notdef			/* Possible future enhancement... */
  sprintf(Nmax_str, "%d", Nmax);
  panel_create_item(pn, PANEL_CYCLE,
		    PANEL_ITEM_X, 350,
		    PANEL_ITEM_Y, 5,
		    PANEL_LABEL_STRING, "Image size:",
		    PANEL_LABEL_BOLD, TRUE,
		    PANEL_CHOICE_STRINGS, Nmax_str, "512", "256", 0,
		    PANEL_VALUE, 1,
		    0);
#endif

  panel_create_item(pn, PANEL_BUTTON,
		    PANEL_ITEM_X, 360,
		    PANEL_ITEM_Y, 30,
		    PANEL_LABEL_IMAGE, panel_button_image(pn, "DRAW", 0, bf),
		    PANEL_NOTIFY_PROC, draw,
		    0);
  panel_create_item(pn, PANEL_BUTTON,
		    PANEL_ITEM_X, 360,
		    PANEL_ITEM_Y, 70,
		    PANEL_LABEL_IMAGE, panel_button_image(pn, " UP ", 0, bf),
		    PANEL_NOTIFY_PROC, up,
		    0);
  panel_create_item(pn, PANEL_BUTTON,
		    PANEL_ITEM_X, 360,
		    PANEL_ITEM_Y, 110,
		    PANEL_LABEL_IMAGE, panel_button_image(pn, "STOP", 0, bf),
		    PANEL_NOTIFY_PROC, stop,
		    0);
  panel_create_item(pn, PANEL_BUTTON,
		    PANEL_ITEM_X, 450,
		    PANEL_ITEM_Y, 72,
		    PANEL_LABEL_IMAGE, panel_button_image(pn, "Quit", 0, pf),
		    PANEL_NOTIFY_PROC, quit,
		    0);
  window_fit_height(pn);
  cv = window_create(fr, CANVAS,
		     WIN_WIDTH, N,
		     WIN_HEIGHT, N,
		     WIN_CONSUME_PICK_EVENT, LOC_DRAG,
		     WIN_EVENT_PROC, sel_region,
		     0);

  window_fit(fr);
  notify_interpose_destroy_func(fr, quit);

  pw = canvas_pixwin(cv);
  notify_dispatch();		/* To make the next put_coords work */
  put_coords(0, N-1, N-1, 0);

  bzero((char *) MSet, sizeof(MSet));
  pw_writebackground(pw, 0, 0, N, N, PIX_SRC | PIX_COLOR(1));

  /* Cannot use window_main_loop() because the notifier can't dispatch
     events inside a PANEL_NOTIFY_PROC. */

  while (1) {
    notify_dispatch();
    if (draw_new_image) {
      panel_pw_text(pn, 347, 5 + panel_fonthome(bf), PIX_SRC, bf, "Drawing!");
      compute();
      panel_pw_text(pn, 347, 5 + panel_fonthome(bf), PIX_SRC, bf, "        ");
      draw_new_image = FALSE;
    }
    usleep(50000);
  }
}

put_coords(ixmin, ixmax, iymin, iymax)
     int ixmin, ixmax, iymin, iymax;
{
  char str[20];

  sprintf(str, "%10.7lf", xmin + side * ixmin/(N-1));
  panel_pw_text(pn, 50, 100 + panel_fonthome(pf), PIX_SRC, pf, str);
  sprintf(str, "%10.7lf", xmin + side * ixmax/(N-1));
  panel_pw_text(pn, 245, 100 + panel_fonthome(pf), PIX_SRC, pf, str);
  sprintf(str, "%10.7lf", ymin + side * (N-1-iymin)/(N-1));
  panel_pw_text(pn, 50, 120 + panel_fonthome(pf), PIX_SRC, pf, str);
  sprintf(str, "%10.7lf", ymin + side * (N-1-iymax)/(N-1));
  panel_pw_text(pn, 245, 120 + panel_fonthome(pf), PIX_SRC, pf, str);
}

void sel_region(canvas, event, arg)
     Canvas canvas;
     Event *event;
     char *arg;
{
  static Bool mouseing = FALSE;
  register int maxdist, tmpx, tmpy;

#define mkbox(a,b,c,d) \
  pw_vector(pw, a, b, c, b, PIX_SRC^PIX_DST, 1);\
  pw_vector(pw, a, b, a, d, PIX_SRC^PIX_DST, 1);\
  pw_vector(pw, c, b, c, d, PIX_SRC^PIX_DST, 1);\
  pw_vector(pw, a, d, c, d, PIX_SRC^PIX_DST, 1)

  switch (event_action(event)) {
    case MS_LEFT:
      if (event_is_down(event)) {
	if (selected) {		/* Remove old box first */
	  mkbox(start_x, start_y, new_x, new_y);
	}
	start_x = new_x = event_x(event);
	start_y = new_y = event_y(event);
	put_coords(new_x, new_x, new_y, new_y);
	mouseing = TRUE;
      } else {
	mouseing = FALSE;
	selected = TRUE;
      }
      break;
    case LOC_DRAG:
      if (mouseing) {
	mkbox(start_x, start_y, new_x, new_y); /* Remove old box */

	/* We want to restrict the size to be square */
	tmpx = event_x(event) - start_x;
	tmpy = start_y - event_y(event);
	maxdist = MIN(tmpx * sgn(tmpx), tmpy * sgn(tmpy));
	new_x = start_x + maxdist * sgn(tmpx);
	new_y = start_y - maxdist * sgn(tmpy);

	mkbox(start_x, start_y, new_x, new_y); /* Draw new box */
	put_coords(MIN(start_x, new_x), MAX(start_x, new_x),
		   MAX(start_y, new_y), MIN(start_y, new_y));
      }
      break;
    case MS_MIDDLE:
      if (selected) {
	mkbox(start_x, start_y, new_x, new_y);
	selected = FALSE;
      }
    }
}

void stop()
{
  abort = TRUE;
}

void up()
{
  if (sp > 0) {
    Pixrect *old;
    register int i, j, k;
    Bool *mem;

    selected = FALSE;
    sp--;
    xmin = Stack[sp].xmin;
    xmax = Stack[sp].xmax;
    ymin = Stack[sp].ymin;
    ymax = Stack[sp].ymax;
    side = xmax - xmin;
    put_coords(0, N-1, N-1, 0);
    old = Stack[sp].pr;
    pw_write(pw, 0, 0, N, N, PIX_SRC, old, 0, 0);

    /* Restore MSet */
    /* This is ugly - I'm assuming that sizeof(Bool) == 1. Shame on me! */
    mem = (Bool *) mpr_d(old)->md_image;
    for (i=0; i<N; i++) {
      for (j=0; j<N; j+=8) {
	for (k=0; k<8; k++)
	  MSet[j+k][N-1-i] = ((*mem & (1<<(7-k))) >> (7-k)) == 0 ? 1 : 0;
	mem++;
      }
    }
    pr_destroy(old);
  }
}
    
void draw()
{
  draw_new_image = TRUE;
}

void quit()
{
#ifdef oldSunOS
  exit(0);			/* You won't miss the following fancy stuff.. */
#else
  if (alert_prompt(fr, (Event *)0,
		   ALERT_MESSAGE_STRINGS,
		     "Please confirm:",
		     "Do you know what you're doing??",
		     0,
		   ALERT_BUTTON_YES, "Of course, quit bugging me!",
		   ALERT_BUTTON_NO, "Sorry, I hit the wrong button...",
		   ALERT_MESSAGE_FONT, bf,
		   ALERT_BUTTON_FONT, pf,
		   0)
      == ALERT_YES) {
    exit(0);
  } else
    notify_veto_destroy(fr);
#endif
}

compute()
{
  register int ix, iy;

  if (selected && start_x != new_x) { /* New region selected */
    Pixrect *save = mem_create(N, N, 1);
    mkbox(start_x, start_y, new_x, new_y); /* Remove the box first */
    pw_read(save, 0, 0, N, N, PIX_SRC, pw, 0, 0);
    Stack[sp].pr = save;
    Stack[sp].xmin = xmin;
    Stack[sp].xmax = xmax;
    Stack[sp].ymin = ymin;
    Stack[sp].ymax = ymax;
    if (sp < Stacksize) sp++;	/* Hard to imagine this happen, but... */
    bzero((char *) MSet, sizeof(MSet));
    pw_writebackground(pw, 0, 0, N, N, PIX_SRC | PIX_COLOR(1));

    xmax = xmin + side * MAX(start_x, new_x) /(N-1);
    xmin += side * MIN(start_x, new_x) /(N-1);
    ymax = ymin + side * (N-1-MIN(start_y, new_y)) /(N-1);
    ymin += side * (N-1-MAX(start_y, new_y)) /(N-1);
    selected = FALSE;
  } else {
    /* No region selected, just redraw. Perhaps using new parameters. */
    put_coords(0, N-1, N-1, 0);
  }

  rec = (int) panel_get_value(recur_p);
  incr = (int) panel_get_value(incr_p);
  maxiter = (int) panel_get_value(maxiter_p);

  side = xmax - xmin;
  delta = 0.25 * side / (N-1);	/* 0.25 seems OK */
  recur = rec * delta;

  abort = FALSE;

/*************************************************************************/
/*************************************************************************/

/* From now on, you will find the new Mandelbrot algorithm. No Sun specific
   stuff, except the notify_dispatch() and some pw_put() and pw_line(). */

  for (iy = 0; iy < N; iy += incr) {
    notify_dispatch();		/* Allow user to hit the STOP button */
    if (abort) break;
    for (ix = 0; ix < N; ix += incr)
      if (!MSet[ix][iy])
	MDisk(ix,iy);
  }
}

MDisk(ix, iy)
     register int ix, iy;
{
  register double cx, cy, dist;
  register int irad;

  if (ix<0 || ix>=N || iy<0 || iy>=N || MSet[ix][iy]) return;

  cx = xmin + (side * ix) / (N-1);
  cy = ymin + (side * iy) / (N-1);
  dist = 0.25 * MSetDist(cx, cy, maxiter);
  irad = dist / side * (N-1);	/* Bug in the original algorithm */

  if (irad == 1) {
    MSet[ix][iy] = 1;
    pw_put(pw, ix, N-1-iy, 0);	/* Sun specific */
  } else if (irad > 1)
    FILLDISK(ix, iy, irad);
  else if (dist > delta) {
    MSet[ix][iy] = 1;
    pw_put(pw, ix, N-1-iy, 0);	/* Sun specific */
  }

  if (dist > recur) {
    if (irad > 1) irad++;
    MDisk(ix, iy + irad);
    MDisk(ix, iy - irad);
    MDisk(ix + irad, iy);
    MDisk(ix - irad, iy);

/* It will be slightly faster if I leave out the following "45-degree"
   recursions. The reason is that most of these points will probably
   be filled already and MDisk will return immediately. But since
   they are in the original algorithm and the improvement is only marginal
   I will leave them here. */

    irad = 0.5 + irad / SQRT_2;
    MDisk(ix + irad, iy + irad);
    MDisk(ix - irad, iy - irad);
    MDisk(ix - irad, iy + irad);
    MDisk(ix + irad, iy - irad);
  }
}

double MSetDist(cx, cy, maxiter)
     register double cx, cy;
     register int maxiter;
{
# define overflow 10e10		/* Don't know if this is foolproof */

  register int iter=0;
  register double zx, zy, zx2, zy2;
  register double *xorbit, *yorbit;

  /* Could use a static array for this, if you don't have alloca */
  xorbit = (double *) alloca(maxiter * sizeof(double));
  yorbit = (double *) alloca(maxiter * sizeof(double));

  /* This is the standard Mandelbrot iteration */
  zx = zy = zx2 = zy2 = xorbit[0] = yorbit[0] = 0.0;
  do {
    zy = (zx * zy) + (zx * zy) + cy; /* gcc generates only one mult for this */
    zx = zx2 - zy2 + cx;
    iter++;
    xorbit[iter] = zx;		/* Save the iteration orbits for later */
    yorbit[iter] = zy;
    zx2 = zx * zx;
    zy2 = zy * zy;
  } while ((zx2 + zy2) < 1000.0 && iter<maxiter);

  if (iter < maxiter) {		/* Generate derivatives */
    register double zpx, zpy, tmp;
    register int i;

    zpx = zpy = 0.0;

    for (i=0; i<iter; i++) {
      tmp = 2 * (xorbit[i] * zpx - yorbit[i] * zpy) + 1.0;
      zpy = 2 * (xorbit[i] * zpy + yorbit[i] * zpx);
      zpx = tmp;
      if (fabs(zpx) > overflow || fabs(zpy) > overflow)
	return 0.0;
    }
    /* This is the distance estimation */
    return log(zx2 + zy2) * sqrt(zx2 + zy2) / sqrt(zpx*zpx + zpy*zpy);
  }
  return 0.0;
}

FILLDISK(ix, iy, irad)
     register int ix, iy;
     int irad;
{
  register int x, y, e;

  /* The "Mini"-algorithm. Perhaps I should use display locking around the
     plotline's, but after all, the fun is watching it work... */

  x = 0;
  y = irad;
  e = irad / 2;
  while (x <= y) {
    plotline(ix - x, ix + x, iy + y);
    plotline(ix - y, ix + y, iy + x);
    plotline(ix - x, ix + x, iy - y);
    plotline(ix - y, ix + y, iy - x);
    e -= x;
    if (e < 0) {
      e += y;
      y--;
    }
    x++;
  }
}

plotline(x1, x2, y)
     register int x1, x2, y;
{
  register int i;
  if (y<0 || y>N-1 || (x1<0 && x2<0) || (x1>=N-1 && x2 >=N-1)) return;

  if (x1 < 0) x1 = 0;
  if (x1 > N-1) x1 = N-1;
  if (x2 < 0) x2 = 0;
  if (x2 > N-1) x2 = N-1;

  pw_vector(pw, x1, N-1-y, x2, N-1-y, PIX_SRC, 0); /* Sun specific */

  for (i=x1; i<=x2; i++)
    MSet[i][y] = 1;
}
