/*
 * which [-i] [-a] [--] [<command>]
 * alias which alias !\$ \| /usr/local/bin/which -i !\*
 * alias which 'eval alias \$$# | /usr/local/bin/which -i ${1+"$@"}'
 * which()
 * {
 *	eval last=\"\$$#\"
 *	set | sed -n "/^$last(){$/,/^}$/p" |
 *		/usr/local/bin/which -i ${1+"$@"}
 * }
 *
 * author: Maarten Litmaath @ VU University Amsterdam (maart@cs.vu.nl)
 * first change:
 *	Emile LeBlanc (leblanc%math.Berkeley.EDU@ucbvax.berkeley.edu) notes
 *	the access() system call considering everything executable for
 *	root (!), so we give root a special treatment
 *	'which', 'which -i' and 'which -a' with no further arguments now
 *	return the PATH environment variable, split up into its components
 *	the aliases defined above are slightly different from the previous
 *	version - now it's the shell who's doing the alias checking
 * second change:
 *	added support for shell functions and multiline aliases, added the
 *	`--' option, changed the source style
 * third change:
 *	to hell with access()!
 *	now stat() is used to give the right answer even if the effective
 *	uid (gid) differs from the real uid (gid)
 *	we can't use setuid(geteuid()), because that's nonportable :-(
 */

#include	<sys/types.h>
#include	<sys/stat.h>
#include	<stdio.h>

#define		BUF_SIZE	512
#define		M_USR		0700
#define		M_GRP		0070
#define		M_OTH		0007
#define		X_ALL		0111
#define		R_ALL		0444

char	Version[] =
	"@(#)which 4.0 90/01/24 Maarten Litmaath @ VU Informatika Amsterdam";
char	*Prog;


void	usage()
{
	fprintf(stderr, "Usage: %s [-i] [-a] [--] [<command>]\n", Prog);
	exit(1);
}


main(argc, argv) 
int	argc;
register char	**argv;
{
	register char	*path, *s;
	char	*save, *strcpy(), *getenv(), *fgets(), buf[BUF_SIZE];
	int	all = 0, inter = 0, stop = 0, found = 0, uid, gid, mask,
		xmask, rmask;
	struct	stat	st;
	void	usage(), convert();


	Prog = *argv++;
	--argc;

	while (!stop && (s = *argv) && (*s == '-')) {
		++argv;
		--argc;
		++s;
		while (*s)
			switch (*s++) {
			case 'a':
				all = 1;
				break;
			case 'i':
				inter = 1;
				break;
			case '-':
				stop = 1;
				break;
			default:
				usage();
			}
	}

	if (argc > 1)
		usage();

	if (inter && *argv) {
		while (fgets(buf, sizeof buf, stdin)) {
			if (!found) {
				printf("%s", *argv);
				found = 1;
			}
			printf("\t%s", buf);
		}
		if (found && !all)
			exit(0);
	}

	if (!(save = path = getenv("PATH"))) {
		fprintf(stderr, "%s: no PATH in environment!\n", Prog);
		exit(1);
	}

	if (!*path)
		save = path = ".";

	if (!*argv) {
		convert(path, buf);
		puts(buf);
		exit(0);
	}

	uid = geteuid();
	gid = getegid();
	if (uid == 0) {
		xmask = X_ALL;
		rmask = R_ALL;
	}

	while (*path) {
		s = buf;
		while ((*s++ = *path) && *path++ != ':')
			;
		if (*buf == ':') {
			/*
			 * to deal with the dubious convention that a
			 * spurious colon is equivalent to a dot...
			 */
			*buf = '.';
			++s;
		}
		(void) strcpy(s, *argv);
		*--s = '/';

		if (stat(buf, &st) != 0 || (st.st_mode & S_IFMT) != S_IFREG)
			continue;

		/* file exists and is regular */

		if (uid != 0) {
			mask = st.st_uid == uid ? M_USR :
				st.st_gid == gid ? M_GRP : M_OTH;
			xmask = X_ALL & mask;
			rmask = R_ALL & mask;
		}

		if (!(st.st_mode & xmask))
			continue;

		/* file is executable */

		*s = 0;
		if (stat(buf, &st) != 0) {
			perror(buf);
			continue;
		}

		if (!(st.st_mode & rmask)) {
			fprintf(stderr,
				"%s: %s found in unreadable directory %s!\n",
				Prog, *argv, buf);
			found = 1;
			continue;
		}

		/* directory is readable */

		*s = '/';
		puts(buf);
		if (!all)
			exit(0);
		found = 1;
	}

	if (found)
		exit(0);

	convert(save, buf);
	fprintf(stderr, "%s not found in\n%s\n", *argv, buf);
	exit(1);
}


void	convert(path, buf)
register char	*path, *buf;
{
	for (;;) {
		while ((*buf++ = *path) && *path++ != ':')
			;
		if (!*path)
			break;
		*buf++ = '\n';
	}
	*buf = '\0';		/* to cope with a PATH ending in ':' */
}
