/* tklex.c - lexical input routines
 *
 * 26.Jul.87  jimmc  Initial definition
 * 12.Aug.87  jimmc  Change name TkClean to TkDone
 * 13.Aug.87  jimmc  Allow negative numbers
 * 18.Jan.88  jimmc  Move definition of TkInfo into tklex.h
 * 26.Jan.88  jimmc  Read in negative numbers properly
 */

#include <stdio.h>
#include <ctype.h>
#include <strings.h>
#include "tklex.h"

extern char *malloc(),*calloc(),*realloc();

static int (*TkFatalP)();
static int (*TkWarnP)();
TkInfo *TkGinfo;		/* used for debugging */

/* VARARGS1 */
static TkFatal(msg,a0,a1)
char *msg;
char *a0,*a1;
{
char buf[1200];

	sprintf(buf,msg,a0,a1);
	if (TkFatalP) {
		(*TkFatalP)(buf);
	}
	fprintf(stderr,"Tk: %s\n",buf);
	exit(1);
}

/* VARARGS1 */
static TkWarn(msg,a0,a1,a2)
char *msg;
char *a0,*a1,*a2;
{
char buf[1200];

	sprintf(buf,msg,a0,a1,a2);
	if (TkWarnP) {
		(*TkWarnP)(buf);
	}
	else {
		fprintf(stderr,"Tk: %s\n",buf);
	}
	/* continue processing */
}

/* VARARGS2 */
static TkFileWarn(info,msg,a0,a1)
TkInfo *info;
char *msg;
char *a0,*a1;
{
char buf[1200];

	sprintf(buf,msg,a0,a1);
	TkWarn("File %s, line %d: %s", info->filename, info->lineno, buf);
}

static StoreNextChar(info,c)	/* store another char into stringvalue */
TkInfo *info;
int c;
{
	if (info->stringcount>=info->stringalloc) {	/* need more space */
		if (info->stringalloc==0) info->stringalloc = 60;
		else info->stringalloc *= 2;
		if (info->stringvalue)
			info->stringvalue = realloc(info->stringvalue,
				(unsigned)(info->stringalloc));
		else info->stringvalue = malloc((unsigned)(info->stringalloc));
		if (!info->stringvalue) {
			info->stringalloc = 0;
			TkFatal("no more memory to input string, line %d",
				info->lineno);
		}
	}
	info->stringvalue[info->stringcount++] = c;
}

static int		/* returns the next input character */
TkGetChar(info)
TkInfo *info;
{
int c;
	if (info->pbindex!=info->pballoc) {	/* we have pushback */
		c = info->pbchars[info->pbindex++];
		if (c==255) c=EOF;
	}
	else {
		c = getc(info->f);
	}
	if (c<='\f' && c>='\n') info->lineno++;
	return c;
}

static
TkPushChar(info,c)
TkInfo *info;
int c;
{
char *newspace;
int oldsize;

	if (info->pbindex==0) {	/* no more space for pushback */
		oldsize = info->pballoc;
		if (info->pballoc==0) info->pballoc=58;
		else info->pballoc *= 2;
		newspace = malloc((unsigned)(info->pballoc+1));
		if (!newspace) TkFatal("no memory for pushback");
		info->pbindex = info->pballoc - oldsize;
		if (oldsize) strcpy(newspace+info->pbindex,info->pbchars);
		else newspace[info->pballoc]=0;
		if (info->pbchars) free(info->pbchars);
		info->pbchars = newspace;
	}
	info->pbchars[--(info->pbindex)] = c;
	if (c<='\f' && c>='\n') info->lineno--;
}


static int
backslash(info)	/* originally from LISCH1 */
TkInfo *info;
{
	register int c ;

	switch(c = TkGetChar(info)) {
	case '\\':  return('\\') ;
	case 'b':   return('\b') ;
	case 'e':   return('\033') ;        /* escape */
	case 'f':   return('\f') ;
	case 'n':   return('\n') ;
	case 'r':   return('\r') ;
	case 't':   return('\t') ;
	case 'v':   return('\013') ;        /* vertical tab */
	case '"':   return('"') ;
	case '\'':  return('\'') ;
	case '^':
		c = TkGetChar(info) ;
		if (isprint(c))
			return(c & ~0140) ; /* convert to control char range */
		else {
			TkPushChar(info,c) ;
			return('^') ;
		}
	default:
		if (! isdigit(c))
			return(c) ;
		else {
			int i, n ;
			char buf[4] ;

			buf[0] = c ;
			for (i = 1; i < 3; i++)
			if (! isdigit(buf[i] = TkGetChar(info))) {
				TkPushChar(info,buf[i]) ;
				break ;
			}
			buf[i] = '\0' ;
			sscanf(buf, "%o", &n) ;
			return(n) ;
		}
	}
}

INTFUNCPTR  /* returns previous value (pointer to function returning int) */
TkSetFatalHandler(p)
int (*p)();
{
int (*oldp)();

	oldp = TkFatalP;
	TkFatalP = p;
	return oldp;
}

INTFUNCPTR  /* returns previous value (pointer to function returning int) */
TkSetWarnHandler(p)
int (*p)();
{
int (*oldp)();

	oldp = TkWarnP;
	TkWarnP = p;
	return oldp;
}

TkHandle		/* returns a handle for calls to TkGet, 0 on error */
TkInit(filename)	/* the input file to read */
char *filename;		/* name of the file to read from; stdin is special */
{
TkInfo *info;

	info = (TkInfo *)calloc(sizeof(TkInfo),1);
	if (!info) TkFatal("no memory for info block");
	if (strcmp(filename,"stdin")==0) info->f = stdin;
	else info->f = fopen(filename,"r");
	if (!info->f) {
		TkWarn("Error opening input file %s", filename);
		return 0;
	}
	info->filename = malloc((unsigned)(strlen(filename)+1));
	if (!info->filename) TkFatal("no memory for filename copy");
	strcpy(info->filename,filename);
	info->lineno++;		/* first line in file is line 1 */
	return (TkHandle)info;
}

int			/* returns 0 if all OK */
TkDone(handle)
TkHandle handle;
{
TkInfo *info;
int t=0;

	info = (TkInfo *)handle;
	if (!info) TkFatal("no handle for cleanup");
	if (info->f) {
		t = ferror(info->f) || fclose(info->f);
		if (t) TkFileWarn(info,"error closing file");
	}
	if (info->filename) free(info->filename);
	if (info->pbchars) free(info->pbchars);
	if (info->stringvalue) free(info->stringvalue);
	free((char *)info);
	return t;
}

TkPush(handle,t)	/* allow ONE token to be pushed back */
TkHandle handle;
int t;		/* the token to push back */
{
TkInfo *info;

	info = (TkInfo *)handle;
	if (!info) TkFatal("no handle for input");
	info->pushedtoken = t;
}

int		/* token type as defined in lex.h */
TkGet(handle)	/* returns the next input token */
TkHandle handle; /* the handle for the input stream as returned by TkInit */
{
TkInfo *info;
int c;
int negflag=0;

	info = (TkInfo *)handle;
	if (!info) TkFatal("no handle for input");
	if (info->pushedtoken) {
		c = info->pushedtoken;
		info->pushedtoken = 0;
		return c;
	}
tryAgain:
	c = TkGetChar(info);
	while (isspace(c)) {		/* skip over white space */
		c = TkGetChar(info);
	}
	switch (c) {
	case EOF:	return TkEOF;
	case '(':	return TkOParen;
	case ')':	return TkCParen;
	case '"':		/* start of string */
		info->stringcount = 0;	/* clear string */
		while (1) {
			c = TkGetChar(info);
			switch (c) {
			case EOF:
				TkFileWarn(info,"EOF in string");
				goto stringDone;
			case '\\':
				if ((c=TkGetChar(info))=='\n') break;
				TkPushChar(info,c);
				c = backslash(info);
				StoreNextChar(info,c);
				break;
			case '\n':
				TkPushChar(info,c);	/* for err msg */
TkFileWarn(info,"newline in string - string terminated");
				/* FALL THROUGH */
			case '"':
				goto stringDone;
			default:
				if (isprint(c))
					StoreNextChar(info,c);
				else
TkFileWarn(info,"illegal character (%03o) in string - ignored",c);
				break;
			}
		}
stringDone:
		StoreNextChar(info,'\000');
		return TkString;
	case '/':		/* possibly start of comment */
		if ((c=TkGetChar(info))!='*') {
			TkPushChar(info,c);
			c = '/';
			goto notComment;
		}
		while (1) {	/* exit via goto or return */
			c = TkGetChar(info);
			if (c==EOF) {
				TkFileWarn(info,"EOF in comment");
				return TkEOF;
			}
			if (c=='*') {
				c = TkGetChar(info);
				if (c=='/') goto tryAgain;  /* end of comment */
				else TkPushChar(info,c); /* in case ** */
			}
		}
	case '-':		/* possible negative number */
		c = TkGetChar(info);
		if (!isdigit(c)) {
			TkPushChar(info,c);
			/* if not a digit, put it back and complain
			 * about the '-' below */
			c = '-';
		}
		else {
		    /* if it is a digit, leave it in c to be picked up below */
			negflag = 1;
		}
		break;
	default:	break;	/* fall out of switch, do more checking */
	}
	if (isdigit(c)) {
		info->stringcount = 0;
		if (negflag) StoreNextChar(info,'-');
		while (isdigit(c)) {
			StoreNextChar(info,c);
			c = TkGetChar(info);
		}
		TkPushChar(info,c);	/* put back that last char */
		StoreNextChar(info,'\000');
		return TkNumber;
	}
	if (isalpha(c)||c=='_') {	/* symbol */
		info->stringcount = 0;
		while (isalnum(c)||c=='_') {
			StoreNextChar(info,c);
			c = TkGetChar(info);
		}
		TkPushChar(info,c);	/* put back that last char */
		StoreNextChar(info,'\000');
		return TkSymbol;
	}
notComment:
	TkFileWarn(info, "illegal character (%c, %03o) in input - ignored",
		c, c);
	goto tryAgain;
}

int
TkNumberValue(handle)
TkHandle handle;
{
TkInfo *info;

	info = (TkInfo *)handle;
	if (!info) TkFatal("no handle for input");
	if (!info->stringvalue) TkFatal("no string for numeric value");
	if (info->stringvalue[0]=='-')
		return (-atoi(info->stringvalue+1));
	return atoi(info->stringvalue);
}

static	/* may become public at some point in the future */
char *
TkSStringValue(handle)	/* like TkStringValue, but doesn't copy the string */
TkHandle handle;
{
TkInfo *info;

	info = (TkInfo *)handle;
	if (!info) TkFatal("no handle for input");
	if (!info->stringvalue) TkFatal("no string for string value");
	return info->stringvalue;
}

char *
TkStringValue(handle)
TkHandle handle;
{
char *oldstring,*newstring;

	oldstring = TkSStringValue(handle);
	newstring = malloc((unsigned)(strlen(oldstring)+1));
	if (!newstring) TkFatal("no memory for string copy");
	strcpy(newstring,oldstring);
	return newstring;
}

int
TkLineNumber(handle)
TkHandle handle;
{
TkInfo *info;

	info = (TkInfo *)handle;
	if (!info) TkFatal("no handle for input");
	return info->lineno;
}

/* Now for some debugging routines to help test things */

TkDebugHandle(handle)	/* for debugging purposes */
TkHandle handle;
{
TkInfo *info;

	info = (TkInfo *)handle;
	if (!info) {
		TkWarn("no handle for debug");
		return;
	}
	TkGinfo = info;		/* set the global for debugging */
}

TkPrintToken(handle,token)	/* debugging routine to print token */
TkHandle handle;
int token;		/* token type as returned by TkGet */
{
TkInfo *info;

	info = (TkInfo *)handle;
	if (!info) {
		TkWarn("no handle for debug");
		return;
	}
	printf("line %d: ",info->lineno);
	switch (token) {
	case TkEOF:
		printf("EOF");
		break;
	case TkNumber:
		printf("Number: %d",TkNumberValue(handle));
		break;
	case TkString:
		printf("String: \"%s\"",TkSStringValue(handle));
		break;
	case TkSymbol:
		printf("Symbol: %s",TkSStringValue(handle));
		break;
	case TkOParen:
		printf("(");
		break;
	case TkCParen:
		printf(")");
		break;
	default:
		printf("???? <%d>",token);
		break;
	}
	printf("\n");
}

/* for debugging - scans a file, reads all tokens, and prints them
 * all back out again. */
TkDebugFile(filename)
char *filename;
{
TkHandle handle;
int c;

	handle = TkInit(filename);
	if (!handle) {
		printf("can't get a handle for %s\n", filename);
		return 1;
	}
	for (c=TkGet(handle);c!=TkEOF;c=TkGet(handle)) {
		TkPrintToken(handle,c);
	}
	TkPrintToken(handle,c);
	TkDone(handle);	/* done with that one */
	return 0;
}

/* end */
