/*
	Gamma -- gamma and related functions

	last edit:	88/09/09	D A Gwyn

	SCCS ID:	@(#)gamma.c	1.1 (edited for publication)

Acknowledgement:
	Code based on that found in "Numerical Methods in C".
*/

#include	<assert.h>
#include	<math.h>

#include	"std.h"

double
LGamma( x )
	double			x;
	{
	static const double	cof[6] =
		{
		76.18009173,	-86.50532033,	24.01409822,
		-1.231739516,	0.120858003e-2,	-0.536382e-5
		};
	double			tmp, ser;
	register int		j;

	assert(x > 0.0);

	if ( --x < 0.0 )	/* use reflection formula for accuracy */
		{
		double	pix = PI * x;

		return log( pix / sin( pix ) ) - LGamma( 1.0 - x );
		}

	tmp = x + 5.5;
	tmp -= (x + 0.5) * log( tmp );

	ser = 1.0;

	for ( j = 0; j < 6; ++j )
		ser += cof[j] / ++x;

	return -tmp + log( 2.50662827465 * ser );
	}

double
Gamma( x )
	double	x;
	{
	return exp( LGamma( x ) );
	}

double
Factorial( n )
	register int	n;
	{
	static double	a[33] =
		{
		1.0,	1.0,	2.0,	6.0,	24.0
		};
	static int	ntop = 4;

	assert(n >= 0);

	if ( n > 32 )
		return Gamma( (double)n + 1.0 );

	while ( ntop < n )
		{
		register int	j = ntop++;

		a[ntop] = a[j] * (double)ntop;
		}

	return a[n];
	}

double
LFactorial( n )
	register int	n;
	{
	static double	a[101];		/* init 0.0 */

	assert(n >= 0);

	if ( n <= 1 )
		return 0.0;

	if ( n <= 100 )
		if ( a[n] > 0.0 )
			return a[n];	/* table value already set up */
		else
			return a[n] = LGamma( (double)n + 1.0 );
	else
		return LGamma( (double)n + 1.0 );	/* beyond table */
	}

double
BCoeff( n, k )
	register int	n, k;
	{
	assert(k >= 0);
	assert(n >= k);

	return Round( exp( LFactorial( n )
			 - (LFactorial( k ) + LFactorial( n - k ))
			 )
		    );
	}

double
Beta( z, w )
	double	z, w;
	{
	return exp( LGamma( z ) + LGamma( w ) - LGamma( z + w ) );
	}

#define	ITMAX	100
#define	EPS	3.0e-7

static double
gser( a, x )
	double		a, x;
	{
	double		ap, del, sum;
	register int	n;

	assert(x >= 0.0);

	if ( x <= 0.0 )
		return 0.0;

	del = sum = 1.0 / (ap = a);

	for ( n = 1; n <= ITMAX; ++n )
		{
		sum += del *= x / ++ap;

		if ( Abs( del ) < Abs( sum ) * EPS )
			return sum * exp( -x + a * log( x ) - LGamma( a ) );
		}

	assert(n <= ITMAX);
	/*NOTREACHED*/
	}

static double
gcf( a, x )
	double		a, x;
	{
	register int	n;
	double		gold = 0.0, fac = 1.0, b1 = 1.0,
			b0 = 0.0, a0 = 1.0, a1 = x;

	for ( n = 1; n <= ITMAX; ++n )
		{
		double	anf;
		double	an = (double)n;
		double	ana = an - a;

		a0 = (a1 + a0 * ana) * fac;
		b0 = (b1 + b0 * ana) * fac;
		anf = an * fac;
		b1 = x * b0 + anf * b1;
		a1 = x * a0 + anf * a1;

		if ( a1 != 0.0 )
			{		/* renormalize */
			double	g = b1 * (fac = 1.0 / a1);

			gold = g - gold;

			if ( Abs( gold ) < EPS * Abs( g ) )
				return exp( -x + a * log( x ) - LGamma( a ) )
					* g;

			gold = g;
			}
		}

	assert(n <= ITMAX);
	/*NOTREACHED*/
	}

double
PGamma( a, x )
	double	a, x;
	{
	assert(x >= 0.0);
	assert(a > 0.0);

	return x < a + 1.0 ? gser( a, x ) : 1.0 - gcf( a, x );
	}

double
QGamma( a, x )
	double	a, x;
	{
	assert(x >= 0.0);
	assert(a > 0.0);

	return x < a + 1.0 ? 1.0 - gser( a, x ) : gcf( a, x );
	}

double
Erf( x )
	double	x;
	{
	return x < 0.0 ? -PGamma( 0.5, x * x ) : PGamma( 0.5, x * x );
	}

double
Erfc( x )
	double	x;
	{
	return x < 0.0 ? 1.0 + PGamma( 0.5, x * x ) : QGamma( 0.5, x * x );
	}

double
CPoisson( x, k )
	double	x;
	int	k;
	{
	return QGamma( (double)k, x );
	}

double
PChiSq( chisq, df )
	double	chisq;
	int	df;
	{
	return PGamma( (double)df / 2.0, chisq / 2.0 );
	}

double
QChiSq( chisq, df )
	double	chisq;
	int	df;
	{
	return QGamma( (double)df / 2.0, chisq / 2.0 );
	}
