/*
    $Header: /aware0/aware/piercarl/Src./Commands./strings.c,v 1.8 89/03/26 17:39:13 piercarl Exp $
*/

static char Notice[] =
    "Copyright (C) 1988 Piercarlo Grandi. All rights reserved.";

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the  Free Software Foundation; either version 1, or (at your option)
    any later version.

    This  program is distributed in the hope that it will be useful, but
    WITHOUT   ANY   WARRANTY;  without  even  the  implied  warranty  of
    MERCHANTABILITY  or  FITNESS  FOR  A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You may have received a copy of the GNU General Public License along
    with  this  program;  if not, write to the Free Software Foundation,
    Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
    An introduction to this program.

    This  is  a  clone  of the strings(1) program present under BSD. It is
    much   more   general,   and  most  importantly  supports  COFF  style
    executables.

    It  is  easy to modify this program along two different lines, one way
    is for other types of executables, and another is for a task different
    from printing strings.

    This is possible because scanning a COFF file and printing strings are
    respectively  a  functional  and a function; the scanner may be passed
    any  suitable  function (e.g. a disassenbler), and the function may be
    used with many different scanners.

    It  is  not an exact clone of strings(1); two new options are allowed,
    -p  to  print  the  pathname of the file before each string, and -c to
    print only null terminated strings, that is C strings.
*/

#define public		/* extern */
#define private		static
#define reg		register

#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
extern long		ftell();

#define StringsTRACE	0

#define StringsCOFF	1		/* COFF executable format	*/
#define StringsBSD	0		/* BSD executables		*/
#define StringsV7	0		/* V7 executables		*/
#define StringsX86	0		/* Xenix/Intel executables	*/

#define StringsMAX	(512)

#if (StringsAOUT)
#   include "a.out.h"
#endif
#if (StringsCOFF)
#   include "filehdr.h"
#   include "scnhdr.h"
#   include "ldfcn.h"
#endif

/*
    The options. All but the last are boolean.
*/

short unsigned		dasha = 0;
short unsigned		dasho = 0;
short unsigned		dashp = 0;
short unsigned		dashc = 0;
short unsigned		dashmin = 4;

/*
    This returns non zero if the given path accessible thru the given file
    is an executable image (or an archive of executable images)
*/

private short unsigned	StringsAout(file,path)
    FILE		    *file;
    char		    *path;
{

    /*
	Our first test is that the given file is a regular file, because
	we  MUST  be able to lseek(2) an executable (admittedly we could
	relax this precondition, but the great care adn effort needed is
	not probably worth the utility.
    */

checkForFile:

    {
	struct stat		stat;

	if (fstat(fileno(file),&stat) < 0)
	{
	    perror(path);
	    return 0;
	}

#	if (StringsTRACE)
	    fprintf(stderr,"Aout: mode 0%o\n",stat.st_mode);
#	endif

	if ((stat.st_mode&S_IFMT) != S_IFREG)
	    return 0;
    }

#   if (StringsCOFF)

hasCoffFormat:
    {
	LDFILE			*ldfile;
	short unsigned		aout;

	if ((ldfile = ldopen(path,NULL)) == NULL)
	{
	    extern int		    errno;

	    if (errno != 0)
		perror(path);
	    return 0;
	}

#	if (StringsTRACE)
	    fprintf(stderr,"Aout: TYPE 0%o, f_magic 0%o\n",
		TYPE(ldfile),HEADER(ldfile).f_magic);
#	endif

	aout = (TYPE(ldfile) == ARTYPE) || ISCOFF(HEADER(ldfile).f_magic);
	ldaclose(ldfile);

	return aout;
    }
#   endif
}

/*
    This  will  apply  the  data extractor to the data section(s) of the
    executable  with  the given path and accessible thru the given file.
    It  is  guaranteed that if the file is stdin this procedure will not
    be called.
*/

private void		StringsData(file,path,extractor)
    FILE		    *file;
    char		    *path;
    void		    (*extractor)(/*FILE *,char *,long*/);
#if (StringsCOFF)
{
    LDFILE		    *ldfile;

scanAllArchiveMembers:

    for
    (
	ldfile = ldopen(path,NULL);
	ldfile != NULL;
	ldfile = (ldclose(ldfile) == SUCCESS) ? NULL : ldopen(path,ldfile)
    )
    if (ISCOFF(HEADER(ldfile).f_magic))
    {
	short unsigned		section;
	SCNHDR			scnhdr;

#	if (StringsTRACE)
	    fprintf(stderr,"Data: f_nscns %u\n",HEADER(ldfile).f_nscns);
#	endif

    ForAllSections:
	for (section = 1; section <= HEADER(ldfile).f_nscns; section++)
	if (ldshread(ldfile,section,&scnhdr) != SUCCESS)
	    perror(path);
	else
	{
#	    if (StringsTRACE)
		fprintf(stderr,"Data: section %u, s_flags 0%lo\n",
		    section,scnhdr.s_flags);
#	    endif

	skipNonDataSection:

	    if ((scnhdr.s_flags & 0x0000000f) != STYP_REG
		|| !(scnhdr.s_flags & STYP_DATA))
		continue;

#	    if (StringsTRACE)
		fprintf(stderr,"Data: s_scnptr 0x%08lx, s_size 0x%08lx\n",
		    scnhdr.s_scnptr,scnhdr.s_size);
#	    endif

	seekForSectionStart:

	    if (ldsseek(ldfile,section) < 0)
	    {
		perror(path);
		continue;
	    }

	extractFromSection:

	    (*extractor)(IOPTR(ldfile),path,scnhdr.s_size);
	}
    }
}
#endif

/*
    This defines which characters are part of a good string.
*/
#define StringsGOOD(c)	(isprint(c) || isspace(c))

/*
    This  accumulates  a  good  string,  starting  from the current file
    position  for  the  given  number  of  bytes.  We  consider a string
    terminated  either  by a newline or by a non string character, or if
    strict  C type strings are wanted, by a null character, as we assume
    that non newline, non null terminated strings are spurious.
*/

private void		StringsScan(file,path,bytes)
    FILE		    *file;
    char		    *path;
    long		    bytes;
{
    static char		    string[StringsMAX];
    reg char		    *s;
    reg int		    c;
    reg short unsigned	    l;

#   if (StringsTRACE)
	fprintf(stderr,"Scan: ftell 0x%08lx, bytes 0x%08lx\n",ftell(file),bytes);
#   endif

    for (s = string, l = 0; bytes != 0 && (c = getc(file)) != EOF; --bytes)
    {
	/*
	    A potential string is terminated either by a newline or by a
	    non  string  character. If this is a non terminal character,
	    we  add it to the string as long as the string max length is
	    not overflowed.

	    Note that if we only accept null terminated strings, newline
	    is no longer considered to be a string terminator.
	*/

	if (StringsGOOD(c) && (c != '\n' || dashc))
	{
	    if (s < (string + sizeof string - 1))
		*s++ = c;
	    l++;
	}
	/*
	    The  string  has  been  terminated;  if it is non empty, and
	    longer  than  the prescribed minimum, and (only when flag -c
	    is on) zero terminated, it is a bona fide string and it will
	    be printed.

	    Notice  that with the following condition we will print also
	    zero  length strings IFF they are newline terminated; with a
	    zero  dashmin  we  would  otherwise  recognize a null string
	    between every couple of bad characters.
	*/
	else if (s != string || c == '\n')
	{
	    *s++ = '\n'; if (c == '\n') l++; *s++ = '\0';
	    if (l >= dashmin && (!dashc || c == '\0'))
	    {
		if (dashp)	printf("%s: ",path);
		if (dasho)	printf("0x%08lx: ",ftell(file)-1-l);
		fputs(string,stdout);
	    }
	    s = string, l = 0;
	}
    }

    if (ferror(stdin))
	perror(path);
}

/*
    Our  job  is  easy;  first we collect the options, then we apply the
    extractor,  directly  or thru the executable scanner, to each of the
    files whose name remain as arguments.
*/

/*ARGSUSED*/
public int		main(argc,argv,envp)
    int			    argc;
    char		    **argv;
    char		    **envp;
{

parseOptions:

    for
    (
	--argc,argv++;
	argc > 0 && (*argv)[0] == '-' && (*argv)[1] != '\0';
	--argc, argv++
    )
    {
	register char	  *flags;

	for (flags = *argv + 1; *flags != '\0'; flags++)
	switch (*flags)
	{
	case 'o':	dasho = 1; break;
	case 'a':	dasha = 1; break;
	case 'p':	dashp = 1; break;
	case 'c':	dashc = 1; break;
	default:
	    if (!isdigit(*flags))
	    {
		fputs("Syntax: strings [ -[acop] ] [ -nnnn ] [ - ] [ file ...]\n",stderr);
		exit(1);
	    }

	    dashmin = *flags-'0';
	    for (flags++; isdigit(*flags); flags++)
		dashmin = dashmin*10 + (*flags-'0');
	    --flags;
	    if (dashmin < 1)
		dashmin = 1;
	}
    }

    /*
	A do because we must loop at least once on stdin, even if no arg
	path is given.
    */

    if (argc <= 0)
	*argv = "stdin";

processInputs:

    do
    {
	/*
	    If  this is an actual file (not a pipe) and is an executable
	    image  (and option a is not specified) we will scan just the
	    data sections, else we will scan all the file.

	    An  assumption here we guarantee is that StringsData will be
	    called  ONLY  on  a  regular  file  to which lseek(2) can be
	    applied, and this is ensured by Aout().
	*/

	if (argc > 0 && freopen(*argv,"r",stdin) == NULL)
	{
		perror(argv[0]);
		exit(1);
	}

	if (dasha || !StringsAout(stdin,*argv))
	    StringsScan(stdin,*argv,0x0fffffffL);
	else
	    StringsData(stdin,*argv,StringsScan);

	if (argc > 0)
	    --argc, argv++;
    }
    while (argc > 0);

    fclose(stdin);

    exit(0);
    /*NOTREACHED*/
}
