/*
 * swho -- display users, ttys, and login times using curses
 *
 * usage: swho [-v]
 *
 * -v	Disable use of standout mode (usually reverse video)
 *
 * Written by Paul Pomes, University of Illinois, Computing Services Office
 * Copyright (C) 1985 by Paul Pomes and the University of Illinois Board
 * of Trustees
 *
 * This program is distributed in the hope that it will be useful,
 * but without any warranty.  No author or distributor accepts
 * responsibility to anyone for the consequences of using it or for
 * whether it serves any particular purpose or works at all, unless
 * s/he says so in writing.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this program under the following conditions:
 *
 *    Permission is granted to anyone to make or distribute copies
 *    of program source code, either as received or modified, in any
 *    medium, provided that all copyright notices, permission and
 *    nonwarranty notices are preserved, and that the distributor
 *    grants the recipient permission for further redistribution as
 *    permitted by this document, and gives him and points out to
 *    him an exact copy of this document to inform him of his rights.
 *
 *    Permission is granted to distribute this program in compiled
 *    or executable form under the same conditions applying for
 *    source code, provided that either
 *    A. it is accompanied by the corresponding machine-readable
 *       source code, or
 *    B. it is accompanied by a written offer, with no time limit,
 *       to give anyone a machine-readable copy of the corresponding
 *       source code in return for reimbursement of the cost of
 *       distribution.  This written offer must permit verbatim
 *       duplication by anyone.
 *    C. it is distributed by someone who received only the
 *       executable form, and is accompanied by a copy of the
 *       written offer of source code which he received along with it.
 *
 * In other words, you are welcome to use, share and improve this
 * program.  You are forbidden to forbid anyone else to use, share
 * and improve what you give them.   Help stamp out software-hoarding!
 *
 * UUCP:	{ihnp4,pur-ee,convex}!uiucdcs!uiucuxc!paul
 * ARPANET:	paul@uiucuxc.cso.uiuc.edu
 * CSNET:	paul%uiucuxc@uiuc.csnet
 * ICBMS:	88 13 N / 40 07 W
 * US Mail:	Univ of Illinois, CSO, 1304 W Springfield Ave, Urbana, IL  61801
 *
 * $Log:	swho.c,v $
 * Revision 1.5  86/01/01  13:17:42  paul
 * Made screen re-draw a #ifdef REDRAW option.
 * 
 * Revision 1.4  85/12/09  17:08:02  paul
 * Added -v switch and test for termcap entry sg to disable standout mode.
 * sg entry indicates standout mode needs a space before and after.  Not
 * so good for a program that uses every one.
 * 
 * Revision 1.3  85/11/15  15:56:10  paul
 * Added feature to clear and re-draw the screen every ten minutes.
 * 
 * Revision 1.2  85/10/30  13:18:17  paul
 * Added code to denote ttys with no shell as "  ----  " in the name
 * field.   -pbp
 * 
 * Revision 1.1  85/10/24  17:37:22  paul
 * Initial revision
 * 
 */

#ifndef lint
static char	RcsId[] = "$Header: swho.c,v 1.5 86/01/01 13:17:42 paul Exp $";
#endif

#include	<stdio.h>
#include	<utmp.h>

#ifdef	SYS5
#include	<string.h>
#define		index		strchr
#else
#include	<strings.h>
#endif	SYS5

#include	<sys/time.h>
#include	<signal.h>
#include	<curses.h>
#include	<sys/types.h>
#include	<sys/stat.h>

#define		equal(s1, s2)	(strcmp(s1, s2) == 0)

/* utmp sizes */
#define		NMAX		sizeof(utmp.ut_name)
#define		LMAX		sizeof(utmp.ut_line)

/* width of 1 display column */
#define		COL_WIDTH	20

/* update interval in seconds */
#define		INTERVAL	10

/* count of how many INTERVALs before completely re-drawing the screen.
 * comment this out to disable screen re-draws.
 */
#define		REDRAW		60

/* x/60 rounded */
#define		DIV60(t)	((t+30)/60)

/* number of COL_WIDTH columns on screen */
int		ncols;

/* special terminal types and their symbols */
struct special {
	char	*name;
	char	symbol;
} special[] = {
	"dialup",	'D',
	"lnet",		'L',
	"telenet",	'T',
	"wats",		'W',
	"switch",	'S',
	NULL,		'\0'
};

/* set if terminal inserts blanks when standout mode is used */
int		sg;

/* current time for idle calculations and display */
time_t		clock;

/* one line of utmp information */
struct utmp	utmp;

/* copy of argv[0] for error messages */
char		*self;

/* standout mode disabled if set */
int		video_flag = 0;

/* debug messages printed if set */
int		debug_flag = 0;

/* interrupt handler */
int		quit();

main(argc, argv)
int	argc;
char	**argv;
{
	/* a useful counter */
	int		i;

	/* line and column manipulation */
	int		line, col;

#ifdef	REDRAW
	/* re-draw screen when equal to REDRAW */
	int		redo_timer = 0;
#endif	REDRAW

	/* number of minutes user is idle */
	time_t		idle;

	/* copy of utmp.ut_line */
	char		tty[LMAX+1];

	/* copy of utmp.ut_name */
	char		name[NMAX+1];

	/* stat /etc/utmp every INTERVAL seconds; read only when changed */
	struct stat	stb;
	time_t		*mtime = &stb.st_mtime;
	time_t		lasttime;

	/* stream descriptor for /etc/ttys file */
	register FILE	*tfd;

	/* set to '1' if tty line has a shell */
	int		ttyon;

	/* stream descriptor for utmp file */
	register FILE	*ufd;

	/* pointer to time struct */
	struct tm	*tm;

	/* library routines */
	char		*malloc();

	/*
	 * squirrel a copy of *argv[0] away for use in error messages
	 */
	self = malloc((unsigned) (strlen(*argv) + 1));
	(void) strcpy(self, *argv);

	/* parse arguments */
	i = 1;
	while (i < argc && *argv[i] == '-') {
		if (equal(argv[i]+1, "d")) {
			/* d - set debug level */
			debug_flag++;
			i++;
			fprintf(stderr, "%s: debug option enabled\n", self);
		}
		if (equal(argv[i]+1, "v")) {
			/* v - disable use of standout mode */
			video_flag++;
			i++;
		}
		else {
			/* command line errors */
			fprintf(stderr, "%s: %s - bad flag\n", self, argv[i]+1);
			fprintf(stderr, "Usage: %s [-v]\n", self);
			exit(1);
		}
	}
	if ((tfd = fopen("/etc/ttys", "r")) == NULL) {
		fprintf(stderr, "%s: ", self);
		perror("/etc/ttys");
		exit(1);
	}
	if ((ufd = fopen("/etc/utmp", "r")) == NULL) {
		fprintf(stderr, "%s: ", self);
		perror("/etc/utmp");
		exit(1);
	}
	(void) signal(SIGINT, quit);
	(void) signal(SIGHUP, quit);
	so_chk();
	initscr();
	ncols = (COLS / COL_WIDTH) - 1;
	noecho();
	ttylist();
	*mtime = (time_t) 0;
	while (1) {
#ifdef	REDRAW
		if (redo_timer++ == REDRAW) {
			redo_timer = 0;
			*mtime = (time_t) 0;
			ttylist();
		}
#endif	REDRAW
		(void) time(&clock);
		tm = localtime(&clock);
		if (! video_flag)
			standout();
		mvprintw(0, 0, "%.19s", asctime(tm));
		if (! video_flag)
			standend();
		lasttime = *mtime;
		(void) fstat(fileno(ufd), &stb);
		if (*mtime > lasttime) {
			(void) fread((char *) &utmp, sizeof(utmp), 1, ufd);
			ttyon = fgetc(tfd);
			for (line = 1, col = 0; ;) {
				if (fread((char *) &utmp, sizeof(utmp), 1, ufd) != 1)
					break;
				if (line == LINES) {
					if (col == ncols)
						break;
					else {
						line = 0;
						col++;
					}
				}
				if (ttyon == '0' && utmp.ut_name[0] == '\0')
					mvprintw(line, col*20, "  ----  ");
				else if (utmp.ut_name[0] == '\0') {
					mvprintw(line, col*20, "        ");
					mvprintw(line, col*20+13, "     ");
					if (! sg && ! video_flag)
						standout();
					mvprintw(line, col*20+8, " ");
					if (! sg && ! video_flag)
						standend();
				}
				else {
					tm = localtime((time_t *) &utmp.ut_time);

					/* utmp strings may not have end null */
					(void) strncpy(tty, utmp.ut_line, LMAX);
					(void) strncpy(name, utmp.ut_name, NMAX);
					mvprintw(line, col*20, "%8s", name);
					mvprintw(line, col*20+13, "%2d:%02d", tm->tm_hour, tm->tm_min);
					idle = findidle();
					if (! sg && ! video_flag)
						standout();
					if (idle > (time_t) 60)
						mvprintw(line, col*20+8, "+");
					else if (idle > (time_t) 30)
						mvprintw(line, col*20+8, "-");
					else
						mvprintw(line, col*20+8, " ");
					if (! sg && ! video_flag)
						standend();
				}
				/* advance the pointer in /etc/ttys */
				while ((ttyon = fgetc(tfd)) != EOF && ttyon != '\n')
					;
				if (ttyon == EOF || (ttyon = fgetc(tfd)) == EOF)
					break;
				line++;
			}
			rewind(tfd);
			rewind(ufd);
		}
		refresh();
		sleep(INTERVAL);
	}
}

/*
 * quit -- cleanup after interrupt
 *
 *	parameters:
 *		none
 *	returns:
 *		none
 *	side effects:
 *		none
 *	deficiencies:
 */
quit()
{
	(void) signal(SIGINT, SIG_IGN);
	clear();
	refresh();
	endwin();
	exit(0);
}

/*
 * so_chk -- check whether terminal inserts blanks with standout mode
 *
 *	parameters:
 *		none
 *	returns:
 *		none
 *	side effects:
 *		sets global variable sg
 *	deficiencies:
 */
so_chk()
{
	char	tbuf[1024];
	int	ret_value;

	char	*getenv();

	if ((ret_value = tgetent(tbuf, getenv("TERM"))) != 1) {
		if (ret_value == 0)	/* no entry */
			sg = 0;
		else {
			fprintf(stderr, "%s: so_chk: can't open /etc/termcap\n", self);
			exit(1);
		}
		return;
	}
	if ((sg = tgetnum("sg")) == -1)
		sg = 0;
	return;
}

/*
 * ttylist -- display the ttys
 *
 *	parameters:
 *		none
 *	returns:
 *		none
 *	side effects:
 *		updates the display
 *	deficiencies:
 */
ttylist()
{

	register FILE	*fd0, *fd1;
	register struct special *ps;
	char		entry[25];
	register char	*pe;
	register	c;
	int		line, col;

	/* open files ttys and ttytype.  die gracefully otherwise */
	if ((fd0 = fopen("/etc/ttys", "r")) == NULL) {
		fprintf(stderr, "%s: ttylist: ", self);
		perror("/etc/ttys");
		exit(1);
	}
	if ((fd1 = fopen("/etc/ttytype", "r")) == NULL) {
		fprintf(stderr, "%s: ttylist: ", self);
		perror("/etc/ttytype");
		exit(1);
	}

	clear();
	/* first line is the console and is a special case */
	pe = entry;
	while ((c = getc(fd0)) != '\n')
		*pe++ = c;
	*pe = '\0';
	pe = entry;
	mvprintw(1, 9, "%.3s", pe+2);

	/* first line of ttytype we don't care about */
	while ((c = getc(fd1)) != '\n')
		;

	/* now continue with the rest of the files */
	for (col = 0, line = 2; ; ) {
		if (line == LINES) {
			if (col == ncols)
				break;
			else {
				line = 0;
				col++;
			}
		}
		while ((c = getc(fd0)) != '\n' && c != EOF)
			*pe++ = c;
		if (c == EOF)
			break;
		*pe = '\0';
		pe = entry;
		mvprintw(line, col*20+10, "%2s", pe + (strlen(pe)-2));

		/* read the tty type and mark the special ones */
		while ((c = getc(fd1)) != '\n' && c != EOF)
			*pe++ = c;
		if (c == EOF)
			break;
		*pe = '\0';
		pe = entry;
		if (index(pe, '\t'))
			*(index(pe, '\t')) = '\0';
		else if (index(pe, ' '))
			*(index(pe, ' ')) = '\0';
		if (! sg && ! video_flag)
			standout();
		mvprintw(line, col*20+8, "  ");
		for (ps = special; ps->name != NULL; ps++)
			if (equal(ps->name, pe))
				mvprintw(line, col*20+8, " %c", ps->symbol);
		if (! sg && ! video_flag)
			standend();
		line++;
	}
	(void) fclose(fd0);
	(void) fclose(fd1);

	/* clean up and go back */
	refresh();
}

/*
 * findidle -- find & return number of minutes current tty has been idle
 *
 *	parameters:
 *		none
 *	returns:
 *		idle time in (time_t) minutes
 *	side effects:
 *		none
 *	deficiencies:
 *		idle time is a slippery idea, this routine checks only the
 *		access timestamp on the utmp.ut_line tty.
 */
time_t
findidle()
{
	struct stat stbuf;
	time_t lastaction, diff;
	char ttyname[20];

	(void) strcpy(ttyname, "/dev/");
	(void) strcatn(ttyname, utmp.ut_line, LMAX);
	(void) stat(ttyname, &stbuf);
	lastaction = stbuf.st_atime;
	diff = clock - lastaction;
	diff = DIV60(diff);
	if (diff < 0)
		diff = 0;
	return(diff);
}
