/*
	sample: display random samples
*/
#include "curses.h"
#include <math.h>

int 	BOTTOM;                 /* set to LINES-1 */

#define	MAXCOLS 100             /* max # of columns on screen */
int 	Colcount[MAXCOLS];      /* current count in column bin */
int 	Sampsize = 1;           /* sample size */
int 	Outline = 0;            /* only show shape of distribution */
int 	Plotchar = '*';         /* histogram plotting character */

typedef	double	DATUM;
#define	MAXDISTRIB    1000      /* max size of population distribution */
DATUM	Distrib[MAXDISTRIB];    /* population distribution */
int 	N = 0;                  /* population size */
DATUM	S1 = 0.0;               /* sum of distribution values */
DATUM	S2 = 0.0;               /* sum of squared distribution values */
DATUM	S3 = 0.0;               /* sum of cubed distribution values */
DATUM	S4 = 0.0;               /* sum of quarted distribution values */
DATUM	Max;                    /* distribution maximum */
DATUM	Min;                    /* distribution minimum */

double	sample ();
#define	EPSILON .000000001
#define	scale(x) (((x)-Min)/(Max-Min+EPSILON)) /* in [0, 1) */
#define	fzero(x) (fabs(x) < EPSILON)

main (argc, argv) char **argv;
	{
	double 	x;
	double	x2;
	int 	n = 0;
	double	s1 = 0.0;
	double	s2 = 0.0;
	double	s3 = 0.0;
	double	s4 = 0.0;
	readdata ();
	initial (argc, argv);
	for (;;)
		{
		if (Sampsize < 1) /* just dump out distribution */
			{
			if (n == N) finish ();
			x = Distrib[n];
			}
		else x = sample (Sampsize, Distrib, N);
		x2 = x * x;
		n++;
		s1 += x;
		s2 += x2;
		s3 += x2 * x;
		s4 += x2 * x2;
		if (plotdata (x, n, s1, s2, s3, s4) == 0)
			if (Sampsize > 0) finish ();
			else beep ();
		}
	}

readdata ()
	{
	DATUM	x;
	DATUM	x2;
	char	string[100];
	while (getword (string, stdin))
		{
		if (N == MAXDISTRIB)
			{
			fprintf (stderr, "sample: max # of values = %d\n", MAXDISTRIB);
			exit (1);
			}
		x = atof (string);
		if (N == 0) /* first time around */
			Min = Max = x;
		else if (x > Max)
			Max = x;
		else if (x < Min)
			Min = x;
		Distrib[N++] = x;
		S1 += x;
		x2 = x * x;
		S2 += x2;
		S3 += x2 * x;
		S4 += x2 * x2;
		}
	if (N == 0)
		{
		fprintf (stderr, "sample: missing input distribution\n");
		exit (2);
		}
	}

initial (argc, argv) char **argv;
	{
	long	time ();
	extern	char	*optarg;
	extern	int 	optind;
	int 	errflg = 0;
	int 	C;
	char	*optstring = "c:n:o";
	char	*usage = "[-o] [-c plotchar] [-n size]";
	while ((C = getopt (argc, argv, optstring)) != EOF)
		switch (C)
			{
			case 'c': Plotchar = *optarg; break;
			case 'n': Sampsize = atoi (optarg); break;
			case 'o': Outline = 1; break;
			default: errflg++; break;
			}
	if (errflg)
		{
		fprintf (stderr, "Usage: %s %s\n", argv[0], usage);
		exit (1);
		}
	srandom (time (0) + getpid ());
	initscr ();
	BOTTOM = LINES-1;
	clear ();
	bottomline ();
	inform (0, N, S1, S2, S3, S4);
	refresh ();
	}

bottomline ()
	{
	char	buf[BUFSIZ];
	int 	width = 7;
	move (BOTTOM, 0);
	clrtoeol ();
	standout ();
	printw ("%-.2f", Min);
	move (BOTTOM, COLS-width);
	printw ("%*.2f", width-1, Max);
	if (Sampsize > 0)
		sprintf (buf, "Displaying samples of size %d", Sampsize);
	else
		sprintf (buf, "Displaying the base distribution");
	move (BOTTOM, (COLS-strlen(buf))/2);
	printw (buf);
	standend ();
	}

finish ()
	{
	move (BOTTOM, 0);
	clrtoeol ();
	refresh ();
	endwin ();
	exit (0);
	}

/*
	takes a sample of size "n" from a supplied vector
	returns the mean of the sample
*/
double
sample (n, distrib, size)
int 	n;        /* sample size */
DATUM	*distrib; /* vector of population values */
int 	size;     /* size of population distribution */
	{
	int 	trial;
	DATUM	s1 = 0.0;
	if (n == 0 || size == 0) return (0.0);
	for (trial = 0; trial < n; trial++)
		s1 += distrib [ random () % size ];
	return (s1 / n);
	}

plotdata (x, n, s1, s2, s3, s4)
double 	x;
int 	n;
double	s1;
double	s2;
double	s3;
double	s4;
	{
	int 	dindex = (int) (scale (x) * COLS);
	int 	col = dindex;
	int 	row;
	Colcount[dindex]++;
	if (Colcount[dindex] < BOTTOM)
		{
		inform (40, n, s1, s2, s3, s4);
		row = LINES-Colcount[dindex]-1;
		if (Outline && row < BOTTOM-1)
			{
			move (row+1, col);
			addch (' ');
			}
		move (row, col);
		addch (Plotchar);
		refresh ();
		return (1);
		}
	else return (0);
	}

inform (column, n, s1, s2, s3, s4)
int 	column;
int 	n;
double	s1;
double	s2;
double	s3;
double	s4;
	{
	double	M = 0.0;
	double	var = 0.0;
	double	sd = 0.0;
	double	skew;
	double	kurtosis;
	double	m2;
	if (n > 0)
		{
		move (0, column);
		M = s1/n;
		printw ("#%-4d %5.2f ", n, M);
		m2 = M * M;
		if (n > 1)
			{
			var	= (s2 - M*s1)/(n-1);
			if (fzero (var)) kurtosis = skew = sd = var = 0.0;
			else
				{
				sd	= sqrt (var);
				skew = (s3 - 3.0*M*s2 + 3.0*m2*s1 - m2*s1)/(n*var*sd);
				kurtosis = (s4-4.*M*s3+6.*m2*s2-4.*m2*M*s1+n*m2*m2)/(n*var*var);
				}
			printw (" %5.2f", sd);
			printw (" %5.2f", skew);
			printw (" %5.2f", kurtosis);
			clrtoeol ();
			}
		}
	}
