/*
 * last.c -- examine the wtmp file for last login information
 *
 * =*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=
 *  Copyright (c) 1988  Harvey R. Moran Jr.
 *                      1936 Altavue Rd.
 *                      Catonsville, Md., 21228
 *
 *                      moran%tron.UUCP@umbc3.UMD.EDU   Internet
 *                      {wb3ffv,netsys}!hrmhpc!harvey   UUCP
 *
 *  This software may be freely distributed provided that this
 *  copyright notice is left intact and provided that:
 *     a) The source code in machine readable format 
 *        is included with any binary distribution.
 *     b) If a binary version of this program or a derived work is sold,
 *        the source code must be provided in machine readable
 *        format for no additional charge.
 *     c) No derived works may impose restrictions limiting free
 *        distribution of the source code.
 *  Source code in this context includes the C language source code
 *  provided here as file "last.c" and the manual page provide here as
 *  file "last.1".  It also includes any works derived from either of these.
 *  
 * =*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=
 *  Developed using Microport System V/AT 2.3U
 *
 *   usage:
 *        last [ -N ] [ username ]
 *
 *  This will print the usage of the system as
 * recorded in WTMP_FILE (defined in /usr/include/utmp.h) for
 * all users except the ones listed in exclude[] (below).
 * The option -N specifies that a maximum of N lines of
 * output will be generated.
 * The optional username form will restrict information printed
 * to be logins by the specified user.
 *
 * This command is modeled on the "last" command which is distributed
 * with the Berkeley Software Distribution (BSD) of UNIX(tm).
 * The print formats are slightly different. The field which
 * specifies a network login identification are not provided with this
 * version because the information is not available in the System V.2
 * UNIX(tm) WTMP_FILE file.
 *
 * It will also quit (with error message) if there are more than
 * MAXENTRIES entries in your WTMP_FILE (/etc/wtmp on my system).
 *
 *  Harvey Moran  11/10/88
 */
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <utmp.h>
#include <time.h>

#ifndef MAXENTRIES
#	define MAXENTRIES 1024	/* maximum entries in WTMP_FILE handled */
#endif

#define STILL_IN -1L
#define AVOID    -1L

typedef int BOOL;
#define YES 1
#define NO  0

typedef struct utmp WTMP_t;
WTMP_t proc_tab;

typedef struct {
    char user_name[9];
	short pid;
	time_t x_time;
	time_t y_time;
	} ENTRY;

ENTRY entries[MAXENTRIES];

/*
 * user process user names to be excluded from display
 * change per local installation.  Perhaps these should be
 * in an environment variable instead, but there are already
 * so many of them.
 */
char *exclude[] = { "rc2", "crtsaver", "LOGIN", NULL };

char *keep_list = (char *) NULL;

/*
 * forward declared functions
 */

char *delta_time();
void qsort();
int cmp_ENTRY(), cmp_ENTRY_1();

main(ac, av)
int ac;
char *av[];
{
	int i, j, fd;
	unsigned  count = (unsigned) -1;	/* BIG unsigned int. */
	char intime[27], outtime[27], elapsed[27];
	long time_tmp;

	if ( ac > 1 && (i = atoi(av[1])) < 0 ) {
		count = (unsigned) -i;
		for ( i = 2; i < ac; ++i )
			av[i-1] = av[i];
		av[i] = NULL;
		--ac;
		}
	if ( (fd=open(WTMP_FILE, (O_RDONLY|O_NDELAY))) == -1 ) {
		fprintf(stderr, "%s:Can't open %s\n", av[0], WTMP_FILE);
		exit(1);
		}
	i = 0;
	while ( read(fd, &proc_tab, sizeof(WTMP_t)) == sizeof(WTMP_t) ) {
		switch ( proc_tab.ut_type ) {
			case EMPTY:			/* Empty slot in the WTMP_FILE */
			case INIT_PROCESS:	/* Process started by "init" */
			case LOGIN_PROCESS:	/* A "getty" process waiting for login */
			case ACCOUNTING:
				break;			/* IGNORE all these forever */

			case RUN_LVL:		/* Change of Run Level */
			case BOOT_TIME:		/* Boot time */
			case OLD_TIME:		/* Old time of Day was */
			case NEW_TIME:		/* New Time of Day set */
				break;			/* IGNORE all these for now */

			case USER_PROCESS:	/* A user process was started */
			case DEAD_PROCESS:	/* A user (I think) process died */
				strncpy(entries[i].user_name, proc_tab.ut_user, sizeof(proc_tab.ut_user));
				if ( omit(entries[i].user_name, ac, av) )
					break;
				if ( ac == 2 && strcmp(av[1], entries[i].user_name) )
					break;	/* not the user of interest */
				entries[i].pid = proc_tab.ut_pid;
				entries[i].x_time = proc_tab.ut_time;
				if ( ++i > (sizeof(entries)/sizeof(ENTRY)) ) {
					fprintf(stderr, "Too many entries in %s\n", WTMP_FILE);
					fprintf(stderr, "Fix program %s\n", av[0]);
					exit(2);
					}
				break;

			default:
				fprintf(stderr, "Ignoring GARBAGE in file %s. (ut_type = 0x%02x)\n", WTMP_FILE, proc_tab.ut_type);
				break;
			}
		}
	/*
     * sort by pid, x_time
     */
	qsort((char *) entries, i, sizeof(ENTRY), cmp_ENTRY);
	j = 0;
#define E entries	/* minimize line wrapping of source code */
	do {
		if ( j == (i-1) ) {
			/*
			 * final login has no corresponding logout
			 * avoid referencing array element with is empty (or does not exist)
			 */
			E[j].y_time = STILL_IN;
			break;
			}
		else if ( E[j].pid != E[j+1].pid ) { /* login has no corresponding logout */
			E[j].y_time = STILL_IN;
			++j;
			}
		else { /* E[j].pid == E[j+1].pid , i.e. login/logout pair */
			E[j].y_time = E[j+1].x_time;
			E[j+1].x_time = AVOID;
			j += 2;
			}
		} while ( j < i );
    /*
     * sort by descending x_time
     */
	qsort((char *) entries, i, sizeof(ENTRY), cmp_ENTRY_1);
	for ( j = 0; j < i && count; ++j ) {
		if ( E[j].x_time == AVOID )
			continue;
		else if ( E[j].y_time == STILL_IN ) {
			time_tmp = (long) E[j].x_time;
			(void) strcpy(intime, asctime(localtime(&time_tmp)));
			intime[20] = '\0';	/* strip year */
			printf("%-8s %s %s still-logged-in\n", E[j].user_name, intime, " ");
			--count;
			}
		else {
			time_tmp = (long) E[j].x_time;
			(void) strcpy(intime, asctime(localtime(&time_tmp)));
			time_tmp = (long) E[j].y_time;
			(void) strcpy(outtime, asctime(localtime(&time_tmp)));
			intime[20] = outtime[20] = '\0';	/* strip year */
			time_tmp = (long) (E[j].y_time - E[j].x_time);
			(void) strcpy(elapsed, delta_time(time_tmp));
			printf("%-8s %s-%s %s\n", E[j].user_name, intime, outtime+10, elapsed);
			--count;
			}
		}
#undef E
}

/*
 * return comparison of 2 ENTRY type records
 * suitable for sorting by: pid, x_time
 */

cmp_ENTRY(a, b)
ENTRY *a, *b;
{
	if ( a->pid < b->pid )
		return (-1);
	if ( a->pid > b->pid )
		return (1);
	if ( a->x_time < b->x_time )
		return (-1);
	if ( a->x_time > b->x_time )
		return (1);
	return (0);
}

/*
 * return comparison of 2 ENTRY type records
 * suitable for a descending sort by: x_time
 */
cmp_ENTRY_1(a,b)
ENTRY *a, *b;
{
	if ( a->x_time < b->x_time )
		return (1);
	if ( a->x_time > b->x_time )
		return (-1);
	return (0);
}


/*
 * return whether to omit this name
 * It is omitted if:
 *       a) the name matches something in the exclude list
 *       b) the name does not match something in the invocation argument list
 *          and the argument list is not null
 */
BOOL
omit(name, ac, av)
char *name;
int ac;
char *av[];
{
	char **p;
	int i;

	for ( p = &exclude[0]; *p != (char *) NULL; ++p ) {
		if ( strcmp(*p, name) == 0 )
			return (YES);
		}
	if ( ac == 1 )	/* empty argument list, do not omit any */
		return (NO);
	for ( i = 1; i < ac; ++i ) {
		if ( strcmp(name, av[i]) == 0 )
			return (NO);
		}
	return (YES);
}


char *
delta_time(t)
long t;
{
	static char AscTime[11];
	int sec, min, hr, da;
	long x, y;

	sec = t % 60L;
	x = (t - (long) sec)/60L;
	min = x % 60L;
	x = (x - (long) min)/60L;
	hr  = x % 24;
	x   = (x - (long) hr)/24L;
	da  = x;
	if ( da )
		(void) sprintf(AscTime, "%3d+%02d:%02d:%02d", da, hr, min, sec);
	else
		(void) sprintf(AscTime, "    %02d:%02d:%02d", hr, min, sec);
	return (AscTime);
}
