/****************************************************************

Module:
	LogCmd

Description:
	Commands related to logarithms (The LOGS menu on HP28C)

Modification history:

	0.0	hjp	89-12-03

		initial version
		LN and EXP extracted from ArithCmd.

****************************************************************/

#include "errors.h"
#include "globvar.h"
#include "rpl.h"
#include "intcmd.h"
#include "logcmd.h"
#include "stackcmd.h"

/*
	ln	--	compute natural logarithm of object in level 1

	x	->	ln(x)

	REAL	->	REAL
	REAL	->	COMPLEX
	COMPLEX ->	COMPLEX
*/

void c_ln (void)
{
	genobj		* a;
	realobj 	* c;
	complexobj	* b;

	if (! stack) {
		error ("ln", ERR_2FEWARG);
		return;
	}

	if ((a = stack->obj)->id == REAL) {
		if (((realobj *)a)->val >= 0.0) {
			if (!(c = mallocobj (REAL)))
			{
				error ("LN", ERR_NOMEM);
				return;
			}
			c->id = REAL;
			c->link = 0;
			c->size = sizeof (realobj);
			c->val = log (((realobj *) a)->val);
			c_drop ();
			push (c);
		} else {
			if (! (b = mallocobj (COMPLEX)))
			{
				error ("LN", ERR_NOMEM);
				return;
			}
			b->val.x = log (-((realobj *) a)->val);
			b->val.y = M_PI;
			c_drop ();
			push (b);
		}
	} else if (a->id == COMPLEX) {
		double	x = ((complexobj *) a)->val.x,
			y = ((complexobj *) a)->val.y;

		if (! (b = mallocobj (COMPLEX)))
		{
			error ("LN", ERR_NOMEM);
			return;
		}
		b->val.x = log (sqrt (x * x + y * y));
		b->val.y = x || y ? atan2 (y, x) : 0.0;
		c_drop ();
		push (b);
	} else {
		error ("inv", ERR_WRTYPE);
	}
}

/*
	EXP	compute e to the power of object in level 1

	x	->	e ^ x

	real	->	real
	complex ->	complex
*/

void c_exp (void)
{
	genobj		* a;
	realobj 	* c;
	complexobj	* b;

	if (! stack) {
		error ("EXP", ERR_2FEWARG);
		return;
	}

	if ((a = stack->obj)->id == REAL) {
		if (!(c = mallocobj (REAL)))
		{
			error ("EXP", ERR_NOMEM);
			return;
		}
		c->val = exp (((realobj *) a)->val);
		c_drop ();
		push (c);
	} else if (a->id == COMPLEX) {
		double	x = ((complexobj *) a)->val.x,
			y = ((complexobj *) a)->val.y;

		if (! (b = mallocobj (COMPLEX)))
		{
			error ("EXP", ERR_NOMEM);
			return;
		}
		b->val.x = exp (x) * cos (y);
		b->val.y = exp (x) * sin (y);
		c_drop ();
		push (b);
	} else {
		error ("EXP", ERR_WRTYPE);
	}
}
