/****************************** MINIT ******************************************
 * This file contains procedures to solve a Linear Programming
 * problem of n variables and m constraints, the last p of which
 * are equality constraints by the dual simplex method.
 *
 * The code was originally in Algol 60 and was published in Collected
 * Algorithms from CACM (alg #333) by Rudolfo C. Salazar and Subrata
 * K. Sen in 1968 under the title MINIT (MINimum ITerations) algorithm
 * for Linear Programming.
 * It was directly translated into C by Badri Lokanathan, Dept. of EE,
 * University of Rochester, with no modification to code structure. 
 *
 * The problem statement is
 * Maximise z = cX
 *
 * subject to
 * AX <= b
 * X >=0
 *
 * c is a (1*n) row vector
 * A is a (m*n) matrix
 * b is a (m*1) column vector.
 * e is a matrix with with (m+1) rows and lcol columns (lcol = m+n-p+1)
 *   and forms the initial tableau of the algorithm.
 * td is read into the procedure and should be a very small number,
 *   say 10**-8
 * 
 * The condition of optimality is the non-negativity of
 * e[1,j] for j = 1 ... lcol-1 and of e[1,lcol] = 2 ... m+1.
 * If the e[i,j] values are greater than or equal to -td they are
 * considered to be non-negative. The value of td should reflect the 
 * magnitude of the co-efficient matrix.
 *
 ******************************************************************************/

/***************************** INCLUDES ***************************************/
#include <stdio.h>
/************************ CONSTANT DEFINITIONS ********************************/
#ifndef LARGENUMBER
#define LARGENUMBER 1e6
#endif  LARGENUMBER
#ifndef VLARGENUMBER
#define VLARGENUMBER 1e8
#endif  VLARGENUMBER
#ifndef VSMALLNUMBER
#define VSMALLNUMBER 1e-8
#endif  VSMALLNUMBER
/************************** MACRO DEFINITIONS  ********************************/
#define ABS(A) (A > 0 ? A: -A)
#define MAX(A,B) (A > B ? A : B)
#define MIN(A,B) (A < B ? A : B)
/************************ FILE-LIMITED VARIABLES ******************************/
static int k, m, n, p, lcol, L, im, jmin, jm, imax, debug;
static float td=VSMALLNUMBER;	/* This is application specific. */
static float gmin, phimax;
/************************** FILE-LIMITED ARRAYS *******************************/
static int *imin, *jmax, *ind, *ind1, *chk;
static float **e, *thmin, *delmax;

/********************************* ZX3LP ***************************************
 * This is a user-specific front end to minit.
 * In its present form, it has been written for compatibility with ifplan. 
 * Note that zx3lp was originally a fortran subroutine. The weird return
 * values and flags (such as IER) are inherited from there.
 * Variable IA and arrays RW, IW are not necessary.
 ******************************************************************************/
zx3lp(A,IA,B,C,N,M1,M2,S,PSOL,DSOL,RW,IW,IER)
float **A, *B, *C, *PSOL, *DSOL, *RW, *S;
int *IA, *N, *M1, *M2, *IW, *IER;
{
	int err;

	/* Allocate space for arrays and initialize globals. */
	if(!minit_space_handler(*N,*M1+*M2,*M2))
	{
		(void) fprintf(stderr,"Space problems in LP handler.\n");
		return;
	}

	/* Invoke minit here. */
	err = minit(A,B,C,PSOL,DSOL,S);
	switch(err)
	{
		case 1:
		case 2:
		*IER = 133;
		break;

		case 3:
		*IER = 131;
		break;

		default:
		*IER = 0;
	}
}

/************************ minit_space_handler **********************************
 * This routine performs space maintenance and initializes global arrays.     
 * It looks at n, m and p, which are static in this file, to look for
 * previously allocated space.
 ******************************************************************************/
int
minit_space_handler(N,M,P)
int N, M, P;
{
	register int i, j;
	float **tmp;
	static int MS, LS;		/* To keep track of array sizes. */
	char *calloc(), *malloc(), *realloc();

	/* Initialize globals. */
	m = M;
	n = N;
	p = P;
	lcol = m + n - p + 1;

	if(LS == 0)
	{
		/* First pass. Allocate space for every array. */

		MS = m + 1;
		LS = m + n - p + 1;
		if(!(jmax = (int *) calloc((unsigned) MS,sizeof(int))))
			return(0);

		if(!(ind1 = (int *) calloc((unsigned) MS,sizeof(int))))
			return(0);

		if(!(chk = (int *) calloc((unsigned) MS,sizeof(int))))
			return(0);

		if(!(delmax = (float *) calloc((unsigned) MS,sizeof(float))))
			return(0);

		if(!(imin = (int *) calloc((unsigned) LS,sizeof(int))))
			return(0);

		if(!(ind = (int *) calloc((unsigned) LS,sizeof(int))))
			return(0);

		if(!(thmin = (float *) calloc((unsigned) LS,sizeof(float))))
			return(0);

		if(!(tmp=e=(float **) calloc((unsigned) MS,sizeof(float *))))
			return(0);

		for(i = 0; i < MS; i++)
			if(!(*(tmp++)=(float *) calloc((unsigned) LS,
					sizeof(float))))
				return(0);

		return(1);
	}

	if(M + 1 > MS)
	{
		/* Need to reallocate space. */
		MS = M + 1;
		if(!(jmax = (int *) realloc((char *)jmax,
				(unsigned)(MS*sizeof(int)))))
			return(0);

		if(!(ind1 = (int *) realloc((char *) ind1,
				(unsigned)(MS*sizeof(int)))))
			return(0);

		if(!(chk = (int *) realloc((char *) chk,
				(unsigned)(MS*sizeof(int)))))
			return(0);

		if(!(delmax = (float *) realloc((char *) delmax,
				(unsigned)(MS*sizeof(float)))))
			return(0);

		if(!(e=(float **) realloc((char *) e,
				(unsigned)(MS*sizeof(float *)))))
			return(0);

		for(tmp = e + m, i = 0; i < MS; i++)
			if(!(*(tmp++)=(float *)
					malloc((unsigned)(LS*sizeof(float)))))
				return(0);
	}

	if(lcol > LS)
	{
		LS = lcol;
		if(!(ind = (int *) realloc((char *) ind,
				(unsigned)(LS*sizeof(int)))))
			return(0);

		if(!(imin = (int *) realloc((char *) imin,
				(unsigned)(LS*sizeof(int)))))
			return(0);

		if(!(thmin = (float *) realloc((char *) thmin,
				(unsigned)(LS*sizeof(float)))))
			return(0);

		for(tmp = e, i = 0; i < MS; tmp++, i++)
			if(!(*tmp=(float *) realloc((char *) tmp,
					(unsigned)(LS*sizeof(float)))))
				return(0);
	}

	/* Finally, initialize all arrays to 0. */
	for(i = 0; i <= m; i++)
	{
		jmax[i] = ind1[i] = chk[i] = 0;
		delmax[i] = 0.0;
		for(j = 0; j < lcol; j++)
			e[i][j] = 0.0;
	}

	for(j = 0; j < lcol; j++)
	{
		imin[j] = ind[j] = 0;
		thmin[j] = 0.0;
	}

	return(1);
}

/********************************* MINIT ***************************************
 * This is the main procedure. It is invoked with the various matrices,
 * after space for other arrays has been generated elsewhere (in zx3lp.)
 * It returns
 * 0 if everything went OK and x, w, z as the solutions.
 * 1 if no solution existed.
 * 2 if primal had no feasible solutions.
 * 3 if primal had unbounded solutions.
 ******************************************************************************/
int
minit(A,b,c,x,w,z)
float **A, *b, *c, *x, *w, *z;
{
	register int i, j;
	void tab();

	/* Generate initial tableau. */
	for(j = 0; j < n; j++)
		e[0][j] = -c[j];

	for(i = 0; i < m; i++)
	{
		for(j = 0; j < n; j++)
			e[i+1][j] = A[i][j];
	}

	for(j = 0; j < m; j++)
		e[j+1][lcol-1] = b[j];

	for(i = 0; i < m - p; i++)
		e[1+i][n+i] = 1.0;

	/* Now begins the actual algorithm. */
	for(i = 1; i < m+1; i++)
		chk[i] = -1;	/* Indexing problem; Algol
				 * version treates 0 as out
				 * of bounds, in C we prefer -1.
				 */

	if(!p)	/* There are no equality constraints */
		goto RCS;
	else
	{
	if(phase1())
		return(1);
	}
		
	RCS: L = 1; k = 1;

	if(debug)
		tab();	/* Print the tableau. */

	for(j = 0; j < lcol - 1; j++)
	{
		if(e[0][j] < -td)
			ind[(L++)-1] = j; /* ind[L] keeps track of the
					   * columns in which e[0][j]
					   * is negative.
					   */
	}

	for(i = 1; i < m + 1; i++)
	{
		if(e[i][lcol-1] < -td)
			ind1[(k++)-1] = i; /* ind1[k] keeps track of the
					    * rows in which e[i][lcol]
					    * is negative.
					    */
	}

	if(L == 1)
	{
		if(k == 1) /* Results */
			goto RESULTS;
		else
		{
			if(k == 2)
			{
				for(j = 0; j < lcol - 1; j++)
				{
					if(e[ind1[0]][j] < 0)
						goto R;
				}
				/* Primal problem has no feasible solutions. */
				return(2);
			}
			else
				goto R;
		}
	}
	else
	{
		if(L == 2)
		{
			if(k == 1)
			{
				for(i = 1; i < m + 1; i++)
				{
					if(e[i][ind[0]] > 0)
						goto C;
				}

				/* Primal problem is unbounded. */
				return(3);
			}
			else
				goto S;
		}

		if(k == 1)
			goto C;
		else
			goto S;
	}

	R:
	prophi();
	if(rowtrans(imax,jm))
		return(1);
	goto RCS;

	C:
	progamma();
	if(rowtrans(im,jmin))
		return(1);
	goto RCS;

	S:
	progamma();
	prophi();
	if(gmin == LARGENUMBER)
	{
		if(rowtrans(imax,jm))
			return(1);
		goto RCS;
	}

	if(phimax == - LARGENUMBER)
	{
		if(rowtrans(im,jmin))
			return(1);
		goto RCS;
	}

	if(ABS(phimax) > ABS(gmin))
	{
		if(rowtrans(imax,jm))
			return(1);
	}
	else
	{
		if(rowtrans(im,jmin))
			return(1);
	}
	goto RCS;

	RESULTS:
	/* Output results here. */
	*z = e[0][lcol-1];
	for(i = 0; i < n; i++)
		x[i] = 0.0;

	for(j = 0; j < m; j++)
		x[j] = 0.0;

	for(i = 1; i < m + 1; i++)
	{
		if(chk[i] >= n)
			chk[i] = -1;

		if(chk[i] > -1)
			x[chk[i]] = e[i][lcol-1];
	}

	for(j = n; j < lcol - 1; j++)
		w[j-n] = e[0][j];
	
	return(0);
}

/****************************** rowtrans ***************************************
 * Performs the usual tableau transformations in a linear programming
 * problem, (p,q) being the pivotal element.
 * Returns the following error codes:
 * 0: Everything was OK.
 * 1: No solution.
 ******************************************************************************/
int
rowtrans(p,q)
int p, q;
{
	register int i, j;
	float dummy;

	if(p == -1 || q == -1) /* No solution. */
		return(1);

	dummy = e[p][q];

	if(debug)
		(void) printf("--\nPivot element is e[%d][%d] = %f\n",p,q,dummy);

	for(j = 0; j < lcol; j++)
		e[p][j] /= dummy; 

	for(i = 0; i < m + 1; i++)
	{
		if(i != p)
		{
			if(e[i][q] != 0.0)
			{
				dummy = e[i][q];
				for(j = 0; j < lcol; j++)
					e[i][j] -= e[p][j] * dummy;
			}
		}
	}

	chk[p] = q;
	return(0);
} /* rowtrans */

/****************************** progamma ***************************************
 * Performs calculations over columns to determine the pivot element.
 ******************************************************************************/
progamma()
{
	float theta, gamma;
	register int i, L1;

	gmin = LARGENUMBER;	/* gmin is set to a large no. for
				 * initialization purposes.
				 */
	jmin = -1;		/* Array indices in C start from 0 */

	for(L1 = 0; L1 < L - 1; L1++)
	{
		imin[ind[L1]] = 0;
		thmin[ind[L1]] = LARGENUMBER;
		for(i = 1; i < m + 1; i++)
		{
			if(e[i][ind[L1]] > td && e[i][lcol-1] >= -td)
			{
				theta = e[i][lcol-1] / e[i][ind[L1]];
				if(theta < thmin[ind[L1]])
				{
					thmin[ind[L1]] = theta;
					imin[ind[L1]] = i;
				}
			}
		}

		if(thmin[ind[L1]] == LARGENUMBER)
			gamma = VLARGENUMBER;
		else
			gamma = thmin[ind[L1]] * e[0][ind[L1]];

		if(gamma < gmin)
		{
			gmin = gamma;
			jmin = ind[L1];
		}
	}
	if(jmin > -1)
		im = imin[jmin];
} /* progamma */

/****************************** prophi *****************************************
 * Performs calculations over rows to determine the pivot element.
 ******************************************************************************/
prophi()
{
	float delta, phi;
	register int j, k1;

	phimax = - LARGENUMBER;	/* phimax is set to a small no. for
				 * initialization purposes.
				 */
	imax = -1;		/* Array indices in C start from 0 */
	
	for(k1 = 0; k1 < k - 1; k1++)
	{
		jmax[ind1[k1]] = 0;
		delmax[ind1[k1]] = - LARGENUMBER;
		for(j = 0; j < lcol - 1; j++)
		{
			if(e[ind1[k1]][j] < -td && e[0][j] >= -td)
			{
				delta = e[0][j] / e[ind1[k1]][j];
				if(delta > delmax[ind1[k1]])
				{
					delmax[ind1[k1]] = delta;
					jmax[ind1[k1]] = j;
				}
			}
		}

		if(delmax[ind1[k1]] == - LARGENUMBER)
			phi = - VLARGENUMBER;
		else
			phi = delmax[ind1[k1]] * e[ind1[k1]][lcol-1];
		
		if(phi > phimax)
		{
			phimax = phi;
			imax = ind1[k1];
		}
	}
	if(imax > -1)
		jm = jmax[imax];
} /* prophi */

/****************************** phase1 *****************************************
 * Applied only to equality constraints if any.
 ******************************************************************************/
phase1()
{
	float theta, gamma;
	register int i, j, r;
	/* Fix suggested by Holmgren, Obradovic, Kolm. */
	register int im1, jmin1, first;
	void tab();

	im1 = jmin1 = -1;
	/* Fix suggested by Messham to allow negative coeffs. in
	 * equality constraints.
	 */
	for(i = m - p + 1; i < m + 1; i++)
	{
		if(e[i][lcol - 1] < 0)
		{
			for(j = 0; j < lcol; j++)
				e[i][j] = -e[i][j];
		}
	}

	for(r = 0; r < p; r++)
	{
		gmin = LARGENUMBER;	/* gmin is set to a large no. for
					 * initialization purposes.
					 */
		L = 1;
		jmin = -1;
		first = 1;
		for(j = 0; j < n; j++)
		{
			thmin[j] = LARGENUMBER;
			/* Fix suggested by Kolm and Dahlstrand */
			/* if(e[0,j] < 0) */
			if(e[0][j] < -td)
				ind[(L++)-1] = j;
		}

	L1:	if(L == 1)
		{
			for(j = 0; j < n; j++)
				ind[j] = j;

			L = n + 1;
		}

		for(k = 0; k < L - 1; k++)
		{
			for(i = m - p + 1; i < m + 1; i++)
				if(chk[i] == -1)
				{
					/* Fix suggested by Kolm
					 * and Dahlstrand
					 */
					/* if(e[i][ind[k]] > 0.0) */
					if(e[i][ind[k]] > td)
					{
						if((theta = e[i][lcol-1] /
						  e[i][ind[k]]) < thmin[ind[k]])
						{
							thmin[ind[k]] = theta;
							imin[ind[k]] = i;
						}
					}
				}

			/* Fix suggested by Obradovic overrides
			 * fixes suggested by Kolm and Dahstrand
			 * as well as Messham.
			 */
			if(thmin[ind[k]] < LARGENUMBER)
			{
				gamma = thmin[ind[k]] * e[0][ind[k]];
				if(gamma < gmin)
				{
					gmin = gamma;
					jmin = ind[k];
				}
			}
		}
		if(jmin == -1)
		{
			if(first)
			{
				first = 0;
				L = 1;
				goto L1;
			}
			else
				im = -1;
		}
		else
			im = imin[jmin];

		if(im == im1 && jmin == jmin1)
		{
			L = 1;
			goto L1;
		}

		if(debug)
			tab();	/* Print the tableau. */

		if(rowtrans(im,jmin))
			return(1);
		
		im1 = im;
		jmin1 = jmin;
	}
	return(0);
} /* phase1 */

/****************************** tab *****************************************
 * The following procedure is for debugging. It simply prints the
 * current tableau.
 ******************************************************************************/
static void
tab()
{
	register int i, j;

	(void) printf("\n");

	for(i = 0; i < 35; i++)
		(void) printf("-");

	(void) printf(" TABLEAU ");

	for(i = 0; i < 35; i++)
		(void) printf("-");

	(void) printf("\n");

	for(i = 0; i < m+1; i++)
	{
		for(j = 0; j < lcol; j++)
			(void) printf("%6.3f ",e[i][j]);

		(void) printf("\n");
	}
}
