/*
 *	man - view the on-line manual
 *
 *	man [ section ] title
 *
 *	man -k keyword
 *
 *	This manual program is designed primarily for a speed increase
 * in viewing the manual. Rather than run nroff every time a man page is
 * requested, pre-formatted packed versions of the pages are kept in the
 * /usr/catman directory and pcat is used to read them. This agrees with
 * System V's catman program. This program stats both the unformatted and
 * formatted versions of the requested page and, if the formatted page is
 * up-to-date, it is printed, otherwise a new packed page is created with
 * nroff and pack. This automates the task of keeping up-to-date man pages
 * on-line. The environmental variable PAGER is checked for a preferred
 * pager, if none is specified /usr/bin/pg is used. To keep things simple,
 * nroff uses no fancy options.
 *	With the -k option, the /usr/man/xref database is searched for the
 * given keyword.
 *
 * AUTHOR
 *	Edward C. Bennett, edward@ukecc.UUCP
 *
 * Copyright 1985 by Edward C. Bennett
 *
 * Permission is given to alter this code as needed to adapt it to forign
 * systems provided that this header is included and that the original
 * author's name is preserved.
 */
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <string.h>

#define	DEFPAGER	"/usr/bin/pg"
#define	GREP		"/bin/grep"
#define	XREF		"/usr/man/xref"

/*
 * These are the directories under /usr/man that are to be searched for the
 * requested man page. The order in the array is the order of the search
 * so to minimize searching effort the more frequently used sections should
 * go at the top.
 */
char	*mandirs[] = {
	"u_man/man1",
	"p_man/man2",
	"p_man/man3",
	"u_man/man6",
	"p_man/man4",
	"p_man/man5",
	"a_man/man1",
	"a_man/man7",
	"a_man/man8",
};
#define	NUMDIRS		sizeof(mandirs)/sizeof(char *)

main(argc, argv)
int argc;
char **argv;
{
	char	*manpage, *pager, *section, *title, *Findpage(), *getenv();
	char	cmd[BUFSIZ], catbuf[BUFSIZ], catbufz[BUFSIZ];
	int	status;
	void	exit();
	struct	stat	manstat, catstat;

	if (!strcmp(argv[1], "-k")) {
		execl(GREP, GREP, argv[2], XREF, 0);
	}

	switch (argc) {
	case 2:
		title = argv[1];
		section = NULL;
		break;

	case 3:
		title = argv[2];
		section = argv[1];
		break;

	default:
		fprintf(stderr, "Usage: %s [ section ] title\n", argv[0]);
		exit(1);
	}

	if ((pager = getenv("PAGER")) == NULL)
		pager = DEFPAGER;

	chdir("/usr/man");

	if ((manpage = Findpage(title, section)) == NULL) {
		if (section)
			printf("No entry for %s in section %s of the manual\n", title, section);
		else
			printf("No manual entry for %s\n", title);
		exit(1);
	}

	strcpy(catbuf, "/usr/catman/");
	strcat(catbuf, manpage);
	strcpy(catbufz, catbuf);
	strcat(catbufz, ".z");

	stat(manpage, &manstat);
	if (stat(catbufz, &catstat) == -1)
		catstat.st_mtime = -1;

	if (catstat.st_mtime < manstat.st_mtime) {
		unlink(catbufz);
		printf("Reformatting page, wait...");
		fflush(stdout);
		if (fork() == 0) {
			sprintf(cmd, "nroff -man %s > %s", manpage, catbuf);
			execl("/bin/sh", "sh", "-c", cmd, 0);
		}
		wait(&status);
		printf("\nCompressing output, wait...");
		fflush(stdout);
		if (fork() == 0) {
			sprintf(cmd, "pack -f %s", catbuf);
			execl("/bin/sh", "sh", "-c", cmd, 0);
		}
		wait(&status);
	}

	/*
	 * Do this in case the user does something like 'man curses > file'
	 */
	if (isatty(1))
		sprintf(cmd, "pcat %s | %s", catbuf, pager);
	else
		sprintf(cmd, "pcat %s", catbuf);

	execl("/bin/sh", "sh", "-c", cmd, 0);
}

/*
 * Findpage - determine the pathname of the requested page
 *
 * path = Findpage(title, sect);
 *	path	is a pointer to the requested path
 *	title	is a pointer to the title of the requested page
 *	sect	is a pointer to the section to search, NULL if all sections
 *
 *	Findpage() returns a pointer to a buffer containing the path name
 * of the unformatted version of the request man page in the form
 * [apu]_man/man[1-8]/title.[1-8]. The user can specify a section to search.
 * This is needed for cases like write(1) and write(2). If 'section' is given
 * as NULL, all sections are searched. If no match for title.sect can be found,
 * NULL is returned.
 */
char *
Findpage(title, sect)
char *title, *sect;
{
	int	fd, i, len, tlen;
	long	lseek();
	static	char	manbuf[BUFSIZ];
	struct	direct	manent;

	tlen = strlen(title);
	for (i = 0; i < NUMDIRS; i++) {
		if (sect && *sect != *(mandirs[i] + 9))
			continue;

		if ((fd = open(mandirs[i], O_RDONLY)) == -1)
			continue;

		/*
		 * Skip "." and ".."
		 */
		lseek(fd, 2*sizeof(manent), 0);
		while (read(fd, &manent, sizeof(manent)) == sizeof(manent)) {
			if (manent.d_ino == 0)
				continue;

			len = (int)(strchr(manent.d_name, '.') - manent.d_name);
			if (!strncmp(manent.d_name, title, (len > tlen ? len : tlen))) {
				if (sect && strcmp(sect, manent.d_name+len+1))
					continue;

				sprintf(manbuf, "%s/%s", mandirs[i], manent.d_name);
				return(manbuf);
			}
		}
	}
	return(NULL);
}
