/*  File   : lit.c
    Author : Richard A. O'Keefe
    Updated: 5 December 1987
    Purpose: Replacement for echo(1).

    There are two versions of the echo(1) command:
	4.2 BSD
	    echo [-n] argument...
        System V
	    echo argument...
    Both are badly designed.  The 4.2 version has no way of printing
    something which starts with -n.  The System V one has no way of
    switching off the escape character interpretation which is its
    great new feature, and lacks an equivalent of -n.

    The solution is to have a new command which does things RIGHT.

    SYNOPSIS
	lit [-n] [-dlist] [-echar] [--] argument...

    OPTIONS
    -n	suppress the new-line which is normally printed at the end.

    -dl	normally, a single space is written between the arguments.
	With the option "-d" on its own, a tab is written instead.
	When there are characters after the "-d", these characters
	are used in turn, e.g. -d": , " would use ":" then " " then
	"," then " " then ":" again.  The default is thus equivalent
	to -d" ".

    -ex	normally, the arguments are copied literally.
	With the option "-e" on its own, C-style character escapes
	are interpreted.  In fact, rather more than most Cs.
	With the option "-ex", the character "x" is used instead of
	C's backslash;  this makes it easier to use in scripts.
	For example,
	    lit -e@ a@tb@tc
	writes out a, TAB, b, TAB, c, NEWLINE.

    --	indicates the end of the options.  This is only needed if the
	first argument begins with a "-".
*/

#include <stdio.h>

char *program_name = "lit";

void OutputError()
    {
	(void) fprintf(stderr,
	    "%s: I/O error near byte %ld of output\n",
	    program_name, ftell(stdout));
	exit(1);
    }


void WriteChar(c)
    int c;
    {
	if (putchar(c) < 0) OutputError();
    }


void Display(string, escape_character)
    char *string;
    int escape_character;
    {
	register FILE *output = stdout;
	register char *s;
	register int c;
	int i, n;

	for (s = string; c = *s++; ) {
	    if (c == escape_character) switch (c = *s++) {
		case 'n': case 'N':		/* newline */
		    c = 10; break;
		case 't': case 'T':		/* tab */
		    c =  9; break;
		case 'r': case 'R':		/* return */
		    c = 13; break;
		case 'v': case 'V':		/* vertical tab */
		    c = 11; break;
		case 'b': case 'B':		/* backspace */
		    c =  8; break;
		case 'f': case 'F':		/* formfeed */
		    c = 12; break;
		case 'e': case 'E':		/* escape */
		    c = 27; break;
		case 'd': case 'D':		/* delete */
		    c =127; break;
		case 's': case 'S':		/* space */
		    c = 32; break;
		case 'a': case 'A':		/* alarm */
		    c =  7; break;
		case '^':			/* control */
		    c = *s++;
		    c = !c ? '^' : c == '?' ? 127 : c&31;
		    break;
		case 'x': case 'X':		/* hexadecimal */
		    for (i = 2, n = 0; --i >= 0; s++) {
			c = *s;
			if (c >= '0' && c <= '9') {
			    n = (n<<4) - '0' + c;
			} else
			if (c >= 'a' && c <= 'f') {
			    n = (n<<4) - ('a'-10) + c;
			} else
			if (c >= 'A' && c <= 'F') {
			    n = (n<<4) - ('A'-10) + c;
			} else {
			    break;
			}
		    }
		    c = n&255;
		    break;
		case 'o': case 'O':		/* octal */
		    c = *s++;
		    if (c < '0' || c > '7') {
			c = s[-1];
			break;
		    }
		case '0': case '1': case '2': case '3':
		case '4': case '5': case '6': case '7':
		    n = c-'0';
		    for (i = 2; --i >= 0 && (c = *s) >= '0' && c <= '7'; s++) {
			n = (n<<3) - '0' + c;
		    }
		    c = n&255;
		    break;
		default:
		    break;
	    }
	    if (putc(c, output) < 0) OutputError();
	}
    }


main(argc, argv)
    int argc;
    char **argv;
    {
	int escape_character = -1;	/* no such character */
	int write_trailing_newline = 1;	/* do it by default */
	int options_done = 0;		/* options not completed yet */
	char *separators = " ";		/* list of separators */
	char *next_separator = "";
	int argno;			/* argument number */

	if (argc > 0) program_name = argv[0];

	for (argno = 1; argno < argc && !options_done; argno++) {
	    if (argv[argno][0] != '-') {
		break;
	    } else
	    switch (argv[argno][1]) {
		case 'n': case 'N':	/* suppress trailing \n */
		    write_trailing_newline = 0;
		    break;
		case 'd': case 'D':	/* delimiters */
		    separators = argv[argno][2] ? argv[argno]+2 : "\t";
		    break;
		case 'e': case 'E':	/* escape */
		    escape_character = argv[argno][2];
		    if (!escape_character) escape_character = '\\';
		    break;
		case '-':		/* end of options */
		    options_done = 1;
		    break;
		default:
		    (void) fprintf(stderr,
			"%s: unknown option: %s\n",
			program_name, argv[argno]);
		    (void) fprintf(stderr,
			"Usage: lit [-n] [-dlist] [-echar] [--] arg...\n");
		    exit(1);
	    }
	}
	while (argno < argc) {
	    Display(argv[argno], escape_character);
	    argno++;
	    if (argno != argc) {
		if (!*next_separator) next_separator = separators;
		WriteChar(*next_separator++);
	    }
	}
	if (write_trailing_newline) WriteChar('\n');
	if (fflush(stdout) < 0) OutputError();
    }


