/* lprint.c -- print out statement counts */

#include <stdio.h>
#include <string.h>
#include "bool.h"
#include "new.h"

#define	streq(s, t)	(strcmp((s), (t)) == 0)

extern FILE *efopen();

char *progname = "lprint";

void usage()
{
	fprintf(stderr, "usage: %s [-blaipf] [-c] [-r file] [file ...]\n", progname);
	fprintf(stderr, "	-b	print basic block counts (default)\n");
	fprintf(stderr, "	-l	print counts for all lines\n");
	fprintf(stderr, "	-a	print all basic blocks\n");
	fprintf(stderr, "	-i	print instruction counts\n");
	fprintf(stderr, "	-p	print function summaries\n");
	fprintf(stderr,	"	-f	print file summaries\n");
	fprintf(stderr, "	-c	compress prof.out file\n");
	fprintf(stderr, "	-r	select alternate prof.out file\n");
	exit(1);
}

int flags = 0;

#define	BLOCKS	0x01
#define	LINES	0x02
#define	ALL	0x04
#define	INST	0x08
#define	FILESUM	0x10
#define	FUNCSUM	0x20

#define	DEFAULT	BLOCKS

typedef struct Mapfile Mapfile;
typedef struct Mapblock Mapblock;
typedef struct Sourcefile Sourcefile;
typedef struct Block Block;
typedef struct Func Func;

struct Mapfile {
	char	*name;
	int	n, nfunc;
	struct Mapblock {
		long	count;
		int	inst;
		Sourcefile *source;
	} *block;
	Mapfile	*next;
};

Mapfile *maplist = NULL;

struct Block {
	int	line, last, inst;
	long	count;
};

struct Sourcefile {
	char	*name;
	int	nblock, maxblock;
	Block	*block;
	Sourcefile *next;
};
Sourcefile *sourcelist = NULL;

struct Func {
	char	*name;
	Mapfile	*map;
	int	block;
};

Func *functab = NULL;
int nfunc = 0, maxfunc = 0;

void gather(fp)
	FILE *fp;
{
	int i, n;
	char name[BUFSIZ];
	while (fscanf(fp, "%s %d\n", name, &n) != EOF) {
		Mapfile *p;
		for (p = maplist; p != NULL; p = p->next)
			if (streq(name, p->name)) {
				if (n != p->n)
					panic("prof.out: bad counts for %s: %d and %d\n",
					      name, p->n, n);
				goto found;
			}
		p = new(Mapfile, 1);
		p->next = maplist;
		maplist = p;
		p->name = strdup(name);
		p->n = n;
		p->block = new(struct Mapblock, n);
		for (i = 0; i < n; i++)
			p->block[i].count = 0;
	found:
		for (i = 0; i < n; i++) {
			long count;
			if (fscanf(fp, "%ld", &count) == EOF)
				panic("prof.out: early EOF\n");
			p->block[i].count += count;
		}
	}
}

void writeprof(fp)
	FILE *fp;
{
	Mapfile *p;
	for (p = maplist; p != NULL; p = p->next) {
		int i;
		fprintf(fp, "%s %d\n", p->name, p->n);
		for (i = 0; i < p->n; i++)
			fprintf(fp, "%ld\n", p->block[i].count);
	}
}

Sourcefile *install(name, line, last, inst, count, n)
	char	*name;
	int	line, last, inst, n;
	long	count;
{
	int i;
	static Sourcefile *p = NULL;
	if (p != NULL && streq(name, p->name))
		goto found;
	for (p = sourcelist; p != NULL; p = p->next)
		if (streq(name, p->name))
			goto found;

	p = new(Sourcefile, 1);
	p->name	    = strdup(name);
	p->block    = new(Block, n);
	p->maxblock = n;
	p->nblock   = 0;
	p->next     = sourcelist;
	sourcelist  = p;

found:
	if (p->nblock >= p->maxblock) {
		p->maxblock *= 4;
		p->block = renew(Block, p->block, p->maxblock);
	}

	/* insertion sort, but (in practice) very rarely executed */
	for (i = p->nblock++; i > 0 && line < p->block[i - 1].line; i--)
		p->block[i] = p->block[i - 1];

	p->block[i].line  = line;
	p->block[i].last  = last;
	p->block[i].inst  = inst;
	p->block[i].count = count;
	return p;
}

void correlate(map)
	Mapfile *map;
{
	int i;
	FILE *fp = efopen(map->name, "r");
	for (i = 0; i < map->n; i++) {
		char filename[BUFSIZ];
		int line, lastline;
		if (fscanf(fp, "%s %d %d %d\n", filename,
			   &line, &lastline, &map->block[i].inst) != 4)
			panic("%s, line %d: bad map entry\n", map->name, i + 1);
		map->block[i].source = install(filename, line, lastline,
				map->block[i].inst, map->block[i].count, map->n);
	}
	if (fscanf(fp, "%d functions\n", &map->nfunc) != 1)
		panic("%s, line %d: bad function line\n", map->name, map->n);
	while (nfunc + map->nfunc >= maxfunc) {
		if (maxfunc == 0)
			maxfunc = 200;
		else
			maxfunc *= 8;
		functab = renew(Func, functab, maxfunc);
	}
	for (i = 0; i < map->nfunc; i++, nfunc++) {
		char name[BUFSIZ];
		int block;
		if (fscanf(fp, "%s %d\n", name, &block) != 2)
			panic("%s, line %d: bad function entry\n",
			      map->name, map->n + 1 + 1);
		functab[nfunc].name  = strdup(name);
		functab[nfunc].map   = map;
		functab[nfunc].block = block;
	}
	fclose(fp);
}

void printline(s, count, inst)
	char	*s;
	long	count;
	int	inst;
{
	if (flags & BLOCKS)
		printf("%10ld  ", count);
	if (flags & INST) {
		if (count == 0) {
			char buf[20];
			sprintf(buf, "(%d)", inst);
			printf("%11s ", buf);
		} else
			printf("%10ld  ", count * inst);
	}
	printf("%s", s);
}

void printsource(p)
	Sourcefile *p;
{
	int	line = 0, last = 0, i = 0, inst = 0;
	long	count = 0;
	char	buf[BUFSIZ];
	FILE	*fp = efopen(p->name, "r");

	static int fill = 0;
	static bool firsttime = TRUE;
	if (firsttime) {
		firsttime = FALSE;
		if (flags & BLOCKS)
			fill += 12;
		if (flags & INST)
			fill += 12;
	} else
		printf("\f");

	while (fgets(buf, sizeof buf, fp) != NULL) {
		for (; i < p->nblock && line >= p->block[i].line; i++) {
			if (flags & ALL)
				printline("\n", p->block[i].count,
						p->block[i].inst);
			if (last <= p->block[i].last)
				last = p->block[i].last;
		}
		line++;
		if (i < p->nblock && line == p->block[i].line && last < line) {
			count = p->block[i].count;
			inst  = p->block[i].inst;
			last  = p->block[i].last;
			printline(buf, count, inst);
			while (++i < p->nblock && line == p->block[i].line)
				if (flags & ALL)
					printline("\n", p->block[i].count,
							p->block[i].inst);
		} else if (flags & LINES)
			printline(buf, count, inst);
		else
			printf("%*s%s", fill, "", buf);
	}
	fclose(fp);
}

typedef struct Sum {
	long	calls, i, ie, ine, bb, bbe, bbne;
} Sum;

void printsumline(name, sum, isfunc)
	char	*name;
	Sum	sum;
	bool	isfunc;
{
	printf("%9ldbbe %4ldbb %4ldbbne ", sum.bbe, sum.bb, sum.bbne);
	printf("%9ldie %4ldi %4ldine ", sum.ie, sum.i, sum.ine);
	if (isfunc)
		printf("%7ldcalls ", sum.calls);
	else
		printf("%13s", "");
	printf(" %s\n", name);
}

static Sum zerosum = { 0, 0, 0, 0, 0, 0, 0 };

void printfunctionsummary(p)
	Sourcefile *p;
{
	Func *func;

	for (func = &functab[0]; func < &functab[nfunc]; func++)
		if (func->map->block[func->block].source == p) {
			Sum sum;
			Mapblock *block = func->map->block;
			int i, last;
			sum = zerosum;
			if (func < &functab[nfunc] && func->map == (func + 1)->map)
				last = (func + 1)->block;
			else
				last = func->map->n;
			sum.calls += block[func->block].count;
			for (i = func->block; i < last; i++) {
				Mapblock *b = &block[i];
				sum.bb	+= 1;
				sum.bbe	+= b->count;
				sum.i	+= b->inst;
				sum.ie	+= b->inst * b->count;
				if (b->count == 0) {
					sum.bbne += 1;
					sum.ine  += b->inst;
				}
			}
			printsumline(func->name, sum, TRUE);
		}
}

void printfilesummary(p)
	Sourcefile *p;
{
	Sum sum;
	Mapfile *map;
	Mapblock *block;

	sum = zerosum;
	for (map = maplist; map != NULL; map = map->next)
		for (block = &map->block[0]; block < &map->block[map->n]; block++)
			if (block->source == p) {
				sum.bb	+= 1;
				sum.bbe	+= block->count;
				sum.i	+= block->inst;
				sum.ie	+= block->inst * block->count;
				if (block->count == 0) {
					sum.bbne += 1;
					sum.ine  += block->inst;
				}
			}
	printsumline(p->name, sum, FALSE);
}

void analyze(name)
	char *name;
{
	Sourcefile *p;
	for (p = sourcelist; p == NULL || !streq(p->name, name); p = p->next)
		if (p == NULL) {
			fprintf(stderr, "%s: not in prof.out\n", name);
			return;
		}

	if (flags & (INST|BLOCKS)) {
		printsource(p);
		if (flags & (FUNCSUM|FILESUM))
			printf("\n\n\n\n");
	}
	if (flags & FUNCSUM) {
		printfunctionsummary(p);
		if (flags & FILESUM)
			printf("\n");
	}
	if (flags & FILESUM) {
		printfilesummary(p);
		if ((flags & FUNCSUM) && (flags & (INST|BLOCKS)) == 0)
			printf("\n\n");
	}
}

int main(argc, argv)
	int argc;
	char *argv[];
{
	int c;
	bool compact = FALSE;
	char *prof = "prof.out";
	Mapfile *map;

	extern int optind;
	extern char *optarg;

	progname = argv[0];
	while ((c = getopt(argc, argv, "blaipfcr:?")) != EOF)
		switch(c) {
		case 'b':	flags |= BLOCKS;	break;
		case 'l':	flags |= LINES;		break;
		case 'a':	flags |= ALL;		break;
		case 'i':	flags |= INST;		break;
		case 'p':	flags |= FUNCSUM;	break;
		case 'f':	flags |= FILESUM;	break;
		case 'c':	compact = TRUE;		break;
		case 'r':	prof = optarg;		break;
		case '?':	usage();
		}

	if (streq(prof, "-"))
		gather(stdin);
	else {
		FILE *fp = efopen(prof, "r");
		gather(fp);
		fclose(fp);
	}
	if (compact) {
		if (streq(prof, "-"))
			writeprof(stdout);
		else {
			FILE *fp = efopen(prof, "w");
			writeprof(fp);
			fclose(fp);
		}
		if (optind == argc && flags == 0)
			return 0;
	}

	if (flags == 0)
		flags = DEFAULT;
	if ((flags & (ALL|LINES)) && (flags & (INST|BLOCKS)) == 0)
		flags |= BLOCKS;

	for (map = maplist; map != NULL; map = map->next)
		correlate(map);
	if (optind == argc) {
		Sourcefile *p;
		for (p = sourcelist; p != NULL; p = p->next)
			analyze(p->name);
	} else
		for (; optind < argc; optind++)
			analyze(argv[optind]);
	return 0;
}
