/* bb.c -- insert basic block counting code */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "new.h"
#include "bool.h"
#include "bb.h"

extern FILE *efopen();
extern char *memcpy();

char *progname = "bb";

static void usage()
{
	fprintf(stderr, "usage: %s file.s ...\n", progname);
	exit(1);
}


/* globals shared with $TARGET.l */

FILE *out = NULL;		/* generated .s file */
FILE *map = NULL;		/* generated .sL file */

int lineno = 0;			/* original source line number */
char filename[WORDSIZE];	/* original source filename */

char *yystring;			/* input to lex */
char *tail;			/* tail of line (everything after opcode) */
char line[BUFSIZ];		/* input line */
char label[WORDSIZE];		/* most recent label */

bool text = TRUE;		/* in text segment? */
bool newblock = TRUE;		/* started new basic block? */
int block = 0;			/* current basic block number */
int instructions = NOPRINT;	/* counter of instructions */
int reached = 0;		/* last line reached in basic block */


/* associate functions with basic blocks*/

#define	MAXFUNC	1000
int nfunc;
struct {
	char *name;
	int bb;
} func[MAXFUNC];

function(s, bb)
	char *s;
	int bb;
{
	if (nfunc >= MAXFUNC)
		panic("too many functions (>%d), on %s\n", MAXFUNC, s);
	func[nfunc].name = strdup(s);
	func[nfunc].bb = bb;
	nfunc++;
}

functionmap()
{
	int i;
	fprintf(map, "%d functions\n", nfunc);
	for (i = 0; i < nfunc; i++) {
		fprintf(map, "%s %d\n", func[i].name, func[i].bb);
		free(func[i].name);
		func[i].name = NULL;
	}
	nfunc = 0;
}


/* parse file, pass to yylex from $TARGET.l */
void bb(infile)
	char *infile;
{
	char outfile[BUFSIZ], mapfile[BUFSIZ];
	static char pid[10] = "";
	FILE *in;

	in = efopen(infile, "r");

	strcpy(mapfile, infile);
	strcat(mapfile, "L");
	map = efopen(mapfile, "w");

	if (pid[0] == '\0')
		sprintf(pid, ".%d", getpid());
	strcpy(outfile, infile);
	strcat(outfile, pid);
	out = efopen(outfile, "w");

	lineno = 0;
	filename[0] = '\0';

	while (fgets(line, sizeof line, in) != NULL) {
		char *s = line, opcode[WORDSIZE];
		int i;
startofline:
		for (; isspace(*s); s++)
			;
		if (*s == '\0') {
			fputs(line, out);
			continue;
		}
		for (i = 0; isgraph(s[i]); i++)
			if (s[i] == ':') {
				if (text) {
					memcpy(label, s, i);
					label[i] = '\0';
					if (labelstartsblock(label))
						newblock = TRUE;
				}
				s += i + 1;
				goto startofline;
			}
		tail = s + i;
		memcpy(opcode, s, i);
		opcode[i] = '\n';
		opcode[i + 1] = '\0';
		yystring = opcode;
		yylex();
	}

	epilogue(mapfile);
	functionmap();

	fclose(in);
	fclose(out);
	fclose(map);
	map = out = NULL;

	if (unlink(infile) == -1)
		panic("couldn't unlink %s -- output in %s\n", infile, outfile);
	if (rename(outfile, infile) == -1)
		panic("couldn't rename %s to %s\n", outfile, infile);
}

int main(argc, argv)
	int argc;
	char *argv[];
{
	progname = argv[0];
	if (argc == 1)
		usage();
	for (--argc, ++argv; argc != 0; --argc, ++argv)
		bb(*argv);
	return 0;
}

/* functions for use in $TARGET.l -- common to most machines */

void passline()
{
	fputs(line, out);
}

void inst()
{
	if (text) {
		if (newblock)
			increment();
		reached = lineno;
		instructions++;
	}
	passline();
}

void safe()
{
	if (text) {
		if (newblock)
			safeincrement();
		reached = lineno;
		instructions++;
	}
	passline();
}

void branch()
{
	if (text) {
		if (newblock)
			safeincrement();
		reached = lineno;
		instructions++;
		newblock = TRUE;
	}
	passline();
}

#define	STAB_LINE	0104	/* N_SLINE from <stab.h> */
#define	STAB_FILE	0144	/* N_SO from <stab.h> */

void stabd()
{
	char *s;
	passline();
	if (atoi(tail) != STAB_LINE)
		return;
	if ((s = strchr(line, ',')) == NULL || (s = strchr(s + 1, ',')) == NULL)
		panic("bad directive: .stabn%s", tail);
	lineno = atoi(s + 1);
}

void stabs()
{
	char *s, *t;
	int len;
	passline();
	if ((s = strchr(tail, ',')) == NULL)
		panic("bad directive: .stabs%s", tail);
	if (atoi(s + 1) != STAB_FILE)
		return;
	if ((s = strchr(tail, '"')) == NULL || (t = strchr(s + 1, '"')) == NULL)
		panic("bad directive: .stabs%s", tail);
	len = t - s - 1;
	memcpy(filename, s + 1, len);
	filename[len] = '\0';
}
