static char *RCSid = "$Header: /ecn1/src/ecn/RCS/help.c,v 1.9 87/09/11 09:39:15 davy Exp $";

/*
 * apropos, help, man, whatis
 *
 * This program is really four programs in one, all the above names should
 * be linked together.  Apropos searches for given keywords in the system
 * manuals, whatis prints the "NAME" line of the documentation.  Help and
 * man, if called with non-switch (-X) arguments, behave identically.  Help,
 * if called with no arguments (except switches), behaves interactively, 
 * displaying the documentation root, and permitting the user to scan through
 * it.
 *
 * Help and man begin by searching DOCROOT (/usr/man).  Man searches through
 * the eight chapters only, while help also searches the directory itself.
 * If nothing is found, the environment variable HELPPATH is checked, and any
 * directories set in it are then checked.
 *
 * Flags are:
 *		-!	No shell escapes
 *		-1, -P	Pdp 11 document (currently ignored)
 *		-N	No searching un-nroffed doc directories
 *		-c	Continuous output
 *		-f	Run whatis
 *		-h	Prompt for hardcopy output
 *		-k	Run apropos
 *		-p	Paged output (default)
 *
 * To compile:
 *		cc -O -s -o help help.c -ltermcap
 *		ln help man
 *		ln help apropos
 *		ln help whatis
 *
 * David A. Curry, Purdue-ECN, 7/20/83
 *
 * Converted to 4.2 BSD, 9/25/83, David A. Curry
 *
 *	NOTE: For 4.2 BSD, dbuf is a character array.  I tried using
 *	      an array of pointers to type struct direct, but for some
 *	      reason this blew up on large directories, although it
 *	      worked fine on small ones.  I screwed with it for 3 hours,
 *	      and finally gave up.  I think it might be a qsort bug, 
 *	      although I may be wrong.
 *					--DAC
 *
 * $Log:	help.c,v $
 * Revision 1.9  87/09/11  09:39:15  davy
 * Fixed to not rely on existence of _sobuf.
 * 
 * Revision 1.8  85/10/02  11:08:28  klimbal
 * Added case "q" to Quit so you dont have to interrupt out
 * any longer.
 * 
 * --Kenny Wickstrom
 * 
 * Revision 1.7  85/09/20  09:21:34  davy
 * Changed order of chapter search to match /usr/ucb/man...
 * 
 * Revision 1.6  85/08/29  20:49:20  klimbal
 * Added "(? for help)" to prompt line so that
 * the "typical" user can use the program.
 * 
 * --Ken Wickstrom
 *   8/29/85
 * 
 * Revision 1.5  85/08/21  10:48:46  davy
 * Miscellaneous 4.3 changes... understands window size, uses /usr/man/whatis,
 * and searches chapters in right order.
 * --Dave
 * 
 * Revision 1.4  85/05/13  14:18:49  root
 * Removed the -h (hardcopy) option.
 * 
 * Revision 1.3  83/10/26  09:44:03  root
 * 1. Fixed bug which caused help to exit when it encountered
 *    an empty directory.
 * 2. Modified init() to get the current working directory,
 *    and modified main() to search this directory path instead
 *    of using ".".
 * 3. Modified dis_dir() to use absolute pathnames instead of
 *    relative pathnames. This involved some fairly simple
 *    changes in dis_dir() to call push() and pop(), and the
 *    addition of the routines push() and pop().
 * 
 * The above changes from Dave Curry.....
 * 
 * Revision 1.2  83/10/24  10:00:47  tony
 * Added chapters "l", "n", and "o" to search list.
 * 
 * Revision 1.1  83/10/24  09:40:35  tony
 * Initial revision
 * 
 */
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <sgtty.h>
#include <ctype.h>
#include <stdio.h>

/*
 * Various constants.  MAN and HELP are for which mode the program is in,
 * CAT and MORE are the continuous and paged file printers respectively,
 * NROFF is the text formatter to use, PRINT is the line-printer command,
 * DOCROOT is the default directory to look for documentation in, CHAPTERS
 * refer to "cat1", "man3", etc., MAXPATHS is the number of directories 
 * allowed in HELPPATH, and MAXFILES is the number of files in a directory.
 */
#define MAN		0
#define CAT		"/bin/cat"
#define HELP		1
#define MORE		"/usr/ucb/more -s"
#define NROFF		"/usr/bin/nroff"
#define PRINT		"/usr/ucb/lpr"
#define DOCROOT		"/usr/man"
#define COLWIDTH	19
#define CHAPTERS	"1nl6823457po"
#define MAXPATHS	16
#define MAXFILES	1024
#define USRCUSTOM	"/usr/custom/doc"

/*
 * Constants returned by getpath().  REPRINT implies redraw the screen,
 * RESTART is begin the listing from 1 again, NXTPAGE is display the 
 * next page of choices, RPTROOT is go back to the top-level directory,
 * and NXTROOT is go to the next root directory to search.  
 */
#define REPRINT		(char *) 0
#define RESTART		(char *) 1
#define NXTPAGE		(char *) 2
#define RPTROOT		(char *) 3
#define NXTROOT		(char *) 4
#define LSTROOT		(char *) 5

/*
 * Terminal screen control macros.
 */
#define clear()		tputs(term.cl, 1, putch)
#define bstandout()	tputs(term.so, 1, putch)
#define estandout()	tputs(term.se, 1, putch)
#define cursor(x, y)	tputs(tgoto(term.cm, x, y), 1, putch)

DIR *opendir();
struct direct *readdir();
char dbuf[MAXFILES][MAXNAMLEN+1];

struct sgttyb termbuf;

struct term {
	char *cm;	/* Cursor motion sequence for terminal		*/
	char *cl;	/* Clear screen sequence for terminal		*/
	char *so;	/* Start standout mode sequence for terminal	*/
	char *se;	/* End standout mode sequence for terminal	*/
	int  cols;	/* Number of columns on terminal screen		*/
	int  lines;	/* Number of lines on terminal screen		*/
} term;

int  hmode;				/* MAN or HELP - mode of prog.	*/
char *arg0;				/* program name, no pathname	*/
int  npaths;				/* number of paths in HELPPATH	*/

short cflg = 0;				/* 0 if -p, 1 if -c		*/
short hflg = 0;				/* 1 if -h			*/
short nflg = 1;				/* 0 if -N			*/
short noshell = 0;			/* 1 if -!			*/
short pdp11doc = 0;			/* 1 if -1 or -P		*/

char paths[MAXPATHS][256];		/* Paths from HELPPATH variable	*/

int putch();
int stackptr;				/* for dis_dir's "stack"	*/
extern char *PC;			/* Terminal's padding string	*/
extern int  ospeed;			/* Terminal's output baud rate	*/
char sobuf[BUFSIZ];
	
main(argc, argv)
int argc;
char **argv;
{
	int i, j;
	int didone;
	char cwd[128];
	char file[256];
	char *display();
	char *f, *chaps;
	char *rindex(), *search();
	
	/*
	 * Buffer stderr, which is where we write.
	 */
	setbuf(stderr, sobuf);
	
	/*
	 * If stdout isn't a tty, use continuous output.
	 */
	if (!isatty(1))
		cflg = 1;

	/*
	 * Set arg0 to name of program minus the pathname.
	 */
	if ((arg0 = rindex(argv[0], '/')) == NULL)
		arg0 = argv[0];
	else
		arg0++;
		
	/*
	 * Decide how we've been called.
	 */
	switch (*arg0) {
	case 'a':		/* apropos */
		hmode = MAN;
		
		if (argc < 2)
			fprintf(stderr, "Usage: %s commands ...\n", arg0);
		else 
			apropos(argc+1, argv+1);
		exit(0);
		break;
	case 'h':		/* help */
		hmode = HELP;
		break;
	case 'm':		/* man */
		hmode = MAN;
		break;
	case 'w':		/* whatis */
		hmode = MAN;
		
		if (argc < 2)
			fprintf(stderr, "Usage: %s commands ....\n", arg0);
		else 
			whatis(argc+1, argv+1);
		exit(0);
		break;
	}
	
	/*
	 * Initialize - get HELPPATH and term info.  On
	 * return, cwd will contain the absolute path of
	 * the current working directory.
	 */
	init(cwd);
	
	/*
	 * If no arguments were given, then print a usage message
	 * for man, go interactive for help.
	 */
	if (argc < 2) {
		if (hmode == MAN) {
			fprintf(stderr, "Usage: %s ", arg0);
			fprintf(stderr, "[-1NPcfhkp] [chapters] ");
			fprintf(stderr, "commands\n");
			exit(0);
		}
		else {
			/*
			 * Display each possible document directory.
			 */
noargs:			for (i=(-1); i < npaths; i++) {
				/*
	 			 * If we get LSTROOT, reset i to
				 * the proper value.
				 */
runpaths:			if (i < 0) {
					if (display(DOCROOT) == LSTROOT) {
						i = (npaths - 2);
						continue;
					}
				}
				else {
					if (display(paths[i]) == LSTROOT) {
						i -= 2;
						continue;
					}
				}
				
				/*
				 * If i is at the end, reset it to the
				 * beginning.
				 */
				if (i == (npaths - 1))
					i = (-2);
			}
		}
		fputc('\n', stderr);
		exit(0);
	}
	
	/*
	 * Process arguments.
	 */
	chaps = NULL;
	didone = 0;
	for (i=1; i < argc; i++) {
		/*
		 * Chapters given.
		 */
		if ((argv[i][0] >= '0') && (argv[i][0] <= '9')) {
			chaps = argv[i];
			continue;
		}
		/*
		 * Flag.
		 */
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
			case 'c':
				cflg = 1;
				continue;
			case 'f':
				i++;
				if (i >= argc) {
					fprintf(stderr, "Usage: %s ", arg0);
					fprintf(stderr, "[-!1NPcfhkp] ");
					fprintf(stderr, "[chapters] ");
					if (hmode == MAN)
						fprintf(stderr, "commands\n");
					else
						fprintf(stderr,"[commands]\n");
					exit(0);
				}
				argc -= i;
				argv += i;
				whatis(argc, argv);
				exit(0);
				break;
#ifdef HARDCOPY
			case 'h':
				hflg = 1;
				continue;
#endif
			case 'k':
				i++;
				if (i >= argc) {
					fprintf(stderr, "Usage: %s ", arg0);
					fprintf(stderr, "[-!1NPcfhkp] ");
					fprintf(stderr, "[chapters] ");
					if (hmode == MAN)
						fprintf(stderr, "commands\n");
					else
						fprintf(stderr,"[commands]\n");
					exit(0);
				}
				argc -= i;
				argv += i;
				apropos(argc, argv);
				exit(0);
				break;
			case 'p':
				cflg = 0;
				continue;
			case 'N':
				nflg = 0;
				continue;
			case '1':
			case 'P':
				pdp11doc = 1;
				continue;
			case '!':
				noshell = 1;
				continue;
			default:
				fprintf(stderr, "%s: ", arg0);
				fprintf(stderr, "bad flag %s.\n", argv[i]);
				fflush(stderr);
				continue;
			}
		}
		
		/*
		 * If chapters were given, we search only DOCROOT in
		 * the specified chapters.  Otherwise, things vary.
		 */
		if (chaps) {
			f = search(argv[i], DOCROOT, chaps);
		}
		else {
			/*
			 * If man, search the chapters in DOCROOT, then
			 * search HELPPATH.
			 */
			if (hmode == MAN) {
				f = search(argv[i], DOCROOT, CHAPTERS);
				
				if (f == NULL) {
					for (j=0; (j < npaths) && !f; j++)
						f = search(argv[i],paths[j],0);
				}
			}
			else {
				/*
				 * If help, search DOCROOT, search all the
				 * chapters, search HELPPATH, search ".".  As
				 * a last resort, we try accessing the path
				 * directly.
				 */
				f = search(argv[i], DOCROOT, 0);
				
				if (f == NULL)
					f = search(argv[i], DOCROOT, CHAPTERS);
					
				if (f == NULL) {
					for (j=0; (j < npaths) && !f; j++)
						f = search(argv[i],paths[j],0);
				}
				
				if (f == NULL)
					f = search(argv[i], cwd, 0);
					
				if (f == NULL) {
					if (argv[i][0] == '/')
						strcpy(file, argv[i]);
					else
						sprintf(file, "%s/%s", cwd, argv[i]);

					if (access(file, 0) == 0)
						f = file;
				}
			}
		}
		
		/*
		 * If f is non-null, display the file (directory).
		 */
		if (f) {
			didone++;
			if (display(f) == LSTROOT) {
				i = (-1);
				goto runpaths;
			}
		}
		else {
			didone++;
			fprintf(stderr, "Can't find documentation on %s.\n", argv[i]);
		}
	}
	
	/*
	 * If we didn't look for any, and the mode is
	 * help, go interactive.  This is so they can
	 * give us flags.
	 */
	if ((didone == 0) && (hmode == HELP))
		goto noargs;
}
/*
 * init - gets HELPPATH and terminal info.  Puts current directory into cwd.
 */
init(cwd)
char *cwd;
{
	int i;
	char *s, *t;
	char *tgetstr(), *getenv();
	static char tbuf[256];
#ifdef TIOCGWINSZ
	struct winsize win;
#endif
	
	/*
	 * Get current working directory.
	 */
	getwd(cwd);

	/*
	 * We only need terminal info for help.
	 */
	if (hmode == HELP) {
		ioctl(2, TIOCGETP, &termbuf);
		ospeed = termbuf.sg_ospeed;
		
		if ((t = getenv("TERM")) == NULL) {
			PC = NULL;
			term.cm = NULL;
			term.so = NULL;
			term.se = NULL;
			term.cl = "\n\n\n";
			term.cols = 80;
			term.lines = 24;
		}
		else {
			if (tgetent((char *)malloc(1024), t) < 0) {
				PC = NULL;
				term.cm = NULL;
				term.so = NULL;
				term.se = NULL;
				term.cl = "\n\n\n";
				term.cols = 80;
				term.lines = 24;
			}
			else {
				s = tbuf;
				term.cm = tgetstr("cm", &s);
				term.so = tgetstr("so", &s);
				term.se = tgetstr("se", &s);
				PC = tgetstr("pc", &s);
			
				if ((term.cl = tgetstr("cl", &s)) == NULL)
					term.cl = "\n\n\n";
				if ((term.cols = tgetnum("co")) <= 0)
					term.cols = 80;
				if ((term.lines = tgetnum("li")) <= 0)
					term.lines = 24;
			}
		}
	}

#ifdef TIOCGWINSZ
	if (ioctl(0, TIOCGWINSZ, &win) == 0) {
		if (win.ws_col > 0)
			term.cols = win.ws_col;
		if (win.ws_row > 0)
			term.lines = win.ws_row;
	}
#endif

	
	/*
	 * Store the paths.  The variable should be like :dir1:dir2:...:
	 */
	npaths = 0;
	if ((s = getenv("HELPPATH")) != NULL) {
		if (*s == ':')
			s++;
		
loop:		t = paths[npaths];
		
		while ((*s != ':') && (*s != NULL))
			*t++ = *s++;
		*t = NULL;

		switch (*s) {
		case ':':
			s++;
			npaths++;
			goto loop;
			break;
		case NULL:
			*t = NULL;
			if (paths[npaths][0] != NULL)
				npaths++;
#ifdef USRCUSTOM
			strcpy(paths[npaths++], USRCUSTOM);
#endif
			return;
		}
	}
#ifdef USRCUSTOM
	else {
		strcpy(paths[0], USRCUSTOM);
		npaths=1;
	}
#endif
}
/*
 * search - looks for what in where.  If how is given, it implies chapters.
 */
char *search(what, where, how)
char *what, *where, *how;
{
	int i, j;
	DIR *dirp;
	struct stat sbuf;
	struct direct *dp;
	char *p, tmp[MAXNAMLEN+1];
	static char fpath[256];
	
#ifdef DEBUG
	fprintf(stderr, "SEARCH(%s, %s, %s)\n", what, where, how);
	fflush(stderr);
#endif
	
	if (stat(where, &sbuf) < 0) {
		error("%s does not exist.", where);
		return(NULL);
	}
		
	if ((sbuf.st_mode & S_IFMT) != S_IFDIR) {
		error("%s is not a directory.", where);
		return(NULL);
	}
	
	/*
	 * If they told us how, search where/cat[0-n] and where/man[0-n]
	 * Each search is done as follows: open directory and read it in,
	 * skip ., .., and any entries which begin with a different letter
	 * than what does.  Try to match what against the root of the name
	 * (strip the .1, .1g, etc. off).
	 */
	if (how) {
		p = how;
		
		while (*p) {
			sprintf(fpath, "%s/cat%c", where, *p);
		
#ifdef DEBUG
			fprintf(stderr, "Opening %s\n", fpath);
			fflush(stdout);
#endif

			if ((dirp = opendir(fpath)) == NULL) {
				p++;
				continue;
			}
			
			while ((dp = readdir(dirp)) != NULL) {
				if (dp->d_ino == 0)
					continue;
				if (!strcmp(dp->d_name, ".") ||
				    !strcmp(dp->d_name, ".."))
				    	continue;
				if (dp->d_name[0] != what[0])
					continue;
				
				strcpy(tmp, dp->d_name);
				
				if (match(what, tmp) == 0) {
					strcat(fpath, "/");
					strcat(fpath, dp->d_name);
					closedir(dirp);
					return(fpath);
				}
			}
			
			p++;
			closedir(dirp);
		}
		
		if (nflg) {
			p = how;
		
			while (*p) {
				sprintf(fpath, "%s/man%c", where, *p);
		
#ifdef DEBUG
				fprintf(stderr, "Opening %s\n", fpath);
				fflush(stdout);
#endif

				if ((dirp = opendir(fpath)) == NULL) {
					p++;
					continue;
				}
			
				while ((dp = readdir(dirp)) != NULL) {
					if (dp->d_ino == 0)
						continue;
				
					strcpy(tmp, dp->d_name);
				
					if (match(what, tmp) == 0) {
						strcat(fpath, "/");
						strcat(fpath, dp->d_name);
						closedir(dirp);
						return(fpath);
					}
				}
				
				p++;
				closedir(dirp);
			}
		}
		return(NULL);
	}
	
	/*
	 * If they didn't tell us how, we just look in where.  The
	 * search is the same.
	 */
	if ((dirp = opendir(where)) == NULL) {
		error("Cannot open %s.", where);
		return(NULL);
	}
		
	while ((dp = readdir(dirp)) != NULL) {
		if (dp->d_ino == 0)
			continue;
			
		if (!strcmp(dp->d_name, ".") || !strcmp(dp->d_name, ".."))
		    	continue;
			
		if (dp->d_name[0] != what[0])
			continue;
				
		strcpy(tmp, dp->d_name);
				
		if (match(what, tmp) == 0) {
			sprintf(fpath, "%s/%s", where, dp->d_name);
			closedir(dirp);
			return(fpath);
		}
	}

	closedir(dirp);
	
	/*
	 * As a last resort, try just accessing the thing directly.
	 * This is for things like "help graphics/crc".
	 */
	sprintf(fpath, "%s/%s", where, what);
	
	if (stat(fpath, &sbuf) < 0)
		return(NULL);
	else
		return(fpath);
}
/*
 * match - takes b and strips off the suffix, then does strcmp.
 */
match(a, b)
char *a, *b;
{
	char *c, *rindex();
	
	/*
	 * Before stripping the suffix, see if they match.
	 */
	if (!strcmp(a, b))
		return(0);
		
	/*
	 * Strip the suffix.
	 */
	if ((c = rindex(b, '.')) != NULL)
		*c = NULL;
		
#ifdef DEBUG
	fprintf(stderr, "MATCH(%s, %s)\n", a, b);
	fflush(stderr);
#endif
	
	/*
	 * Compare them.
	 */
	return(strcmp(a, b));
}
/*
 * display - descides how to print file (directory) on screen.
 */
char *display(file)
char *file;
{
	char *dis_dir();
	struct stat sbuf;
	
#ifdef DEBUG
	fprintf(stderr, "DISPLAY(%s)\n", file);
	fflush(stderr);
#endif
	
	if (stat(file, &sbuf) < 0) {
		error("%s does not exist.", file);
		return(NULL);
	}
	
	/*
	 * If it's a directory, only display it if we're in help.  If 
	 * it's a file, display it.
	 */
	if ((sbuf.st_mode & S_IFMT) == S_IFDIR) {
		if (hmode == HELP)
			return(dis_dir(file));
	}
	else {
		dis_file(file);
		return(NULL);
	}
}
/*
 * dis_file - prints a file on the screen
 */
dis_file(file)
char *file;
{
	int w;
	char c;
	char syscmd[128];
	int fd, pid, status;
	char *rsp, *getrsp();
	char *tmp, *mktemp();
	
	if ((fd = open(file, O_RDONLY)) < 0) {
		error("Can't open %s.", file);
		return(NULL);
	}
	
	if (read(fd, &c, 1) < 0) {
		error("Can't read %s.", file);
		close(fd);
		return(NULL);
	}
	
	tmp = NULL;
	
	/*
	 * If the first character is a ".", and we're allowed to nroff,
	 * nroff the file.  Put output in tmp file so we can print it later
	 * if necessary.
	 */
	if ((c == '.') && nflg) {
		error("Formatting %s, please wait.....", file);
		
		tmp = mktemp("/tmp/helpXXXXXX");
		
		sprintf(syscmd, "%s -man %s > %s", NROFF, file, tmp);
		system(syscmd);
		fputc('\n', stderr);
		fflush(stderr);
	}
	
	/*
	 * Print the file.
	 */
	if (cflg) {
		sprintf(syscmd, "%s %s", CAT, (tmp ? tmp : file));
	}
	else {
		if (hmode == MAN)
			sprintf(syscmd, "%s %s", MORE, (tmp ? tmp : file));
		else
			sprintf(syscmd, "%s -d %s", MORE, (tmp ? tmp : file));
	}

	fputc('\n', stderr);
	fflush(stderr);
	status = system(syscmd);
	
#ifdef HARDCOPY
	/*
	 * If -h given, see if they want a hardcopy of file.
	 */
	if (hflg && isatty(1)) {
		rsp = getrsp("Hardcopy? ");
		
		if ((*rsp == 'y') || (*rsp == 'Y')) {
			error("Making hardcopy of %s, please wait.....", file);
			
#ifdef vax
			if ((pid = vfork()) == 0) {
#else
			if ((pid = fork()) == 0) {
#endif
				if (tmp)
					execl(PRINT, "print", "-nr", tmp, 0);
				else
					execl(PRINT, "print", file, 0);
			}
			fputc('\n', stderr);
			fflush(stderr);
			while ((w = wait(&status)) != pid && w != -1);
		}
	}
#endif

	close(fd);
	
	if (tmp)
		unlink(tmp);
}
/*
 * dis_dir - display a directory in interactive mode.  Sorry about the
 *	     goto's here, but it beats recursing like crazy.
 */
char *dis_dir(path)
char *path;
{
	DIR *dirp;
	short level;
	char *f, file[1024];
	int co, li, nfiles, spread, dir_inx;
	int col, base, lines, batch, files_left;
	struct direct *dp;
	struct stat sbuf;
	char *getpath();
	char tmp[MAXNAMLEN+1];
	extern comp();
	
	/*
	 * file contains the absolute path of what we are
	 * looking at.  This is our "stack" on which we 
	 * "push" and "pop" pieces of pathnames.  stackptr
	 * initially points to the end (top) of the stack.
	 */
	level = 1;
	strcpy(file, path);
	stackptr = strlen(file);

	/*
	 * Open the directory.  If we can't, level tells
	 * us whether we came from a got or a function call.
	 */
newdir:	if ((dirp = opendir(file)) == NULL) {
		/*
		 * If we can't open it, pop
		 * this part of the path back off.
		 */
		error("Can't open %s.", file);
		pop(file);
		
		if (level == 1)
			return(NULL);
		else
			goto retry;
	}
	
	if (chdir(file) < 0) {
		/*
		 * If we didn't make it, pop this
		 * part of the path back off.
		 */
		error("Can't chdir to %s.", file);
		pop(file);

		if (level == 1) {
			closedir(dirp);
			return(NULL);
		}
		else {
			goto retry;
		}
	}
	
	nfiles = 0;

	/*
	 * Read the directory.
	 */
	while ((nfiles < MAXFILES) && ((dp = readdir(dirp)) != NULL)) {
		if ((dp->d_ino > 0) && (dp->d_name[0] != '.')) {
			strcpy(dbuf[nfiles], dp->d_name);
			nfiles++;
		}
	}
	closedir(dirp);

	/*
	 * Sort everything so it's purty...
	 */
	qsort(dbuf, nfiles, MAXNAMLEN+1, comp); 
	
	/*
	 * Display the directory on the screen.  I swiped this
	 * from the old help, and it wasn't commented there
	 * either (sorry....)
	 */
	if (nfiles) {
		files_left = nfiles;
		base = 0;
		
		col = (term.cols - 1) / COLWIDTH;
		lines = term.lines - 5;
		
retry:		while (files_left > 0) {
			clear();
			
			batch = (files_left > lines * col) ? lines * col : files_left;
			spread = (batch + col - 1) / col;
			
			if (base == 0)
				fprintf(stderr, "Current topics in %s are:\n\n", file);
			else
				fprintf(stderr, "Additional topics in %s are:\n\n", file);
				
			for (li=0; li < spread; li++) {
				for (co=0; co < col; co++) {
					dir_inx = co * spread + li + base;
					
					if (dir_inx < nfiles)
						fprintf(stderr, "%3d:%-*.*s ", dir_inx+1, COLWIDTH-5, COLWIDTH-5, dbuf[dir_inx]);
				}
				fputc('\n', stderr);
			}
			fflush(stderr);
			
			/*
			 * See what they want to do.
			 */
			switch (f=getpath(path, files_left > batch, nfiles)) {
			case REPRINT:
				continue;
			case RESTART:
				base = 0;
				files_left = nfiles;
				continue;
			case NXTPAGE:
				if ((files_left - batch) <= 0)
					continue;
				files_left -= batch;
				base += batch;
				continue;
			case RPTROOT:
				strcpy(file, path);
				level = 1;
				goto newdir;
			case NXTROOT:
				return(NULL);
			case LSTROOT:
				return(LSTROOT);
			}
			
			/*
			 * If we get through the switch() above, then
			 * we're displaying a file or directory.  Push
			 * the relative path onto the stack.
			 */
			strcpy(tmp, f);
			f = tmp;
			push(f, file);
			
			if (stat(f, &sbuf) < 0) {
				/*
				 * If it isn't there, pop
				 * back to where we were.
				 */
				error("Can't stat %s.", f);
				pop(file);
				continue;
			}
			if ((sbuf.st_mode & S_IFMT) == S_IFDIR) {
				level++;
				goto newdir;
			}
			else {
				/*
				 * After displaying a file, pop
				 * the filename.
				 */
				dis_file(file);
				pop(file);
				fputc('\n', stderr);
				getrsp("Press RETURN to continue: ");
				continue;
			}
		}
	}
	else {
		error("Directory %s is empty.", file);
		closedir(dirp);

		if (level == 1) {
			return(NULL);
		}
		else {
			/*
			 * Pop this directory.
			 */
			pop(file);
			level--;
			goto newdir;
		}
	}
}
/*
 * comp - comparison routine for qsort.
 */
comp(d1, d2)
char *d1, *d2;
{
	return(strcmp(d1, d2));
}
/*
 * getpath - prompting routine for interactive mode.
 */
char *getpath(path, more, nfiles)
char *path;
int more, nfiles;
{
	int i, dirx;
	char tmp[32];
	char *getrsp();
	char *rsp, *bp, *prstr;
	char *p1 = "Number or title to select topic (? for help): ";
	char *p2 = "Number or title to select topic, RETURN for more topics (? for help): ";

	if (more)
		prstr = p2;
	else
		prstr = p1;
	
	/*
	 * Prompt for input, and do what they say.  If they give us
	 * a filename, we use match() to find it, so that they don't
	 * have to type the suffixes for the commands.  If they give
	 * us a number, we return the appropriate filename.
	 */
	do {
		rsp = getrsp(prstr);
		
		switch (*rsp) {
		case '.':
			if (rsp[1] == NULL)
				return(RESTART);
			break;
		case '?':
			printhelp(path);
			getrsp("Press RETURN to continue: ");
			return(REPRINT);
		case '!':
			if (noshell) {
				error("Shell commands not permitted.");
				continue;
			}
			system(rsp+1);
			getrsp("Press RETURN to continue: ");
			return(REPRINT);
		case '/':
			if (rsp[1] == NULL)
				return(RPTROOT);
			break;
		case '>':
			return(NXTROOT);
		case '<':
			return(LSTROOT);
		case '^':
			return(RESTART);
		case 'q':
			exit(0);
		case NULL:
			return(NXTPAGE);
		}
		
		for (bp=rsp; ((*bp >= '0') && (*bp <= '9')); bp++);
		
		if (*bp == NULL) {
			dirx = atoi(rsp) - 1;
			
			if ((dirx < 0) || (dirx >= nfiles)) {
				error("%s: illegal selector.", rsp);
				continue;
			}
			
#ifdef DEBUG
			fprintf(stderr, "RETURNING %s\n", dbuf[dirx]);
			fflush(stderr);
#endif

			return(dbuf[dirx]);
		}
		else {
			for (i=0; i < nfiles; i++) {
				if (dbuf[i][0] != rsp[0])
					continue;
				
				strcpy(tmp, dbuf[i]);
				
				if (match(rsp, tmp) == 0)
					return(dbuf[i]);
			}
			return(rsp);
		}
	} while (1);
}
/*
 * error - print error messages.
 */
/*VARARGS*/
error(fmt, arg)
char *fmt, *arg;
{
	char errbuf[80];
	
	sprintf(errbuf, fmt, arg);
	
	if (term.cm != NULL) 
		cursor(0, term.lines-1);
	if (term.so != NULL)
		bstandout();
	
	fputs(errbuf, stderr);
	fputc('\007', stderr);
	
	if (term.se != NULL)
		estandout();
		
	fflush(stderr);
	sleep(4);
}
/*
 * getrsp - print a prompt and return input.
 */
char *getrsp(prompt)
char *prompt;
{
	static char rspbuf[128];
	
	if (term.cm != NULL)
		cursor(0, term.lines-2);
	else 
		fputc('\n', stderr);
	
	if (term.so != NULL)
		bstandout();
		
	fputs(prompt, stderr);
	
	if (term.se != NULL)
		estandout();
		
	fflush(stderr);
	
	if (fgets(rspbuf, 80, stdin) == NULL) {
		fputc('\n', stderr);
		exit(0);
	}

	rspbuf[strlen(rspbuf)-1] = NULL;
	return(rspbuf);
}
/*
 * printhelp - print possible inputs
 */
printhelp(root)
char *root;
{
	clear();
	
	fprintf(stderr, "- Enter a name or number to look at a document.\n");
	fprintf(stderr, "- Type '..' to return to the previous level in\n");
	fprintf(stderr, "  this documentation list.\n");
	fprintf(stderr, "- Type '^' to restart the listing of this level.\n");
	fprintf(stderr, "- Type '/' to return to the first level of this\n");
	fprintf(stderr, "  documentation list (%s).\n", root);
	fprintf(stderr, "- Type '>' to go to the next documentation list.\n");
	fprintf(stderr, "- Type '<' to go to the previous documentation list.\n");
	if (!noshell) {
		fprintf(stderr, "- Type '!' followed by a shell command to execute\n");
		fprintf(stderr, "  that shell command.\n");
	}
	fprintf(stderr, "- Type '?' to see this message.\n");
	fprintf(stderr, "- Type 'q' to QUIT.\n");
}
/*
 * putch - output routine for tputs.
 */
putch(ch)
char ch;
{
	fputc(ch, stderr);
}
/*
 * Everything from here on down was swiped from Berkeley's man program,
 * except that I renamed their match() to match2().  It wasn't commented
 * then, and it ain't commented now (sorry....)
 */
apropos(argc, argv)
int argc;
char **argv;
{
	char buf[BUFSIZ];
	char *gotit;
	register char **vp;

	if (freopen("/usr/man/whatis", "r", stdin) == NULL) {
		perror("/usr/man/whatis");
		exit (1);
	}
	gotit = (char *) calloc(1, blklen(argv));
	while (fgets(buf, sizeof buf, stdin) != NULL)
		for (vp = argv; *vp; vp++)
			if (match2(buf, *vp)) {
				printf("%s", buf);
				gotit[vp - argv] = 1;
				for (vp++; *vp; vp++)
					if (match2(buf, *vp))
						gotit[vp - argv] = 1;
				break;
			}
	for (vp = argv; *vp; vp++)
		if (gotit[vp - argv] == 0)
			printf("%s: nothing apropriate\n", *vp);
}

match2(buf, str)
char *buf, *str;
{
	register char *bp;

	bp = buf;
	for (;;) {
		if (*bp == 0)
			return (0);
		if (amatch(bp, str))
			return (1);
		bp++;
	}
}

amatch(cp, dp)
register char *cp, *dp;
{

	while (*cp && *dp && lmatch(*cp, *dp))
		cp++, dp++;
	if (*dp == 0)
		return (1);
	return (0);
}

lmatch(c, d)
char c, d;
{

	if (c == d)
		return (1);
	if (!isalpha(c) || !isalpha(d))
		return (0);
	if (islower(c))
		c = toupper(c);
	if (islower(d))
		d = toupper(d);
	return (c == d);
}

blklen(ip)
register int *ip;
{
	register int i = 0;

	while (*ip++)
		i++;
	return (i);
}

whatis(argc, argv)
int argc;
char **argv;
{
	char *trim();
	register char **avp;

	if (argc == 0) {
		fprintf(stderr, "man: -f what?\n");
		exit(1);
	}
	if (freopen("/usr/man/whatis", "r", stdin) == NULL) {
		perror("/usr/man/whatis");
		exit (1);
	}
	for (avp = argv; *avp; avp++)
		*avp = trim(*avp);
	whatisit(argv);
	exit(0);
}

whatisit(argv)
char **argv;
{
	char buf[BUFSIZ];
	register char *gotit;
	register char **vp;

	gotit = (char *)calloc(1, blklen(argv));
	while (fgets(buf, sizeof buf, stdin) != NULL)
		for (vp = argv; *vp; vp++)
			if (wmatch(buf, *vp)) {
				printf("%s", buf);
				gotit[vp - argv] = 1;
				for (vp++; *vp; vp++)
					if (wmatch(buf, *vp))
						gotit[vp - argv] = 1;
				break;
			}
	for (vp = argv; *vp; vp++)
		if (gotit[vp - argv] == 0)
			printf("%s: not found\n", *vp);
}

wmatch(buf, str)
char *buf, *str;
{
	register char *bp, *cp;

	bp = buf;
again:
	cp = str;
	while (*bp && *cp && lmatch(*bp, *cp))
		bp++, cp++;
	if (*cp == 0 && (*bp == '(' || *bp == ',' || *bp == '\t' || *bp == ' '))
		return (1);
	while (isalpha(*bp) || isdigit(*bp))
		bp++;
	if (*bp != ',')
		return (0);
	bp++;
	while (isspace(*bp))
		bp++;
	goto again;
}

char *trim(cp)
register char *cp;
{
	register char *dp;

	for (dp = cp; *dp; dp++)
		if (*dp == '/')
			cp = dp + 1;
	if (cp[0] != '.') {
		if (cp + 3 <= dp && dp[-2] == '.' && any(dp[-1], "cosa12345678npP"))
			dp[-2] = 0;
		if (cp + 4 <= dp && dp[-3] == '.' && any(dp[-2], "13") && isalpha(dp[-1]))
			dp[-3] = 0;
	}
	return (cp);
}
any(c, sp)
register int c;
register char *sp;
{
	register int d;

	while (d = *sp++)
		if (c == d)
			return (1);
	return (0);
}
/*
 * push - pushes what onto the stack.  The theory behind this makes the
 *        code look a little weird, but if you walk through it by
 *	  hand once, it shouldn't be too hard to follow.
 */
push(what, stack)
char *what, *stack;
{
	char *s, *rindex();

	/*
	 * If it's just ".", we aren't going anywhere,
	 * the stack remains the same.
	 */
	if (!strcmp(what, "."))
		return;

	/*
	 * If it's just "..", we simply go
	 * up one level.  So, all we do is
	 * pop the stack.  This avoids paths 
	 * like "/usr/man/cat1/../cat2/../cat3"
	 */
	if (!strcmp(what, "..")) {
		pop(stack);
		return;
	}

	/*
	 * If it's "./something", we only need the
	 * something.  Skip over the "./", and if
	 * there's something there, push it, otherwise,
	 * it's just like "." above.
	 */
	if (!strncmp(what, "./", 2)) {
		what += 2;

		if (*what == NULL)
			return;
	}

	/*
	 * If it's "../something", we can just pop and then
	 * do the something.  If there isn't anything there, 
	 * it's just like ".." above.
	 */
	if (!strncmp(what, "../", 3)) {
		pop(stack);
		what += 3;

		if (*what == NULL)
			return;
	}

	/*
	 * If we are supposed to push an absolute path, 
	 * we let it overwrite the current stack.  This
	 * sounds wrong, but it makes sense when you are 
	 * running the program.
	 */
	if (*what == '/') {
		strcpy(stack, what);
		stackptr = strlen(stack);
		return;
	}

	/*
	 * Set stackptr to top of present stack.
	 */
	stackptr = strlen(stack);

	/*
	 * Only put in a slash if we need one.
	 */
	if (stack[strlen(stack)-1] != '/')
		strcat(stack, "/");

	/*
	 * Push what onto the stack.
	 */
	strcat(stack, what);

	/*
	 * If what contained slashes, set the stackptr 
	 * to the last slash.  This is so going from
	 * "/" to "usr/man" to ".." puts us in /usr,
	 * not back in /.
	 */
	s = rindex(what, '/');

	if (s != NULL) {
		stackptr = strlen(stack);
		while (stack[stackptr] != '/')
			stackptr--;
	}
}
/*
 * pop - pops the stack.
 */
pop(stack)
char *stack;
{
	char *s, *t, *rindex();

	/*
	 * s is set to the last slash in the stack,
	 * t to wherever stackptr is pointing.
	 */
	s = rindex(stack, '/');
	t = (char *) &stack[stackptr];

	/*
	 * If there aren't any slashes, we can't
	 * very well pop - we'd leave an empty 
	 * stack.
	 */
	if (s == NULL)
		return;

	/*
	 * If the only slash is in stack[0], then
	 * we want the stack to contain just "/".
	 */
	if (s == stack) {
		*(s+1) = NULL;
	}
	/*
	 * Otherwise, pop back to s or t, whichever will
	 * pop more off.  Note that since s is set to the 
	 * last slash, this won't pop any more than one 
	 * "piece" of a pathname.
	 */
	else {
		if (s <= t) 
			*s = NULL;
		else
			*t = NULL;
	}

	/*
	 * Reset the stackptr.
	 */
	stackptr = strlen(stack);
}
