/*--------------------------------------------------------------------------
 Generate inline fft code for a N-point radix-2 DIT FFT.
 Reads six lines from stdin giving the format of six macros:
 1. radix-2 butterfly for any twiddle factor
 2. radix-2 butterfly for twiddle factor of 1              (k / N = 0)
 3. radix-2 butterfly for twiddle factor of (i+1)/sqrt(2)  (k / N = 1/8)
 4. radix-2 butterfly for twiddle factor of i              (k / N = 1/4)
 5. radix-2 butterfly for twiddle factor of (i-1)/sqrt(2)  (k / N = 3/8)
 6. radix-4 butterfly for stages 1 and 2; twiddle factors are all trivial

 In all cases, the first 'radix' %d's are expanded to the index
 of the locations the butterfly is to operate upon.

 In the radix-2 butterflies, the third and fourth %d's are expanded to
 the real and imaginary componants of the twiddle factor, expressed
 as 13-bit fixed-point fractions.

 See Oppenheim & Schafer, "Digital Signal Processing", 1975, p. 318; 
 the particular dataflow graph implemented is that of figure 6.10, 
 which uses bit-reversed input, and accesses coefficients in normal order.

 Dan Kegel, rochester!srs!dan, S.R. Systems, Oct '88
 This code is hereby placed in the public domain without any promise
 that it functions as intended, or at all.
--------------------------------------------------------------------------*/
#include <stdio.h>
#include <math.h>

#ifndef PI
#define PI 3.1415926
#endif

/* Symbolic constants for the indices of the format array, read in from stdin.
 * Each entry handles a different special case, except BTRFLY2_ANY,
 * which is general.
 */
#define BTRFLY2_ANY      0
#define BTRFLY2_0_4TH_PI 1
#define BTRFLY2_1_4TH_PI 2
#define BTRFLY2_2_4TH_PI 3
#define BTRFLY2_3_4TH_PI 4
#define BTRFLY4_STAGE0   5
#define COMMENT 6
#define N_FORMATS 7

#define FRAC_BITS 12		/* for TMS320 inline multiply */

/* The following global should be set to (1 << FRAC_BITS) if
 * multiplies by unity are optimized away, and to
 * (1 << FRAC_BITS) - 1 if multiplies by unity are done in the same
 * manner as other multiplies.
 * I think.
 */
static float i_unity = (float) (1 << FRAC_BITS);

/*--------------------------------------------------------------------------
 Routine to convert floating point number to integer format.
 Result is a fixed-point number with FRAC_BITS of fraction.

 Constructed so that ftoi(x) = -ftoi(-x), for whatever that's worth.
--------------------------------------------------------------------------*/
int
ftoi(f)
    float f;
{
    int i;
    if (f < 0.0)
	i = - (int) (0.5 - f * i_unity);
    else
	i = (int) (0.5 + f * i_unity);
    return i;
}

/*--------------------------------------------------------------------------
 Routine to calculate real part of j'th twiddle factor for an n-point DIT fft.
--------------------------------------------------------------------------*/
int
icos(j, n)
    int j, n;
{
    return ftoi(cos((2 * PI * j) / (float) n));
}

/*--------------------------------------------------------------------------
 Routine to calculate imaginary part of j'th twiddle factor for an 
 n-point DIT fft.
--------------------------------------------------------------------------*/
int
isin(j, n)
    int j, n;
{
    return ftoi(sin((2 * PI * j) / (float) n));
}

main(argc, argv)
    int argc;
    char **argv;
{
    int log_n;
    int i, j, k, n;
    int stage;
    char format[N_FORMATS][256];
    char comment[256];

    if (argc < 2) {
	(void) fprintf(stderr, "usage: %s log_fft_size < instruction_formats\n\
Outputs inline code to implement given size DIT fft.\n\
Input is six butterfly template lines, in which %%d is expanded to array \n\
indices and real and imaginary parts of the twiddle factors; for instance,\n\
	BFLY2	X%%03d,X%%03d,%%d,%%d\n\
	BFLY2_0PI4	X%%03d,X%%03d\n\
	BFLY2_1PI4	X%%03d,X%%03d\n\
	BFLY2_2PI4	X%%03d,X%%03d\n\
	BFLY2_3PI4	X%%03d,X%%03d\n\
	BFLY4_SPECIAL	X%%03d,X%%03d,X%%03d,X%%03d\n\
followed by a comment template line, in which %%s is expanded to a remark\n\
about the following code, for example\n\
	* %%s\n\
", argv[0]);
	exit(1);
    }

    /* Get arguments. */
    log_n = atoi(argv[1]);
    if (log_n < 2 || log_n > 13) {
	(void) fprintf(stderr, 
	    "log_fft_size %s out of range, must be 2 to 13\n", argv[1]);
	exit(1);
    }
    n = 1 << log_n;

    /* Read templates. */
    for (i=0; i<N_FORMATS; i++) {
	if (gets(format[i]) == NULL) {
	    (void) fprintf(stderr, 
		"%s: couldn't read all %d formats from stdin\n",
		argv[0], N_FORMATS);
	    exit(1);
	}
    }

    (void) sprintf(comment, "Inline code for %d-point FFT.", n);
    (void) printf(format[COMMENT], comment);
    (void) putchar('\n');

    (void) printf(format[COMMENT], "Radix-4 butterflies for stages 1 and 2.");
    (void) putchar('\n');
    /* For each 4-point DFT making up the combined first two stages: */
    for (j=0; j < n/4; j++) {
	(void) printf(format[BTRFLY4_STAGE0], 4*j, 4*j+1, 4*j+2, 4*j+3);
	(void) putchar('\n');
    }

    /* For each following stage of the Cooley-Tukey FFT */
    for (stage=3; stage<=log_n; stage++) {
	int n_dft;
	int n_butterfly;
	int dft_size;

	(void) sprintf(comment, "Stage %d.", stage);
	(void) printf(format[COMMENT], comment);
	(void) putchar('\n');

	n_dft = 1 << (log_n - stage);		/* # of dft's */
	n_butterfly = 1 << (stage-1);		/* # of butterflies per dft */
	dft_size = 1 << stage;			/* size of each dft */

	/* For each (dft_size)-point dft making up this stage: */
	for (j=0; j < n_dft; j++) {

	    (void) sprintf(comment, 
		"Radix-2 butterflies for %d-point dft %d of stage %d.", 
		    dft_size, j, stage);
	    (void) printf(format[COMMENT], comment);
	    (void) putchar('\n');

	    /* For each butterfly making up this dft: */
	    for (k=0; k<n_butterfly; k++) {
		int f;
		int index1, index2;

		/* Calculate the indices of the two locations this butterfly
		 * operates upon.
		 */
		index1 = j * 2 * n_butterfly + k;
		index2 = index1 + n_butterfly;

		/* Decide which butterfly to use.
		 * Special cases that can be optimized are those where
		 *    ln (twiddle factor)
		 * is a multiple of sqrt(-1) * pi/4.
		 */
		if ( ((8 * k) % dft_size) != 0) {
		    /* Must use general case. */
		    f = BTRFLY2_ANY;
		} else {
		    /* Use one of four optimized cases. */
		    int pi_quarter = (8 * k) / dft_size;
		    f = pi_quarter + BTRFLY2_0_4TH_PI;
		}

		/* Output the butterfly. */
		(void) printf(format[f],
		    index1, index2, icos(k, dft_size), isin(k, dft_size));
		(void) putchar('\n');
	    }
	}
    }


    exit(0);
}
