
/*  @(#)mon.c 1.17 90/06/26
 *
 *  Monitoring routines used by the faces program.
 * 
 *  Copyright (c) Rich Burridge - Sun Microsystems Australia.
 *                                All rights reserved.
 *
 *  Permission is given to distribute these sources, as long as the
 *  copyright messages are not removed, and no monies are exchanged. 
 * 
 *  No responsibility is taken for any errors on inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  to me, then an attempt will be made to fix them.
 */

#include <stdio.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/file.h>
#include "faces.h"
#include "extern.h"


add_face(dtype, itype, name)
enum disp_type dtype ;
enum icon_type itype ;
char *name ;
{
  unsigned short buf[256] ;  /* Ikon/icon image. */

  if (itype == XFACE)
    {
      itype = ORDINARY ;
      destroy_image(itype) ;
      get_xface(face_buf, buf) ;
      load_icon(itype, buf, 0) ;
    }
  else if (itype == ORDINARY)
    {
      destroy_image(itype) ;
      if (get_icon(name, buf) == -1) itype = NOFACE ;
      else load_icon(itype, buf, 0) ;
    }
 
  switch (dtype)
    {
      case DISP_ALL   : adjust_image(DISP_NAME,  itype, row, column) ;
                        adjust_image(DISP_OTHER, itype, row, column) ;
                        adjust_image(DISP_ICON,  itype, 0, 0) ;
                        break ;
      case DISP_BOTH  : adjust_image(DISP_NAME,  itype, row, column) ;
                        adjust_image(DISP_OTHER, itype, row, column) ;
                        break ;
      case DISP_ICON  : adjust_image(dtype, itype, 0, 0) ;
                        break ;
      case DISP_NAME  :
      case DISP_OTHER : adjust_image(dtype, itype, row, column) ;
    }
}


adjust()          /* Adjust the row and column position. */
{
  struct psinfo *this ;

  if (mtype != MONNEW)
    {
      if (++column == maxcols)
        {
          column = 0 ;
          row++ ;
        }
      return ;
    }
  if (psrecs != NULL)           /* Adjust animation positions for MONNEW */
    {
      this = psrecs ;           /* Point to beginning of chain. */
      while (this != NULL)
        {
          this->column++ ;      /* Adjust column position. */
          if (facetype == NEWSTYPE && this->next == NULL)
            this->column-- ;    /* Reset for brand new animate record. */
          this = this->next ;
        }
    }
}


do_check()        /* Perform another check of the appropriate type. */
{
  switch (mtype)
    {
      case MONALL     : do_mail(MONALL) ;  /* Monitor all of the mail file. */
                        break ;
      case MONNEW     : do_mail(MONNEW) ;  /* Monitor new mail only. */
                        break ;
      case MONPRINTER : do_printer() ;     /* Monitor the print queue. */
                        break ;
      case MONPROG    : do_prog() ;        /* Run user supplied program. */
                        break ;
      case MONUSERS   : do_users() ;       /* Monitor users on a machine. */
    }
  firsttime = 0 ;
}


do_face_update(name, buf)   /* Send mail to update faces database. */
char *name, buf[2048] ;
{
  FILE *fp ;                /* File descriptor for users mail spool file. */
  char command[MAXLINE] ;   /* Command to send mail to special alias. */

  SPRINTF(command, UPDATEDEF, update_alias) ;
  if ((fp = popen(command, "w")) == NULL)     /* Open pipe to mail process. */
    {
      FPRINTF(stderr,"%s: couldn't send mail to update database.\n", progname) ;
      return ;
    }
  FPRINTF(fp, "To: %s\n", update_alias) ;
  FPRINTF(fp, "Subject: %s\n\n", name) ;  /* Send icon name. */
  FPRINTF(fp, "%s\n", buf) ;              /* Send ikon data. */
  PCLOSE(fp) ;
}


do_mail(mtype)              /* Monitor a mail file for new or all mail. */
enum mon_type mtype ;
{        
  FILE *fp ;                /* File descriptor for users mail spool file. */
  time_t ubuf[2] ;          /* For reseting the access time on spoolfile. */

  column = row = 0 ;        /* Start in top left corner of pixrect. */
  newmail = 0 ;             /* Assume no new mail. */
  noicons = 0 ;

  if (mtype == MONNEW) make_pixrect(maxcols) ;
  if (stat(spoolfile, &buf) == -1)
    {
      lastsize = 0 ;
      add_face(DISP_ICON, NOMAIL, "") ;
      if (mtype == MONNEW) show_display() ;   /* Show new mail. */
      else make_display() ;     /* Output icons and tidyup chain of records. */
      return ;
    }
  if (buf.st_size > lastsize) newmail = 1 ;   /* New mail found. */
  if (!buf.st_size) add_face(DISP_ICON, NOMAIL, "") ;

  ubuf[0] = buf.st_atime ;                    /* Save for possible reset. */
  ubuf[1] = buf.st_mtime ;

  if (mtype == MONNEW)
    if (buf.st_size <= lastsize)   /* Is the size of mail folder bigger? */
      if (mhflag && (buf.st_size < lastsize) && (buf.st_size != 0))
        {
          lastsize = 0 ;    /* User uses MH. Any shrinkage means new mail. */
          newmail = 1 ;
        }
      else
        {
          lastsize = buf.st_size ;   /* No: save new size and exit. */
          show_display() ;
          return ;
        }

  if ((fp = fopen(spoolfile, "r")) == NULL)   /* Open spoolfile. */
    {
      if (mtype == MONNEW) show_display() ;   /* Show new mail. */
      else make_display() ;     /* Output icons and tidyup chain of records. */
      return ;
    }
  if (mtype == MONNEW) FSEEK(fp, lastsize, 0) ;
  lastsize = buf.st_size ;
  while (fgets(line, MAXLINE, fp) != NULL)
    {
      if (EQUAL(line, "From "))
        {
          if (froms_found) process_info() ;  /* Process previous mail. */
          process_from() ;                   /* Save new from details. */
        }
      else if (EQUAL(line, "From:"))  process_from() ;
      else if (EQUAL(line, "X-Face")) process_face() ;
      else if (doing_xface)           process_face() ;
    }
  FCLOSE(fp) ;

  if (!stat(spoolfile, &buf))    /* Reset access time if no change in size. */
    if (buf.st_size == lastsize)
      UTIME(spoolfile, ubuf) ;

  if (froms_found) process_info() ;       /* Process remaining mail item. */
  if (mtype == MONNEW) show_display() ;   /* Show new mail. */
  else make_display() ;     /* Output icons and tidyup chain of records. */
}


do_printer()                 /* Monitor printer queue. */
{
  struct recinfo *this, *next ;
  FILE *fp ;                 /* File descriptor for users mail spool file. */
  char command[MAXLINE] ;    /* Print status command for this printer. */
  char owner[MAXLINE] ;      /* Owner of job in the print queue. */
  int size ;                 /* Size of this print job in bytes. */

  noicons = 0 ;
  SPRINTF(command, PRINTDEF, printer) ;
  if ((fp = popen(command, "r")) == NULL)     /* Open spoolfile. */
    {
      FPRINTF(stderr,"%s: couldn't get printer stats.\n", progname) ;
      return ;
    }
  column = row = 0 ;          /* Start in top left corner of pixrect. */
  FGETS(nextline, MAXLINE, fp) ;
  if (EQUAL(nextline, "no entries"))
    {
      make_pixrect(1) ;                    /* Just the "no print" icon. */
      add_face(DISP_ALL, NOPRINT, "") ;    /* Set to "no print" icon. */
      text(DISP_ALL, LEFT, printer) ;      /* Output printer name. */
    }
  else if (EQUAL(nextline, "Printer Error: may need attention!"))
    {
      make_pixrect(1) ;                    /* Just the "no paper" icon. */
      add_face(DISP_ALL, NOPAPER, "") ;    /* Set to "no paper" icon. */
      text(DISP_ALL, LEFT, printer) ;      /* Output printer name. */
    }
  else
    {
      FGETS(nextline, MAXLINE, fp) ;    /* Skip the next line. */
      while (fgets(nextline, MAXLINE, fp) != NULL)
        {
          SSCANF(&nextline[7], "%s", owner) ;
          SSCANF(&nextline[60], "%d", &size) ;
          h_to_c("", community) ;
          found = make_iconname(facedir, community, owner) ;
          add_record("", owner, "", size) ;
        }
      make_pixrect(noicons) ;
      this = recs ;
      while (this != NULL)
        {
          next = this->next ;
          add_face(DISP_BOTH, ORDINARY, this->iconname) ;
          SPRINTF(nextline, "%1d", this->size) ;
          if (!dontshowuser)
            text(DISP_NAME, LEFT, this->username) ;  /* Owner. */
          text(DISP_OTHER, LEFT, nextline) ;         /* Size. */
          if (this == recs)
            {
              add_face(DISP_ICON, ORDINARY, this->iconname) ;
              text(DISP_ICON, LEFT, printer) ;
              SPRINTF(nextline, "%1d %s", noicons,
                                         (noicons == 1 ? "job" : "jobs")) ;
              text(DISP_ICON, RIGHT, nextline) ;   /* Number of jobs. */
            }
          adjust() ;    /* Adjust column and row. */
          remove_record(this) ;
          this = next ;
        }
      recs = last = NULL ;
    }
  PCLOSE(fp) ;
  show_display() ;
}


do_prog()                   /* Run user supplied program or script. */
{
  FILE *fp ;                /* File descriptor for users command output. */
  enum icon_type ftype ;    /* Type of the current face. */
  char host[MAXLINE] ;      /* Pointer to host name from the "From" line. */
  char ileft[MAXLINE] ;     /* Text for left side of icon display. */
  char iright[MAXLINE] ;    /* Text for right side of icon display. */
  char user[MAXLINE] ;      /* Pointer to user name from the "From" line. */
  char wleft[MAXLINE] ;     /* Text for left side of window display. */
  char wright[MAXLINE] ;    /* Text for right side of window display. */
  int cols = maxcols ;      /* Maximum number of columns for display. */
  int rows = 1 ;            /* Maximum number of rows for display. */

  if ((fp = popen(userprog, "r")) == NULL)   /* Connect to user program. */
    {
      FPRINTF(stderr,"%s: couldn't get program (%s) information.\n",
              progname, userprog) ;
      return ;
    }
  FGETS(nextline, MAXLINE, fp) ;
  SSCANF(nextline, "Cols=%d Rows=%d", &cols, &rows) ;
  if (cols < 1) cols = maxcols ;
  if (rows < 1) rows = 1 ;

  width = cols * imagewidth ;
  height = rows * imageheight ;
  create_pixrects(width, height) ;

  maxcols = cols ;          /* Maximum width in columns. */
  column = row = 0 ;        /* Start in top left corner of pixrect. */
  while (fgets(nextline, MAXLINE, fp) != NULL)
    {
      nextline[80] = '\0' ;
      if (sscanf(&nextline[70], "%s", iright) == EOF) iright[0] = '\0' ;
      nextline[70] = '\0' ;
      if (sscanf(&nextline[60], "%s", ileft)  == EOF) ileft[0] = '\0' ;
      nextline[60] = '\0' ;
      if (sscanf(&nextline[50], "%s", wright) == EOF) wright[0] = '\0' ;
      nextline[50] = '\0' ;
      if (sscanf(&nextline[40], "%s", wleft)  == EOF) wleft[0] = '\0' ;
      nextline[40] = '\0' ;
      if (sscanf(&nextline[20], "%s", host)   == EOF) host[0] = '\0' ;
      nextline[20] = '\0' ;
      if (sscanf(&nextline[ 0], "%s", user)   == EOF) user[0] = '\0' ;

      STRCPY(community, "") ;
      if (!(EQUAL(user, "NOMAIL")  || EQUAL(user, "NOPAPER") ||
            EQUAL(user, "NOPRINT") || EQUAL(user, "NOUSERS")))
        {
          h_to_c(host, community) ;   /* Turn hostname into community name. */
          a_to_u(community, user, realname) ;
          found = make_iconname(facedir, community, realname) ;
        }
           if (EQUAL(user, "NOMAIL"))  ftype = NOMAIL ;
      else if (EQUAL(user, "NOPAPER")) ftype = NOPAPER ;
      else if (EQUAL(user, "NOPRINT")) ftype = NOPRINT ;
      else if (EQUAL(user, "NOUSERS")) ftype = NOUSERS ;
      else                             ftype = ORDINARY ;
      add_face(DISP_BOTH, ftype, iconname) ;
      if (strlen(ileft))  text(DISP_OTHER, LEFT,  ileft) ;
      if (strlen(iright)) text(DISP_OTHER, RIGHT, iright) ;
      if (strlen(wleft))  text(DISP_NAME,  LEFT,  wleft) ;
      if (strlen(wright)) text(DISP_NAME,  RIGHT, wright) ;
      adjust() ;             /* Adjust column and row. */
    }
  PCLOSE(fp) ;
  show_display() ;
}


do_users()                   /* Monitor users on a machine. */
{
  struct recinfo *this, *next ;
  FILE *fp ;                 /* File descriptor for users mail spool file. */
  char command[MAXLINE] ;    /* Rusers system call for this machine. */
  char ts[MAXLINE] ;         /* Pointer to login time from rusers line. */
  char username[MAXLINE] ;   /* Name of user logged in. */
 
  noicons = 0 ;
  SPRINTF(command, USERSDEF, hostname) ;
  if ((fp = popen(command, "r")) == NULL)    /* Connect to rusers command. */
    {
      FPRINTF(stderr,"%s: couldn't get user stats.\n", progname) ;
      return ;
    }
  column = row = 0 ;          /* Start in top left corner of pixrect. */
  while (fgets(nextline, MAXLINE, fp) != NULL)
    {
      SSCANF(&nextline[0], "%s", username) ;
      SSCANF(&nextline[38], "%s", ts) ;
      h_to_c("", community) ;
      found = make_iconname(facedir, community, username) ;
      add_record("", username, ts, 0) ;
    }
  if (!noicons)
    {
      make_pixrect(1) ;                    /* Just the "no users" icon. */
      add_face(DISP_ALL, NOUSERS, "") ;    /* Set to "no users" icon. */
    }
  else
    {
      make_pixrect(noicons) ;
      add_face(DISP_ICON, NOFACE, "") ;
      this = recs ;
      while (this != NULL)
        {
          next = this->next ;
          add_face(DISP_BOTH, ORDINARY, this->iconname) ;
          if (!dontshowtime)
            text(DISP_OTHER, LEFT, this->ts) ;         /* Timestamp. */
          if (!dontshowuser)
            text(DISP_NAME, LEFT, this->username) ;    /* Username. */
          adjust() ;    /* Adjust column and row. */
          remove_record(this) ;
          this = next ;
        }
      SPRINTF(nextline, "%1d %s", noicons, (noicons == 1 ? "user" : "users")) ;
      text(DISP_ICON, RIGHT, nextline) ;   /* Number of jobs. */
      recs = last = NULL ;
    }
  PCLOSE(fp) ;
  show_display() ;
}


make_pixrect(count)   /* Make window pixrect the correct size. */
int count ;
{
  int c, r ;          /* Size in columns and rows of window display. */

  r = ((count-1) / maxcols) + 1 ;   /* Number of rows of faces. */
  c = maxcols ;                     /* Full width display. */
  if (count <= 10)
    {
      r = 1 ;         /* One row. */
      c = count ;     /* Of 'count' columns. */
    }

  height = r * imageheight ;    /* Height of the icon display. */
  width = c * imagewidth ;      /* Width of the icon display. */
  create_pixrects(width, height) ;
}


make_display()              /* Output icons and tidyup chain of records. */
{
  int count ;               /* Name of faces in icon display. */
  struct recinfo *this, *next ;

  count = noicons ;         /* Number of faces to display. */
  if (!count) count = 1 ;   /* Always one "no mail" icon. */
  make_pixrect(count) ;

  count = 0 ;
  if (!noicons) add_face(DISP_ALL, NOMAIL, "") ;
  else
    {
      this = recs ;
      while (this != NULL)
        {
          next = this->next ;
          if (!this->total)
            {
              this = next ;
              continue ;
            }
          if (this->faceimage != NULL)
            {
              STRCPY(face_buf, (char *) this->faceimage) ;
              add_face(DISP_ALL, XFACE, this->iconname) ;
              if (!this->update && update)               /* Update database */
                {
                  do_face_update(this->iconname, face_buf) ;
                  this->update = 1 ;
                }
            }
          else add_face(DISP_ALL, ORDINARY, this->iconname) ;
          count += this->total ;
          if (!dontshowno)
            {
              SPRINTF(nextline, "%1d", this->total) ;
              text(DISP_OTHER, RIGHT, nextline) ;
            }
          if (!dontshowtime) text(DISP_OTHER, LEFT, this->ts) ;
          if (!dontshowuser)
            {
              text(DISP_NAME, LEFT, this->username) ;
              text(DISP_ICON, LEFT, this->username) ;
            }
          adjust() ;
          this = next ;
        }
      SPRINTF(nextline, "%1d", count) ;
      text(DISP_ICON, RIGHT, nextline) ;
      garbage_collect() ;                 /* Remove zero length records. */
    }
  show_display() ;         /* Display the latest set of faces. */
}


process_face()        /* Extract next line of face image information. */
{

/*  Read in the X-Face: data. By default, the first line will contain an
 *  initial X-Face: (which is ignored), followed by 72 characters of
 *  compressed data. Second and subsequent lines will contain an initial
 *  space (which is ignored), followed by 79 characters of compressed data.
 *  The last line may contain less characters.
 *
 *  The trouble is that users may try to pretty up this output. In order to
 *  handle that possibility, the following rules will apply:
 *
 *  On the initial line, the "X-Face: " will be removed, any immediate
 *  whitespace (tabs and spaces) will be removed, and the remainder of
 *  the line will be placed in the data buffer (minus the trailing newline).
 *
 *  On subsequent lines, initial whitespace will be removed, and the
 *  remainder of the data appended to the data buffer (minus the trailing
 *  newline).
 *
 *  When a blank line, or a line with the initial non whitespace character,
 *  then this will signify the end of X-Face data, and the buffer will be
 *  passed to the uncompression routine.
 *
 *  Note that this extraction process is not perfect, and it is possible that
 *  more data may be appended to the face buffer then necessary. This should
 *  not be a problem, because the face uncompression routine should ignore
 *  that extra input data.
 */

  char *ptr ;

  if (!doing_xface)
    {
      bzero(face_buf, 2048) ;
      ptr = &line[7] ;
      while (*ptr == ' ' || *ptr == '\t') ptr++ ;
      STRNCPY(face_buf, ptr, strlen(ptr)-1) ;
      face_buf[strlen(ptr)-1] = '\0' ;
      doing_xface = 1 ;
    }
  else if (!strlen(line) || !(line[0] == ' ' || line[0] == '\t'))
    {
      doing_xface = 0 ;
      if (uncompface(face_buf) < 0) return ;
      x_face = 1 ;                       /* We have an "on-the-fly" face. */
    }
  else
    {
      ptr = line ;
      while (*ptr == ' ' || *ptr == '\t') ptr++ ;
      STRNCAT(face_buf, ptr, strlen(ptr)-1) ;
    }
}


process_from()             /* Extract information from the From line. */
{
  char *host, *user ;
  char temp[MAXLINE], ts[9] ;

  if (EQUAL(line, "From "))
    {
      SSCANF(line, "%s %s %s %s %s %s", temp, temp, temp, temp, temp, ts) ;
      ts[5] = '\0' ;
      STRCPY(face_ts, ts) ;
      froms_found = 1 ;
    }
  else
    {
      if (fromc_found) return ;    /* Only process first From: line. */
      fromc_found = 1 ;
    }

  if (parsefrom(line, &user, &host))
    {
      STRCPY(face_user, strlower(user)) ;
      STRCPY(face_host, host) ;
    }
  else FPRINTF(stderr, "%s: couldn't parse '%s'\n", progname, line) ;
}


process_info()          /* Process From line and face information. */
{
  struct recinfo *crec ;    /* Pointer to current mail record for updating. */

  h_to_c(face_host, community) ;    /* Turn hostname into community name. */
  a_to_u(community, face_user, realname) ;
  if (mtype == MONNEW)
    {
      if (x_face)
        {
          SPRINTF(iconname, "%s/%s/%s/48x48x1",
                  strlen(facedir) ? facedir : defdir, community, realname) ;
          add_face(DISP_ALL, XFACE, iconname) ;
          if (update) do_face_update(iconname, face_buf) ;
        }
      else
        {
          if (!(found = make_iconname(facedir, community, realname)))
            add_face(DISP_ALL, NOFACE, (char *) NULL) ;
          else add_face(DISP_ALL, ORDINARY, iconname) ;
        }

      if (!dontshowtime) text(DISP_OTHER, LEFT, face_ts) ;
      if (!dontshowuser)
        {
          text(DISP_NAME, LEFT, realname) ;
          text(DISP_ICON, LEFT, realname) ;
        }
      adjust() ;
    }
  else
    {
      found = make_iconname(facedir, community, realname) ;
      if (x_face)
        SPRINTF(iconname, "%s/%s/%s/48x48x1",
                strlen(facedir) ? facedir : defdir, community, realname) ;
      if ((crec = rec_exists(community, realname)) != NULL)
        {
          STRCPY(crec->ts, face_ts) ;
          if (!crec->total) noicons++ ;
          crec->total++ ;
        }
      else add_record(community, realname, face_ts, 0) ;
    }
  fromc_found = froms_found = x_face = 0 ;  /* Reset for the next message. */
}
