/* RCS      -- $Header: /u2/dvadura/src/generic/dmake/src/RCS/sysintf.c,v 1.1 90/10/06 12:04:17 dvadura Exp $
-- SYNOPSIS -- system independent interface
-- 
-- DESCRIPTION
--	These are the routines constituting the system interface.
--	The system is taken to be essentially POSIX conformant.
--	The original code was extensively revised by T J Thompson at MKS,
--	and the library cacheing was added by Eric Gisin at MKS.  I then
--	revised the code yet again, to improve the lib cacheing, and to
--	make it more portable.
--
--	The following is a list of routines that are required by this file
--	in order to work.  These routines are provided as functions by the
--	standard C lib of the target system or as #defines in system/sysintf.h
--	or via appropriate C code in the system/ directory for the given
--	system.
--
--	The first group must be provided by a file in the system/ directory
--	the second group is ideally provided by the C lib.  However, there
--	are instances where the C lib implementation of the specified routine
--	does not exist, or is incorrect.  In these instances the routine
--	must be provided by the the user in the system/ directory of dmake.
--	(For example, the bsd/ dir contains code for putenv(), and tempnam())
--
--	DMAKE SPECIFIC:
--		seek_arch()
--		touch_arch()
--		void_lcache()
--		runargv()
--		STAT()
--		Remove_prq()
--
--	C-LIB SPECIFIC:  (should be present in your C-lib)
--		utime()
--		time()
--		getenv()
--		putenv()
--		getcwd()
--		signal()
--		chdir()
--		tempnam()
-- 
-- AUTHOR
--      Dennis Vadura, dvadura@watdragon.uwaterloo.ca
--      CS DEPT, University of Waterloo, Waterloo, Ont., Canada
--
-- COPYRIGHT
--      Copyright (c) 1990 by Dennis Vadura.  All rights reserved.
-- 
--      This program is free software; you can redistribute it and/or
--      modify it under the terms of the GNU General Public License
--      (version 1), as published by the Free Software Foundation, and
--      found in the file 'LICENSE' included with this distribution.
-- 
--      This program is distributed in the hope that it will be useful,
--      but WITHOUT ANY WARRANTY; without even the implied warrant of
--      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--      GNU General Public License for more details.
-- 
--      You should have received a copy of the GNU General Public License
--      along with this program;  if not, write to the Free Software
--      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--
-- LOG
--     $Log:	sysintf.c,v $
 * Revision 1.1  90/10/06  12:04:17  dvadura
 * dmake Release, Version 3.6
 * 
*/

#include <stdio.h>
#include "extern.h"
#include "sysintf.h"
#include "alloc.h"

/*
** Tries to stat the file name.  Returns 0 if the file
** does not exist.  Note that if lib is not null it tries to stat
** the name found inside lib.
**
** If member is NOT nil then look for the library object which defines the
** symbol given by name.  If found _strdup the name and return make the
** pointer pointed at by sym point at it.  Not handled for now!
*/
time_t
Do_stat(name, lib, member)
char *name;
char *lib;
char **member;
{
   struct stat buf;
   time_t seek_arch();

   if( member != NIL(char *) )
      Fatal("Library symbol names not supported");

   if( lib != NIL(char) )
      return( seek_arch(basename(name), lib) );
   else
      return( (STAT(name, &buf) == -1) ? (time_t)0 : buf.st_mtime );
}



/* Touch existing file to force modify time to present.
 */
int
Do_touch(name, lib, member)
char *name;
char *lib;
char **member;
{
   if( member != NIL(char *) )
      Fatal("Library symbol names not supported");

   if (lib != NIL(char))
      return( touch_arch(basename(name), lib) );
   else
      return( utime(name, NIL(time_t)) );
}



void
Void_lib_cache( lib_name, member_name )/*
=========================================
   Void the library cache for lib lib_name, and member member_name. */
char *lib_name;
char *member_name;
{
   VOID_LCACHE( lib_name, member_name );
}



/*
** return the current time
*/
time_t
Do_time()
{
   extern time_t time();
   return (time((time_t*)0));
}



/*
** Execute the string passed in as a command and return
** the return code. The command line arguments are
** assumed to be separated by spaces or tabs.  The first
** such argument is assumed to be the command.
**
** If group is true then this is a group of commands to be fed to the
** the shell as a single unit.  In this case cmd is of the form
** "file" indicating the file that should be read by the shell
** in order to execute the command group.
*/
int
Do_cmnd(cmd, group, do_it, target, how, ignore, shell, last)
char   *cmd;
int     group;
int	do_it;
CELLPTR target;
HOWPTR  how;
int     ignore;
int     shell;
int	last;
{
   int  i;

   if( !do_it ) {
      if( last && !Doing_bang ) Update_time_stamp( target, how );
      return(0);
   }

   if( Max_proc == 1 ) Wait_for_completion = TRUE;
   if( (i = runargv(target, how, ignore, group, last, shell, cmd)) == -1 )
      Quit();

   /* NOTE:  runargv must return either 0 or 1, 0 ==> command executed, and
    * we waited for it to return, 1 ==> command started and is running
    * concurrently with make process. */
   return(i);
}


#define MINARGV 64
/* Take a command and pack it into an argument vector to be executed. */
char **
Pack_argv( group, shell, cmd )
int    group;
int    shell;
char  *cmd;
{
   static char **av = NIL(char *);
   static int   avs = 0;
   int i = 0;

   if( av == NIL(char *) ) {
      TALLOC(av, MINARGV, char*);
      avs = MINARGV;
   }

   if( (Packed_shell = shell||group||(*_strpbrk(cmd, Shell_metas)!='\0')) ) {
      char* sh = group ? GShell : Shell;

      if( sh != NIL(char) ) {
         av[i++] = sh;
         if( (av[i] = (group?GShell_flags:Shell_flags)) != NIL(char) ) i++;

	 av[i++] = cmd;
	 av[i]   = NIL(char);
      }
      else
	 Fatal("%sSHELL macro not defined", group?"GROUP":"");
   }
   else {
      do {
         while( iswhite(*cmd) ) ++cmd;
         if( *cmd ) av[i++] = cmd;

         while( *cmd != '\0' && !iswhite(*cmd) ) ++cmd;
         if( *cmd ) *cmd++ = '\0';

	 if( i == avs ) {
	    avs += MINARGV;
	    av = (char **) realloc( av, avs*sizeof(char *) );
	 }
      } while( *cmd );

      av[i] = NIL(char);
   }

   return(av);
}


/*
** Return the value of ename from the environment
** if ename is not defined in the environment then
** NIL(char) should be returned
*/
char *
Read_env_string(ename)
char *ename;
{
   extern char *getenv();
   return( getenv(ename) );
}



/*
** Set the value of the environment string ename to value.
**  Returns 0 if success, non-zero if failure
*/
int
Write_env_string(ename, value)
char *ename;
char *value;
{
   extern int putenv();
   char*   p;
   char*   envstr = _stradd(ename, value, FALSE);

   p = envstr+strlen(ename);	/* Don't change this code, _stradd does not */
   *p++ = '=';			/* add the space if *value is 0, it does    */
   if( !*value ) *p = '\0';	/* allocate enough memory for one though.   */

   return( putenv(envstr) );
}



void
ReadEnvironment()
{
   extern char **Rule_tab;
   extern char **environ;
   char **rsave;

   rsave    = Rule_tab;
   Rule_tab = environ;
   Readenv  = TRUE;

   Parse( NIL(FILE) );

   Readenv  = FALSE;
   Rule_tab = rsave;
}



/*
** All we have to catch is SIG_INT
*/
void
Catch_signals(fn)
void (*fn)();
{
   if( signal(SIGINT, SIG_IGN) != SIG_IGN )
      signal( SIGINT, fn );
   if( signal(SIGQUIT, SIG_IGN) != SIG_IGN )
      signal( SIGQUIT, fn );
}



/*
** Clear any previously set signals
*/
void
Clear_signals()
{
   if( signal(SIGINT, SIG_IGN) != SIG_IGN )
      signal( SIGINT, SIG_DFL );
   if( signal(SIGQUIT, SIG_IGN) != SIG_IGN )
      signal( SIGQUIT, SIG_DFL );
}



/*
** Set program name
*/
void
Prolog(argc, argv)
int   argc;
char* argv[];
{
   Pname = (argc == 0) ? DEF_MAKE_PNAME : argv[0];
   Root_how.hw_files = NIL(FILELIST);
}



/*
** Do any clean up for exit.
*/
void
Epilog(ret_code)
int ret_code;
{
   Unlink_temp_files(&Root_how);
   exit( ret_code );
}



/*
** Use the built-in functions of the operating system to get the current
** working directory.
*/
char *
Get_current_dir()
{
   static char buf[MAX_PATH_LEN+1];

   return( getcwd(buf, sizeof(buf)) );
}



/*
** change working directory
*/
int
Set_dir(path)
char*   path;
{
   return( chdir(path) );
}



/*
** return switch char
*/
char
Get_switch_char()
{
   return( getswitchar() );
}



/*
** Generate a temporary file name and open the file for writing.
** If a name cannot be generated or the file cannot be opened
** return -1, else return the fileno of the open file.
** and update the source file pointer to point at the new file name.
** Note that the new name should be freed when the file is removed.
*/
FILE*
Open_temp(path, suff)
char **path;
char *suff;
{
   extern char *tempnam();

   *path = _strjoin( tempnam(NIL(char), "mk"), suff, -1, TRUE );
   Def_macro( "TMPFILE", *path, M_MULTI|M_EXPANDED );

   return( fopen(*path, "w") );
}


/*
** Open a new temporary file and set it up for writing.
*/
FILE *
Start_temp( suffix, cp, how, fname )
char     *suffix;
CELLPTR   cp;
HOWPTR    how;
char    **fname;
{
   FILE	       *fp;
   char        *tmpname;
   char	       *name;
   FILELISTPTR new;

   name = (cp != NIL(CELL))?cp->CE_NAME:"makefile text";
   if( how == NIL(HOW) ) how = &Root_how;

   if( (fp = Open_temp( &tmpname, suffix)) == NIL(FILE) )
      Fatal("Cannot open temp file `%s' while processing `%s'", tmpname, name );

   TALLOC( new, 1, FILELIST );

   new->fl_next = how->hw_files;
   new->fl_name = tmpname;
   new->fl_file = fp;		/* indicates temp file is open */

   how->hw_files = new;
   *fname = tmpname;

   return( fp );
}


/*
** Close a previously used temporary file.
*/
void
Close_temp(how, file)
HOWPTR  how;
FILE    *file;
{
   FILELISTPTR fl;
   if( how == NIL(HOW) ) how = &Root_how;

   for( fl=how->hw_files; fl && fl->fl_file != file; fl=fl->fl_next );
   if( fl ) {
      fl->fl_file = NIL(FILE);
      fclose(file);
   }
}


/*
** Clean-up, and close all temporary files associated with a target.
*/
void
Unlink_temp_files( how )/*
==========================
   Unlink the tempfiles if any exist.  Make sure you close the files first
   though.  This ensures that under DOS there is no disk space lost. */
HOWPTR how;
{
   FILELISTPTR next;

   while( how->hw_files != NIL(FILELIST) ) {
      if( how->hw_files->fl_file ) fclose( how->hw_files->fl_file );

      if( Verbose )
         printf( "%s:  Left temp file [%s]\n", Pname, how->hw_files->fl_name );
      else
         (void) unlink( how->hw_files->fl_name );

      FREE( how->hw_files->fl_name );
      next = how->hw_files->fl_next;
      FREE(how->hw_files);
      how->hw_files = next;
   }
}


void
Handle_result(status, ignore, abort_flg, target)
int	status;
int	ignore;
int	abort_flg;
CELLPTR target;
{
   status = ((status&0xff)==0 ? status>>8
	    : (status & 0xff)==SIGTERM ? -1
	    : (status & 0x7f)+128);

   if( status )
      if( !abort_flg ) {
	 fprintf( stderr, "%s:  Error code %d, while making '%s'",
		  Pname, status, target->ce_fname );

	 if( ignore || Continue ) {
	    fputs( " (Ignored)\n", stderr );
	 }
	 else {
	    fputc( '\n', stderr );

	    if( !(target->ce_attr & A_PRECIOUS) )
	       if( unlink( target->ce_fname ) == 0 )
		  fprintf(stderr,"%s:  '%s' removed.\n",Pname,target->ce_fname);

	    Quit();
	 }
      }
      else if( !(target->ce_attr & A_PRECIOUS) )
	 unlink( target->ce_fname );
}


void
Update_time_stamp( cp, how )
CELLPTR cp;
HOWPTR  how;
{
   HASHPTR hp;
   CELLPTR tcp;
   int     tmpflg; 

   how->hw_flag |= F_MADE;
   Unlink_temp_files( how );

   /* do the time only at end, of MULTI recipe targets, or immediately
    * for non-MULTI recipe targets. */
   tmpflg = cp->ce_flag & F_MULTI;

   if( (tmpflg && cp->CE_HOW == how) || !tmpflg ) {
      tcp = cp;
      do {
	 if( tcp->ce_attr & A_LIBRARY )
	    Void_lib_cache( tcp->ce_fname, NIL(char) );
	 else if( !Touch && (tcp->ce_attr & A_LIBRARYM) )
	    Void_lib_cache( tcp->ce_lib, tcp->ce_fname );

	 if( Trace ) {
	    tcp->ce_time  = Do_time();
	    tcp->ce_flag |= F_STAT;		/* pretend we stated ok */

	    if( tcp->ce_fname == NIL(char) )
	       tcp->ce_fname = tcp->CE_NAME;
	 }
	 else {
	    Stat_target( tcp, TRUE );
	    if( tcp->ce_time == (time_t) 0L )
	       tcp->ce_time = Do_time();
	 }

	 if( Verbose )
	    printf( "%s:  <<<< Set [%s] time stamp to %ld\n",
		    Pname, tcp->CE_NAME, tcp->ce_time );

	 tcp->ce_flag |= F_MADE;
	 tcp = tcp->ce_all;
      }
      while( tcp != NIL(CELL) && tcp != cp );
   }
   else if( tmpflg )
      cp->ce_flag |= F_STAT;


   /* Scan the list of prerequisites and if we find one that is
    * marked as being removable, (ie. an inferred intermediate node
    * then remove it.  We remove a prerequisite by running the recipe
    * associated with the special target .REMOVE, with $< set to
    * the list of prerequisites to remove. */

   if( (hp = Get_name( ".REMOVE", Defs, FALSE, NIL(CELL) )) != NIL(HASH) ) {
      register LINKPTR dp;
      int flag = FALSE;
      int rem;
      t_attr attr;

      tcp = hp->CP_OWNR;
      tcp->ce_flag |= F_TARGET;
      Clear_prerequisites( tcp->CE_HOW );

      for( dp = how->hw_prq; dp != NIL(LINK); dp = dp->cl_next ) {
	 register CELLPTR prq = dp->cl_prq;

	 attr = Glob_attr | prq->ce_attr;
	 rem  = (prq->ce_flag & F_REMOVE) &&
		(prq->ce_flag & F_MADE  ) &&
		!(attr & A_PRECIOUS) &&
		!Force;

	 if( rem ) {
	    CELLPTR tmp = prq;
	    do {
	       (Add_prerequisite(tcp->CE_HOW, prq, FALSE))->cl_flag |= F_TARGET;
	       prq->ce_flag &= ~F_REMOVE;
	       prq = prq->ce_all;
	    }
	    while( prq != NIL(CELL) && prq != tmp );
	    flag = TRUE;
	 }
      }

      if( flag ) Remove_prq( tcp );
   }
}
