/*
 * calc - double precision calculator
 *
 * If called with arguments, calc takes the arguments as a single
 * expression, and evaluates them.  Otherwise, it prompts for input,
 * reads and processes a line at a time, and exits when EOF is
 * received.  
 *
 * Commands are regular algebraic expressions consisting of numbers,
 * the operators +, -, *, /, % (modulus), and ^ (exponentiation), and
 * the functions sin, cos, tan, asin, acos, atan, ln, log, exp, and sqrt.
 *
 * In addition, the letters a-z represent register names which may be
 * assigned to and used in expressions.  The special register "scale"
 * represents the number of decimal places to be printed in results.
 * The constant names "@e" and "@pi" are the constants e and pi 
 * respectively.
 *
 * David A. Curry, 3/13/84
 */
%{

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <math.h>

char *ip;				/* present input character	*/
int scale = 5;				/* no. of decimal places	*/
short cmdline = 0;			/* 1 if args given		*/
double registers[26];			/* register variables		*/
char inbuffer[BUFSIZ];			/* input buffer			*/

double e = 2.7182818284590452353602874;		/* e			*/
double pi = 3.1415926535897932384626433;	/* pi			*/

extern int errno;

%}

/*
 * Start with the lines nonterminal.
 */
%start lines

/*
 * The parser deals with integers for miscellaneous tokens and register
 * references, doubles for numbers, and a pointer to a function for
 * function calls.
 */
%union {
	int ival;
	double dval;
	double (*func)();
}

/*
 * Expressions are doubles.
 */
%type <dval> expr

/*
 * NL - newline
 * HELP - help command
 * SCALE - scale variable
 * EOFILE - end of file
 * REGISTER - register reference
 * FUNCTION - function name
 * NUMBER - constant
 */
%token <ival> NL HELP SCALE EOFILE
%token <ival> REGISTER
%token <func> FUNCTION
%token <dval> NUMBER

%left '+' '-'
%left '*' '/' '%'
%left '^'
%left UMINUS

%%

lines	:
	|	lines line
	;

line	:	EOFILE
			{ if (!cmdline)
				putchar('\n');
			  exit(0);
			}
	|	expr NL
			{ if (errno == ERANGE)
				yyerror("range error");
			  else if (errno == EDOM)
			  	yyerror("domain error");
			  else
			  	printf("%.*f\n", scale, $1);
			  errno = 0;
			}
	|	SCALE '=' expr NL
			{ scale = (int) $3; }
	|	REGISTER '=' expr NL
			{ registers[$1] = $3; }
	|	SCALE NL
			{ printf("Scale presently set to %d digits.\n", scale); }
	|	HELP NL
			{ help(); }
	|	error 
			{ *ip = NULL; yyclearin; yyerrok; }
	;

expr 	:	expr '+' expr
			{ $$ = $1 + $3; }
	|	expr '-' expr
			{ $$ = $1 - $3; }
	|	expr '*' expr
			{ $$ = $1 * $3; }
	|	expr '/' expr
			{ if ($3 == 0) {
				errno = ERANGE;
				$$ = 1;
			  }
			  else {
			  	$$ = $1 / $3;
			  }
			}
	|	expr '%' expr
			{ $$ = (int) $1 % (int) $3; }
	|	expr '^' expr
			{ $$ = pow($1, $3); }
	|	FUNCTION '(' expr ')'
			{ $$ = (*$1)($3); }
	|	'(' expr ')'
			{ $$ = $2; }
	|	'-' expr		%prec UMINUS
			{ $$ = - $2; }
	|	REGISTER
			{ $$ = registers[$1]; }
	|	NUMBER
			{ $$ = $1; }
	;
%%

main(argc, argv)
int argc;
char **argv;
{
	register i;
	register char *s;
	
#ifdef YYDEBUG
	/*
	 * Turn on debugging.
	 */
	extern int yydebug;
	yydebug = 1;
#endif

	ip = inbuffer;
	*ip = NULL;
	
	/*
	 * Initialize registers.
	 */
	for (i=0; i < 26; i++)
		registers[i] = 0.0;

	/*
	 * If we have arguments, stick them into the
	 * input buffer.
	 */
	if (argc > 1) {
		cmdline = 1;
		
		while (--argc)
			strcat(inbuffer, *++argv);

		s = inbuffer;
		
		while (*s) {
			if (isupper(*s))
				*s = tolower(*s);
			s++;
		}

		*s++ = '\n';
		*s = NULL;
	}
	
	/*
	 * We exit in yyparse.
	 */
	for (;;) {
		yyparse();
	}
}

/*
 * yylex - lexical analysis routine
 */
yylex()
{
	double frac;
	register char *s;
	
	/*
	 * If the present input character is NULL, 
	 * return EOFILE if we had arguments, otherwise,
	 * prompt for another line of input.
	 */
	if (*ip == NULL) {
		if (cmdline) {
			return(EOFILE);
		}
		else {
			printf("-> ");
			
			if (gets(inbuffer) == NULL)
				return(EOFILE);
			
			s = inbuffer;

			while (*s) {
				if (isupper(*s))
					*s = tolower(*s);
				s++;
			}

			*s++ = '\n';
			*s = NULL;
			
			ip = inbuffer;
		}
	}
	
	for (;;) {
		/*
		 * Should never see this, but just in case.
		 */
		if (*ip == EOF) {
			return(EOFILE);
		}

		/*
		 * Skip whitespace.
		 */
		if ((*ip == ' ') || (*ip == '\t')) {
			ip++;
			continue;
		}

		/*
		 * Newlines or semicolons both count as newlines.
		 */
		if ((*ip == '\n') || (*ip == ';') || (*ip == NULL)) {
			ip++;
			return(NL);
		}
		
		/*
		 * If it's a letter...
		 */
		if (isalpha(*ip)) {
			/*
			 * Check for function calls.  If we find one,
			 * set function pointer to the function, skip
			 * over the function name in the input buffer.
			 */
			if (!strncmp(ip, "asin", 4) && !isalpha(*(ip+4))) {
				yylval.func = asin;
				ip = ip + 4;
				return(FUNCTION);
			}
			if (!strncmp(ip, "acos", 4) && !isalpha(*(ip+4))) {
				yylval.func = acos;
				ip = ip + 4;
				return(FUNCTION);
			}
			if (!strncmp(ip, "atan", 4) && !isalpha(*(ip+4))) {
				yylval.func = atan;
				ip = ip + 4;
				return(FUNCTION);
			}
			if (!strncmp(ip, "sqrt", 4) && !isalpha(*(ip+4))) {
				yylval.func = sqrt;
				ip = ip + 4;
				return(FUNCTION);
			}
			if (!strncmp(ip, "sin", 3) && !isalpha(*(ip+3))) {
				yylval.func = sin;
				ip = ip + 3;
				return(FUNCTION);
			}
			if (!strncmp(ip, "cos", 3) && !isalpha(*(ip+3))) {
				yylval.func = cos;
				ip = ip + 3;
				return(FUNCTION);
			}
			if (!strncmp(ip, "tan", 3) && !isalpha(*(ip+3))) {
				yylval.func = tan;
				ip = ip + 3;
				return(FUNCTION);
			}
			if (!strncmp(ip, "exp", 3) && !isalpha(*(ip+3))) {
				yylval.func = exp;
				ip = ip + 3;
				return(FUNCTION);
			}
			if (!strncmp(ip, "log", 3) && !isalpha(*(ip+3))) {
				yylval.func = log10;
				ip = ip + 3;
				return(FUNCTION);
			}
			if (!strncmp(ip, "ln", 2) && !isalpha(*(ip+2))) {
				yylval.func = log;
				ip = ip + 2;
				return(FUNCTION);
			}

			/*
			 * Check for the scale variable.
			 */
			if (!strncmp(ip, "scale", 5) && !isalpha(*(ip+5))) {
				ip = ip + 5;
				return(SCALE);
			}
			
			/*
			 * Check for the help command.
			 */
			if (!strncmp(ip, "help", 4) && !isalpha(*(ip+4))) {
				ip = ip + 4;
				return(HELP);
			}
			
			/*
			 * If we make it this far, we must have a
			 * register reference.
			 */
			yylval.ival = *ip++ - 'a';
			return(REGISTER);
		}
		
		/*
		 * Constant.
		 */
		if (*ip == '@') {
			ip++;
			
			if (!strncmp(ip, "pi", 2) && !isalpha(*(ip+2))) {
				ip = ip + 2;
				yylval.dval = pi;
				return(NUMBER);
			}
			
			if ((*ip == 'e') && !isalpha(*(ip+1))) {
				ip++;
				yylval.dval = e;
				return(NUMBER);
			}
			
			yylval.ival = '@';
			return('@');		/* cause parser error */
		}
		
		/*
		 * Operator.
		 */
		if ((*ip == '+') || (*ip == '-') || (*ip == '*') ||
		    (*ip == '/') || (*ip == '%') || (*ip == '^') ||
		    (*ip == '(') || (*ip == ')') || (*ip == '=')) {
		        yylval.ival = *ip;
		    	return(*ip++);
	    	}

		/*
		 * Number.
		 */
		if (isdigit(*ip) || (*ip == '.')) {
			yylval.dval = 0.0;
			frac = 10.0;
			
			while (isdigit(*ip)) {
				yylval.dval = yylval.dval * 10.0 + *ip - '0';
				ip++;
			}
			
			if (*ip == '.') {
				ip++;
				
				while (isdigit(*ip)) {
					yylval.dval = yylval.dval + (*ip - '0') / frac;
					frac = frac * 10.0;
					ip++;
				}
			}
			
			return(NUMBER);
		}

		/*
		 * Produce a parser error.
		 */
		yylval.ival = '@';
		return('@');			/* cause parser error */
	}
}

/*
 *yyerror - print error message.
 */
yyerror(s)
char *s;
{
	fprintf(stderr, "%s\n", s);
}

/*
 * help - print a help message.
 */
help()
{
	printf("\nOperators:\n");
	printf("\t+  -  *  /  %  ^\n");
	printf("\nFunctions:\n");
	printf("\tsin, cos, tan, asin, acos, atan, ");
	printf("exp, ln, log, sqrt\n");
	printf("\nConstants:\n");
	printf("\t@e, @pi\n");
	printf("\nRegisters:\n");
	printf("\ta-z, scale\n");
	printf("\nCommands may be separated by newlines or semicolons.\n");
	printf("\n");
}
