/*
 * Update the display of the line being edited on the screen.
 * Copyright (C) 1989 by Kenneth Almquist.
 */


#include "attyed.h"
#include "ed.h"
#include <ctype.h>

#ifndef NULL
#define NULL 0
#endif


struct displine {
      char *text;		/* text of line */
      int len;			/* length of line */
};


int scrnwidth;			/* width of screen */
int eddisplay;			/* if nonzero, editor display is valid */
int needbeep;			/* if nonzero, beep on next refresh */
struct displine virtscrn[60];	/* what the screen should look like */
struct displine realscrn[60];	/* what the screen does look like */
int vslines;			/* number of lines on virtual screen */
int rslines;			/* number of lines on real screen */
int virtline;			/* cursor position in virtscrn */
int virtcol;			/* cursor position in virtscrn */
char *virtptr;			/* &virtscrn[virtline].text[virtcol] */
int *virtlen;			/* &virtscrn[virtline].len */
int realline;			/* position of cursor on real screen */
int realcol;			/* position of cursor on real screen */
int lastrealline;		/* last real line on the screen */
int startline;			/* end of prompt */
int startcol;			/* end of prompt */
char tcspace[120];		/* strings from termcap are stored here */
int tc_am;			/* cursor wraps around at right of screen */
char *tc_bc;			/* backspace */
char *tc_bl;			/* bell */
int tc_bs;			/* if set, terminal can backspace */
char *tc_ce;			/* clear to end of line */
int tc_co;			/* default width of screen */
char *tc_do;			/* cursor down one line */
char *tc_nc;			/* cursor right on column */
char *tc_up;			/* cursor up one line */
int tc_xn;			/* a newline is ignored after wrap around */
char PC;			/* pad character (for termcap routines) */
extern int errno;


#ifdef __STDC__
void virtgoto(int, int);
void drawch(int);
void realgoto(int, int);
void copytoreal(int, int);
void allocline(struct displine *);
#else
void virtgoto();
void drawch();
void realgoto();
void copytoreal();
void allocline();
#endif

#ifdef __STDC__
void ttyoutc(int);
char *getenv(char *);
int tgetent(char *, char *);
int tgetnum(char *);
int tgetflag(char *);
char *tgetstr(char *, char **);
void tputs(char *, int, void (*)(int));
char *malloc(unsigned);
#else
void ttyoutc();
char *getenv();
int tgetent();
int tgetnum();
int tgetflag();
char *tgetstr();
void tputs();
char *malloc();
#endif



/*
 * Read the termcap entry for this terminal.
 */

void
gettermcap() {
      char *termtype;
      char buffer[1024];
      char *tcp = tcspace;
      char *tc_pc;

      errno = 0;
      if ((termtype = getenv("TERM")) == NULL)
	    badinit("$TERM not set");
      if (tgetent(buffer, termtype) <= 0)
	    badinit("Termcap entry not found");
      tc_am = tgetflag("am");
      tc_bs = tgetflag("bs");
      tc_bc = tgetstr("bc", &tcp);
      tc_bl = tgetstr("bl", &tcp);
      tc_ce = tgetstr("ce", &tcp);
      tc_co = tgetnum("co");
      tc_do = tgetstr("do", &tcp);
      tc_nc = tgetstr("nc", &tcp);
      tc_pc = tgetstr("pc", &tcp);
      tc_up = tgetstr("up", &tcp);
      tc_xn = tgetflag("xn");
      PC = (tc_pc != NULL)? *tc_pc : '\0';
      if (tc_up == NULL)
	    badinit("Atty can only run a terminal with a cursor-up escape sequence");
      scrnwidth = tc_co;
}



void
newscrnwidth(width) {
      struct displine *lp;
      int disp = eddisplay;

      if (width <= 1)
	    width = tc_co;
      if (width == scrnwidth)
	    return;
      if (disp)
	    dispoff(column);
      for (lp = virtscrn ; lp->text ; lp++) {
	    free(lp->text);
	    lp->text = NULL;
      }
      vslines = 0;
      for (lp = realscrn ; lp->text ; lp++) {
	    free(lp->text);
	    lp->text = NULL;
      }
      scrnwidth = width;
      if (disp)
	    refresh();
}



void
refresh() {
      struct displine *lp;
      char *p;
      int pointline, pointcol;

      if (eddisplay == 0) {
	    if (realscrn[0].text == NULL)
		  allocline(&realscrn[0]);
	    if (virtscrn[0].text == NULL)
		  allocline(&virtscrn[0]);
	    for (lp = realscrn ; lp->text != NULL ; lp++)
		  lp->len = 0;
	    startcol = column % scrnwidth;
	    startline = 0;
	    realcol = startcol;
	    realline = 0;
	    if (column != 0) {
		  bcopy(outline, realscrn[0].text, realcol);
		  realscrn[0].len = realcol;
		  bcopy(outline, virtscrn[0].text, realcol);
		  virtscrn[0].len = realcol;
		  if (promptset) {
			startcol = 0;
			startline = 1;
		  }
	    }
	    if (promptset) {
		  virtgoto(startline, startcol);
		  for (p = prompt ; *p ; p++)
			drawch(*p);
		  startline = virtline;
		  startcol = virtcol;
	    }
	    lastrealline = 0;
	    eddisplay = 1;
      }
      /* erase everything following the prompt from the virtual screen */
      virtgoto(startline, startcol);
      lp = &virtscrn[startline];
      lp->len = startcol;
      while ((++lp)->text != NULL)
	    lp->len = 0;
      if (editprompt)
	    drawch(editprompt);
      pointcol = -1;
      if (ttymode.echo) {
	    for (p = curline ; p < endcurline ; p++) {
		  if (p == point) {
			pointline = virtline;
			pointcol = virtcol;
		  }
		  drawch(*p);
	    }
      }
      if (pointcol < 0) {
	    pointline = virtline;
	    pointcol = virtcol;
      }
      copytoreal(pointline, pointcol);
}


/*
 * Copy the virtual screen to the real screen.
 */

void
copytoreal(pointline, pointcol) {
      char *p;
      char *r;
      int line;
      int len;
      int limit;
      int col;

      for (line = 0 ; line <= virtline ; line++) {
	    len = virtscrn[line].len;
	    p = virtscrn[line].text;
	    limit = realscrn[line].len;
	    r = realscrn[line].text;
	    if (r == NULL) {
		  allocline(&realscrn[line]);
		  r = realscrn[line].text;
	    }
	    if (limit > len)
		  limit = len;
	    for (col = 0 ; col < limit ; col++) {
		  if (*p != *r) {
			if (realline != line || realcol != col)
			      realgoto(line, col);
			ttyoutc(*p);
			realcol++;
			*r = *p;
		  }
		  p++;
		  r++;
	    }
	    for ( ; col < len ; col++) {
		  if (*p != ' ') {
			if (realline != line || realcol != col)
			      realgoto(line, col);
			ttyoutc(*p);
			realcol++;
		  }
		  *r++ = *p++;
	    }
	    limit = realscrn[line].len;
	    if (col < limit) {
		  if (tc_ce == NULL || limit - col == 1) {
			for ( ; col < limit ; col++) {
			      if (*r++ != ' ') {
				    if (realline != line || realcol != col)
					  realgoto(line, col);
				    ttyoutc(' ');
				    realcol++;
			      }
			}
		  } else {
			realgoto(line, col);
			tputs(tc_ce, 1, ttyoutc);
		  }
	    }
	    if (realcol >= scrnwidth) {
		  if (tc_xn || ! tc_am)
			outreal("\r\n", 2);
		  realcol = 0;
		  realline++;
	    }
	    realscrn[line].len = virtscrn[line].len;
      }
      while (line <= lastrealline) {
	    if (realscrn[line].len > 0) {
		  if (tc_ce) {
			realgoto(line, 0);
			tputs(tc_ce, 1, ttyoutc);
		  } else {
			r = realscrn[line].text;
			limit = realscrn[line].len;
			for (col = 0 ; col < limit ; col++) {
			      if (*r++ != ' ') {
				    if (realline != line || realcol != col)
					  realgoto(line, col);
				    ttyoutc(' ');
				    realcol++;
			      }
			}
			if (realcol >= scrnwidth) {
			      if (tc_xn || ! tc_am)
				    outreal("\r\n", 2);
			      realcol = 0;
			      realline++;
			}
		  }
		  realscrn[line].len = 0;
	    }
	    line++;
      }
      realgoto(pointline, pointcol);
      if (needbeep) {
	    if (tc_bl)
		  tputs(tc_bl, 1, ttyoutc);
	    needbeep = 0;
      }
      lastrealline = virtline;
}



/*
 * Move to beyond the end of the edited stuff, and turn the display off.
 */

void
movetoend() {
      realgoto(lastrealline, 0);
      ttyoutc('\n');
      eddisplay = 0;
      column = 0;
      promptset = 0;
}


/*
 * Output the line being edited to the display, and freeze it there.
 * If the data flag is zero, only freeze the prompt.
 */

void
freezedisp(data) {
      int i;
      struct displine *lp;
      int save_echo;

      save_echo = ttymode.echo;
      ttymode.echo = data;
      refresh();
      ttymode.echo = save_echo;
      eddisplay = 0;
      column = realcol;
      lp = &realscrn[realline];
      for (i = 0 ; i < column ; i++) {
	    outline[i] = (i < lp->len)? lp->text[i] : ' ';
      }
      promptset = 0;
}



/*
 * Turn display off.
 */

void
dispoff(col) {
      int line;

      if (eddisplay == 0)
	    return;
      col = col % scrnwidth;
      virtscrn[0].len = col;
      for (line = 1 ; line <= lastrealline ; line++)
	    virtscrn[line].len = 0;
      copytoreal(0, col);
      eddisplay = 0;
}



void
virtgoto(line, col) {
      struct displine *lp;
      char *p;

      while (vslines <= line) {
	    allocline(&virtscrn[vslines]);
	    vslines++;
      }
      lp = &virtscrn[line];
      if (lp->len < col) {
	    p = &lp->text[lp->len];
	    do *p++ = ' ';
	    while (++lp->len < col);
      }
      virtline = line;
      virtcol = col;
      virtptr = &lp->text[col];
      virtlen = &lp->len;
}


void
drawch(c)
      char c;
      {
      if (! isprint(c)) {
	    if (c == '\t') {
		  do drawch(' ');
		  while ((virtcol & 07) != 0);
		  return;
	    } else {
		  drawch('^');
		  c ^= 0100;
	    }
      }
      *virtptr++ = c;
      virtcol++;
      if (*virtlen < virtcol)
	    *virtlen = virtcol;
      if (virtcol >= scrnwidth) {
	    virtgoto(virtline + 1, 0);
      }
}


/*
 * Goto the specified location on the real screen.  Note that we may
 * scroll the screen, so we must use line feed rather than do for
 * downward motion.
 */

void
realgoto(line, col) {
      char *text;
      int len;

      while (realline > 0 && realline > line) {
	    tputs(tc_up, 1, ttyoutc);
	    realline--;
      }
      while (realline < line) {
#ifdef notdef
	    tputs(tc_do, 1, ttyoutc);
#else
	    ttyoutc('\n');
#endif
	    realline++;
      }
      if (col < realcol - col) {
	    ttyoutc('\r');
	    realcol = 0;
      }
      if (realcol > col) {
	    if (tc_bc) {
		  do {
			tputs(tc_bc, 1, ttyoutc);
		  } while (--realcol > col);
	    } else {
		  do {
			ttyoutc('\b');
		  } while (--realcol > col);
	    }
      }
      if (realcol < col) {
	    text = realscrn[realline].text;
	    len = realscrn[realline].len;
	    do {
		  ttyoutc(realcol < len? text[realcol] : ' ');
	    } while (++realcol < col);
      }
      if (line == 0 && realline > 0) {
	    tputs(tc_up, 1, ttyoutc);
	    realline--;
      }
}



void
allocline(lp)
      struct displine *lp;
      {
      char *p;

      if (lp->text != NULL)
	    return;
      if ((p = malloc(scrnwidth)) == NULL)
	    fatal("Malloc line failed");
      lp->text = p;
      lp->len = 0;
}
