/*
 * The kbind program - compile .bind files.
 *
 * Copyright (C) 1989 by Kenneth Almquist.  All rights reserved.
 * This file is part of atty, which is distributed under the terms specified
 * by the Atty General Public License.  See the file named LICENSE.
 */

#include <stdio.h>
#include "bind.h"


#define VERSION 21	/* change if you change the .bindc format */


/*
 * List of commands.
 */

char *command[] = {
	"universal-argument",
	"digit-argument",
	"negative-argument",
	"self-insert",
	"quoted-insert",
	"newline",
	"end-of-file",
	"eof-or-delete-char",
	"newline-and-insert",
	"tty-intr",
	"tty-quit",
	"tty-susp",
	"delete-backward-char",
	"delete-char",
	"kill-word",
	"backward-kill-word",
	"kill-region",
	"kill-line",
	"kill-input",
	"copy-region-as-kill",
	"forward-char",
	"backward-char",
	"forward-word",
	"backward-word",
	"beginning-of-line",
	"end-of-line",
	"yank",
	"set-mark",
	"exchange-point-and-mark",
	"upcase-char",
	"upcase-word",
	"upcase-region",
	"gosling-transpose-chars",
	"transpose-chars",
	"transpose-words",
	"next-history",
	"previous-history",
	"beginning-of-history",
	"end-of-history",
	"re-search-forward",
	"re-search-backward",
	"file-complete",
	"list-file-completions",
	"get-history-word",
	"last-output-line",
	"mode-0",
	"mode-1",
	0
};



/* Token types */
#define TWORD	1	/* generic word */
#define TNL	2	/* newline */
#define TSTRING	3	/* quoted string */


#define C_UNSET 127	/* command is not set */


#define NMODES 2	/* maximum number of modes */
#define TBLSIZE 128	/* size of a key table */
#define MAXTBL 32	/* maximum number of key tables */
#define STRINGSPACE 5000 /* space for holding strings */
#define MAXABBREV 500	/* maximum number of abbreviations */


struct cmd {
      unsigned char type;	/* type of command */
      unsigned char index;	/* which command */
      int linno;		/* line where defined */
};


struct mode {
      int used;			/* set if mode is in use */
      struct cmd dft;		/* default binding */
      struct cmd *keytab;	/* base key table for this mode */
};


struct syntax {
      char *name;		/* name of this syntax class */
      int defined;		/* set if syntax defined */
      char bitmap[16];		/* bit map specifying which characters are
				   in the class */
};


struct abbrev {
      int fromlen;		/* length of "from" */
      char *from;		/* the abbreviation */
      int tolen;		/* length of "to" */
      char *to;			/* what it is defined as */
};


char *inputfile;		/* name of input file */
FILE *infp;			/* input file */
char outputfile[1024];		/* name of output file */
FILE *outfp;			/* output file */
struct mode mode[NMODES];	/* modes */
struct mode *curmode;		/* current mode */
int nmodes;			/* number of modes actually used */
struct cmd *keytab[MAXTBL];	/* key binding tables */
int nkeytab;			/* number of key tables */
struct syntax syntax[NSYNTAX];	/* syntax tables */
char stringspace[STRINGSPACE];	/* space for holding strings */
char *strnext;			/* next entry in string table */
int nstrings;			/* number of strings in string table */
int nabbrev;			/* number of abbreviations */
struct abbrev abbrev[MAXABBREV];/* abbreviations */
int abbrevsize;			/* total size of abbreviation table */
int mode_1_line;		/* line that mode-1 command occured on */
int lasttoken;			/* type of last token read */
int tokpushback;		/* set to push back token */
char toktext[1024];		/* text of last token read */
char tokstring[512];		/* input string */
char stringlen;			/* length of tokstring */
int linno;			/* current line of input */
int nerrors;			/* number of errors encountered */
int debug;			/* set for debugging info */


#ifdef __STDC__
void makefuncs(void);
char *cname(char *);
void parsefile(void);
void parseline(void);
void parsecmd(struct cmd *);
void setmode(int);
void bind(char *, int, struct cmd *);
struct cmd *newkeytab(void);
void defaultbind(void);
void setdftbind(struct cmd *, struct cmd *);
void setsyntax(char *, char *, int);
void chknl(void);
int gettoken(void);
int getstring(void);
void cvttext(int);
int ateof(void);
char *saveblock(char *, int);
void error(char *);
int writeout(void);
void putkeytab(struct cmd *);
void putsh(int, FILE *);
#else
void makefuncs();
char *cname();
void parsefile();
void parseline();
void parsecmd();
void setmode();
void bind();
struct cmd *newkeytab();
void defaultbind();
void setdftbind();
void setsyntax();
void chknl();
int gettoken();
int getstring();
void cvttext();
int ateof();
char *saveblock();
void error();
int writeout();
void putkeytab();
void putsh();
#endif

char *malloc();
#define equal(s1, s2)	(strcmp(s1, s2) == 0)



main(argc, argv)
      char **argv;
      {
      char **ap;
      int status;

      if (argc <= 1) {
	    fputs("Usage:  kbind file.bind ...\n", stderr);
	    exit(2);
      }
      if (argc == 2 && equal(argv[1], "-makefuncs")) {
	    makefuncs();
	    return 0;
      }
      if (equal(argv[1], "-d")) {
	    argv++;
	    debug = 1;
      }
      status = 0;
      for (ap = argv + 1 ; *ap ; ap++) {
	    inputfile = *ap;
	    if ((infp = fopen(inputfile, "r")) == NULL) {
		  fprintf(stderr, "Can't open %s\n", inputfile);
		  status = 2;
	    }
	    nerrors = 0;
	    parsefile();
	    if (nerrors != 0) {
		  if (status == 0)
			status = 1;
		  continue;
	    }
	    if (writeout() < 0)
		  status = 2;
      }
      exit(status);
}


/*
 * Generate function index to be compiled with atty.
 */

char head[] = "\
/*\n\
 * This file was created by the kbind program.\n\
 */\n\
\n";

void
makefuncs() {
      char **pp;

      if ((outfp = fopen("edfuncs.c", "w")) == NULL) {
	    fputs("Can't create edfuncs.c\n", stderr);
	    exit(2);
      }
      fputs(head, outfp);
      for (pp = command ; *pp ; pp++) {
	    fprintf(outfp, "int %s();\n", cname(*pp));
      }
      putc('\n', outfp);
      fputs("int (*edfunc[])() = {\n", outfp);
      for (pp = command ; *pp ; pp++) {
	    fprintf(outfp, "      %s,\n", cname(*pp));
      }
      fputs("};\n\n", outfp);
      fprintf(outfp, "int version = %d;\n", VERSION);
}


/*
 * Convert a function name to the way that it appears in C.
 */

char *
cname(s)
      char *s;
      {
      static char buf[128];
      char *p;

      strcpy(buf, s);
      for (p = buf ; *p ; p++) {
	    if (*p == '-')
		  *p = '_';
      }
      return buf;
}


void
parsefile() {
      int i;

      linno = 1;
      curmode = NULL;
      nkeytab = 0;
      nmodes = 0;
      strnext = stringspace;
      nstrings = 0;
      nabbrev = 0;
      syntax[0].name = "word";
      syntax[0].defined = 0;
      syntax[1].name = "filename";
      syntax[1].defined = 0;
      syntax[2].name = "abbrev";
      syntax[2].defined = 0;
      for (i = 0 ; i < NMODES ; i++) {
	    mode[i].used = 0;
	    mode[i].dft.type = C_UNSET;
	    mode[i]. keytab = NULL;
      }

      while (! ateof()) {
	    parseline();
      }
      for (i = 0 ; i < NSYNTAX ; i++) {
	    if (! syntax[i].defined)
		  setsyntax(syntax[i].bitmap, "^ \t", 3);
      }
      if (curmode == NULL) {
	    printf("%s:%d: No bindings specified\n", inputfile, linno);
	    nerrors++;
      } else if (mode[0].used == 0) {
	    printf("%s:%d: Mode zero not defined\n", inputfile, linno);
	    nerrors++;
      }
      if (mode_1_line != 0 && mode[1].used == 0) {
	    printf("%s:%d: Mode 1 referenced (line %d), but never defined\n",
		   inputfile, linno, mode_1_line);
	    nerrors++;
      }
      nmodes = 1;
      for (i = 1 ; i < NMODES ; i++) {
	    if (mode[i].used) {
		  if (! mode[i - 1].used) {
			printf("%s:%d: Mode %d is defined, but mode %d is not\n",
			      inputfile, linno, i, i - 1);
			nerrors++;
		  }
		  nmodes = i + 1;
	    }
      }
      defaultbind();
      abbrevsize = 0;
      for (i = 0 ; i < nabbrev ; i++) {
	    abbrevsize += abbrev[i].fromlen + abbrev[i].tolen + 2;
      }
}



void
parseline() {
      struct cmd cmd;
      char binding[128];
      int bindlen;
      int i;

      if (gettoken() == TNL)
	    goto out;
      if (lasttoken != TWORD) {
	    error("Expecting command");
	    goto out;
      }
      if (toktext[0] == '#') {
	    /* a comment, ignore it */
      } else if (equal(toktext, "b")) {
	    if (curmode == NULL) {
		  error("No mode specified");
		  setmode(0);
	    }
	    if (gettoken() != TWORD) {
		  error("Expecting key sequence for b command");
		  goto out;
	    }
	    cvttext(1);		/* parse and save value */
	    bindlen = stringlen;
	    bcopy(tokstring, binding, bindlen);
	    parsecmd(&cmd);
	    bind(binding, bindlen, &cmd);
	    chknl();
      } else if (equal(toktext, "mode")) {
	    if (gettoken() != TWORD || *toktext < '0'
	     || *toktext > '9' || toktext[1] != '\0') {
		  error("Expecting mode number");
		  goto out;
	    }
	    setmode(atoi(toktext));
	    chknl();
      } else if (equal(toktext, "default")) {
	    if (curmode == NULL) {
		  error("No mode specified");
		  setmode(0);
	    }
	    parsecmd(&cmd);
	    curmode->dft = cmd;
	    chknl();
      } else if (equal(toktext, "syntax")) {
	    if (gettoken() != TWORD) {
		  error("Expecting syntax class name");
		  goto out;
	    }
	    for (i = 0 ; ; i++) {
		  if (equal(toktext, syntax[i].name))
			break;
		  if (i == NSYNTAX - 1)
			error("Unimplemented syntax class");
	    }
	    if (syntax[i].defined)
		  error("Redefining syntax class");
	    if (getstring() < 0)
		  goto out;
	    cvttext(0);
	    setsyntax(syntax[i].bitmap, tokstring, stringlen);
	    syntax[i].defined = 1;
	    chknl();
      } else if (equal(toktext, "abbrev")) {
	    if (nabbrev >= MAXABBREV) {
		  error("Too many abbreviations");
		  nabbrev = 0;
	    }
	    if (getstring() < 0)
		  goto out;
	    cvttext(0);
	    if (stringlen == 0)
		  error("Can't define the null string to be an abbreviation");
	    abbrev[nabbrev].fromlen = stringlen;
	    abbrev[nabbrev].from = saveblock(tokstring, stringlen);
	    if (getstring() < 0)
		  goto out;
	    cvttext(0);
	    abbrev[nabbrev].tolen = stringlen;
	    abbrev[nabbrev].to = saveblock(tokstring, stringlen);
	    nabbrev++;
	    chknl();
      } else {
	    error("Unrecognized command");
      }
out:
      while (lasttoken != TNL)
	    gettoken();
}


void
parsecmd(cmdp)
      struct cmd *cmdp;
      {
      char **pp;

      cmdp->type = C_UNDEF;
      cmdp->linno = linno;
      if (gettoken() != TWORD) {
	    error("Expecting function name");	
	    return;
      }
      if (equal(toktext, "insert")) {
	    if (getstring() < 0)
		  return;
	    if (nstrings >= 255) {
		  error("Too many strings");
		  return;
	    }
	    cvttext(0);
	    if ((strnext - stringspace) + 1 + stringlen > STRINGSPACE) {
		  error("Out of space for string constants");
		  return;
	    }
	    *strnext++ = stringlen;
	    bcopy(tokstring, strnext, stringlen);
	    strnext += stringlen;
	    cmdp->type = C_INSERT;
	    cmdp->index = nstrings++;
      } else if (equal(toktext, "undefined")) {
	    cmdp->type = C_UNDEF;
	    cmdp->index = 0;
      } else {
	    for (pp = command ; *pp && ! equal(*pp, toktext) ; pp++);
	    if (*pp == NULL) {
		  error("Unrecognized function");
		  return;
	    }
	    if (equal(toktext, "mode-1"))
		  mode_1_line = linno;
	    cmdp->type = C_FUNC;
	    cmdp->index = pp - command;
      }
}


void
setmode(modenum) {
      if (modenum < 0 || modenum >= NMODES) {
	    error("Mode number out of range");
	    return;
      }
      curmode = &mode[modenum];
      curmode->used = 1;
      if (curmode->keytab == NULL)
	    curmode->keytab = newkeytab();
}


void
bind(keyseq, seqlen, cmdp)
      char *keyseq;
      int seqlen;
      struct cmd *cmdp;
      {
      struct cmd *k;
      struct cmd *cp;
      char *p;
      int lno;

      if (seqlen == 0) {			/* can't happen */
	    error("Empty binding sequence");
	    return;
      }
      if (curmode == NULL) {			/* can't happen */
	    error("No current mode");
	    setmode(0);
      }
      k = curmode->keytab;
      p = keyseq;
      while (--seqlen > 0) {
	    cp = &k[*p++ & 0177];
	    if (cp->type == C_UNSET) {
		  if (nkeytab >= MAXTBL) {
			error("Too many key tables required");
			return;
		  }
		  keytab[nkeytab++] = k = newkeytab();;
		  cp->type = C_PFXTBL;
		  cp->index = nkeytab - 1;
		  cp->linno = cmdp->linno;
	    } else if (cp->type == C_PFXTBL) {
		  k = keytab[cp->index];
	    } else {
		  goto redef;
	    }
      }
      cp = &k[*p & 0177];
      if (cp->type != C_UNSET) {
redef:
	    printf("%s:%d: Redefining key.  (Initial definition line %d)\n",
		   inputfile, cmdp->linno, cp->linno);
	    nerrors++;
	    return;
      }
      *cp = *cmdp;
}


/*
 * Create a new key table.
 */

struct cmd *
newkeytab() {
      struct cmd *keytab;
      struct cmd *p;

      if ((keytab = (struct cmd *)malloc(TBLSIZE * sizeof *keytab)) == NULL) {
	    error("Out of space");
	    exit(2);
      }
      p = keytab + TBLSIZE;
      do {
	    (--p)->type = C_UNSET;
      } while (p > keytab);
      return keytab;
}


/*
 * Set the default bindings.
 */

void
defaultbind() {
      int tabno;
      struct cmd *cp;
      int i;
      static struct cmd dft = {C_UNDEF, 0, 0};

      for (tabno = 0 ; tabno < nmodes ; tabno++) {
	    if (mode[tabno].dft.type == C_UNSET)
		  setdftbind(mode[tabno].keytab, &dft);
	    else
		  setdftbind(mode[tabno].keytab, &mode[tabno].dft);
      }
      for (tabno = 0 ; tabno < nkeytab ; tabno++) {
	    setdftbind(keytab[tabno], &dft);
      }
}



void
setdftbind(tab, dft)
      struct cmd *tab;
      struct cmd *dft;
      {
      struct cmd *cp;
      int i;

      for (i = 128, cp = tab ; --i >= 0 ; cp++) {
	    if (cp->type == C_UNSET)
		  *cp = *dft;
      }
}



/*
 * Set up a syntax table.
 */

void
setsyntax(table, string, len)
      char table[16];
      char *string;
      {
      register char *p;
      int c;
      int invert;

      for (p = table ; p < table + 16 ; p++)
	    *p = 0;
      p = string;
      invert = 0;
      if (len > 0 && *p == '^') {
	    len--;
	    p++;
	    invert++;
      }
      while (--len >= 0) {
	    if (len >= 2 && p[1] == '-') {
		  for (c = p[0] ; c <= p[2] ; c++) {
			table[(c & 0177) >> 3] |= 1 << (c & 07);
		  }
		  p += 3;
		  len -= 2;
	    } else {
		  c = *p++;
		  table[(c & 0177) >> 3] |= 1 << (c & 07);
	    }
      }
      if (invert) {
	    for (p = table ; p < table + 16 ; p++)
		  *p = ~*p;
      }
}


/*
 * Be sure the next token is a newline.
 */

void
chknl() {
      if (gettoken() != TNL)
	    error("Expecting NEWLINE");
}


/*
 * Return the next input token.
 */

int
gettoken() {
      int c;
      char *p;

      if (tokpushback) {
	    tokpushback = 0;
	    return lasttoken;
      }
      lasttoken = 0;
      while ((c = getc(infp)) == ' ' || c == '\t');
      if (c == '\n') {
	    linno++;
	    return lasttoken = TNL;
      }
      p = toktext;
      do {
	    if (c == EOF) {
		  *p = '\0';
		  error("EOF in middle of line");
		  exit(2);
	    }
	    if (p >= toktext + sizeof toktext - 2) {
		  *p = '\0';
		  error("Input token too long");
		  p = toktext;
	    }
	    *p++ = c;
      } while ((c = getc(infp)) != ' ' && c != '\t' && c != '\n');
      ungetc(c, infp);
      *p = '\0';
      return lasttoken = TWORD;
}



/*
 * Parse a string.
 */

int
getstring() {
      int c;
      char *p;

      toktext[0] = '\0';
      lasttoken = 0;
      while ((c = getc(infp)) == ' ' || c == '\t');
      if (c == EOF) {
	    error("EOF in middle of line");
	    exit(2);
      }
      if (c != '"') {
	    ungetc(c, infp);
	    gettoken();
	    error("Expecting a quoted string");
	    return -1;
      }
      p = toktext;
      for (;;) {
	    c = getc(infp);
	    if (c == '"')
		  break;
	    if (c == '\\') {
		  *p++ = c;
		  c = getc(infp);
	    }
	    if (c == EOF) {
		  *p = '\0';
		  error("EOF in middle of line");
		  exit(2);
	    }
	    if (c == '\n') {
		  ungetc(c, infp);
		  error("Unterminated string");
		  return -1;
	    }
	    if (p >= toktext + sizeof toktext - 2) {
		  *p = '\0';
		  error("String too long");
		  p = toktext;
	    }
	    *p++ = c;
      }
      *p = '\0';
      lasttoken = TSTRING;
}


/*
 * Parse the text in the last token string.
 */


void
cvttext(doctl) {
      register char *p, *q;
      char *endp = toktext + strlen(toktext);
      int n;
      int c;

      q = tokstring;
      for (p = toktext ; p < endp ; p++) {
	    if (q >= tokstring + 127) {
		  error("String exceeds 127 characters");	
		  break;
	    }
	    if (*p == '\\') {
		  switch (*++p) {
		  case 'b':  *q++ = '\b';  break;
		  case 'e':  *q++ = '\033'; break;	/* escape */
		  case 'f':  *q++ = '\f';  break;
		  case 'n':  *q++ = '\n';  break;
		  case 'r':  *q++ = '\r';  break;
		  case 's':  *q++ = ' ';   break;
		  case 't':  *q++ = '\t';  break;
		  case '0': case '1': case '2': case '3':
		  case '4': case '5': case '6': case '7':
			c = *p - '0';
			n = 3;
			while (*++p >= '0' && *p <= '7' && --n > 0) {
			      c = (c << 3) + *p - '0';
			}
			*q++ = c;
			break;
		  default:
			*q++ = *p;
			break;
		  }
	    } else if (*p == '^' && doctl) {
		  if (*++p == '?')
			*q++ = '\177';
		  else
			*q++ = *p & 037;
	    } else {
		  *q++ = *p;
	    }
      }
      stringlen = q - tokstring;
}


int
ateof() {
      int c;

      if (feof(infp) || (c = getc(infp)) == EOF)
	    return 1;
      ungetc(c, infp);
      return 0;
}


char *
saveblock(block, len)
      char *block;
      int len;
      {
      char *p;

      if ((p = malloc(len)) == NULL) {
	    error("Out of space");
	    exit(2);
      }
      bcopy(block, p, len);
      return p;
}


void
error(msg)
      char *msg;
      {
      int lno;
      static int lasterror = -1;

      lno = (lasttoken == TNL)? linno - 1 : linno;
      if (lno != lasterror) {
	    lasterror = lno;
	    printf("%s:%d: %s", inputfile, lno, msg);
	    if (lasttoken == TNL)
		  printf(" (input token = NEWLINE)\n");
	    else
		  printf(" (input token = \"%s\")\n", toktext);
      }
      nerrors++;
}


int
writeout() {
      int i;
      int tabno;
      int stringno;
      int stringloc;
      struct abbrev *ap;

      if (debug)
	    return dbwriteout();
      strcpy(outputfile, inputfile);
      if ((i = strlen(inputfile)) < 5 || ! equal(inputfile + i - 5, ".bind"))
	    strcat(outputfile, ".bindc");
      else
	    strcat(outputfile, "c");
      if ((outfp = fopen(outputfile, "w")) == NULL) {
	    fprintf(stderr, "Can't create %s\n", outputfile);
	    return -1;
      }
      putsh(BINDMAGIC, outfp);
      putc(VERSION, outfp);
      putc(nmodes + nkeytab, outfp);
      putsh(strnext - stringspace, outfp);
      putsh(abbrevsize, outfp);
      for (tabno = 0 ; tabno < nmodes ; tabno++) {
	    putkeytab(mode[tabno].keytab);
      }
      for (tabno = 0 ; tabno < nkeytab ; tabno++) {
	    putkeytab(keytab[tabno]);
      }
      for (i = 0 ; i < NSYNTAX ; i++)
	    fwrite(syntax[i].bitmap, sizeof syntax[i].bitmap, 1, outfp);
      fwrite(stringspace, strnext - stringspace, 1, outfp);
      for (i = 0, ap = abbrev ; i < nabbrev ; i++, ap++) {
	    putc(ap->fromlen, outfp);	abbrevsize -= 1;
	    fwrite(ap->from, ap->fromlen, 1, outfp); abbrevsize -= ap->fromlen;
	    putc(ap->tolen, outfp);	abbrevsize -= 1;
	    fwrite(ap->to, ap->tolen, 1, outfp);  abbrevsize -= ap->tolen;
      }
      if (abbrevsize != 0)
	    abort();
      if (ferror(outfp) || fclose(outfp) == EOF) {
	    fprintf(stderr, "Write error on %s\n", outputfile);
	    return -1;
      }
      return 0;
}


void
putkeytab(tab)
      struct cmd *tab;
      {
      struct cmd *cp;
      int ch;

      cp = tab;
      for (ch = 0 ; ch <= 127 ; ch++) {
	    putc(cp->type, outfp);
	    cp++;
      }
      cp = tab;
      for (ch = 0 ; ch <= 127 ; ch++) {
	    if (cp->type == C_PFXTBL)
		  putc(cp->index + nmodes, outfp);
	    else
		  putc(cp->index, outfp);
	    cp++;
      }
}


void
putsh(i, fp)
      FILE *fp;
      {
      putc(i, fp);
      putc(i >> 8, fp);
}


#if 1
int
dbwriteout() {
      int i;
      int tabno;
      int stringno;
      int stringloc;
      struct abbrev *ap;
      char *p;

      strcpy(outputfile, inputfile);
      if ((i = strlen(inputfile)) < 5 || ! equal(inputfile + i - 5, ".bind"))
	    strcat(outputfile, ".bindc");
      else
	    strcat(outputfile, "c");
      if ((outfp = fopen(outputfile, "w")) == NULL) {
	    fprintf(stderr, "Can't create %s\n", outputfile);
	    return -1;
      }
      for (tabno = 0 ; tabno < nmodes ; tabno++) {
	    fprintf(outfp, "keytab for mode %d:\n", tabno);
	    dbputkeytab(mode[tabno].keytab);
      }
      for (tabno = 0 ; tabno < nkeytab ; tabno++) {
	    fprintf(outfp, "keytab %d:\n", tabno + nmodes);
	    dbputkeytab(keytab[tabno]);
      }
      for (i = 0 ; i < NSYNTAX ; i++)
	    dbputsyntax(syntax[i].name, syntax[i].bitmap);
      fprintf(outfp, "\nstrings:\n");
      stringloc = 0;
      for (stringno = 0 ; stringno < nstrings ; stringno++) {
	    fprintf(outfp, "%-4d ", i = stringspace[stringloc++]);
	    while (--i >= 0)
		  outchar(stringspace[stringloc++]);
	    putc('\n', outfp);
      }
      fprintf(outfp, "\nAbbreviations:\n");
      for (stringno = 0, ap = abbrev ; stringno < nabbrev ; stringno++, ap++) {
	    i = ap->fromlen, p = ap->from;
	    while (--i >= 0)
		  outchar(*p++);
	    fputs(": ", outfp);
	    i = ap->tolen, p = ap->to;
	    while (--i >= 0)
		  outchar(*p++);
	    putc('\n', outfp);
	    abbrevsize -= ap->fromlen + ap->tolen + 2;
      }
      if (abbrevsize != 0)
	    abort();
}



dbputkeytab(tab)
      struct cmd *tab;
      {
      struct cmd *cp;
      int ch;

      cp = tab;
      for (ch = 0 ; ch <= 127 ; ch++) {
	    outchar1(ch);
	    putc(' ', outfp);
	    if (cp->type == C_FUNC) {
		  fprintf(outfp, "%s\n", command[cp->index]);
	    } else if (cp->type == C_PFXTBL) {
		  fprintf(outfp, "keytab %d\n", cp->index + nmodes);
	    } else if (cp->type == C_INSERT) {
		  fprintf(outfp, "insert %d\n", cp->index);
	    } else if (cp->type == C_UNDEF) {
		  fprintf(outfp, "<undef>\n");
	    } else if (cp->type == C_UNSET) {
		  fprintf(outfp, "{}\n");
	    } else {
		  fprintf(outfp, "type = %d, index = %d\n", cp->type, cp->index);
	    }
	    cp++;
      }
      putc('\n', outfp);

}



dbputsyntax(name, syntax)
      char *name;
      char *syntax;
      {
      int c;
      int start;

      fprintf(outfp, "%s syntax: ", name);
      start = -1;
      for (c = 0 ; c <= 128 ; c++) {
	    if (c != 128 && syntax[c >> 3] & 1 << (c & 07)) {
		  if (start < 0) {
			start = c;
		  }
	    } else {
		  if (start >= 0) {
			outchar(start);
			if (start != c - 1) {
			      putc('-', outfp);
			      outchar(c - 1);
			}
			start = -1;
		  }
	    }
      }
      putc('\n', outfp);
}



outchar1(ch) {
      if (ch < ' ')
	    fprintf(outfp, "^%c", ch ^ 0100);
      else if (ch == ' ')
	    fprintf(outfp, "SP");
      else if (ch == '\177')
	    fprintf(outfp, "^?");
      else
	    fprintf(outfp, "%c ", ch);
}



outchar(ch) {
      if (ch < ' ')
	    fprintf(outfp, "^%c", ch ^ 0100);
      else if (ch == '\177')
	    fprintf(outfp, "^?");
      else
	    putc(ch, outfp);
}
#endif
