/*
 * Atty - tty(4) replacement.
 *
 * Copyright (C) 1989 by Kenneth Almquist.  All rights reserved.
 * This file is part of atty, which is distributed under the terms specified
 * by the Atty General Public License.  See the file named LICENSE.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include <sgtty.h>
#include <errno.h>
#include <fcntl.h>
#include <setjmp.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include "atty.h"
#include "attyed.h"


#ifndef __STDC__
#define const
#define volatile
#endif


#define ECHO_NL 1		/* echo input line terminated by a newline */
#define ECHO_EOF 2		/* echo input line terminated by EOF */
#define ECHO_SIG 3		/* echo interrupt character */


struct indata {
      struct indata *next;	/* next structure on queue */
      short nchars;		/* number of characters */
      short nwrt;		/* number of characters already written */
      char data[4];		/* characters; should be "char data[nchars]" */
};


struct tty origtty;		/* original tty modes */
struct tty realtty;		/* tty modes of real tty */
struct tty ptymodes;		/* modes of pty */
int realfcntl;			/* flags for real tty file descriptor */
int ptyfd;			/* file descriptor of pty */
char ptydevice[12];		/* pty device */
int childpid;			/* pid of child process */
jmp_buf sigjmp;			/* where to jump on signal */
volatile int canjump;		/* set if should do longjmp on signal */
volatile int gotsigchild;	/* child is dead */
volatile int gotsigwinch;	/* window size has changed */
volatile int childstatus;	/* status of child */
struct indata *inputq;		/* queue of data to be sent to pty */
struct indata *inqlast;		/* last entry in inputq */
struct mode ttymode;		/* tty characters for editor */
int remoteon;			/* true if TIOCREMOTE mdoe is on */
int needclearin;		/* true if we must clear the tty input */
int echoflag;			/* set if output echoed */
char sigchar;			/* signal character to be echoed */
int noedit;			/* true if not using input editor */
int logall;			/* save all output data in the trace file */
int loginshell;			/* true if should start up login shell */
int nosuspend;			/* don't permit atty to be suspended */
char *shellcmd;			/* prompt to be used by shell */
short ospeed;			/* output speed, for termcap */
int gottty;			/* true if we have read the tty modes */

const struct ltchars real_ltchars = { -1, -1, -1, -1, -1, -1 };

/* C library routines */
#ifdef __STDC__
int ioctl(int, int, char *);
int isatty(int);
int getpid(void);
char *getenv(char *);
#else
int ioctl();
int isatty();
int getpid();
char *getenv();
#endif


#ifdef __STDC__
int openpty(void);
void spawnshell(void);
void runtty(void);
void ttyoutc(int);
void outreal(char *, int);
void writereal(void);
void senddata(char *, int);
void writepty(void);
int stripecho(int, char *, int);
void flushptyin(void);
void onchild();
void onwinch();
void indata(char *, int);
void newttymodes();
void fillttymode(void);
void sendsig(int);
void flushoutput(void);
void fatal(char *);
void done(int);
void cleanup(void);
int getttymodes(int, struct tty *);
int setttymodes(int, struct tty *);
int copyttysize(void);
#else
int openpty();
void spawnshell();
void runtty();
void ttyoutc();
void outreal();
void writereal();
void senddata();
void writepty();
int stripecho();
void flushptyin();
void onchild();
void onwinch();
void indata();
void newttymodes();
void fillttymode();
void sendsig();
void flushoutput();
void fatal();
void done();
void cleanup();
int getttymodes();
int setttymodes();
int copyttysize();
#endif

#define scopy(s1, s2)	strcpy(s2, s1)	/*TEMPORARY*/


main(argc, argv)
      char **argv;
      {
      int one = 1;
      int i;
      char **ap;
      char *p;

      if (! isatty(2) || ! isatty(0)) {
	    fputs("Atty can only be run from a terminal\n", stderr);
	    exit(2);
      }
      ap = argv;
      if (argc > 0) {
	    ap++;
	    for (;;) {
		  if ((p = *ap) == NULL || *p++ != '-')
			break;
		  ap++;
		  if (p[0] == '-' && p[1] == '\0')	/* "--" */
			break;
		  while (*p) {
			switch (*p++) {
			case 'l':
			      loginshell++;
			      break;
			case 's':
			      nosuspend++;
			      break;
			case 'c':
			      if (*p == '\0' && (p = *ap++) == NULL) {
				    fputs("atty:  no arg for -c option\n");
				    exit(2);
			      }
			      shellcmd = p;
			      p = "";
			      break;
			default:
			      fprintf(stderr, "atty: Illegal option -%c\n", p[-1]);
			      exit(2);
			}
		  }
	    }
      }
      signal(SIGCHLD, onchild);
      ptyfd = openpty();
      if (getttymodes(2, &origtty) < 0)
	    fatal("getttymodes");
      realfcntl = fcntl(2, F_GETFL, 0);
      gottty = 1;
      ospeed = origtty.sgtty.sg_ospeed;
      edinit(*ap);
      ptymodes = origtty;
      ptymodes.ldisc = NTTYDISC;	/* we must have literal next char */
      ptymodes.sgtty.sg_flags |= EVENP|ODDP;	/* no parity checking */
      realtty = origtty;
      realtty.tchars.t_intrc = -1;
      realtty.tchars.t_quitc = -1;
      realtty.ltchars = real_ltchars;
      realtty.sgtty.sg_flags |= CBREAK;
      realtty.sgtty.sg_flags &=~ (CRMOD|ECHO|LCASE);
      if ((realtty.sgtty.sg_flags & TBDELAY) == XTABS)
	    realtty.sgtty.sg_flags &=~ TBDELAY;
      if (setttymodes(2, &realtty) < 0) {
	    perror("setttymodes");
	    exit(2);
      }
      /* copyttysize(); */
      realfcntl |= FNDELAY;
      fcntl(2, F_SETFL, realfcntl);
      i = fcntl(ptyfd, F_GETFL, 0);
      i |= FNDELAY;
      fcntl(ptyfd, F_SETFL, i);
#ifdef REMOTE
      if (ioctl(ptyfd, TIOCREMOTE, (char *)&one) < 0)
	    fatal("TIOCREMOTE");
      remoteon = 1;
#endif
      if (ioctl(ptyfd, TIOCPKT, (char *)&one) < 0)
	    fatal("TIOCPKT");
      if (setttymodes(ptyfd, &ptymodes) < 0)
	    fatal("set pty modes");
      fillttymode();
      copyttysize();
      if ((childpid = fork()) == -1) {
	    fatal("fork");
      }
      if (childpid == 0)
	    spawnshell();
      else {
	    signal(SIGTTOU, SIG_IGN);
	    signal(SIGWINCH, onwinch);
	    runtty();
      }
      /*NOTREACHED*/
}
      

/*
 * Search for an unused pty.  It returns a file descriptor for the master
 * and leaves the name of the slave in the global variable ptydevice.
 * This code starts at the beginning; a more efficient version might start
 * at a randomly chosen point in the middle.
 */

int
openpty() {
      char c;
      int i;
      int fd;

      scopy("/dev/ptyXX", ptydevice);
      for (c = 'p'; c <= 's'; c++) {
            for (i = 0 ; i < 16 ; i++) {
                  ptydevice[8] = c;
                  ptydevice[9] = "0123456789abcdef"[i];
                  fd = open(ptydevice, 2);
                  if (fd >= 0) {
			ptydevice[5] = 't';	/* change "pty" to "tty" */
                        return fd;		/* success */
                  }
                  if (errno == ENOENT)
                        goto failure;
            }
      }
failure:
      fprintf(stderr, "Out of pty's\n");
      exit(2);
}


/*
 * This routine exec's the shell.
 */

void
spawnshell() {
      int fd;
      int mypid = getpid();
      char *shell;
      char **env;
      char **ep;
      extern char **environ;
      char arg0[16];
      char *p;
      char *basename;
      char *argv[4];

      if ((shell = getenv("SHELL")) == NULL)
	    shell = "/bin/sh";
      for (ep = environ ; *ep ; ep++);
      if ((env = (char **)malloc((char *)(ep + 2) - (char *)environ)) == NULL)
	    fatal("Malloc failed");
      for (ep = env ; *environ ; environ++) {
	    if (strncmp(*environ, "ATTY=", 5) != 0)
		  *ep++ = *environ;
      }
      *ep++ = "ATTY=";
      *ep = NULL;
      close(0);
      close(1);
      close(ptyfd);
      fd = open("/dev/tty", O_RDWR);
      if (fd >= 0) {
	    ioctl(fd, TIOCNOTTY, (char *)0);
	    close(fd);
      }
      setpgrp(mypid, 0);
      if (open(ptydevice, O_RDWR) != 0) {
	    perror(ptydevice);
	    exit(2);
      }
      setpgrp(mypid, mypid);
      close(2);
      dup(0);				/* file descriptor 1 = dup of 0 */
      dup(0);				/* file descriptor 2 = dup of 0 */
      if (ioctl(2, TIOCSPGRP, (char *)&mypid) < 0)
	    printf("TIOCSPGRP failed, errno=%d\n", errno);
      basename = shell;
      for (p = shell ; *p ; p++) {
	    if (*p == '/')
		  basename = p + 1;
      }
      if (loginshell) {
	    arg0[0] = '-';
	    for (p = arg0 + 1 ; p < arg0 + 15 && *basename ; *p++ = *basename++);
	    *p = '\0';
	    basename = arg0;
      }
      argv[0] = basename;
      argv[1] = NULL;
      if (shellcmd) {
	    argv[1] = "-sc";
	    argv[2] = shellcmd;
	    argv[3] = NULL;
      }
      execve(shell, argv, env);
      perror(shell);
      exit(2);
}


#define OUTRSIZE 512

char outrbuf[OUTRSIZE];
char *outrp = outrbuf;
char *outrnext = outrbuf;
int outrsize;


/*
 * This routine contains the main loop which is run after initialization.
 */

void
runtty() {
      volatile int ptybit = 1 << ptyfd;
      volatile int realbit = 1 << 2;
      volatile int nfds = ptyfd + 1;
      volatile int inbits, outbits, exceptbits;
      volatile int sel;
      int i, j;
      char buf[260];

      if (setjmp(sigjmp))
	    goto gotsig;
      for (;;) {
	    inbits = realbit;
	    outbits = 0;
	    if (outrsize == 0)
		  inbits |= ptybit;
	    else
		  outbits = realbit;
	    if (inputq != NULL)
		  outbits |= ptybit;
	    exceptbits = 0;
	    canjump = 1;
	    if (gotsigchild || gotsigwinch)
		  goto gotsig;
	    sel = select(nfds, &inbits, &outbits, &exceptbits, (struct timeval *)NULL);
	    canjump = 0;
	    if (0) {	/* if interrupted by signal */
gotsig:
		  if (gotsigchild) {
			gotsigchild = 0;
			i = childstatus;
			if ((i & 0xFF) == 0177
			 && (i >> 8 == SIGTSTP || i >> 8 == SIGSTOP)) {
			      if (! nosuspend) {
				    setttymodes(2, &origtty);
				    fcntl(2, F_SETFL, realfcntl &~ FNDELAY);
				    kill(getpid(), SIGTSTP);
				    setttymodes(2, &realtty);
				    fcntl(2, F_SETFL, realfcntl);
			      }
			      killpg(childpid, SIGCONT);
			} else {
			      if ((i & 0xFF) == 0)
				    i >>= 8;
			      else
				    i = i & 0xFF | 0x80;
			      done(i);
			}
		  }
		  if (gotsigwinch) {
			gotsigwinch = 0;
			newscrnwidth(copyttysize());
		  }
	    } else if (sel < 0) {
		  if (errno != EINTR)
			fatal("select");
	    } else {
		  if (inbits & realbit) {
			if ((i = read(2, buf, sizeof buf)) < 0)
			      fatal("tty read");
			if (i == 0)
			      fatal("tty eof");
			indata(buf, i);
		  }
		  if (inbits & ptybit) {
			if ((i = read(ptyfd, buf, sizeof buf)) < 0) {
			      /*
			       * Probably the problem is that the child
			       * process has terminated, which causes the
			       * ptc device to return EIO.
			       */
			      if (gotsigchild)
				    goto gotsig;
			      if (errno == EIO) {
				    sleep(1);
				    errno = EIO;
				    if (gotsigchild)
					  goto gotsig;
			      }
			      fatal("pty read");
			}
			if (i == 0)
			      fatal("pty eof");
#ifndef TIOCPKT_IOCTL
			newttymodes();
#endif
			if (buf[0] == TIOCPKT_DATA) {
			      j = 0;
			      if (echoflag) {
				    j = stripecho(echoflag, buf + 1, i - 1);
				    echoflag = 0;
			      }
			      j++;
#ifdef notdef
			      if (logall)
				    fwrite(buf + 1, i - 1, 1, trace);
#endif
			      outchars(buf + j, i - j);
			      writereal();
			      if (noedit == 0)
				    refresh();
			} else {
#ifdef TIOCPKT_IOCTL
			      if (buf[0] & TIOCPKT_IOCTL)
				    newttymodes();
#endif
			      if (buf[0] & TIOCPKT_FLUSHREAD) {
				    clearinput();
			      }
			}
		  }
		  if (outbits & realbit) {
			writereal();
		  }
		  if (outbits & ptybit) {
			writepty();
		  }
	    }
      }
}


/*
 * Output a character to the real tty device.
 */

void
ttyoutc(c)
      char c;
      {
      outreal(&c, 1);
}


/*
 * Output a block of characters to the real tty device.
 */

void
outreal(p, size)
      char *p;
      {
      int n = &outrbuf[OUTRSIZE] - outrnext;

      if (size + outrsize > OUTRSIZE)
	    fatal("output overflow");
      if (n > size)
	    n = size;
      bcopy(p, outrnext, n);
      outrnext += n;
      if (outrnext == &outrbuf[OUTRSIZE])
	    outrnext = outrbuf;
      if (n < size) {
	    p += n;
	    n = size - n;
	    bcopy(p, outrnext, n);
	    outrnext += n;
      }
      outrsize += size;
}


/*
 * Write as much data as possible to the real tty, until we block.
 */

void
writereal() {
      int i;
      int n;

      while (outrsize > 0) {
	    n = &outrbuf[OUTRSIZE] - outrp;
	    if (n > outrsize)
		  n = outrsize;
	    if ((i = write(2, outrp, n)) < 0) {
		  if (errno == EWOULDBLOCK)
			break;
		  else
			fatal("tty write");
	    }
	    outrsize -= i;
	    outrp += i;
	    if (outrp == &outrbuf[OUTRSIZE])
		  outrp = outrbuf;
      }
}


/*
 * Send data to the pty.
 */

#define INDATAHDR ((char *)((struct indata *)0)->data - (char *)0)

void
senddata(p, size)
      char *p;
      {
      struct indata *ip;

      if ((ip = (struct indata *)malloc(INDATAHDR + size)) == NULL)
	    fatal("malloc pty data");
      ip->nchars = size;
      ip->nwrt = 0;
      bcopy(p, ip->data, size);
      ip->next = NULL;
      if (inputq == NULL)
	    inputq = ip;
      else
	    inqlast->next = ip;
      inqlast = ip;
}



/*
 * Like writereal, but for pty.
 */

void
writepty() {
#ifdef REMOTE
      int i;
      struct indata *ip;
      int one = 1;

      if ((ip = inputq) == NULL)
	    return;
      if (! remoteon) {
	    if (ioctl(ptyfd, TIOCREMOTE, (char *)&one) < 0)
		  fatal("TIOCREMOTE");
	    remoteon = 1;
      }
      if ((i = write(ptyfd, ip->data, ip->nchars)) < 0) {
	    if (errno != EWOULDBLOCK)
		  fatal("pty write");
	    return;
      } else if (i != ip->nchars) {
	    fatal("partial pty write");
      }
      inputq = ip->next;
      free((char *)ip);
#else
      int i;
      struct indata *ip;
      char buf[512];
      int buflen;
      char c;
      char *p;
      int echotype;

      if ((ip = inputq) == NULL)
	    return;
      if (noedit) {
	    i = write(ptyfd, ip->data + ip->nwrt, ip->nchars - ip->nwrt);
	    if (i < 0)
		  fatal("pty write");
	    if ((ip->nwrt += i) >= ip->nchars) {
		  inputq = ip->next;
		  free((char *)ip);
	    }
      } else {	  
	    newttymodes();	/* just in case */
	    p = buf;
	    for (i = ip->nwrt ; i < ip->nchars ; i++) {
		  c = ip->data[i];
		  if (c == ptymodes.sgtty.sg_erase
		   || c == ptymodes.sgtty.sg_kill
		   || c == ptymodes.tchars.t_intrc
		   || c == ptymodes.tchars.t_quitc
		   || c == ptymodes.tchars.t_startc
		   || c == ptymodes.tchars.t_stopc
		   || c == ptymodes.tchars.t_eofc
		   || c == ptymodes.tchars.t_brkc
		   || c == ptymodes.ltchars.t_suspc
		   || c == ptymodes.ltchars.t_dsuspc
		   || c == ptymodes.ltchars.t_rprntc
		   || c == ptymodes.ltchars.t_flushc
		   || c == ptymodes.ltchars.t_werasc
		   || c == ptymodes.ltchars.t_lnextc)
			*p++ = ptymodes.ltchars.t_lnextc;
		  *p++ = c;
	    }
	    echotype = ECHO_NL;
	    if (p == buf || p[-1] != '\n') {
		  *p++ = ptymodes.tchars.t_eofc;
		  echotype = ECHO_EOF;
	    }
	    buflen = p - buf;
	    i = write(ptyfd, buf, buflen);
	    if (i < 0) {
		  if (errno != EWOULDBLOCK)
			fatal("pty write");
	    } else if (i < buflen) {
		  for (p = buf ; p < buf + buflen ; p++) {
			if (*p == ptymodes.ltchars.t_lnextc)
			      p++;
			ip->nwrt++;
		  }
	    } else {
		  inputq = ip->next;
		  free((char *)ip);
	    }
	    if (ptymodes.sgtty.sg_flags & ECHO) {
		  echoflag = echotype;
	    }
      }
#endif
}



/*
 * Locate the end of the echoed characters.
 */

int
stripecho(type, p, n)
      char *p;
      {
      register char *q;

      q = p;
      if (type == ECHO_NL) {
	    do {
		  if (--n < 0)
			return 0;
	    } while (*q++ != '\n');
      } else if (type == ECHO_EOF) {
	    do {
		  if (--n <= 0)
			return 0;
	    } while (*q++ != '\b' || *q != '\b');
	    q++;
      } else {
	    if (n < 2 || p[0] != '^' || p[1] != (sigchar ^ 0100))
		  return 0;
	    q += 2;
      }
      return q - p;
}



/*
 * Clear pty input queue.  The input queue is the one we write to.
 */

void
flushptyin() {
      struct indata *ip;

      while (inputq != NULL) {
	    ip = inputq;
	    inputq = ip->next;
	    free((char *)ip);
      }
      /* ioctl(ptyfd, TIOCFLUSH, (char *)&mode); */
}


/*
 * Signal handler invoked when a child dies or is stopped.
 */

void
onchild() {
      int pid;
      int status;
      int e = errno;

      if ((pid = wait3(&status, WNOHANG|WUNTRACED, (struct rusage *)0)) == childpid) {
	    childstatus = status;
	    gotsigchild = 1;
	    if (canjump)
		  longjmp(sigjmp, 1);
      }
      errno = e;
}


/*
 * Signal handler called when the terminal changes size.
 */

void
onwinch() {
      gotsigwinch = 1;
      if (canjump)
	    longjmp(sigjmp, 1);
}


/*
 * Called to process data received from the real tty.
 */

void
indata(p, n)
      char *p;
      {
      if (noedit) {
	    senddata(p, n);
#ifdef REMOTE
	    if (ptymodes.sgtty.sg_flags & ECHO) {
		  outchars(p, n);
		  writereal();
	    }
#endif
      } else {
	    while (--n >= 0)
		  inchar(*p++);
	    refresh();
	    if (needclearin) {
		  clearinput();
		  needclearin = 0;
	    }
	    if (outrsize)
		  writereal();
      }
}


clearinput() {
      edflush();
      flushptyin();
}


/*
 * Send a signal to the user process.
 */

void
sendsig(signo) {
#ifdef REMOTE
      int pgrp;
      int zero = 0;

      if (ioctl(ptyfd, TIOCFLUSH, (char *)&zero) < 0)
	    fatal("TIOCFLUSH");
      needclearin++;
      if (ioctl(ptyfd, TIOCGPGRP, (char *)&pgrp) < 0)
	    fatal("TIOCGPRGP");
      killpg(pgrp, signo);
#else
      char c;

      switch (signo) {
	case SIGINT:	c = ttymode.intr;  break;
	case SIGQUIT:	c = ttymode.quit;  break;
	case SIGTSTP:	c = ttymode.susp;  break;
	default:	fatal("sendsig arg");
      }      
      if ((ptymodes.lmode & LNOFLSH) == 0 && signo != SIGTSTP)
	    flushoutput();
      echoflag = ECHO_SIG;
      sigchar = c;
      if (write(ptyfd, &sigchar, 1) != 1)
	    fatal("sendsig write");
#endif
}


void
flushoutput() {
      int two = 2;

      outrp = outrnext = outrbuf;
      outrsize = 0;
      ioctl(2, TIOCFLUSH, (char *)&two);
}


puttty(c)
      char c;
      {
      write(2, &c, 1);
}


puttstr(s)
      char *s;
      {
      write(2, s, strlen(s));
}


/*
 * Called when the tty modes of the pty are changed by the user.
 */

#define SGCOPY	(ALLDELAY|RAW)	/* bits to copy from pty to real tty */

void
newttymodes() {
      char startc, stopc;
      int changereal;
      int flags;

      if (getttymodes(ptyfd, &ptymodes) < 0)
	    fatal("get pty modes");
      noedit = 0;
#ifdef notdef
      if (ptymodes.sgtty.sg_flags & (RAW|CBREAK))
#else
      if ((ptymodes.sgtty.sg_flags & (RAW|CBREAK|ODDP)) != ODDP)
#endif
	    noedit = 1;
      changereal = 0;
      startc = ptymodes.tchars.t_startc;
      stopc = ptymodes.tchars.t_stopc;
#ifdef notdef
      if (ptymodes.sgtty.sg_flags & RAW)
	    startc = stopc = -1;
#endif
      if (realtty.tchars.t_startc != startc
       || realtty.tchars.t_stopc != stopc) {
	    realtty.tchars.t_startc = startc;
	    realtty.tchars.t_stopc = stopc;
	    changereal = 1;
      }
      flags = ptymodes.sgtty.sg_flags & SGCOPY;
      if ((flags & TBDELAY) == XTABS)
	    flags &=~ TBDELAY;
      if ((realtty.sgtty.sg_flags & SGCOPY) !=  flags) {
	    realtty.sgtty.sg_flags &=~ SGCOPY;
	    realtty.sgtty.sg_flags |= flags;
	    changereal = 1;
      }
      if ((realtty.lmode & LDECCTQ) != (ptymodes.lmode & DECCTQ)) {
	    realtty.lmode &=~ LDECCTQ;
	    if (ptymodes.lmode & DECCTQ)
		  realtty.lmode |= LDECCTQ;
	    changereal = 1;
      }
      if (changereal) {
	    if (setttymodes(2, &realtty) < 0)
		  fatal("change real");
      }
      if (ttymode.intr != ptymodes.tchars.t_intrc
       || ttymode.quit != ptymodes.tchars.t_quitc
       || ttymode.erase!= ptymodes.sgtty.sg_erase
       || ttymode.kill != ptymodes.sgtty.sg_kill
       || ttymode.eof  != ptymodes.tchars.t_eofc
       || ttymode.susp != ptymodes.ltchars.t_suspc
       || ttymode.echo != (ptymodes.sgtty.sg_flags & ECHO)) {
	    fillttymode();
	    newttychars();
      }
}


void
fillttymode() {
      ttymode.intr = ptymodes.tchars.t_intrc;
      ttymode.quit = ptymodes.tchars.t_quitc;
      ttymode.erase= ptymodes.sgtty.sg_erase;
      ttymode.kill = ptymodes.sgtty.sg_kill;
      ttymode.eof  = ptymodes.tchars.t_eofc;
      ttymode.susp = ptymodes.ltchars.t_suspc;
      ttymode.echo = ptymodes.sgtty.sg_flags & ECHO;
}


/*
 * A unexpected error occurred.
 */

void
fatal(msg)
      char *msg;
      {
      int e = errno;

      cleanup();
      errno = e;
      perror(msg);
      fflush(stderr);
      abort();
}


/*
 * Called if initialization fails.
 */

void
badinit(msg)
      char *msg;
      {
      cleanup();
      fputs(msg, stderr);
      putc('\n', stderr);
      exit(2);
}


/*
 * Exit the program, resetting the terminal modes first.  Status is the
 * exit status.
 */

void
done(status) {
      cleanup();
      exit(status);
}


void
cleanup() {
      if (gottty) {
	    setttymodes(2, &origtty);
	    realfcntl &=~ FNDELAY;
	    fcntl(2, F_SETFL, realfcntl);
      }
}



int
getttymodes(fd, tty)
      int fd;
      struct tty *tty;
      {
      if (ioctl(fd, TIOCGETD, (char *)&tty->ldisc) < 0
       || ioctl(fd, TIOCGETP, (char *)&tty->sgtty) < 0
       || ioctl(fd, TIOCGETC, (char *)&tty->tchars) < 0
       || ioctl(fd, TIOCLGET, (char *)&tty->lmode) < 0
       || ioctl(fd, TIOCGLTC, (char *)&tty->ltchars) < 0)
	    return -1;
      return 0;
}



int
setttymodes(fd, tty)
      int fd;
      struct tty *tty;
      {
      if (ioctl(fd, TIOCSETD, (char *)&tty->ldisc) < 0
       || ioctl(fd, TIOCSETP, (char *)&tty->sgtty) < 0
       || ioctl(fd, TIOCSETC, (char *)&tty->tchars) < 0
       || ioctl(fd, TIOCLSET, (char *)&tty->lmode) < 0
       || ioctl(fd, TIOCSLTC, (char *)&tty->ltchars) < 0)
	    return -1;
      return 0;
}



/*
 * Copy the size of the tty from the real device to the pty.
 * Returns the number of columns.
 */

int
copyttysize() {
#ifdef TIOCGWINSZ
      struct winsize size;

      if (ioctl(2, TIOCGWINSZ, (char *)&size) >= 0)
	    ioctl(ptyfd, TIOCSWINSZ, (char *)&size);
      return size.ws_col;
#else
      return 0;
#endif
}
