#!	/bin/csh	-f

# Archive management scripts.  Use in conjunction with narc

# Geoffrey Leach
# LatiCorp Inc.	
# {att,bellcore,sun,ames,pyramid}!pacbell!laticorp!geoff

# assemble	Assemble a group of files into an archive directory
# archive	Put and archive directory in its place
# retrieve	Get an archive directory into the cwd

# This shell script implements an archive management facility.  The
# idea is that you have a archive whose path is in the environment
# variable ARCHIVE.  The directories here are your archive.  
# Files in an archive directory are compressed tar files if they were
# put there by this archive.  If they were put there by narc, then they
# are plain text files, which you will assemble into compressed tar, depending
# on how they got there.  Moderated archives are handled in a way that 
# lets you refer to a series of postings with shorthand.  BTW, moderated
# groups are assumed to be organized by volume in your archive.  Retrieve
# just helps you get a particular archive file into your cwd.

# Usage - options apply as stated in the usage messages
#
#	-a	Specify ARCDIR.  Forces ARCVOL to ""
#	-e	Extract.  Overrides shorthand for moderated groups
#	-n	Specify archive file.  $1 defaults
#	-s	"Short" assemble; just get the files
#	-v	Specify ARCVOL


# Environment variable usage 

#    ARCHIVE	base directory of archive
#    ARCPATH	directory name
#    ARCVOL	subdirectory name for moderated groups

# Commands that are used by this script that you might not have

#    unshar

set MYNAME   = $0
set MYNAME   = $MYNAME:t

# usage
if ( $#argv == 0 ) then
    echo -n "Usage: $MYNAME [-a archive [-v volume]] "
    switch ( $MYNAME )
	case "assemble":
	    echo "[-s] [-e name] [-n archive-name] base-issue [number-of-issues]"
	    exit
	case "archive":
	    echo "file"
	    exit
	case "retrieve":
	    echo "file"
	    exit
	default:
	    echo ""
	    exit
    endsw
endif

# Check for environment usage
if ( ! $?ARCHIVE ) set ARCHIVE
if ( ! $?ARCDIR )  set ARCDIR
if ( ! $?ARCVOL )  set ARCVOL

# Check for switches
set SWSCAN
while ( $?SWSCAN )
    switch ( $1 )
    case -a:
        set ARCDIR = $2
	set ARCVOL = ""
	shift
	shift
	breaksw
    case -e:
	if ( ! $?EXTRACT ) then 
	    set EXTRACT
	else
	    unset EXTRACT
	endif
	shift
	breaksw
    case -n:
	set ARCFILE = $2
	shift
	shift
	breaksw
    case -s:
	set SHORT
	shift
	breaksw
    case -v:
	if ( $2 <= 9 ) then 
	    set ARCVOL = v0$2
	else
	    set ARCVOL = v$2
	endif
	shift
	shift
	breaksw
    case -*:
	echo "Unrecognized option $1"
	exit
    default:
	unset SWSCAN
	breaksw
    endsw
end

if ( $ARCDIR == "" ) then
    echo -n "Archive directory? "
    set ARCDIR = $<
endif

if ( $ARCVOL == "" ) then
    switch ( $ARCDIR )
	case x:
	case unix:
	case misc:
	case games:
	    echo -n "Archive volume? "
	    set ARCVOL = $<
	    if ( $ARCVOL <= 9 ) then
		set ARCVOL = v0$ARCVOL
	    else
		set ARCVOL = v$ARCVOL
	    endif
	    breaksw
	default:
	    breaksw
    endsw
endif
set PREFIX = ${ARCVOL}i

if ( $ARCVOL == "" ) then
    set ARCPATH = $ARCHIVE/$ARCDIR
else
    set ARCPATH = $ARCHIVE/$ARCDIR/$ARCVOL
endif

# Here is where we do the real work
switch ( $MYNAME )
    case assemble:

	# For moderated groups, we usually have the number of the first posting,
	# and (optionally) the number of postings.  These
	# are combined to give a sequence of file names.  Otherwise, we assume
	# that the argument is the prefix of the file(s) to be retrieved.
	# There may be more than one file with that name as prefix.
	switch ( $ARCDIR )
	    case "x":
	    case "unix":
	    case "misc":
	    case "games":
		if ( ! $?EXTRACT ) then
		    set MODERATED
		    set BASE = $1
		    if ( $#argv == 2 ) then
			set COUNT = $2
		    else
			set COUNT = 1
		    endif
		endif
		breaksw
	    default:
		breaksw
	endsw

	# Set up the directory structure for the archive.
	# Conditional so that we can assemble into an existing directory.
	# ARCF		is where we put the raw input for later deletion
	# FILES		is the lis of original files, for later deletion
	# HDRS		contains the .hdr files produced by unshar, so you can
	#		find our later where all of this came from.
	# PATCHES	is where you accumulate patch files.  I save these
	#		so they can be un-applied later.  Useful if you are
	#		applying unofficial patches to a maintained source.
	if ( ! $?ARCFILE ) then
	    if ( $?MODERATED ) then
		echo -n "Archive name? "
		set ARCFILE = $<
	    else
		set ARCFILE = $1
	    endif
	endif
	if ( ! -d $ARCFILE ) mkdir $ARCFILE
	cd $ARCFILE
	if ( ! -d HDRS ) mkdir HDRS
	if ( ! -d ARCF ) mkdir ARCF
	if ( ! -d PATCHES ) mkdir PATCHES

	if ( $?MODERATED ) then
	    while ( $COUNT )
		if ( $BASE <  10 ) set BASE = 0$BASE
		if ( $BASE < 100 ) set BASE = 0$BASE
		set f = $ARCPATH/${PREFIX}${BASE}
		if ( ! -e $f ) echo $f not found
		chmod +w $f
		cp $f .
		echo $f >> FILES
		unshar $f:t
		mv $f:t ARCF
		@ COUNT = $COUNT - 1
		@ BASE = $BASE + 1
	    end
	else
	    foreach f ( $ARCPATH/$1* )
		chmod +w $f
		cp $f .
		echo $f >> FILES
		# You may want to assemble pre-existing archives.  We handle
		# Compressed files, which may also be tar or shar.
		set f = $f:t
		if ( $f:e == "Z" ) then
		    set f = $f:r
		    uncompress $f
		endif
		if ( $f:e == "tar" ) then
		    tar xvf $f
		    set f = $f:r
		else
		    unshar $f:t
		endif
		mv $f:t ARCF
	    end
	endif
	mv *hdr HDRS

	if ( -e ark1isdone ) mv ark*isdone ARCF

	if ( $?SHORT ) exit

	# This maintains an archive of formatted documentation
	roffit

	CheckManifest
	ls -C
	exit

    case archive:
	if ( ! $?ARCFILE ) then
	    if ( $#argv != 1 ) then
		echo -n "Archive name? "
		set ARCFILE = $<
	    else
		set ARCFILE = $1
	    endif
	endif
	if ( ! -d $ARCFILE ) then
	    echo archive: $ARCFILE not found
	    exit
	endif
	if ( -e $ARCFILE/FILES ) then
	    set F = ( `cat $ARCFILE/FILES` )
	    rm -r $ARCFILE/FILES
	else
	    set F
	endif
	if ( ! -d $ARCPATH ) then
	    echo "$ARCPATH not found"
	    exit
	endif
	if ( -d $ARCFILE/ARCF ) rm -r $ARCFILE/ARCF
	tar cvf $ARCPATH/$ARCFILE.tar $ARCFILE
	if ( -e $ARCPATH/$ARCFILE.tar.Z ) rm -f $ARCPATH/$ARCFILE.tar.Z
	compress $ARCPATH/$ARCFILE.tar
	chmod -w $ARCPATH/$ARCFILE.tar.Z
	rm -rf $ARCFILE $F
	exit

    case retrieve:
	set f = $ARCPATH/$1.tar.Z
	if ( ! -e $f ) then
	    echo $f not found
	    exit
	endif
	zcat $f | tar xvf -
	exit
    default:
	exit
endsw
