/*
	headers

	Retrieve selected news article headers.  Only reads into the article
	file as far as the blank line separating the headers from the body.
	This is more efficient than other searching methods that may read
	the entire file.  A convenient way to see what's in a newsgroup:

		cd /usr/spool/news/comp/unix/wizards
		headers * | more

	syntax:  headers [ -f specfile ] [ -hheaderspec ... ] file ...

	The set of headers to pull is found in the file specfile or in
	the specs given after -h option(s).  If neither -f nor -h are
	given, and the HEADERS environment variable names a file, that
	file will be used.  Otherwise the Subject:, From: and Date:
	headers are the default (with reasonable alternates for
	each if they are missing).  Capitalization is not important.

	The program reads each of the headers and determines whether
	they are any of the ones desired.  When all headers have been
	read, the ones found are printed out according to the order
	of the specifications.

	This code is in the public domain.

	06 Dec 84 Version 1.0.  Paul DuBois, dubois@primate.wisc.edu
	22 Nov 88 v1.1 Revised to do case-insensitive comparisons.
		Added -h option capability.
*/

# include	<stdio.h>
# include	<ctype.h>
# include	<varargs.h>
# ifdef BSD
# include	<strings.h>
# else if SYSV
# include	<string.h>
# endif


# define	New(x)		((x *) calloc (1, sizeof (x)))

extern char	*calloc ();

char	*newstr ();
void	panic ();

typedef struct Header	Header;

struct Header
{
	char	*hName;		/* name of header field */
	char	hBuf[BUFSIZ];	/* line from article for this header */
	Header	*nextGrp;	/* next group of headers */
	Header	*nextHdr;	/* next header in this group */
};


Header	*head = NULL;		/* pointer to output spec structure */


main (argc, argv)
int	argc;
char	**argv;
{
char	*p, *getenv ();
	
	if (*++argv != NULL && strncmp (*argv, "-h", 2) == 0)	/* cmd line */
	{
		for (;;)
		{
			GroupSpecs (&argv[0][2]);
			if (strncmp (*++argv, "-h", 2) != 0)
				break;
		}
	}
	else if (strcmp ("-f", *argv) == 0)			/* named file */
	{
		if (*++argv == NULL)
			panic ("No header file named after -f");
		FileSpecs (*argv++);
	}
	else if ((p = getenv ("HEADERS")) != NULL)		/* env var */
		FileSpecs (p);
	else							/* default */
	{
		GroupSpecs ("subject summary keywords");
		GroupSpecs ("from reply-to sender");
		GroupSpecs ("date");
	}

	while (*argv != NULL)		/* process input files */
		Headers (*argv++);
	exit (0);
}


/*
	Read specifications from file
*/

FileSpecs (fname)
char	*fname;
{
char	buf[BUFSIZ];

	if (freopen (fname, "r", stdin) == NULL)
		panic ("Can't open specfile %s", fname);
	while (fgets (buf, BUFSIZ, stdin) != NULL)
		GroupSpecs (buf);
}

/*
	Process specification for one group of header names
*/

GroupSpecs (bp)
char	*bp;
{
static Header	*gtail;		/* last group in list of groups */
Header		*htail;		/* last header in current group */

	if ((bp = strtok (bp, " ,\t\n")) != NULL)
	{
		if (head == NULL)	/* first group? */
		{
			if ((head = New (Header)) == NULL)
				panic ("GroupSpecs: out of memory");
			gtail = head;
		}
		else			/* add list to last one */
		{
			if ((gtail->nextGrp = New (Header)) == NULL)
				panic ("GroupSpecs: out of memory");
			gtail = gtail->nextGrp;
		}
		gtail->hName = newstr (bp);
		lower (gtail->hName);
		htail = gtail;
		while ((bp = strtok (NULL, " ,\t\n")) != NULL)
		{
			if ((htail->nextHdr = New (Header)) == NULL)
					panic ("GroupSpecs: out of memory");
			htail = htail->nextHdr;
			htail->hName = newstr (bp);
			lower (htail->hName);
		}
	}
}


/*
	Clear header buffers so won't get debris from previous articles,
	then read headers from article and save any that are present in
	the specifications, and print 'em out.
*/

Headers (article)
char	*article;
{
char	c;
char	buf[BUFSIZ];
char	hdrName[BUFSIZ];
char	*hp, *bp;
Header	*lp, *ep;

	if (freopen (article, "r", stdin) == NULL)
	{
		fprintf (stderr, "%s: cannot open article\n", article);
		return;
	}

	for (lp = head; lp != NULL; lp = lp->nextGrp)
		for (ep = lp; ep != NULL; ep = ep->nextHdr)
			ep->hBuf[0] = '\0';

	while (fgets (buf, BUFSIZ, stdin) != NULL)
	{
		if (*buf == '\n' || *buf == '\0')
			break;		/* end of header section */

		hp = hdrName;		/* get header name */
		bp = buf;
		while ((c = *bp) && c != ':' && c != ' ' && c != '\n')
		{
			*hp++ = c;
			++bp;
		}
		*hp = '\0';
		lower (hdrName);
		CheckHeader (hdrName, buf);
	}

	printf ("\n%s\n", article);
	for (lp = head; lp != NULL; lp = lp->nextGrp)
	{
		for (ep = lp; ep != NULL; ep = ep->nextHdr)
		{
			if (ep->hBuf[0] != '\0')
			{
				fputs (ep->hBuf, stdout);
				break;
			}
		}
	}
}


/*
	Check whether the header name is in the specs and save the
	line from the article if so.
*/

CheckHeader (hdrName, artLine)
char	*hdrName, *artLine;
{
Header	*lp, *ep;

	for (lp = head; lp != NULL; lp = lp->nextGrp)
	{
		for (ep = lp; ep != NULL; ep = ep->nextHdr)
		{
			if (strcmp (ep->hName, hdrName) == 0)
			{
				strcpy (ep->hBuf, artLine);
				return;
			}
		}
	}
}


/*
	Convert string to lowercase
*/

lower (s)
char	*s;
{
	while (*s)
	{
		if (isupper (*s))
			*s = tolower (*s);
		++s;
	}
}


/*
	Get space for string, copy arg into it, and return pointer.
*/

char *newstr (s)
char	*s;
{
char	*p;

	if ((p = calloc (1, (strlen (s) + 1))) == NULL)
		panic ("newstr: out of memory");
	strcpy (p, s);
	return (p);
}


/*
	panic - print message and die with status 1.  Uses vprintf
	so that panic can take variable argument lists.
*/

void
panic (va_alist)
va_dcl
{
va_list	args;
char	*fmt;

	va_start (args);
	fmt = va_arg (args, char *);
	vfprintf (stderr, fmt, args);
	va_end (args);
	fprintf (stderr, "\n");
	exit (1);
}
