/*
	bibinfo.h - constants, types and variables to support operations
	on bibliography entries.

	The BibEnt type is the basic unit of bibliographic information.
	It contains a buffer holding the text of the entry (unmodified),
	and structures which are set up when the entry is read in from
	a file.  These structures allow easy access to the various fields
	of the entry.  They also allow quick checks as to whether a field
	is present in the entry, how many authors there are, and so forth.

	Macros and functions are used to access parts of records.  This
	allows the underlying implementation to change while preserving
	the programming interface.  Currently the implementation is
	as follows:

	Lines of a bibliography entry are read one after the other
	into a buffer.  As each is read, the lines are categorized and
	information set up to allow each field to be accessed easily.
	For each field (except authors), one BibFld struct is set up.
	bfStart is the offset into the text buffer of the beginning of the
	field (starts at the % itself), and bfLen is the length of the
	field (includes length of all continuation lines).  Authors are
	anomalous, since there may be many of them; BibFld's for these are
	stored at the end of the regular field array, and there is a count
	variable.
*/

# ifndef	NULL
# define	NULL	(0)
# endif


# define	beMaxBuf	1024	/* max size of entry's text */
# define	beMaxAuth	20	/* max authors allowed */


typedef struct
{
	int	bfStart;	/* index of start of bib field text */
	int	bfLen;		/* length of bib field text */
} BibFld;


typedef struct
{
	int	beTextLen;		/* length of entry text */
	char	beText[beMaxBuf];	/* entry text */
	BibFld	beFld[26+beMaxAuth];	/* field information */
	int	beAuthCnt;		/* number of authors */
	long	beFilPos;		/* starting position in file */
} BibEnt;


/*
	Macros for accessing various pieces of the bibliographic
	entry.  Most of these can be used as lvalues.  Arguments
	should be as follows:

	BibEnt	*be
	BibFld	*bf
	char	f	('B'..'Z')
	int	n	(0..BEAuthCnt(be)-1)

*/

# define	BEText(be)		(be->beText)
# define	BELen(be)		(be->beTextLen)
# define	BEFilPos(be)		(be->beFilPos)

# define	BEHaveFld(be,f)		(BEFldLen(be,f) != 0)
# define	BEFldPtr(be,f)		(&(be->beFld[(f)-'A']))

# define	BEFldStart(bf)		(bf->bfStart)
# define	BEFldLen(bf)		(bf->bfLen)
# define	BEFldText(be,bf)	(&(BEText(be)[BEFldStart(bf)]))

# define	BEAuthPtr(be,n)		(BEFldPtr(be,'Z'+n+1))
# define	BEAuthLen(be,n)		(BEFldLen(be,'Z'+n+1))
# define	BEAuthText(be,n)	(BEFldText(be,'Z'+n+1))

# define	BEAuthCnt(be)		(be->beAuthCnt)


BibEnt *BEAlloc ();
