/*    numeqn - automatic equation numbering filter for troff */
     
/* Bruce Musicus, June 27, 1981 */
/* Revised brm, May 14, 1983 */
/*   from Research Laboratory of Electronics, MIT, Cambridge, MA 02139 */
     
/* Copyright Musicus 1990.  You may use this package for your personal    */
/*   use, but you may not distribute it commercially, nor embed it in a   */
/*   commercial product without approval by Bruce Musicus                 */

/* EMACS_MODES: c, fill */
     
/* numeqn is a general purpose automatic numbering filter for troff, useful
primarily for numbering equations, but powerful enough to number anything
else as well.  Numbers are formed from ASCII characters, portions of the
header number (from .NH or .sh), and one of 10 different sequence counters,
with optional subscripts.
     
Recognized troff macros:
     
    .EQ [L,I,C] [value,#,##,##+,#tag,##tag,##+tag] [=tag] [indent]
     
    .E# [value,#,##,##+,#tag,##tag,##+tag] [=tag] [text]
     
    .NH [level]        - for ms macro package
     
    .sh level title [numbers ...]    - for me macro package
     
If an ASCII string is specified as the equation number (.EQ), then it will
printed as specified without any processing.  The # character signals
numeqn that it should automatically generate a number.  Specifying # by
itself will use the next sequential number using the default format string.
Specifying ## uses the next sequential subscripted number with the
default format.  A ##+ indicates that the equation number should be
incremented, and the subscript 'a' should be appended.
     
The resulting equation number may be optionally stored in a variable called
"tag" by adding the argument "=tag" following the equation number
specification.  This equation can then be referred to later in the text
by using the macro ".E# #tag".  Numeqn will replace this line with the
number of the tagged equation.  Immediately following the number,
without an intervening space, will be any text following the arguments
to the .E# macro.  This feature is convenient for placing punctuation
immediately following the number.
     
In general, the argument #tag to either .E# or .EQ indicates that the
contents of the variable "tag" is to be used as the number.  Note also that the
arguments #, ##, and ##+ can also be specified to .E#, in which case an
appropriate new number will be generated and inserted in place of the .E#
macro.  This new number may be assigned to a tag variable in exactly the
same manner as for .EQ by adding the argument "=tag".  If a number is
assigned to a tag variable by .E#, however, it will NOT be inserted in
the text and any characters following the =tag argument will be ignored.
This feature allows setting tag variables without inserting the number
in the text.
     
Both .E# and .EQ set the troff string variable E# to the number they generate.
This makes it convenient to embed the number in complicated troff
constructions by referring to: \*(E#
     
Numeqn allows extremely flexible control over the formatting of the
numbers it generates by using a mechanism similar to that used by
"printf" in the standard I/O package.  The variable "format" is an ASCII
string which specifies the default format used for all numbers generated
by using the #, ## and ##+ arguments to .E# or .EQ.  Ordinary characters
in this format string will be printed unchanged in every number.
Information such as the section header and sequence number may be
specified using a sequence of characters starting with a '%'.  The
following commands are recognized:
     
    %h    - insert the complete header number at this point.
     
    %nh    - (1<=n<=5) insert only levels 1 through n of the header
        number at this point.
     
    %s    - insert the next sequential value of counter 0.
     
    %ns    - (0<=n<=9) insert the next sequential value of counter n.
     
    %n.ms    - (0<=n<=9 and 0<=m<=5) insert the next sequential value of
        counter n, and make sure this counter is reset to 0 each time
        a new section header at level m or below is created.  If m=0,
        the counter will never reset.
     
    %%    - insert the character '%'
     
The default format is (%2h%0.2s), which will produce numbers like (3.5.6) or
(3.5.6c) in section 3.5 for the sixth item (third subscripted number.)
Sequence counters will be reset to 0 whenever a .NH command (or .sh
command for the -me macro package) is encountered which would change the
part of the header number which is used in forming the equation number.
This rule may be changed by using the full %n.ms specification.
The default format may be changed, for example, by:
    .E# (A:%h%s) =format
which would set the default format to print numbers in the form: (A:3.5.6)
     
The arguments #tag, ##tag, ##+tag to .EQ and .E# actually allow using the
contents of a tag variable to specify the for this number.  If the tag
variable contains a string with no '%' characters, then its contents
will be used unmodified as the number to be inserted.  If '%h' or '%s'
arguments are in the tag variable, then the header and sequence number
will be inserted appropriately, with the sequence and subscripts
advanced as specified.  Thus, for example, we could define a numbering
format T for tables by:
    .E# %1h%2s =T
No text is generated for this line.  After this, however, a new table
number could be generated by:
    .E# #T
which will insert a number such as: 3.4  if we are in section 3 and this
is the fourth table.  Note that the T format uses sequential counter
number 2, and therefore tables will be numbered independently of
equations, which will use the default format and sequential counter
number 0.
     
Tag variable names may be any ASCII string.  To include spaces or tabs in
the name, simply enclose the name in quotes.  Certain names are reserved
and have special meaning:
     
    format    - contains the default numbering format string.  Setting
        this variable changes the default numbering style.
     
    %h    - contains the present header number.  To change the header
        level, use something like:  ".E# 2.5 =%h" which sets the
        current header to 2.5.
     
    %nh    - (1<=n<=5) contains the header number truncated to
        levels 1 through n.
     
    %s    - contains the last sequence number used on sequence counter
        0.  Changing this variable by setting it to:  ".E# 5c =%s"
        would change the next sequential number to 5c.
     
    %ns    - (0<=n<=9) contains the last sequence number used on
        sequence counter n.
     
Tag variable names formed solely out of digits are considered "local",
and may be redefined at any time.  Tag names which include alphabetic
characters and/or punctuation are considered "global", and must not be
redefined.
     
Various options are available on the command line:
     
    -f string    use "string" as the default numbering format
    -h level    use "level" as the initial header number (this also
              affects the first .NH command
    -s number    use "number" as the initial number in counter 0
    -t tagfile    when finished, output a list of all tag variable
              values to "tagfile"
    -ms        -ms macro package format (default)
    -me        -me macro package format
     
The -t option constructs an output file containg lines of the form:
    .E# (3.5.6c) =tag
for every tag variable name which is defined in the file.  This is
convenient, for example, when running off very long text files in sections.
Simply include the tag file from the previous section as the first text
file when troffing the next section - this will correctly define all the
tag variables, and will allow referring to equation numbers from
previous sections.  A similar trick also allows forward referencing of
equation numbers - run numeqn on the file once with the -t option in
order to collect all the tag variable definitions in a tagfile.  Then
run the tagfile followed by the same text file through numeqn again.
Now single level forward references can be resolved.
     
*******************
     
BUGS -
   Each sequence counter should be used in only one format string.  If a
counter must be shared between several formats, be sure that all use
exactly the same number of header levels.  Otherwise, the sequence
counter will reset according to the last defined format string header level.
     
     
*/
     
     
/* #define DEBUG 1 */
     
     
     
/* Global Variables */
     
#include <stdio.h>
#include <ctype.h>
     
#define max(x, y)    (((x) < (y)) ? (y) : (x))
#define mabs(x)        ((x < 0) ? -(x) : (x))
#define min(x, y)    (((x) < (y)) ? (x) : (y))
     
char *malloc(),*realloc(),*calloc() ;
char *skipsp(),*skipqt(),*eval(),*getstr(),*getstrq(),*gethead(),*newhead() ;
char *getseq(),*newseq() ;
int parse(),findtag(),getline() ;
     
/* Global variables */
     
char buff[100] ;    /* junk string buffer */
     
#define LINLEN 600
char chline[LINLEN+2] ;    /* input line buffer */
     
int argflag = 0 ;    /* flags if any file names found on command line */
int meflag = 0 ;    /* flags -me option */
     
char *tname = NULL;  /* tagfile name (=0 if no -t option) */
     
char *fname = "stdin" ;    /* name of file being processed */
int fline ;        /* line number of file being processed */
     
#define NUMLEN 80
char cnum[NUMLEN+1] ;    /* buffer for evaluated number */
     
#define TAGLEN 80
char ctag[TAGLEN+1] ;    /* buffer for tag name */
     
char **taglist ;    /* pointer to list of pointers to tag,value pairs */
            /*   list organized alphabetically */
unsigned ntags ;    /* count of defined tag variables */
unsigned maxtags ;    /* allocated number of tags */
     
#define FORMAT '\001'
#define HEADER '\002'
#define SEQUENCE '\014'
     
#define FORMLEN 80
char cform2[FORMLEN+1] = "(%2h%0.2s)" ;  /* ASCII default format */
char cform[FORMLEN+1] = "(\004\014)" ;   /* processed default format */
     
#define HNUM 6
int hlev[HNUM] = {
    0, 0, 0, 0, 0, 0 } ;    /* header number at each level */
int nh = 1 ;        /* present level */
int nhflag = 0 ;    /* flags whether header variable %h has been set */
     
#define SNUM 10
struct {
    int no,sub,lev ;
} seq[SNUM] = {
    { 0, 0, 2 },
    { 0, 0, 0 },
    { 0, 0, 0 },
    { 0, 0, 0 },
    { 0, 0, 0 },
    { 0, 0, 0 },
    { 0, 0, 0 },
    { 0, 0, 0 },
    { 0, 0, 0 },
    { 0, 0, 0 } } ;    /* sequence counters, subscripts, reset level */
     
char *string[] = {    /* predefined variables */
    "%0h\0\002",
    "%0s\0\014",
    "%1h\0\003",
    "%1s\0\015",
    "%2h\0\004",
    "%2s\0\016",
    "%3h\0\005",
    "%3s\0\017",
    "%4h\0\006",
    "%4s\0\020",
    "%5h\0\007",
    "%5s\0\021",
    "%6h\0\010",
    "%6s\0\022",
    "%7h\0\011",
    "%7s\0\023",
    "%8h\0\012",
    "%8s\0\024",
    "%9h\0\013",
    "%9s\0\025",
    "%h\0\002",
    "%s\0\014",
    "format\0\001",
    ""
} ;
     
/* MAIN PROGRAM */
/* checks for options, picks off file names, calls "doline" to do the work,
   prints the tag,value pairs at end if -t option used  */
     
main(argc,argv)
int argc ;
char **argv ;
{
    char c ;
     
    init() ;    /* initialize variables and tag storage */
     
    while(--argc>0) {
        if(**++argv == '-') {
#ifdef DEBUG
fprintf(stderr,"switch %c\n",argv[0][1]) ;
#endif
            switch(c = argv[0][1]) {
            case 'f':    if(--argc>0) {
                        strcpy(cform2,*++argv) ;
                        newform(cform,cform2) ;
                    } else {
                        erropt(c) ;
                    }
                    break ;
     
            case 'h':    if(--argc>0) {
                        newhead(0,*++argv) ;
                    } else {
                        erropt(c) ;
                    }
                    break ;
     
            case 's':    if(--argc>0) {
                        newseq(0,*++argv) ;
                    } else {
                        erropt(c) ;
                    }
                    break ;
     
            case 't':    if(--argc>0) {
                        tname = *++argv ;
                    } else {
                        erropt(c) ;
                    }
                    break ;
     
            case 'm':    if(argv[0][2]=='e') meflag = 1 ;
                    break ;
     
            default:    fprintf(stderr,
                      "numeqn: unknown option -%c\n",c) ;
                    break ;
            }
        } else {
            fname = *argv ;
#ifdef DEBUG
fprintf(stderr,"file %s\n",fname) ;
#endif
            argflag++ ;
            if(freopen(fname,"r",stdin)==NULL) {
                fprintf(stderr,
                  "numeqn:  failed to open file    %s\n",fname) ;
            } else {
                dofile() ;
            }
        }
    }
    if(argflag==0) {  /* if no files specified, read stdin */
#ifdef DEBUG
fprintf(stderr,"file stdin") ;
#endif
        dofile() ;
    }
     
/* At end, print all tag,value pairs if -t option */
     
    if(tname != NULL) {
        if(freopen(tname,"w",stdout)==NULL) {
            fprintf(stderr,
              "numeqn:  failed to open tag file %s\n",tname) ;
            exit(1) ;
        } else {
            prtags() ;
        }
    }
     
    exit(0) ;
}
     
     
/* Prints option error message */
erropt(c)
char c ;
{
    fprintf(stderr,"numeqn:  mising -%c option argument\n",c) ;
    return ;
}
     
     
/* Read file line by line, looking for lines beginning with '.'
   Calls handlers for ".EQ",".E#",".NH"
   Otherwise outputs the line and reads the next until reaches EOF
*/
dofile()
{
    char *p ;
     
    fline = 0 ;
     
    while(getline(chline,LINLEN,stdin) > 0) {
#ifdef DEBUG
fprintf(stderr,"getline:  %s",chline) ;
#endif
        if(*chline=='.') {
            p = skipsp(chline+1) ;
            if(*p=='E' && *(p+1)=='Q') {
#ifdef DEBUG
fprintf(stderr,".EQ found\n") ;
#endif
                parseq(p+2) ;
            } else if (*p=='E' && *(p+1)=='#') {
#ifdef DEBUG
fprintf(stderr,".E# found\n") ;
#endif
                parscmd(p+2) ;
            } else if (meflag==0 && *p=='N' && *(p+1)=='H') {
#ifdef DEBUG
fprintf(stderr,".NH found\n") ;
#endif
                parsnh(p+2) ;
            } else if (meflag==1 && *p=='s' && *(p+1)=='h') {
#ifdef DEBUG
fprintf(stderr,".sh found\n") ;
#endif
                parssh(p+2) ;
            } else {
                fputs(chline,stdout) ;
            }
        } else {
            fputs(chline,stdout) ;
        }
    }
    return ;
}
     
     
/* Reads next text line up to newline or up to n chars from stdin
   Increments fline line counter
   Handles invisible newline correctly (even in troff comment)
   Returns count of characters read
*/
int getline(p,n,fp)
char *p ;
register int n ;
FILE *fp ;
{
    char *p2 = p ;
    register int c ;
    int c2 ;
    int nlflag = 0 ;
     
    while(--n>0 && (c=getc(fp))!=EOF) {
        if(!nlflag && c=='\\') {
            switch (c2=getc(fp)) {
     
            case '\n':    fline++ ; /* invisible newline */
                    break ;
     
            case '\"':    nlflag = 1 ; /* troff comment */
     
            default:    *p++ = c ;
                    *p++ = c2 ;
                    --n ;
                    break ;
            }
        } else {
            if((*p++ = c)=='\n')
                break ;
        }
    }
    fline++ ;
    *p = '\0' ;
    return(p-p2) ;
}
     
     
/* Parse .EQ macro command
   Skips L,I,C argument if there
   Calls parse() to create number, set tag, and set troff string variable E#
   Outputs new .EQ macro command with correct number inserted
*/
parseq(p)
char *p ;
{
    char *pl,*pr ;
     
    pr = pl = skipsp(p) ;
    if((*pl=='L' || *pl=='I' || *pl=='C')
                && (isspace(*(pl+1)) || *(pl+1)=='\0')) {
        pr = pl = skipsp(pl+1) ;
    }
    parse(&pr) ;
    p = chline-1 ;
    while(++p<pl) {
        putchar(*p) ;
    }
    if(*cnum) fputs(cnum,stdout) ;
    while(*pr!='\0') {
        putchar(*pr) ;
        pr++ ;
    }
    return ;
}
     
     
/* Parse .E# macro command
   Call parse() to calculate number, set tag, set troff string variable E#
   Any strings following the value, which do not start with '=', will be
     appended directly after the number without any space.
   If no tag set, then outputs number
*/
parscmd(p)
char *p ;
{
    p = skipsp(p) ;
    if(!parse(&p)) {
        if(*cnum) fputs(cnum,stdout) ;
        p = skipsp(p) ;
        if(*p) fputs(p,stdout) ;
    }
    return ;
}
     
     
/* Parse .NH macro command
   If header level has been changed by setting the tag %h or %nh, first
     outputs a set of troff commands to effectively execute most of an .NH
     command to set the level.
   Updates the header numbers properly.
   Resets all sequence numbers whose header level is less than or equal
     to this level.
   Outputs the .NH line unchanged.
*/
parsnh(p)
char *p ;
{
    int i ;
     
    if(nhflag) {        /* level has been changed in numeqn */
        nhflag = 0 ;
        printf(".RT\n.if \\\\n(1T .sp 1\n.if !\\\\n(1T .BG\n.RT\n") ;
                /* initializes if this is first .NH */
        printf(".nr NS %d\n",nh) ;
                /* Sets .NH header level */
        for(i=1 ; i<HNUM ; i++)
            printf(".nr H%d %d\n",i,hlev[i]) ;
                /* Sets individual level registers */
    }
     
    p = skipsp(p) ;
    nh = -1 ;
    if(isdigit(*p) && *p < HNUM+'0') {
        nh = (*p++) - '0' ;
    }
    if(nh==0) {
        nh = 1 ;
        hlev[1] = 0 ;
    } else if(nh < 0) {
        nh = 1 ;
    }
     
    hlev[nh]++ ;
    for(i=nh+1 ; i<HNUM ; i++)
        hlev[i] = 0 ;
     
    for(i=0 ; i<SNUM ; i++) {    /* reset sequence counters */
        if(seq[i].lev >= nh)
            seq[i].no = seq[i].sub = 0 ;
    }
     
    fputs(chline,stdout) ;
    return ;
}
     
     
/* Parse .sh macro command (-me macro package)
   Resets all sequence numbers whose header level is less than or equal
     to this level.
   Outputs the .sh line unchanged.
*/
parssh(p)
char *p ;
{
    int i,j ;
     
    p = skipsp(p) ;
    if(isdigit(*p) && *p < HNUM+'0') {
        nh = (*p++) - '0' ;
        if(nh==0) {
            nh = 1 ;
            hlev[1] = 0 ;
        } else if(nh < 0) {
            nh = 1 ;
        }
        while(*p && !isspace(*p)) ++p ;
    }
    hlev[nh]++ ;
    for(i=nh+1 ; i<HNUM ; i++)
        hlev[i] = 0 ;
     
    for(i=0 ; i<SNUM ; i++) {    /* reset sequence counters */
#ifdef DEBUG
fprintf(stderr,"parssh: seq[%d].lev=%d, nh=%d\n",i,seq[i].lev,nh) ;
#endif
        if(seq[i].lev >= nh)
            seq[i].no = seq[i].sub = 0 ;
    }
     
    p = skipsp(p) ;
#ifdef DEBUG
fprintf(stderr,"parssh: before title %s\n",p) ;
#endif
    p = skipqt(p) ;
    p = skipsp(p) ;
    i = 1 ;
    while(i<HNUM && (isdigit(*p) || *p=='-')) {
#ifdef DEBUG
fprintf(stderr,"parssh: explicit section numbers %s\n",p) ;
#endif
        if(*p != '-') {
#ifdef DEBUG
fprintf(stderr,"parssh: reset at section level %d\n",i) ;
#endif
            hlev[i] = atoi(p) ;
            for(j=0 ; j<SNUM ; j++) {
                if(seq[j].lev>=i)
                    seq[j].no = seq[j].sub = 0 ;
            }
        }
        i++ ;
        p = skipqt(p) ;
        p = skipsp(p) ;
    }
     
    fputs(chline,stdout) ;
    return ;
}
     
     
/* Skips spaces and tabs */
char *skipsp(p)
char *p ;
{
    while(*p=='\t' || *p==' ') ++p ;
    return(p) ;
}
     
     
/* Skip text or quoted text */
char *skipqt(p)
char *p ;
{
    char delim ;
    if(*p=='\'' || *p=='\"') {
        delim = *p ;
        while(*++p) {
            if(*p=='\\') {
                ++p ;
            } else if(*p==delim) {
                return(p+1) ;
            }
        }
    } else {
        while(*p && !isspace(*p)) ++p ;
    }
    return(p) ;
}
     
     
/* Parses value and =tag fields, sets tag variable, handles special tag
     variables, adds new tags to list, output troff command to set E#
   Calls eval() to evaluate number
   Returns 1 if detects "=tag" argument, else returns 0
*/
int parse(pp)
char **pp ;
{
    int fndtag = 0 ;
    char *p = *pp ;
    char **ptag,*pval ;
     
    p = eval(p) ;
    p = skipsp(p) ;
    if(*p=='=') {
        fndtag = 1 ;
        p = skipsp(p+1) ;
        p = getstr(p,ctag,TAGLEN) ;
        if(findtag(ctag,&pval,&ptag)) {
            if(*pval==FORMAT) {
                strcpy(cform2,cnum) ;
                newform(cform,cnum) ;
            } else if(*pval>=HEADER && *pval<HEADER+10) {
                if(meflag==0) newhead(*pval-HEADER,cnum) ;
                else error("Cannot set variable %%%dh",
                                *pval-HEADER) ;
            } else if(*pval>=SEQUENCE && *pval<SEQUENCE+10) {
                newseq(*pval-SEQUENCE,cnum) ;
            } else if(strcmp(cnum,pval)!=0) {
                pval = ctag ;
                while(*pval) {
                    if(isdigit(*pval++)) continue ;
                    else {
                        error("duplicate tag %s",ctag);
                        break ;
                    }
                }
                reptag(ctag,cnum,ptag) ;
            }
        } else {
            addtag(ctag,cnum,ptag) ;
        }
    }
    if(*cnum) printf(".ds E# \"%s\n",cnum) ;
    *pp = p ;
    return(fndtag) ;
}
     
     
/* Evaluates number
   Handles explicit value, #, ##, ##+, #tag, ##tag, ##+tag
   Handles special tag variables also
   Returns pointer to next char after value
   Puts evaluated number in "cnum"
*/
char *eval(p)
char *p ;
{
    int sub = 1 ;
    char delim = '\0' ;
    char *pform, **ptag ;
     
    p = skipsp(p) ;
    if(*p=='\"' || *p=='\'')
        delim = *p++ ;
     
    if(*p=='#') {
        if(*++p=='#') {
            sub = 2 ;
            if(*++p=='+') {
                sub = 3 ;
                ++p ;
            }
        }
        if(delim=='\0')
            p = getstr(p,ctag,TAGLEN) ;
        else
            p = getstrq(p,ctag,TAGLEN,delim) ;
        if(*ctag=='\0') {
            form(cform,sub) ;
        } else {
            if(!findtag(ctag,&pform,&ptag)) {
                error("tag %s not found",ctag) ;
                sprintf(cnum,"(???%s???)",ctag) ;
            } else {
                if(*pform==FORMAT) {
                    form(cform,sub) ;
                } else if(*pform>=HEADER && *pform<HEADER+10) {
                    gethead(*pform-HEADER,cnum) ;
                } else if(*pform>=SEQUENCE &&
                            *pform<SEQUENCE+10){
                    getseq(*pform-SEQUENCE,cnum,0) ;
                } else {
                    newform(buff,pform) ;
                    form(buff,sub) ;
                }
            }
        }
    } else {
        if(delim=='\0')
            p = getstr(p,cnum,NUMLEN) ;
        else
            p = getstrq(p,cnum,NUMLEN,delim) ;
    }
    return(p) ;
}
     
     
/* Get string at p into pdest up to nmax chars, until space character
   Terminate with null
*/
char *getstr(p,pdest,nmax)
char *p,*pdest ;
int nmax ;
{
    if(*p=='\"' || *p=='\'')
        return(getstrq(p+1,pdest,nmax,*p)) ;
    while(!isspace(*p) && *p!='\0' && --nmax>0)
        *pdest++ = *p++ ;
    if(nmax<=0)
        error("string too long",p) ;
    *pdest = '\0' ;
    return(p) ;
}
     
/* Get string at p into pdest up to nmax chars, up to delim character.
   Terminate with null
*/
char *getstrq(p,pdest,nmax,delim)
char *p,*pdest,delim ;
int nmax ;
{
    while(*p!=delim && *p!='\n' && *p!='\0' && --nmax>0)
        *pdest++ = *p++ ;
    if(nmax <= 0) {
        error("string too long",p) ;
    } else if(*p==delim) {
        ++p ;
    } else {
        error("imbalanced quotes",p) ;
    }
    *pdest = '\0' ;
    return(p) ;
}
     
     
/* Evaluate format string, substituting in header and generating next sequence
     number as specified
   Uses format string processed by newform()
   Calls getseq() and gethead() to do the work
   Inputs: p=pointer to format string, seqflag=0 if no increment, =1 if
     increment number, =2 if increment subscript, =3 if increment
     number and set subscript to 'a'.
*/
form(p,seqflag)
char *p ;
int seqflag ;
{
    char *pnum = cnum ;
     
    while(*p) {
#ifdef DEBUG
fprintf(stderr,"entering form:  hlev[1]=%d, hlev[2]=%d, seq[0]=%d,%d\n",
    hlev[1],hlev[2],seq[0].no,seq[0].sub) ;
#endif
        if(*p>=SEQUENCE+10) {
            *pnum++ = *p++ ;
        } else if(*p>=SEQUENCE && *p<SEQUENCE+10) {
            pnum = getseq((*p++)-SEQUENCE,pnum,seqflag) ;
        } else if(*p>=HEADER && *p<HEADER+10) {
            pnum = gethead((*p++)-HEADER,pnum) ;
        }
    }
    *pnum = '\0' ;
    return ;
}
     
     
/* Converts ASCII format string into internal format which can be processed
     more conveniently (this scheme is intended primarily for
     speeding up processing of the default format string.  It will slow down
     processing of other format strings.)
   Interprets %h, %nh, %s, %ns, %n.ms, %% correctly.  If sequence counter
     specified, calculates header level at which counter should be reset
   Inputs: p=pointer ASCII to string, pform=pointer to processed format
*/
newform(pform,p)
char *pform,*p ;
{
    int hmax = 0 ;
    int sflag = -1 ;
    int n,m ;
     
    while(*p) {
        if(*p!='%') {
            *pform++ = *p++ ;
        } else {
            n = -1 ;
            m = -1 ;
            ++p ;
            if(isdigit(*p)) {
                n = (*p++)-'0' ;
            }
            if(*p=='.') {
                if(isdigit(*++p)) m = (*p++)-'0' ;
            }
            switch(*p) {
     
            case 'h':
                if(n<0 || n>=HNUM) n = HNUM-1 ;
                if(n>hmax) hmax = n ;
                *pform++ = HEADER+n ;
                break ;
     
            case 's':
#ifdef DEBUG
fprintf(stderr,"newform: %%s detected: n=%d, m=%d, SEQUENCE=%d\n",n,m,SEQUENCE);
#endif
                if(n<0) n=0 ;
                *pform++ = SEQUENCE+n ;
                if(m>=0) {
                    seq[n].lev = m ;
                } else {
                    seq[n].lev = HNUM-1 ;
                    sflag = n ;
                }
                break ;
     
            case '%':
                *pform++ = '%' ;
                break ;
     
            default:
                error("unknown format character %%%c",*p) ;
                break ;
            }
            ++p ;
        }
    }
#ifdef DEBUG
fprintf(stderr,"parssh: sflag=%d, hmax=%d\n",sflag,hmax) ;
#endif
    if(sflag>=0)
        seq[sflag].lev = hmax ;
    *pform = '\0' ;
    return ;
}
     
     
/* Convert header numbers in hlev[] array into an ASCII header number
   Inputs: n=header level 0-HNUM (0 means full header, n>0 means
     truncate number to n levels), p=string buffer
*/
char *gethead(n,p)
int n ;
char *p ;
{
    int i ;
     
    if(n==0)
        n = nh ;
    else
        n = min(n,nh) ;
     
    for(i=1 ; i<=n ; i++) {
        sprintf(p,"%d.",hlev[i]) ;
        p += strlen(p) ;
    }
    return(p) ;
}
     
     
/* Reads ASCII string and converts into a set of header numbers in hlev[]
   Handles both level 0 (full header like 4.5.6.7.) and levels 1-HNUM
     (single header number like 5)
*/
char *newhead(n,p)
int n ;
char *p ;
{
    int i,j = 0 ;
     
    if(n==0) n = HNUM-1 ;
     
    while(j<n && isdigit(*p)) {
        i = (*p++)-'0' ;
        while(isdigit(*p)) {
            i = 10*i + (*p++) - '0' ;
        }
        hlev[++j] = i ;
        if(*p=='.') ++p ;
    }
    p = skipsp(p) ;
    if(j<=0 || *p!='\0') {
        error("To set header, try something like: 2.4.6 =%%h",i) ;
    }
    if(j>0) {
        nh = j ;
        nhflag = 1 ;
    }
    return(p) ;
}
     
     
/* Convert specified sequence counter to ASCII, optionally incrementing it
   Inputs: n=counter number, p=ASCII buffer, seqflag=0 if simple fetch,
     =1 if increment counter and clear subscript, =2 if increment subscript
*/
char *getseq(n,p,seqflag)
int n, seqflag ;
char *p ;
{
    int j ;
     
    if(seqflag==3) {
        if(seq[n].no < 0)
            seq[n].no += 32000 ;
        else
            seq[n].no++ ;
        seq[n].sub = 1 ;
    } else if(seqflag==2) {
        if(seq[n].no < 0)
            seq[n].no += 32000 ;
        else {
            if(seq[n].sub==0) seq[n].no++ ;
            seq[n].sub++ ;
        }
    } else if(seqflag==1) {
        seq[n].sub = 0 ;
        if(seq[n].no < 0)
            seq[n].no += 32000 ;
        else
            seq[n].no++ ;
    }
    sprintf(p,"%d",seq[n].no) ;
    p += strlen(p) ;
    if((j=seq[n].sub)>0) {
        if(j>26) {
            *p++ = (j-1)/26+'a'-1 ;
        }
        *p++ = ((j-1)%26)+'a' ;
    }
    *p='\0' ;
    return(p) ;
}
     
     
/* Converts ASCII string into new sequence number (should have form like
     5a =%2s)
   Inputs: n=counter number, p=ASCII buffer
*/
char *newseq(n,p)
int n ;
char *p ;
{
    int i = 0 ;
    int flag = 0 ;
     
    if(isdigit(*p)) {
        flag = 1 ;
        while(isdigit(*p))
            i = 10*i+(*p++)-'0' ;
        seq[n].no = i ;
        seq[n].sub = 0 ;
    }
    if(islower(*p)) {
        flag = 1 ;
        seq[n].sub = (*p++)-'a'+1 ;
        if(islower(*p))
            seq[n].sub = seq[n].sub*26+(*p++)-'a'+1 ;
    }
    if(flag==0 || *p!='\0') {
         error("To set sequence number, try for example:  5c = %%%ds",n);
    } else {
        seq[n].no -= 32000 ;
    }
    return(p) ;
}
     
     
/* Initialization of variables and tag table */
init()
{
    int i ;
     
    maxtags = 100 ;
    if((taglist=(char**)malloc((unsigned)maxtags*sizeof(char*)))
                            == (char**)NULL) {
        error("malloc failed - out of memory",i) ;
        exit(1) ;
    }
    for(i=0 ; *string[i]!='\0' ; i++)
        taglist[i] = string[i] ;
    ntags = i ;
    return ;
}
     
     
/* Looks up tag name in list of tag variables using binary search algorthm
   Returns pointer to list of pointers where tag should be, and pointer
     to the string corresponding to the tag variable
   Inputs: tag=tag name, pform=address of pointer to tag value,
     ptag=address of pointer to list of pointers where tag name should be
   Returns 1 if tag found, 0 if not
*/
int findtag(tag,pform,ptag)
char *tag,**pform,***ptag ;
{
    char **ptag1 = taglist ;
    unsigned size = ntags ;
    int i ;
    char *p2 ;
     
    while(size!=0) {
        p2 = *(ptag1+size/2) ;
        if((i=strcmp(tag,p2))==0) {
            *pform = p2+strlen(p2)+1 ;
            *ptag = ptag1+size/2 ;
#ifdef DEBUG
fprintf(stderr,"findtag: tag %s found, format=%s\n",**ptag,*pform) ;
#endif
            return(1) ;
        } else if(i>0) {
            ptag1 += size/2+1 ;
            size -= size/2+1 ;
        } else {
            size /= 2 ;
        }
    }
    *ptag = ptag1 ;
    *pform = "" ;
#ifdef DEBUG
fprintf(stderr,"findtag: tag %s not found, format=%s, pointer=%s\n",tag,*pform,
  **ptag) ;
#endif
    return(0) ;
}
     
     
/* Adds tag name and value to taglist, expanding the list of tags if necessary
   Inserts tag name at location returned by findtag()
   Inputs: tag=tag name, pform=value, ptag=pointer to insertion location
     in pointer list
*/
addtag(tag,pform,ptag)
char *tag,*pform,**ptag ;
{
    int i ;
     
    if(ntags==maxtags) {    /* If present taglist is full, expand it */
        i = ptag - taglist ;
        maxtags += 100 ;
     
        if((taglist=(char**)realloc((char*)taglist,
                (unsigned)maxtags*sizeof(char*)))==(char**)NULL) {
            error("expanding to %d tags, realloc failed",maxtags);
            exit(1) ;
        }
        ptag = taglist+i ;
    }
     
    {
        register char **p2 = taglist+ntags ;
        register char **p1 = p2-1 ;
        register j=p2-ptag ;
        while(j-->0) *p2-- = *p1-- ;
    }
    ntags++ ;
    filltag(tag,pform,ptag) ;
    return ;
}
     
     
/* Changes tag value of tag that is already in tag list
   Inputs: tag=tag name, pform=tag value, ptag = pointer to point of insertion
*/
reptag(tag,pform,ptag)
char *tag,*pform,**ptag ;
{
    free(*ptag) ;
    filltag(tag,pform,ptag) ;
}
     
     
     
/* Gets space for tag name, format string pair, inserts pointer in tag list,
     inserts name and format string
   Inputs: tag=tag name, pform=format string, ptag=point of insertion
*/
filltag(tag,pform,ptag)
char *tag,*pform,**ptag ;
{
    char *p ;
     
    if((p= *ptag=malloc((unsigned)(strlen(tag)+strlen(pform)+2)))==NULL) {
        error("malloc failed - too many tags (%d)",ntags) ;
        exit(1) ;
    }
    while(*p++ = *tag++) ;
    while(*p++ = *pform++) ;
    return ;
}
     
     
/* Prints the entire tag list in format ".E# value =tag".  Also sets format
string to the value used
*/
     
prtags()
{
    char **p = taglist ;
    char *p2,*p3 ;
    int i = ntags ;
     
    while(i-->0) {
        p2 = *p++ ;
        p3 = p2+strlen(p2)+1 ;
        if(*p3>=SEQUENCE+10)
            printf(".E# \"%s\" =\"%s\"\n",p3,p2) ;
    }
    printf(".E# \"%s\" =format\n",cform2) ;
    return ;
}
     
     
/* Prints error message to stderr, together with file name and line number
   Inputs: str=error message in printf format, var=value to be printed by
     fprintf.  Also uses fname and fline global variables
*/
/*VARARGS1*/
error(str,var)
char *str,*var ;
{
    fputs("numeqn: ",stderr) ;
    fprintf(stderr,str,var) ;
    fprintf(stderr,"\n  file %s, line %d\n",fname,fline) ;
    return ;
}

