/*
	table (version 2.0) - a utility for creating nroff driver tables.

	table:  a program to compile nroff terminal driver tables.
		Written by Bruce Townsend Oct, 1987.

	Copyright (c) 1987 by Bruce Townsend and Bell-Northern Research.
	Permission is granted to use and distribute, except for profit,
	providing this copyright notice and the author's name is included.
	No warranty of any kind is expressed or implied, and no liability of
	any kind is assumed by either the author or Bell-Northern Research.

	The contributions of Ian Darwin are gratefully acknowledged.
*/

#define C_SIZE	20000	/* The maximum amount of character data allowed
			   in the initialized structure t - increase if
			   necessary */

#include <stdio.h>
#include "table.h"	/* This file contains the definition of the
			   the structures t and t_stor */

/*	The compiled tab file contains three primary elements:
	-  An integer (c_size) which gives the size in bytes of the character
	   data (the third element of the file).
	-  The structure t_stor, which contains integer data and integer
	   indices into the character data that follows.
	-  c_size bytes of character data.
	Note that the file size in bytes is therefore:
	c_size + sizeof (int) + sizeof (t_stor)
*/

extern struct t t;	/* Defined in the tab file source */
struct t_stor t_stor;	/* This structure is stored in the compiled tab file */

char	c_data[C_SIZE];	/* The character data to be stored in the tab file */
int	c_size;		/* The amount of character data in bytes */

main (argc, argv)
int	argc;
char	*argv[];
{
	FILE	*table;
	int	i;

	if (argc != 2) {	/* Need a file name argument */
	    fprintf (stderr, "Usage: table tabfilename\n");
	    exit (1);
	}

	if ((table = fopen (argv[1], "w")) == NULL) {	/* Open the file */
	    fprintf (stderr, "Could not open file %s for writing\n", argv[1]);
	    exit (1);
	}

	/* Copy the integer values from the initialized structure t
	   to the storage structure t_stor */
	t_stor.bset = t.bset;
	t_stor.breset = t.breset;
	t_stor.Hor = t.Hor;
	t_stor.Vert = t.Vert;
	t_stor.Newline = t.Newline;
	t_stor.Char = t.Char;
#ifdef KANJI
	t_stor.Kchar = t.Kchar;
#endif KANJI
	t_stor.Em = t.Em;
	t_stor.Halfline = t.Halfline;
	t_stor.Adj = t.Adj;

	/* Find each string in the character data table c_data, or add it to
	   the table if it is not there, and provide an index to it which
	   is stored in t_stor */
	t_stor.twinit = addstring (t.twinit);
	t_stor.twrest = addstring (t.twrest);
	t_stor.twnl = addstring (t.twnl);
	t_stor.hlr = addstring (t.hlr);
	t_stor.hlf = addstring (t.hlf);
	t_stor.flr = addstring (t.flr);
	t_stor.bdon = addstring (t.bdon);
	t_stor.bdoff = addstring (t.bdoff);
	t_stor.iton = addstring (t.iton);
	t_stor.itoff = addstring (t.itoff);
	t_stor.ploton = addstring (t.ploton);
	t_stor.plotoff = addstring (t.plotoff);
	t_stor.up = addstring (t.up);
	t_stor.down = addstring (t.down);
	t_stor.right = addstring (t.right);
	t_stor.left = addstring (t.left);
	for (i = 0; i < 256 - 32; i++)
	    t_stor.codetab[i] = addchar (t.codetab[i]);
	t_stor.zzz = 0;		/* The null terminator */

	/* Write to the tab file the amount of character data in bytes,
	   the structure t_stor, and the character data */
	if (fwrite (&c_size, sizeof (c_size), 1, table) != 1 ||
	    fwrite (&t_stor, sizeof (t_stor), 1, table) != 1 ||
	    fwrite (c_data, sizeof (*c_data), c_size, table) != c_size) {
	    fprintf (stderr, "Write to file failed\n");
	    exit (1);
	}

	/* Close the tab file */
	if (fclose (table)) {
	    fprintf (stderr, "File %s not closed properly\n", argv[1]);
	    exit (1);
	}
}

addstring (string)
char	*string;
{
	if (string)	/* If pointer is non-zero add string to table */
	    return (string_search (string, strlen (string) + 1));

	else		/* If pointer is zero return 0 index */
	    return (0);
}

addchar (string)
char	*string;
{
	if (string)	/* If pointer is non-zero add string to table */
			/* Note that the string is at least 2 chars long
			   and that the first 2 bytes may be null */
	    return (string_search (string, strlen (string + 2) + 3));

	else		/* If pointer is zero return 0 index */
	    return (0);
}

/*	string_search searches through the character array c_data to
	find the string.  If found, the routine returns an integer
	index that locates the string in c_data.  If it is not
	found, the string is added to c_data, and c_size, the amount
	of data in c_data, is updated.
*/
string_search (string, length)
char	*string;	/* The string to find in or add to the table */
int	length;		/* The string length including the null terminator */
{
	int	s_index, c_index, match;
	char	*pointer;

	for (s_index = 0; s_index <= c_size - length; s_index++) {
	    pointer = string;
	    match = 1;
	    for (c_index = s_index; c_index < s_index + length;) {
		if (*pointer++ != c_data[c_index++]) {
		    match = 0;
		    break;
		}
	    }
	    if (match) return (s_index);
	}

	s_index = c_size;
	while (length--)
	    c_data[c_size++] = *string++;

	return (s_index); 
}
