/*
	table (version 2.0) - a utility for creating nroff driver tables.

	elbat:  a program to de-compile nroff terminal driver tables.
		Written by Bruce Townsend Oct, 1987.
		Based on a program by:
		Matt Crawford, University of Chicago, 10 May 1984

	Copyright (c) 1987 by Bruce Townsend and Bell-Northern Research.
	Permission is granted to use and distribute, except for profit,
	providing this copyright notice and the author's name is included.
	No warranty of any kind is expressed or implied, and no liability of
	any kind is assumed by either the author or Bell-Northern Research.
*/

#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "table.h"

extern char	*structure;

char	*codelabel[] = {
	"space",	"!",	"\"",	"#",	"$",	"%",	"&",
	"' close",	"(",	")",	"*",	"+",	",",	"- hyphen",
	".",	"/",	"0",	"1",	"2",	"3",	"4",	"5",
	"6",	"7",	"8",	"9",	":",	";",	"<",	"=",
	">",	"?",	"@",	"A",	"B",	"C",	"D",	"E",
	"F",	"G",	"H",	"I",	"J",	"K",	"L",	"M",
	"N",	"O",	"P",	"Q",	"R",	"S",	"T",	"U",
	"V",	"W",	"X",	"Y",	"Z",	"[",	"\\",	"]",
	"^",	"_ dash",	"` open",	"a",	"b",	"c",
	"d",	"e",	"f",	"g",	"h",	"i",	"j",	"k",
	"l",	"m",	"n",	"o",	"p",	"q",	"r",	"s",
	"t",	"u",	"v",	"w",	"x",	"y",	"z",	"{",
	"|",	"}",	"~",	"narrow sp",	"hyphen",	"bullet",
	"square",	"3/4 em",	"rule",	"1/4",	"1/2",	"3/4",
	"minus",	"fi",	"fl",	"ff",	"ffi",	"ffl",	"degree",
	"dagger",	"section",	"foot mark",	"acute accent",
	"grave accent",	"underrule",	"slash (longer)",
	"half narrow space",	"unpaddable space",	"alpha",
	"beta",	"gamma",	"delta",	"epsilon",	"zeta",
	"eta",	"theta",	"iota",	"kappa",	"lambda",
	"mu",	"nu",	"xi",	"omicron",	"pi",	"rho",	"sigma",
	"tau",	"upsilon",	"phi",	"chi",	"psi",	"omega",
	"Gamma",	"Delta",	"Theta",	"Lambda",
	"Xi",	"Pi",	"Sigma",	"Tau",	"Upsilon",	"Phi",
	"Psi",	"Omega",	"square root",	"terminal sigma",
	"root en",	">=",	"<=",	"identically equal",
	"equation minus",	"approx =",	"approximates",
	"not equal",	"right arrow",	"left arrow",	"up arrow",
	"down arrow",	"eqn equals",	"multiply",	"divide",
	"plus-minus",	"cup (union)",	"cap (intersection)",	"subset of",
	"superset of",	"improper subset",	" improper superset",
	"infinity",	"pt deriv",	"gradient",	"not",	"integral",
	"proportional to",	"empty set",	"member of",
	"equation plus",	"registration mk",	"copyright mk",
	"box rule",	"cent sign",	"dbl dagger",	"right hand",
	"left hand",	"math * ",	"bell system sign",
	"or (was star)",	"circle",	"left top of big curly",
	"left bottom of big curly",	"right top of big curly",
	"right bottom of big curly",	"left center of big curly",
	"right center of big curly",	"bold vertical rule",
	"left bottom of big bracket",	"right bottom of big bracket",
	"left top of big bracket",	"right top of big bracket",
	"???",	"???",	"???",	"???",	"???",	"???",	"???",	"???",
	"???",	"???",	"???",	"???",	"???",	/* The last 13 are unused */
};

struct t t;
struct t_stor t_stor;
char	*mptr;		/* Pointer to the character data */
int	c_size;		/* The amount of character data in bytes */
int	debug;

#define intshow(memb)	printf ("/*%-8s*/\t\t%d,\n", "memb", t.memb)
#define show(memb)	printf ("/*%-8s*/\t\t\"", "memb");\
			vprint (t.memb);\
			if (debug) printf("\",\t\t/*\t%d\t*/\n", t_stor.memb);\
			else printf ("\",\n")
#define getptr(memb)	t.memb = mptr + t_stor.memb

main(argc, argv)
char	**argv;
{
	int	c, i;
	char	labelbuf[64];

	if (argc == 3 && argv[1][0] == '-' && argv[1][1] == 'd') debug = 1;
	if (argc != 2 && !debug) {
	    fprintf (stderr, "Usage: elbat [-d] tabfile > tabfile.c\n");
	    exit(1);
	}

	loadtab(argv[argc - 1]);
	
	printf ("%s} t =\t{\n", structure);
	printf ("/*bset    */\t\t0%o,\n", t.bset);
	printf ("/*breset  */\t\t0%o,\n", t.breset);
	intshow (Hor);
	intshow (Vert);
	intshow (Newline);
	intshow (Char);
#ifdef KANJI
	intshow (Kchar);
#endif KANJI
	intshow (Em);
	intshow (Halfline);
	intshow (Adj);
	if (debug) printf ("/* name\t\t\tstring\t\t\tindex into c_data */\n");
	show (twinit);
	show (twrest);
	show (twnl);
	show (hlr);
	show (hlf);
	show (flr);
	show (bdon);
	show (bdoff);
	show (iton);
	show (itoff);
	show (ploton);
	show (plotoff);
	show (up);
	show (down);
	show (right);
	show (left);

	for (c = 0; c < 256 - 32 - 13; c++) {
	    sprintf (labelbuf, "/* %s */", codelabel[c]);
	    printf ("%-20s\t\"", labelbuf);
	    if (t.codetab[c][0])
		vprint (t.codetab[c]);
	    else if (t.codetab[c][1]) {
		printf ("\\000");
		vprint (t.codetab[c]+1);
	    }
	    else
		printf ("\\000\\0");

	    if (debug) printf("\",\t\t/*\t%d\t*/\n", t_stor.codetab[c]);
	    else printf ("\",\n");
	}
	printf ("};\n");

	if (debug) {
	    printf ("\n/* Character data dump: Size = %d bytes\n", c_size);
	    for (i = 0; i < c_size; i++) {
		if (! (i % 10)) printf ("\n%6d  ", i);
		chprint ((unsigned char) mptr[i]);
	    }
	    printf ("\n*/\n");
	}

	exit (0);
}

vprint (str)
char	*str;
{
	if (!str) return;
	while (*str) {
	    char	c[5];

	    if (isascii (*str) && isprint (*str)
			       && *str != '\\' && *str != '"' ) {
		c[0] = *str;
		c[1] = '\0';
	    }
	    else {
		switch (*str) {
		case '\\':
		    strcpy (c, "\\\\");
		    break;
		case '"':
		    strcpy (c, "\\\"");
		    break;
		case '\b':
		    strcpy (c, "\\b");
		    break;
		case '\t':
		    strcpy (c, "\\t");
		    break;
		case '\n':
		    strcpy (c, "\\n");
		    break;
		case '\r':
		    strcpy (c, "\\r");
		    break;
		default:
		    sprintf (c, "\\%3.3o", (int)*str & 0377);
		    break;
		}
	    }
	printf ("%s", c);
	str++;
	}
}

chprint (c)
unsigned char	c;
{
	if (isascii (c) && isgraph (c)) printf ("    %c", c);
	else if (c) printf (" \\%.3o", c);
	else printf (" ----");
}

loadtab (tname)
char	*tname;
{
	int	tfd, c;
	char	*malloc();
	struct stat	statbuf;

	if ((tfd=open (tname, O_RDONLY)) < 0 ) {
	    perror (tname);
	    exit (1);
	}
	fstat (tfd, &statbuf);
	read (tfd, &c_size, sizeof (int));
	read (tfd, &t_stor, sizeof (t_stor));
	mptr = malloc (c_size);
	read (tfd, mptr, c_size);
	c = sizeof (int) + sizeof (t_stor) + c_size;
	if (c != statbuf.st_size) fprintf (stderr,
	"WARNING: %s size: %d bytes (should be %d), output likely corrupt\n",
	    tname, statbuf.st_size, c);

	t.bset = t_stor.bset;
	t.breset = t_stor.breset;
	t.Hor = t_stor.Hor;
	t.Vert = t_stor.Vert;
	t.Newline = t_stor.Newline;
	t.Char = t_stor.Char;
#ifdef KANJI
	t.Kchar = t_stor.Kchar;
#endif KANJI
	t.Em = t_stor.Em;
	t.Halfline = t_stor.Halfline;
	t.Adj = t_stor.Adj;
	getptr (twinit);
	getptr (twrest);
	getptr (twnl);
	getptr (hlr);
	getptr (hlf);
	getptr (flr);
	getptr (bdon);
	getptr (bdoff);
	getptr (iton);
	getptr (itoff);
	getptr (ploton);
	getptr (plotoff);
	getptr (up);
	getptr (down);
	getptr (right);
	getptr (left);

	for (c = 0; c < 256 - 32; c++)
	    getptr (codetab[c]);
}
