/* headings.c prettyprints section headings from the user's nroff macros */
#include <stdio.h>
#include <ctype.h>

#define	MACROS_LIB "/usr/lib"  /* do not follow .so (include) commands here */
#define MAXHEADERS 20          /* maximum number of section headings */

char	*Header[MAXHEADERS];   /* section heading names */
int 	Nheaders = 0;          /* number of section headings */
int 	Mindent = 0;           /* lowest section number to indent */
int 	Section[MAXHEADERS];   /* section numbers */
int 	Countlines = 0;        /* should lines be counted and printed? */
int 	Linecount = 0;         /* count of number of lines */
int 	Countchars = 0;        /* should characters be counted and printed? */
int 	Charcount = 0;         /* count of number of characters */
int 	Numbersections = 0;    /* should sections be numbered? */
int 	Maxheaders = (MAXHEADERS-1); /* only print sections to this depth */
int 	Followso = 0;          /* should .so (include) macros be followed? */
char	*Paragraph;            /* paragraph macro */
char	*Parashow;             /* how should paragraphs be shown? */
int 	Lastindent;            /* indentation (depth) of previous macro */
char	*Argv0;                /* name of the program */

initial (argc, argv) char **argv;
	{
	extern	int 	optind;
	extern	char	*optarg;
	int 	C;
	int 	errflg = 0;
	char	*usage =
		"[-cflns] [-h header] [-p para] [-P mark]\n\t[-m min] [-M max] [-] [files]";
	Argv0 = argv[0];
	while ((C = getopt (argc, argv, "M:m:P:p:lcnsh:f")) != EOF)
		switch (C)
			{
			case 'f': Followso = 1; break;
			case 'h':
				Header[Nheaders] = optarg;
				Nheaders++;
				break;
			case 's':
			case 'n': Numbersections = 1; break;
			case 'c': Countchars = 1; break;
			case 'l': Countlines = 1; break;
			case 'p': Paragraph = optarg; break;
			case 'P': Parashow = optarg; break;
			case 'm': Mindent = atoi (optarg); break;
			case 'M': Maxheaders = atoi (optarg); break;
			case '?': errflg++; break;
			}
	if (errflg)
		{
		fprintf (stderr, "Usage: %s %s\n", Argv0, usage);
		exit (1);
		}
	if (Nheaders < 1) /* assume -me macros for no particular reason */
		{
		Header[0] = "sh";
		Nheaders = 1;
		if (Mindent == 0)
			Mindent = 1;
		}
	return (optind);
	}

main (argc, argv) char **argv;
	{
	int  	head ();
	exit (filter (argc, argv, initial (argc, argv), head));
	}

int
head (filename, infile)
char	*filename;
FILE	*infile;
	{
	char	line[BUFSIZ];  /* input lines read here */
	char	*ptr;          /* work pointer in line */
	int 	i;

	Linecount = 0;
	printf ("Headings for %s\n", filename);

	while (fgets (line, BUFSIZ, infile))
		{
		Linecount++;
		Charcount += strlen (line);
		if (*line != '.') /* ignore any lines that are not breaking macros */
			continue;
		trimend (line);
		for (ptr = line+1; isspace (*ptr); ptr++); /* skip optional space */
		if (Followso && begins ("so", ptr)) /* nroff source file */
			{
			if (!begins (MACROS_LIB, ptr))
				head (ptr, infile);
			}
		else if (Paragraph && *Paragraph && begins (Paragraph, ptr))
			{
			output (Lastindent, Parashow ? Parashow : ptr, 0);
			}
		else for (i = 0; i < Nheaders; i++)
			if (begins (Header[i], ptr))
				{
				ptr += strlen (Header[i]);
				if ((Nheaders == 1) && (i = atoi (ptr))) /* numbered section */
					{
					while (isspace (*ptr)) ptr++;
					while (!isspace (*ptr)) ptr++;
					}
				while (*ptr == '"' || isspace (*ptr)) ptr++;
				if (*ptr == NULL) /* section heading is on next line */
					{
					if (fgets (line, BUFSIZ, infile))
						{
						Linecount++;
						Charcount += strlen (line);
						trimend (line);
						ptr = line;
						}
					}
				output (i, ptr, 1);
				break;
				}
		}
	}

/* trim off spaces and "'s from end of line */
trimend (line)
char	*line;
	{
	char	*ptr = line;
	while (*ptr)
		ptr++;
	while (ptr > line && (ptr[-1] == '"' || isspace (ptr[-1])))
		ptr--;
	*ptr = '\0';
	}

/* indent the line to the standard output */
output (indent, line, header)
int 	indent;    /* amount to indent the line */
char	*line;     /* heading or paragraph text to be printed */
int 	header;    /* is the line a section heading? */
	{
	int 	loop;
	Lastindent = indent;
	if (!header)
		indent++; /* non-headers are less important */
	if (header && indent > Maxheaders)
		return;
	if (Countlines)
		printf ("%4d\t", Linecount);
	if (Countchars)
		printf ("%6d\t", Charcount);
	for (loop = 0; loop < indent-Mindent; loop++)
		putchar ('\t');
	if (header && Numbersections)
		{
		Section[indent]++;
		for (loop = indent+1; loop < Maxheaders; loop++)
			Section[loop] = 0;
		printsection ();
		putchar (' ');
		}
	puts (line);
	}
	
/* print the section number in #.#.# format */
printsection ()
	{
	int 	loop;
	int 	max;
	for (loop = Maxheaders; loop >= 0; loop--)
		if (Section[loop]) break;
	max = loop;
	for (loop = Mindent; loop <= max; loop++)
		{
		printf ("%d", Section[loop]);
		if (loop < max)
			putchar ('.');
		}
	}

begins (string1, string2)
register	char *string1;
register	char *string2;
	{
	while (*string1)
		if (*string1++ != *string2++) return (0);
	return (1);
	}
