/*
 *  Cpic.c --- produce pic output for graphs in which all vertices lie on a
 *  circle and edge multiplicity is <= 3.
 *
 *  Terry Jones  (tcjones@watdragon)
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <signal.h>
#include <math.h>
#include "cpic.h"

char *this_file;
FILE *in_fp;
FILE *out_fp;
char *myname;
int line_number = 0;
int found_line = 0;
edge_list *avail_edges = NULL;
picture *avail_pics = NULL;

main(argc, argv)
int argc;
char **argv;
{
	extern char *getenv();
	FILE *open_file();
	extern void clean_up();
	int std_in = 0;

	myname=*argv;
	handle_signals();

	set_io(argv, &argc, &std_in);

	if (std_in){
		in_fp = stdin;
		this_file = "(standard input)";
		do_file(stdin, out_fp);
	}
	else{
		if (out_fp != stdout){
			argc -= 2;
			argv += 2;
		}
		while (--argc){
			this_file = *++argv;
			in_fp = open_file(this_file,"r",0);
			if (in_fp == NULL) continue;
			do_file(in_fp, out_fp);
		}
	}
	fclose(out_fp);
	exit(0);
}


set_io(c, n, std_in)
char **c;
int *n;
int *std_in;
{
	switch (*n - 1){

		case 0: {
			*std_in = 1;
			out_fp = stdout;
			break;
		}

		case 1: {
			if (!strcmp(c[1],"-h") || !strcmp(c[1],"-help")) help();
			out_fp = stdout;
			break;
		}

		default: {
			if (!strcmp(c[1],"-o")){
				out_fp = open_file(c[2], "w", 0);  /* open w/o clobbering. */

				if (*n == 3){
					*std_in = 1;
				}
			}

			else if (!strcmp(c[1],"-O")){
				out_fp = open_file(c[2], "w", 1); /* open & clobber. */

				if (*n == 3){
					*std_in = 1; /* All arguments are used up. */
				}
			}

			else out_fp = stdout;
			break;
		}
	}
}
		

help()
{
	fprintf(stderr,"Usage: %s [-[oO] outfile] [infiles...]\n", myname);
	exit(0);
}


handle_signals()
{
    if (signal(SIGINT,SIG_IGN)!=SIG_IGN){
		signal(SIGINT,clean_up);
	}
}

void
clean_up()
{
	fprintf(stderr,"\nInterrupt...\n");
	goodbye();
}

do_file(in_f, out_f)
FILE *in_f;
FILE *out_f;
{
	extern picture *begin_picture();
	extern char *get_quoted_name();
	char *getline();
	char line[MAX];
	picture_list pics;
	picture *curr_pic = NULL;
	char *pic_title = NULL;
	int pics_this_list = 0;
	double picr;
	double ptr;

	pics.pic = NULL;

	while ( GETLINE ){

		line_number++;

		/* A line we are not interested in, print it and go on. */
		if (strlen(line) < 4 || strncmp(line, ".CPS", 4) != 0){
			fprintf(out_f, "%s\n", line);
			continue;
		}


		/* A line we are interested in. */
		/* This line will have format ".CPS [pic_radius] [point_radius]" */

		switch (sscanf(line,".CPS %f %f", &picr, &ptr)){

			case EOF: {
				/* Could not match, but tried. */
				picr = PIC_RADIUS;
				ptr = POINT_RADIUS;
				break;
			}

			case 1: {
				/* Got picr from the sscanf */
				ptr = POINT_RADIUS;
				break;
			}

			case 2: {
				break;
			}

			default: {
				fprintf(stderr,
					"%s: .CPS line syntax is \".CPS [picture radius [", myname);
				fprintf(stderr,
					"point radius]]\". Both are taken to be in inches and\n");
				fprintf(stderr,
					"default to %.2f and %.2f if omitted. (Line=%d, File=%s)\n",
					PIC_RADIUS, POINT_RADIUS, line_number, this_file);

				goodbye();
				break;
			}

		}

		/* Start pic processing. */
		/* fprintf(out_f, ".PS\n#%s\n", line); */
		fprintf(out_f, ".PS\n");


		if ( ! GETLINE ) goodbye();
		line_number++;

		while (allwhite(line) || iscomment(line)){
			/* fprintf(out_f,"%s\n",line); */
			if ( ! GETLINE ) goodbye();
			line_number++;
		}

		/* Get the number of pictures that will be on this line. */
		if (sscanf(line, "%d", &pics_this_list) != 1) goodbye();
		found_line = line_number;

		pic_title = get_quoted_name(line);
		begin_picture_list(&pics, pics_this_list, pic_title);

		/* Used to print out the definition commented out, but don't worry. */
		/* fprintf(out_f, "#%s\n", line); */

		while ( GETLINE ){

			int n1, n2, n3;
			int pictures_done = 0;

			line_number++;

			/* skip blank and comment lines */
			while (allwhite(line) || iscomment(line)){
				/* fprintf(out_f,"%s\n", line); */
				if ( ! GETLINE ) goodbye();
				line_number++;
			}

			/* Now we have a non-blank, non-comment line. */
			/* Don't worry about echoing it. */
			/* fprintf(out_f, "#%s\n", line); */

			switch(sscanf(line, "%d %d %d", &n1, &n2, &n3)){

				/* The number of vertices that this picture will be on. */
				case 1: {

					extern char *get_quoted_name();
					extern double get_width();

					double width = get_width(line);
					char *pic_name = get_quoted_name(line);

					curr_pic = 
						begin_picture(&pics, curr_pic, n1, pic_name, width);
					break;
				}


				/* Two vertices which are to be joined (once). */
				case 2: {
					add_edges(curr_pic, n1, n2, 1);
					break;
				}

				/* Two vertices to be joined and the edge multiplicity */
				case 3: {
					add_edges(curr_pic, n1, n2, n3);
					break;
				}

				default: {
					if (strlen(line) < 4 || strncmp(line, ".CPE", 4) != 0){
						fprintf(stderr,"Whoops! bad input \n");
						
						goodbye();
					}

					end_picture(&pics, picr, ptr, out_f);
					pictures_done = 1;
					break;
				}
			}

		if (pictures_done) break;
		}
	}
	fclose(in_f);
}

begin_picture_list(pl, max_vertices, pic_title)
picture_list *pl;
int max_vertices;
char *pic_title;
{
	if (pl->pic != NULL) free_picture_list(pl);
	pl->pic_limit = max_vertices;
	pl->pics_in_list = 0;
	pl->total_label_width = 0.0;
	strcpy(pl->pic_title, pic_title);
	pl->pic = NULL;
}


picture *
begin_picture(pl, cp, vertices, pic_name, width)
picture_list *pl;
picture *cp;
int vertices;
char *pic_name;
double width;
{
	extern picture *new_picture();

	if (pl->pic_limit == pl->pics_in_list){
		fprintf(stderr,"%s: %d pictures promised on line %d of file %s,\n", 
			myname, pl->pic_limit, found_line, this_file);
		
		fprintf(stderr,"but the picture beginning line %d exceeds this!\n",
			line_number);
		goodbye();
	}

	if (vertices == 0){
		fprintf(stderr,
			"%s: A graph with 0 vertices? Interesting. Line=%d, File=%s.\n",
			myname, line_number, this_file);
		goodbye();
	}

	if (width < 0.0){
		fprintf(stderr,
			"%s: A label with negative width? Interesting. Line=%d, File=%s.\n",
			myname, line_number, this_file);
		goodbye();
	}


	pl->pics_in_list++;

	if (pl->pic == NULL){
		pl->pic = cp = new_picture();
	}
	else{
		cp->next_pic = new_picture();
		cp = cp->next_pic;
	}


	/* Try to estimate the label's width. */
	/* Actual string to be printed does not include the 2 "'s */
	if (pic_name && width == 0.0) 
		width = CHAR_WIDTH * (double)(strlen(pic_name) - 2);

	/* Add to the total amount of label space we have seen for this pic. */
	if (pic_name) 
		pl->total_label_width += width;

	cp->vertices = vertices;
	strcpy(cp->fig_name, pic_name);
	cp->label_width = width;
	return(cp);
}


add_edges(cp, v1, v2, multiplicity)
picture *cp;
int v1;
int v2;
int multiplicity;
{
	extern edge_list *new_edge_list();
	edge_list *temp_e_list;

	if (cp == NULL){
		fprintf(stderr,"%s: Oops! you tried to define edges before saying\n",
			myname);
		fprintf(stderr,
			"how many vertices there were going to be. Line=%d, File=%s\n",
			line_number, this_file);
		goodbye();
	}

	if (v1 < 1 || v2 < 1 || v1 > cp->vertices || v2 > cp->vertices){
		fprintf(stderr,
			"%s: Vertex index outside legal range on line %d, in file %s.\n", 
			myname, line_number, this_file);
		goodbye();
	}

	if (multiplicity < 1 || multiplicity > 3){
		fprintf(stderr,
			"%s: Illegal edge multiplicity of %d give on line %d in file %s.\n",
			myname, multiplicity, line_number, this_file);
		goodbye();
	}

	if (v1 == v2){
		fprintf(stderr,
"%s: Self loops not supported - edge %d,%d ignored on line %d in file %s.\n",
			myname, v1, v1, line_number, this_file);
		return;
	}

	temp_e_list = new_edge_list();

	temp_e_list->v1 = v1;
	temp_e_list->v2 = v2;
	temp_e_list->times = multiplicity;

	temp_e_list->next_edge = cp->e_list;
	cp->e_list = temp_e_list;
}

FILE *
open_file(s,mode,flag)
char *s;
char *mode;
int flag;
{
	/* 
	 * fopen() the file whose name is "s", exit on error.
	 * flag is clobber mode for "w". 1 = clobber existing file.
	 *
	 */


	FILE *fp, *fopen();
	struct stat buf;

	if (strcmp(mode,"w") == 0 && flag != 1 && stat(s, &buf) != -1){
		fprintf(stderr,
			"%s: file \"%s\" already exists! (Use -O to overwrite.)\n", 
			myname, s);
		exit(1);
	}

	if (!(fp=fopen(s,mode))){
		fprintf(stderr,"%s: could not open %s (ignored).\n", myname, s);
	}

	return(fp);
}



int
file_exists(s)
char *s;
{
	/* 
	 * return 1 if the file s exists, 0 if not.
	 * This is silly - I should have used access(). Didn't know about it.
	 *
	 */


	extern int errno;
	struct stat buf;
	int status;

	status = stat(s, &buf);

	if (status == 0){
		return(1);
	}
	else if (status == -1 && errno == ENOENT){
		return(0);
	}
	else{
		fprintf(stderr,"file_exists: stat failed, errno=%d\n", errno);
		perror("stat");
		exit(1);
	}
}

char *
getline(fp,max,where)
FILE *fp;
int max;
char *where;
{
	/* returns null terminated line in "where" and NULL on EOF or error */
	/* essentially does what gets() does, except for any open FILE * */

	extern char* index();
	char *fred;
	
	if ((fred=fgets(where,max,fp))){
		char *tmp;
		if (!(tmp=index(where,'\n'))){
			printf("getline: Line with no newline!\n");
			exit(1);
		}
		*tmp='\0';
	}
	return(fred);
}


int 
allwhite(s)
char *s;
{
	/*
	 * Return 1 if the char array s is all white space, or empty.
	 *
	 */

	if (!strlen(s)) return(1);

	while (*s){
		if (*s != ' ' && *s != '\t'){
			return(0);
		}
		s++;
	}
	return(1);
}


picture *
new_picture()
{
	picture *temp;

	if (avail_pics == NULL){
		if ((temp = (picture *)malloc((unsigned long)sizeof(picture))) == NULL){
			fprintf(stderr,"Could not malloc! giving up.\n");
			exit(1);
		}
	}
	else{
		temp = avail_pics;
		avail_pics = avail_pics->next_pic;
	}

	temp->vertices = 0;
	temp->fig_name[0] = '\0';
	temp->label_width = 0.0;
	temp->e_list = NULL;
	temp->next_pic = NULL;
	return(temp);
}

edge_list *
new_edge_list()
{
	edge_list *temp;

	if (avail_edges == NULL){
		if ((temp = 
			(edge_list *)malloc((unsigned long)sizeof(edge_list))) == NULL){

			fprintf(stderr,"Could not malloc! giving up.\n");
			exit(1);
		}
	}
	else{
		temp = avail_edges;
		avail_edges = avail_edges->next_edge;
	}

	temp->v1 = temp->v2 = temp->times = 0;
	temp->next_edge = NULL;
	return(temp);
}

goodbye()
{
	fflush(stderr);
	fclose(in_fp);
	fclose(out_fp);
	fprintf(stderr,"%s: Execution aborted on line %d in file %s.\n",
		myname, line_number, this_file);
	exit(1);
}



free_picture_list(pl)
picture_list *pl;
/* This does not seem to free anything! Is it wrong?  TERRY! */
{
	picture *this_pic = pl->pic;
	edge_list *this_e_list;

	if (this_pic == NULL) return;

	while (this_pic->next_pic != NULL){
		if (this_pic->e_list != NULL){
			this_e_list = this_pic->e_list;
			while (this_e_list->next_edge != NULL){
				this_e_list = this_e_list->next_edge;
			}

			this_e_list->next_edge = avail_edges;
			avail_edges = this_pic->e_list;
		}

		this_pic = this_pic->next_pic;
	}
	this_pic->next_pic = avail_pics;
	avail_pics = pl->pic;
}


end_picture(pl, picr, ptr, f)
picture_list *pl;
double picr;
double ptr;
FILE *f;
{
	/* Produce pic output for each of the pictures in the picture list. */

	/* 
	 * Will need to do some looking at the list to determine the width
	 * and other things like that - saving on recomputation of sins etc.
	 * but for now just be dumb so we can get some output!
	 */

	picture *p = pl->pic;
	int rad_count = 1;
	int sep_count = 1;
	double sep;
	double pic_y;
	double arc_rad;
	double x_adjust = BOUNDARY;

	if (pl->pic_limit != pl->pics_in_list){
		fprintf(stderr,
			"%s: You promised %d graphs but gave %d. (Line %d, File %s)\n",
			myname, pl->pic_limit, pl->pics_in_list, line_number, this_file);
		goodbye();
	}

	/* Calculate inter-picture separator. */
	sep = (PAGE_WIDTH - 
		   (2.0 * (double)(pl->pic_limit) * picr) - pl->total_label_width
		  ) / (double)(pl->pic_limit + 1.0);

	/* Calculate height of figure centres. */
	pic_y = picr + BOUNDARY + (pl->pic_title[0] ? TITLE_SEP : 0.0);

	/* 
	 * debug - point of reference.
	 * fprintf(f,"circle rad 0.1i at (0i,%.2fi) #invisible\n", picr + sep);
	 */

	while (p != NULL){
		/* process each picture in turn. */

		char centre[15];

		x_adjust += p->label_width; /* Will be 0.0 if there is no label. */
		sprintf(centre,"(%.2fi,%.2fi)", 
			rad_count*picr + sep_count*sep + x_adjust, pic_y);
		
		produce_pic(p, centre, picr, ptr, f);

		rad_count += 2;
		sep_count++;
		p = p->next_pic;
	}

	/* 
	 * DISABLED - use .ce in plain troff to get the title for the picture
	 * Do the title for the whole picture.
	 * if (pl->pic_title[0]){
	 *	fprintf(f, "%s at (%.2fi,%2fi)\n", pl->pic_title, 
	 *
	 *		either -- BOUNDARY + sep * (double)(pl->pic_limit + 1) / 2.0 +
	 *		          picr * 2.0 * (double)(pl->pic_limit) / 2.0,
	 *		-- or --  PAGE_WIDTH / 2.0,
	 *
	 *		TITLE_SEP);
	 * }
	 */

	fprintf(f,".PE\n");
}


produce_pic(p, centre, picr, ptr, f)
picture *p;
char *centre;
double picr;
double ptr;
FILE *f;
{
	extern double cos();
	extern double sin();

	int i;
	static int vertex_count = 0;
	static point *points;

	/* See if there is any need to recompute the vertex coordinates. */

	if (p->vertices != vertex_count){

		double inc = 360.0 / (double) p->vertices; /* p->vertices != zero */

		/* Need more space than there is - free the old and calloc again. */

		if (p->vertices > vertex_count){ 
			free(points);
			if ((points = 
				(point *)calloc(p->vertices+1, sizeof(point))) == NULL){
				fprintf(stderr,"%s: Could not calloc! Goodbye.\n",myname);
				goodbye();
			}
		}

		vertex_count = p->vertices;

		for (i=1; i<=p->vertices; i++){
			double fred = (double)(i-1) * inc * RAD_ADJ;
			points[i].x = picr * cos( fred );
			points[i].y = picr * sin( fred );
		}
	}

	plot_label(centre, picr, p->fig_name, p->label_width, f);
	define_vertices(points, p->vertices, centre, ptr, f);
	plot_vertices(p->vertices, f);
	plot_edges(points, p->e_list, centre, f);
}


plot_label(centre, picr, name, width, f)
char *centre;
double picr;
char *name;
double width;
FILE *f;
{
	if (*name){
		fprintf(f,"# Picture title.\n");
		fprintf(f, "%s at %s - (%2fi, 0i)\n", name, centre,(width/2.0) + picr);
		fprintf(f,"#\n");
	}
}


define_vertices(pts, n, centre, ptr, f)
point *pts;
int n;
char *centre;
double ptr;
FILE *f;
{
	/*
	 * Output the defined positions of the vertices.
	 */

	register int i;

	fprintf(f,"# Vertice definitions.\n");
	for (i=1; i<=n; i++){
		fprintf(f,"define V_%d X (%.2f,%.2f) + %s X\n", i, pts[i].x, pts[i].y,
			centre);
	}
	fprintf(f,"define Pt_Radius X %.2fi X\n", ptr);
	fprintf(f,"#\n");
}


plot_vertices(n, f)
int n;
FILE *f;
{
	int i;

	fprintf(f,"# Set default circle radius\n");
	fprintf(f,"circlerad = Pt_Radius\n");
	fprintf(f,"#\n");

	fprintf(f, "# Do the vertices.\n");

	for (i=1; i<=n; i++){

		/* 
		 * This will produce labelled vertices from 1,2,...n - useful in debug.
		 * fprintf(f,"circle \"%d\" at V_%d\n", i, i);
		 */

		fprintf(f,"circle at V_%d\n", i);
	}
	fprintf(f,"#\n");
}


plot_edges(pts, edges, centre, f)
point *pts;
edge_list *edges;
char *centre;
FILE *f;
{
	/* No need to check that vertex numbers are in range - done in add_edges. */

	extern double get_arc_radius();
	extern double distance();
	char arc_radius[15];

	fprintf(f,"# Draw the edges.\n");

	while (edges != NULL){

		int one = edges->v1;
		int two = edges->v2;

		if (edges->times != 1){
			fprintf(f,"arcrad = %.2f\n",
				get_arc_radius(distance(pts[one].x, pts[one].y, 
					pts[two].x, pts[two].y)));
		}

		/* No need to check multiplicities - done in add_edges. */
		switch(edges->times){

			case 1: {
				fprintf(f,"line from V_%d to V_%d\n", one, two);
				break;
			}

			case 2: {
				fprintf(f,"arc from V_%d to V_%d\n", one, two);
				fprintf(f,"arc from V_%d to V_%d\n", two, one);
				break;
			}

			case 3: {

				fprintf(f,"line from V_%d to V_%d\n", one, two);
				fprintf(f,"arc from V_%d to V_%d\n", one, two);
				fprintf(f,"arc from V_%d to V_%d\n", two, one);
				break;
			}
		}

		edges = edges->next_edge;
	}
	fprintf(f,"#\n");
}


double
get_arc_radius(d)
double d;
{
		/*
		 *  Some empirical arc radius values depending on the inter pt dist.
		 *  IT'S PROBABLY NOT A GOOD IDEA TO FIDDLE WITH THESE...
		 *  of course i should now use all the least squares stuff i learned
		 *  for the comps :-)
		 */

		if (d < 0.2) return(0.15);
		if (d < 0.3) return(0.32);
		if (d < 0.35) return(0.45);
		if (d < 0.4) return(0.6);
		/*if (d < 0.45) return(0.65);*/
		if (d < 0.5) return(0.8);
		if (d < 0.6) return(1.05);
		if (d < 0.7) return(1.22);
		if (d < 0.77) return(1.32);
		if (d < 0.85) return(1.75);
		if (d < 1.0) return(2.0);
		if (d < 1.5) return(2.5);
		return(3.0);
}


double 
distance(x1, why1, x2, y2)
double x1;
double why1;
double x2;
double y2;
{
	return( sqrt ((x1-x2)*(x1-x2) + (why1-y2)*(why1-y2)) );
}


char *
get_quoted_name(s)
char *s;
{
	/* 
	 * Try to find a quoted "" string in the array s. If one exists
	 * then return a pointer to the first quote, having zeroed the
	 * character after the closing quote.
	 */

	char *start = index(s, '"');

	if (start){
		char *quotes = index(start + 1, '"');;
		if (quotes == NULL){
			fprintf(stderr, "Closing quote omitted on figure name. Line");
			fprintf(stderr, " %d, file %s.\n", line_number, this_file);
			goodbye();
		}
		*(quotes + 1) = '\0';
	}
	return(start);
}


double
get_width(s)
char *s;
{
	char *quote1, *quote2;
	double w = 0.0;

	if ( (quote1 = index(s,'"')) == NULL ||
		 (quote2 = index(quote1 + 1, '"')) == NULL || 
		 sscanf(quote2 + 1, "%f", &w) != 1 )

		 return(0.0);

	return(w);
}
