/*
    xccisb.c	CIS "B" Routines for XCOMM

    The following code was modified from "BP.C", a "generic" B-protocol
    handler available from CompuServe.

*/
/**
 * Copyright (c) 1985 by Steve Wilhite, Worthington, Ohio
 *
 * Permission is granted to use or distribute this software without any
 * restrictions as long as this entire copyright notice is included intact.
 * You may include it in any software product that you sell for profit.
 *
 * This software is distributed as is, and is not guaranteed to work on any
 * given hardware/software configuration.  Furthermore, no liability is
 * granted with this software.
 *
 * ABSTRACT:
 *
 *      The function, Transfer_File, implements error-free file transfer using
 *      CompuServe's "B" protocol.
 *
 *      It has been assumed that the start-of-packet sequence, DLE "B", has
 *      been detected and the next byte not received yet is the packet
 *      sequence number (an ASCII digit).
 *
 * AUTHOR: Steve Wilhite, CREATION DATE: 21-Jul-85
 *
 * REVISION HISTORY:
 *
 * 4 Dec 1987 -- larry gensch -- Supports QUICK-B Protocol 
 **/

#include <stdio.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#include "xcomm.h"

extern FILE *tfp;		/* Generic tty pointer */

static int Abort_Flag = 0;	/* Equals 1 if user wants to abort */
static int Text_Mode = 0;	/* 1 = Convert /r/n to /n and back */
static int ACK_Count = 0;	/* Number of successful packets */
static int XMIT_Count = 0;	/* Number of successful packets */
static int NAK_Count = 0;	/* Number of consecutive NAKS */
static long io_count = 0L;	/* Number of bytes transferred */
static jmp_buf err_buf;		/* setjmp() buffer */
static int Last_Chr = 0;	/* Last char read from file */
static long start_time = 0L;	/* Start time of transfer */
static long bytes_sec = 0L;	/* Number of bytes transferred/sec */

static int Failure_Sent = 0;	/* CIS knows we aborted? */

#define	Wants_To_Abort()	(Abort_Flag)
#define MAX_WINDOW		2
#define	min(a,b)		((a)<(b) ? (a) : (b))

/*
    The following define allows a hex transcript of the file transfer
    to be generated.  Uncomment this define if you want the file XCOMM.LOG
    generated.  (This really increases overhead, but handy for debugging!)
*/

/* #define CIS_B_DEBUG 	/* Uncomment if you want XCOMM.LOG generated */

#define ETX     0x03
#define ENQ     0x05
#define DLE     0x10
#define XON     0x11
#define XOFF    0x13
#define NAK     0x15

#ifndef TRUE
#define TRUE            1
#define FALSE           0
#endif

#define Success         -1
#define Failure         0

#define MAX_PACKET      1024
#define Max_Errors      10
#define Max_Time        10
#define WACK            ';'             /* wait acknowledge */

/* Sender actions */

#define S_Send_Packet   0
#define S_Get_DLE       1
#define S_Get_Num       2
#define S_Get_Seq       3
#define S_Get_Data      4
#define S_Get_Checksum  5
#define S_Timed_Out     6
#define S_Send_NAK      7
#define S_Got_NAK       8
#define S_Got_ACK       9

/* Receiver actions */

#define R_Get_DLE       0
#define R_Get_B         1
#define R_Get_Seq       2
#define R_Get_Data      3
#define R_Get_Checksum  4
#define R_Send_NAK      5
#define R_Send_ACK      6
#define R_Send_CTL      6

static int
Ch,
Checksum,
Seq_Num,
R_Size,					/* Size of receiver buffer */
XOFF_Flag,
Seen_ETX;

static char S_Buffer[MAX_PACKET+1];	/* Sender buffer */
static char R_Buffer[MAX_PACKET+1];	/* Receiver buffer */

static int QBP_ws;
static int QBP_wr;
static int QBP_bs;
static int QBP_cm;
static int QBP_dq;

static int QBP_Active;			/* Signifies windowing in effect */
static unsigned QBP_Packet;		/* QB Packet Length */
static int QB_Out = 0;			/* Count of outstanding packets */
static int QB_Old = 0;			/* Oldest packet */
static int QB_Current = 0;		/* Current packet */

static struct QB_Hist {			/* Packet Send-ahead buffer */
    int		block;			/* Block number */
    int		size;			/* Size of the packet */
    int		chk;			/* Checksum of packet */
    char	seq;			/* Sequence number of packet */
    char	buf[MAX_PACKET];	/* History buffer */
} QB_Buf[MAX_WINDOW];	

#ifdef CIS_B_DEBUG
static void xclog(dir, val)
 char dir;
 int val;
{
    static int cnt = 0;
    static FILE *fp = NULL;
    static int lastdir = 0;

    if (fp == NULL) {
	fp = fopen("xcomm.log", "w");
	cnt = 0;
	lastdir = dir;
    }

    if (dir == 0) {
	fprintf(fp, "\n");
	fclose(fp);
	return;
    }

    if (++cnt > 20 || lastdir != dir) {
	fprintf(fp, "\n");
	cnt = 1;
    }

    if (lastdir != dir)
	fprintf(fp, "\n");

    fprintf(fp, "%c%1x%1x ", dir, val/16, val % 16);
    lastdir = dir;
}
#else 
#define xclog(dir,val)	/* */
#endif

static long filelength(fp)
 FILE *fp;
{
    struct stat statbuf;

    fstat(fileno(fp), &statbuf);

    return statbuf.st_size;
}

static Put_Msg(Text)
char *Text;
{
    fprintf(tfp,"\r\n%s\r\n", Text);
}

static char *status_msg = "%-7.7s Blk %5d  %7ld bytes %-30.30s\r";

static Show_ACK()
{
    ACK_Count++;
    NAK_Count = 0;
    fprintf(tfp,status_msg, "ACK", ACK_Count, io_count, " ");
    fflush(tfp);
}

static Show_XMIT(msg)
 char *msg;
{
    fprintf(tfp,status_msg, msg, QB_Buf[QB_Current].block, io_count, " ");
    fflush(tfp);
}

static Show_W_ACK()
{
    fprintf(tfp,status_msg, "ACK", QB_Buf[QB_Old].block, io_count, " ");
    fflush(tfp);
}

static Show_NAK(msg)
{
    char NAK_msg[8];

    sprintf(NAK_msg, "NAK(%d)", NAK_Count++);
    fprintf(tfp,status_msg, NAK_msg, ACK_Count, io_count, msg);
    fflush(tfp);
}

static Send_Byte(Ch)
{
    Ch &= 0xff;

    sendbyte(Ch);
    xclog('>', Ch);
}

static Send_Masked_Byte(Ch)
{
    /* Mask any protocol or flow characters */

    Ch &= 0xff;

    if (Ch == ETX || Ch == ENQ || Ch == DLE || 
	Ch == NAK || Ch == XON || Ch == XOFF)
    {
	Send_Byte(DLE);
	Send_Byte(Ch + '@');
    }
    else Send_Byte(Ch);
}

static Send_ACK()
{
    Send_Byte(DLE);
    Send_Byte(Seq_Num + '0');
}

static Read_Byte()
{
    if ((Ch = readbyte(Max_Time)) < 0 || Wants_To_Abort())
	return Failure;
    else {
	xclog('<', Ch);
	return Success;
    }
}

static Read_Masked_Byte()
{
    Seen_ETX = FALSE;
    if (Read_Byte() == Failure) return Failure;

    if (Ch == DLE) {
	if (Read_Byte() == Failure)
	    return Failure;
	Ch -= '@';
    } else if (Ch == ETX)
	Seen_ETX = TRUE;

    return Success;
}

static unsigned short crctab[256] = {
    0x0000,  0x1021,  0x2042,  0x3063,  0x4084,  0x50a5,  0x60c6,  0x70e7,
    0x8108,  0x9129,  0xa14a,  0xb16b,  0xc18c,  0xd1ad,  0xe1ce,  0xf1ef,
    0x1231,  0x0210,  0x3273,  0x2252,  0x52b5,  0x4294,  0x72f7,  0x62d6,
    0x9339,  0x8318,  0xb37b,  0xa35a,  0xd3bd,  0xc39c,  0xf3ff,  0xe3de,
    0x2462,  0x3443,  0x0420,  0x1401,  0x64e6,  0x74c7,  0x44a4,  0x5485,
    0xa56a,  0xb54b,  0x8528,  0x9509,  0xe5ee,  0xf5cf,  0xc5ac,  0xd58d,
    0x3653,  0x2672,  0x1611,  0x0630,  0x76d7,  0x66f6,  0x5695,  0x46b4,
    0xb75b,  0xa77a,  0x9719,  0x8738,  0xf7df,  0xe7fe,  0xd79d,  0xc7bc,
    0x48c4,  0x58e5,  0x6886,  0x78a7,  0x0840,  0x1861,  0x2802,  0x3823,
    0xc9cc,  0xd9ed,  0xe98e,  0xf9af,  0x8948,  0x9969,  0xa90a,  0xb92b,
    0x5af5,  0x4ad4,  0x7ab7,  0x6a96,  0x1a71,  0x0a50,  0x3a33,  0x2a12,
    0xdbfd,  0xcbdc,  0xfbbf,  0xeb9e,  0x9b79,  0x8b58,  0xbb3b,  0xab1a,
    0x6ca6,  0x7c87,  0x4ce4,  0x5cc5,  0x2c22,  0x3c03,  0x0c60,  0x1c41,
    0xedae,  0xfd8f,  0xcdec,  0xddcd,  0xad2a,  0xbd0b,  0x8d68,  0x9d49,
    0x7e97,  0x6eb6,  0x5ed5,  0x4ef4,  0x3e13,  0x2e32,  0x1e51,  0x0e70,
    0xff9f,  0xefbe,  0xdfdd,  0xcffc,  0xbf1b,  0xaf3a,  0x9f59,  0x8f78,
    0x9188,  0x81a9,  0xb1ca,  0xa1eb,  0xd10c,  0xc12d,  0xf14e,  0xe16f,
    0x1080,  0x00a1,  0x30c2,  0x20e3,  0x5004,  0x4025,  0x7046,  0x6067,
    0x83b9,  0x9398,  0xa3fb,  0xb3da,  0xc33d,  0xd31c,  0xe37f,  0xf35e,
    0x02b1,  0x1290,  0x22f3,  0x32d2,  0x4235,  0x5214,  0x6277,  0x7256,
    0xb5ea,  0xa5cb,  0x95a8,  0x8589,  0xf56e,  0xe54f,  0xd52c,  0xc50d,
    0x34e2,  0x24c3,  0x14a0,  0x0481,  0x7466,  0x6447,  0x5424,  0x4405,
    0xa7db,  0xb7fa,  0x8799,  0x97b8,  0xe75f,  0xf77e,  0xc71d,  0xd73c,
    0x26d3,  0x36f2,  0x0691,  0x16b0,  0x6657,  0x7676,  0x4615,  0x5634,
    0xd94c,  0xc96d,  0xf90e,  0xe92f,  0x99c8,  0x89e9,  0xb98a,  0xa9ab,
    0x5844,  0x4865,  0x7806,  0x6827,  0x18c0,  0x08e1,  0x3882,  0x28a3,
    0xcb7d,  0xdb5c,  0xeb3f,  0xfb1e,  0x8bf9,  0x9bd8,  0xabbb,  0xbb9a,
    0x4a75,  0x5a54,  0x6a37,  0x7a16,  0x0af1,  0x1ad0,  0x2ab3,  0x3a92,
    0xfd2e,  0xed0f,  0xdd6c,  0xcd4d,  0xbdaa,  0xad8b,  0x9de8,  0x8dc9,
    0x7c26,  0x6c07,  0x5c64,  0x4c45,  0x3ca2,  0x2c83,  0x1ce0,  0x0cc1,
    0xef1f,  0xff3e,  0xcf5d,  0xdf7c,  0xaf9b,  0xbfba,  0x8fd9,  0x9ff8,
    0x6e17,  0x7e36,  0x4e55,  0x5e74,  0x2e93,  0x3eb2,  0x0ed1,  0x1ef0
};

#define updcrc(ch, crc) (crctab[((crc >> 8) ^ ch) & 0xff] ^ (crc << 8))

static Do_Checksum(Ch)
 unsigned Ch;
{
    if (QBP_cm == 0) {		/* Normal B Checksum */
	Checksum <<= 1;
	if (Checksum > 0xff)
	    Checksum = (Checksum & 0xff) + 1;

	Checksum += (Ch & 0xff);
	if (Checksum > 0xff)
	    Checksum = (Checksum & 0xff) + 1;
    } else 			/* XMODEM CRC-16 */
	Checksum = (updcrc(Ch & 0xff, Checksum) & 0xffff);
				/* The 0xffff is due to the fact that my
				   machine uses four-byte integers! */
}

static int Read_Packet(Action)
int Action;
{
    int Next_Seq;
    int Errors = 0;
    char *msg = "Timeout";
    int his_chk;
    char mesag[30];

    while (Errors < Max_Errors && !Wants_To_Abort())
	switch (Action) {
	case R_Get_DLE:
	    if (Read_Byte() == Failure)
		Action = R_Send_NAK;
	    else if (Ch == DLE)
		Action = R_Get_B;
	    else if (Ch == ENQ)
		Action = R_Send_ACK;
	    break;

	case R_Get_B:
	    if (Read_Byte() == Failure)
	        Action = R_Send_NAK;
	    else if (Ch == 'B')
		Action = R_Get_Seq;
	    else
		Action = R_Get_DLE;
	    break;

	case R_Get_Seq:
	    if (Read_Byte() == Failure)
		Action = R_Send_NAK;
	    else {
		Checksum = QBP_cm ? -1 : 0;
		Next_Seq = Ch - '0';
		Do_Checksum(Ch);
		R_Size = 0;
		Action = R_Get_Data;
	    }

	case R_Get_Data:
	    if (Read_Masked_Byte() == Failure)
		Action = R_Send_NAK;
	    else if (Seen_ETX)
		Action = R_Get_Checksum;
	    else if (R_Size > QBP_Packet) {
		msg = "Overrun";
		Action = R_Send_NAK;
	    } else {
		R_Buffer[R_Size++] = Ch;
		Do_Checksum(Ch);
	    }

	    break;

	case R_Get_Checksum:
	    Do_Checksum(ETX);

	    if (Read_Masked_Byte() == Failure) {
		Action = R_Send_NAK;
		break;
	    }

	    if (QBP_cm) {
		his_chk = Ch<<8;
		if (Read_Masked_Byte() == Failure) {
		    Action = R_Send_NAK;
		    break;
		}
		his_chk |= Ch;
	    } else
		his_chk = Ch;

	    if (Checksum != his_chk) {
		sprintf(mesag, "Checksum %4x s/b %4x", his_chk, Checksum);
		msg = mesag;
		Action = R_Send_NAK;
	    } else if (Next_Seq == Seq_Num)
		Action = R_Send_ACK;
	    else if (Next_Seq != (Seq_Num + 1) % 10) {
		msg = "Bad Block";
		Action = R_Send_NAK;
	    } else {
		Seq_Num = Next_Seq;
		return Success;
	    }

	    break;

	case R_Send_NAK:
	    Errors++;
	    Send_Byte(NAK);
	    Show_NAK(msg);
	    Action = R_Get_DLE;
	    break;

	case R_Send_ACK:
	    Send_ACK();
	    Action = R_Get_DLE;
	    break;
    }

    return Failure;
}

#define Next_W(ind)	(ind++, ind %=QBP_ws)

static int Send_Window(Size)
 int Size;                      /* size of packet to send */
{
    int Action;
    int I;
    char *ptr;
    struct QB_Hist *QB_Ptr;

    if (QB_Out == QBP_ws) {
	if (get_ACK() == Failure)
	    return Failure;
    }

    QB_Ptr = &QB_Buf[QB_Current];

    Seq_Num = (Seq_Num + 1) % 10;

    Checksum = QBP_cm ? -1 : 0;
    QB_Ptr->block = ++XMIT_Count;
    QB_Ptr->seq = Seq_Num + '0';
    Do_Checksum(Seq_Num + '0');

    QB_Ptr->size = Size;
    for (ptr = QB_Ptr->buf, I = 0; I < Size; I++) {
	Do_Checksum(S_Buffer[I] + 0);
	*ptr++ = S_Buffer[I];
    }
    Do_Checksum(ETX);
    QB_Ptr->chk = Checksum;

    Send_Buf(QB_Current);
    QB_Out++;
    Next_W(QB_Current);

    return Success;
}

Send_Buf(Ind)
{
    char *ptr;
    int i;

    struct QB_Hist *QB_Ptr = &QB_Buf[Ind];

    Show_XMIT(XMIT_Count == QB_Ptr->block?"Trans":"Retrans");

    Send_Byte(DLE);
    Send_Byte('B');
    Send_Byte(QB_Ptr->seq);
    for (ptr = QB_Ptr->buf, i = QB_Ptr->size; i > 0; i--)
	Send_Masked_Byte(*ptr++);

    Send_Byte(ETX);
    if (QBP_cm)
	Send_Masked_Byte(QB_Ptr->chk >> 8);

    Send_Masked_Byte(QB_Ptr->chk & 0xff);

}

get_ACK()
{
    int Action = S_Get_DLE;
    int Errors = 0;
    int hold, cnt;

    Errors = 0;

    while (QB_Out >= QBP_ws) {
	if (Errors > Max_Errors || Wants_To_Abort())
	    return Failure;

	switch (Action) {
	case S_Get_DLE:
	    if (Read_Byte() == Failure) {
		Action = S_Timed_Out;
		break;
	    }

	    if (Ch == DLE) {
		Action = S_Get_Num;
		break;
	    }

	    if (Ch == NAK) {
		Action = S_Got_NAK;
		break;
	    }

	    break;

	case S_Timed_Out:
	    Errors++;
	    Action = S_Get_DLE;
	    break;

	case S_Get_Num:
	    if (Read_Byte() == Failure) {
		Action = S_Timed_Out;
		break;
	    }

	    if (Ch < '0' || Ch > '9') {
		Errors++;
		Action = S_Get_DLE;
		break;
	    }
					/* Figure out which block was ACK'd */
	    while (QB_Buf[QB_Old].seq != Ch && QB_Out > 0) {
		QB_Out--;
		Next_W(QB_Old);
	    }

	    if (QB_Buf[QB_Old].seq != Ch)
		return Failure;		/* total loss of sync! */

	    Show_W_ACK();
	    Action = S_Get_DLE;
	    break;

	case S_Got_NAK:
	    Show_NAK();

	    QB_Current = QB_Old;	/* Reset currency point */
	    for (cnt = 0; cnt < QB_Out; cnt++, Next_W(QB_Current)) {
		if (Wants_To_Abort())
		    return Failure;
		Send_Buf(QB_Current);
	    }

	    Action = S_Get_DLE;
	    break;
	}
    }

    return Success;
}

static int Send_Packet(Size)
int Size;                           /* size of packet to send */
{
    int Action, Next_Seq, RCV_Num, I, Errors, his_chk;
    char *msg = "Timeout";
    char mesag[30];

    Next_Seq = (Seq_Num + 1) % 10;
    Errors = 0;
    Action = S_Send_Packet;

    while (Errors < Max_Errors && !Wants_To_Abort())
	switch (Action) {
	case S_Send_Packet:
	    Checksum = 0;
	    Send_Byte(DLE);
	    Send_Byte('B');
	    Send_Byte(Next_Seq + '0');
	    Do_Checksum(Next_Seq + '0');

	    for (I = 0; I < Size; I++) {
		Send_Masked_Byte(S_Buffer[I]);
		Do_Checksum(S_Buffer[I] + 0);
	    }

	    Send_Byte(ETX);
	    Do_Checksum(ETX);
	    Send_Masked_Byte(Checksum & 0xff);
	    Action = S_Get_DLE;
	    break;

	case S_Get_DLE:
	    if (Read_Byte() == Failure)
	        Action = S_Timed_Out;
	    else if (Ch == DLE)
		Action = S_Get_Num;
	    else if (Ch == NAK) {
		Errors++;
		Action = S_Send_Packet;
		Show_NAK("Unknown");
	    }

	    break;

	case S_Get_Num:
	    if (Read_Byte() == Failure) Action = S_Timed_Out;
	    else if (Ch >= '0' && Ch <= '9')
	    {
		if (Ch == Seq_Num + '0')
			Action = S_Get_DLE;
		else if (Ch == Next_Seq + '0')
		{
			Seq_Num = Next_Seq;
			return Success;
		}
		else if (Errors == 0) Action = S_Send_Packet;
		else Action = S_Get_DLE;
	    }
	    else if (Ch == WACK)
	    {
		sleep(5);        /* Sleep for 5 seconds */
		Action = S_Get_DLE;
	    }
	    else if (Ch == 'B') Action = S_Get_Seq;
	    else Action = S_Get_DLE;
	    break;

	case S_Get_Seq:
	/**
	 * Start of a "B" protocol packet. The only packet that makes
	 * any sense here is a failure packet.
	 **/

	    if (Read_Byte() == Failure)
		Action = S_Send_NAK;
	    else
	    {
		Checksum = 0;
		RCV_Num = Ch - '0';
		Do_Checksum(Ch);
		I = 0;
		Action = S_Get_Data;
	    }

	    break;

	case S_Get_Data:
	    if (Read_Masked_Byte() == Failure)
		Action = S_Send_NAK;
	    else if (Seen_ETX)
		Action = S_Get_Checksum;
	    else if (I > QBP_Packet) {
		msg = "Overrun";
		Action = S_Send_NAK;
	    } else {
		R_Buffer[I++] = Ch;
		Do_Checksum(Ch);
	    }

	    break;

	case S_Get_Checksum:
	    Do_Checksum(ETX);

	    if (Read_Masked_Byte() == Failure)
		Action = S_Send_NAK;

	    if (QBP_cm) {
		his_chk = Ch<<8;
		if (Read_Masked_Byte() == Failure) {
		    Action = R_Send_NAK;
		    break;
		}
		his_chk |= Ch;
	    } else
		his_chk = Ch;

	    if (Checksum != his_chk) {
		sprintf(mesag, "Checksum %4x s/b %4x", his_chk, Checksum);
		msg = mesag;
		Action = S_Send_NAK;
	    }
	    else if (RCV_Num != (Next_Seq + 1) % 10)
	    {
		msg = "Sequence";
		Action = S_Send_NAK;
	    }
	    else
	    {
       /**
	 * Assume the packet is failure packet. It makes no
	 * difference since any other type of packet would be
	 * invalid anyway. Return failure to caller.
	 **/

		Errors = Max_Errors;
	    }

	    break;

	case S_Timed_Out:
	    Errors++;
	    Action = S_Get_DLE;
	    break;

	case S_Send_NAK:
	    Errors++;
	    Send_Byte(NAK);
	    Action = S_Get_DLE;
	    Show_NAK(msg);
	    break;
	}

    return Failure;
}

static Send_Failure()
{
    register i;

    if (Failure_Sent++)
	return;

    Send_Byte(DLE);	/* Panic abort (usually works) */
    Send_Byte(DLE);
    Send_Byte(DLE);
    Send_Byte(DLE);
    Send_Byte(DLE);
}

static int Receive_File(Name)
char *Name;
{
    FILE *Data_File;
    int exists = (access(Name, 0) == 0);

    if (!mungmode && exists) {
	Put_Msg("Transfer aborted due to file pre-existance.");
	Send_Failure();
	return Failure;
    }

    if ((Data_File = fopen(Name, "w")) == NULL) {
	Put_Msg("Cannot create file");
	Send_Failure();
	return Failure;
    }

    if ((Data_File = fopen(Name, "w")) == NULL) {
	Put_Msg("Cannot create file");
	Send_Failure();
	return Failure;
    }

    fprintf(tfp,"Receiving %s file %s\r\n", Text_Mode ? "Ascii":"Binary", Name);
    Send_ACK();

    for (;;)
	if (Read_Packet(R_Get_DLE) == Success)
	    switch (R_Buffer[0]) {
	    case 'N':               /* Data packet */
		if (Write(Data_File, &R_Buffer[1], R_Size - 1)) {
		    Put_Msg("Disk write error");
		    Send_Failure();
		    fclose(Data_File);
		    return Failure;
		}

		if (Wants_To_Abort()) {
		    Send_Failure();
		    fclose(Data_File);
		    return Failure;
		}

		Send_ACK();
		Show_ACK();
		break;

	    case 'T':               /* Transfer packet */
		if (R_Buffer[1] == 'C') {
		    Send_ACK();
		    fclose(Data_File);
		    return Success;
		} else {
		    Put_Msg("Unexpected packet type");
		    Send_Failure();
		    fclose(Data_File);
		    return Failure;
		}

	    case 'F':               /* Failure packet */
		Send_ACK();
		fclose(Data_File);
		return Failure;
	    }
    else {
	fclose(Data_File);
	return Failure;
    }
}

static int Send_File(Name)
char *Name;
{
    FILE *Data_File;
    int N;
    long size;

    if ((Data_File = fopen(Name, "r")) == NULL)
    {
	Put_Msg("Cannot access that file");
	Send_Failure();
	return Failure;
    }

    size = filelength(Data_File);

    fprintf(tfp,"Transmitting %s file %s  (%ld bytes -- %ldK)\r\n",
	    Text_Mode ? "Ascii":"Binary", Name, size, size / 1024);

    do {
	S_Buffer[0] = 'N';
	N = Read(Data_File, &S_Buffer[1], QBP_Packet);

	if (N > 0) {
	    if (Send_Window(N + 1) == Failure) {
		fclose(Data_File);
		return Failure;
	    }

	    if (Wants_To_Abort()) {
		Send_Failure();
		fclose(Data_File);
		return Failure;
	    }
	}
    }    while (N > 0);

    if (N == 0) {
	fclose(Data_File);
	S_Buffer[0] = 'T';
	S_Buffer[1] = 'C';
	if (Send_Window(2) == Failure)
	    return Failure;
	return get_ACK();
    } else {
	Put_Msg("Disk read error");
	Send_Failure();
	return Failure;
    }
}

static int Transfer_File()
{
    XOFF_Flag = FALSE;
    Seq_Num = 0;

    for (;;) {
	if (Read_Packet(R_Get_DLE) == Success)
	    switch (R_Buffer[0]) {
	    case '+':			/* Get/Set QB Parameters */
		Get_QB_Parms();
		break;

	    case 'T':			/* Transfer packet */
		return do_transfer();
		break;

	    default:
		Send_Failure();	/* Undefined packet */
		return Failure;
	    }
	else
	    return Failure;
    }
}

static int Get_QB_Parms()
{
    S_Buffer[0] = '+';
    S_Buffer[1] = QBP_wr = min(1, R_Buffer[1]);
    S_Buffer[2] = QBP_ws = min(MAX_WINDOW - 1, R_Buffer[2]);
    S_Buffer[3] = QBP_bs = min(MAX_PACKET / 128, R_Buffer[3]);
    S_Buffer[4] = min(1, R_Buffer[4]);
    S_Buffer[5] = QBP_dq = 1;			/* Normal B Protocol Quoting */
    
    QBP_ws++;					/* Actual window size */

    if (Send_Packet(6) != Success)
	return Failure;

    QBP_cm = S_Buffer[4];
    QBP_Active = TRUE;
    QBP_Packet = QBP_bs * 128;
}

do_transfer()
{
    int I, N;
    char *Name;

    /* Check data direction */

    if (R_Buffer[1] != 'D' && R_Buffer[1] != 'U')
    {
	Send_Failure();      /* not implemented */
	return Failure;
    }

    /* Check the file type */

    if (R_Buffer[2] != 'A' && R_Buffer[2] != 'B')
    {
	Send_Failure();
	return Failure;
    }

    /* Let CIS tell us if it is Ascii or not */

    Text_Mode = (R_Buffer[2] == 'A');

    /* Collect the file name */

    Name = &R_Buffer[3];
    Name[R_Size - 3] = '\0';

    /* Do the transfer */

    start_time = time(NULL);

    if (R_Buffer[1] == 'U') 
	return Send_File(Name);
    else 
	return Receive_File(Name);
}

static Write(fp, buf, cnt)
 FILE *fp;
 char *buf;
 int cnt;
{
    for (; cnt-- > 0; buf++) {
	io_count++;

	if (Text_Mode && (*buf == '\r' && *(buf+1) == '\n'))
	    continue;

	if (putc(*buf, fp) == EOF)
	    return -1;
    }

    return 0;
}

static Read(fp, buf, max)
 FILE *fp;
 char *buf;
 int max;
{
    int c, cnt = 0;

    while (max--)
	switch (c = getc(fp)) {
	case EOF:
	    return cnt;

	case '\n':
	    if (Text_Mode && Last_Chr != '\r') {
		ungetc(c, fp);
		c = '\r';
	    }
	
	default:
	    Last_Chr = *buf++ = c;
	    cnt++; io_count++;
	    break;
	}

    return cnt;
}

int B_Transfer(mode)
{
    int rc;
    void newsigint();
    long elapsed;

    QBP_Active = FALSE;		/* Start in normal B Mode */
    QBP_ws = 0;
    QBP_wr = 0;
    QBP_bs = 0;
    QBP_cm = 0;
    QBP_dq = 0;
    QBP_Packet = 512;		/* Normal packet size maximum */

    fprintf(tfp,"CIS QuickB Protocol Transfer\r\n");

    Failure_Sent = Abort_Flag = Last_Chr = 0;
    XMIT_Count = ACK_Count = NAK_Count = 0;
    io_count = 0L;

    signal(SIGINT, newsigint);
    intdel(1);

    if (setjmp(err_buf) == 0) {
	xc_setxon(0);			/* turn off flow control */

	Send_Byte(NAK);

	if (startup() == Failure)
	    Abort_Flag = 1;
	else
	    rc = Transfer_File();
    }

    if (Abort_Flag) {
	Send_Failure();
	purge();
    }

    if (Abort_Flag || rc == Failure)
	fprintf(tfp,"\r\nFile Transfer aborted\r\n");
    else
	fprintf(tfp,"\r\nFile Transfer successful\r\n");

    if (io_count) {
	elapsed = time(NULL) - start_time;
	if (elapsed > 0) {
	    bytes_sec = io_count / start_time;
	    fprintf(tfp,"Average %ld bytes/sec\r\n", bytes_sec);
	}
    }

    xclog(0,0);

    intdel(0);
    signal(SIGINT, SIG_IGN);

    return rc;
}

static startup()
{
    int ch;

    do {
	Read_Byte();
	if (Ch != ENQ)
	    Send_Byte(NAK);
	if (Wants_To_Abort())
	    return Failure;
    } while (Ch != ENQ);

    Send_Byte(DLE); Send_Byte('+');	/* Quick B Indicator */
    Send_Byte(DLE); Send_Byte('0');

    return Success;
}

static void newsigint()
{
    Abort_Flag = 1;

    signal(SIGINT, SIG_IGN);

    fprintf(tfp,"\r\nUser Abort...\r\n");
}
