/*
    Module:	xcxmdm.c	XMODEM Protocol Module

    This source code is purely public domain
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>

#include "xcomm.h"

#define SOH     001     /* ^A */
#define EOT     004     /* ^D */
#define ACK     006     /* ^F */
#define NAK     025     /* ^U */
#define CAN     030     /* ^X */
#define CPMEOF  032     /* ^Z */
#define WANTCRC 'C'
#define OK       0
#define TIMEOUT  -1     /* -1 is returned by readbyte() upon timeout */
#define ERROR    -2
#define WCEOT    -3
#define RETRYMAX 10
#define SECSIZ  128

/* globals */
int
        badline         = FALSE,        /* bad telephone line?          */
        crcheck         = TRUE;         /* CRC check enabled?           */

extern FILE *tfp;			/* TTY File pointer */

/* locals */
static FILE
        *xfp;                           /* buffered file pointer        */
static int
        firstsec,                       /* first sector of file or not? */
        textmode        = TRUE;         /* Text translations enabled?   */
static char wcbuff[SECSIZ];             /* Ward Christensen sector buffer */
static unsigned short updcrc();

static void newsigint();		/* Our SIGINT handler */
static int  save_crc;			/* Saved crcheck value */
static jmp_buf our_env;

xreceive(c)
 int c;
{
    if (word[0] == '\0') {
	fprintf(tfp,"No file name specified for XMODEM receive\r\n");
	return;
    }

    xc_setxon(0);				/* turn off XON/XOFF */
    save_crc = crcheck;
    signal(SIGINT, newsigint);			/* Install our handler */

    if (!setmode(c)) {
	if (setjmp (our_env) == 0) {
	    crcheck = 0xff;
	    do_xreceive(word);
	}
    }

    signal(SIGINT, SIG_IGN);			/* Reinstall old handler */
    crcheck = save_crc;
    restmode();
}

xsend(c)
 int c;
{
    if (word[0] == '\0') {
	fprintf(tfp,"No file name specified for XMODEM send\r\n");
	return;
    }

    xc_setxon(0);				/* turn off XON/XOFF */
    save_crc = crcheck;
    signal(SIGINT, newsigint);			/* Install our handler */

    if (!setmode(c)) {
	if (setjmp (our_env) == 0) {
	    crcheck = 0xff;
	    do_xsend(word);
	}
    }

    signal(SIGINT, SIG_IGN);			/* Reinstall old handler */
    crcheck = save_crc;
    restmode();
}

static void newsigint()
{
    fprintf(tfp,"\r\nUser abort\r\n");
    signal(SIGINT, SIG_IGN);	/* Ignore subsequent DEL's */
    canit();			/* Abort the transmission */
    longjmp(our_env, 0);
}

static restmode()
{
    intdel(0);

    xc_setflow();				/* restore XON/XOFF */
}

static do_xreceive(p)
 char *p;
{
    long bytes, ftime;
    short fmode;
    char *ctime();

    fprintf(tfp,"XCOMM: Ready to receive single file %s\r\n", p);
    if (wcrx(p) == ERROR)
	canit();
    return;
}

do_xsend(p)
 char *p;
{
    if (wctx(p) == ERROR) {
	fprintf(tfp,"Error transmitting file %s\r\n", p);
	return;
    }
}

/*
    Receive a file using XMODEM protocol
*/

static wcrx(name)
char *name;
{
    register int sectnum, sectcurr, sendchar, eofseen;

    if (!mungmode && !access(name, 0)) {
        canit();
	fprintf(tfp,"Receive of %s aborted due to pre-exsistence.\r\n");
        return(ERROR);
    }

    if ((xfp = fopen(name, "w")) == NULL) {
        canit();
	fprintf(tfp,"Receive of %s aborted due to inabilty to open.\r\n");
        return(ERROR);
    }
    firstsec = TRUE;
    eofseen = FALSE;
    sectnum = 0;
    sendchar = crcheck ? WANTCRC : NAK;
    fprintf(tfp,"Sync...\r\n");

    while(TRUE) {
        if (badline)
            purge();
        sendbyte(sendchar);
        sectcurr = wcgetsec(6);
        if (sectcurr == ((sectnum + 1) & 0xff)) {
            sectnum++;
	    putsec();
	    fprintf(tfp,"Received sector #%d             \r", sectnum);
	    fflush(tfp);

            sendchar = ACK;
            continue;
        }

        if (sectcurr == (sectnum & 0xff)) {
	    fprintf(tfp,"Received duplicate sector #%d   \r", sectnum);
	    fflush(tfp);
            sendchar = ACK;
            continue;
        }

	fprintf(tfp, "\r\n");
        fclose(xfp);

        if (sectcurr == WCEOT) {
	    fprintf(tfp,"File recived OK\r\n");
            sendbyte(ACK);
            return(OK);
        }

        if (sectcurr == ERROR)
            return(ERROR);

	fprintf(tfp,"Sync error ... expected %d(%d), got %d\r\n",
		    (sectnum + 1) & 0xff, sectnum, sectcurr);
        return(ERROR);
    }
}

/*
    Transmit a file using XMODEM protocol
*/

static wctx(name)
char *name;
{
    register int sectnum, eoflg, c, attempts;

    if ((xfp = fopen(name, "r")) == NULL) {
	fprintf(tfp,"Can't open file %s for reading.\r\n", name);
        return(ERROR);
    }
    firstsec = TRUE;
    attempts = 0;
    fprintf(tfp,"Sync...\r\n");

    while((c = readbyte(30)) != NAK && c != WANTCRC && c != CAN)
        if (c == TIMEOUT && ++attempts > RETRYMAX) {
	    fprintf(tfp,"Receiver not responding.\r\n");
            fclose(xfp);
            return(ERROR);
        }
    if (c == CAN) {
	fprintf(tfp,"Receiver CANcelled.\r\n");
        fclose(xfp);
        return(ERROR);
    }
    crcheck = (c == WANTCRC);
    fprintf(tfp,"%s error checking requested.\r\n",
		crcheck ? "CRC check" : "Checksum");
    sectnum = 1;

    do {
        eoflg = getsec();
	fprintf(tfp,"Transmitting sector #%d\r", sectnum);
	fflush(tfp);

        if (wcputsec(sectnum) == ERROR) {
            fclose(xfp);
            return(ERROR);
        }
        sectnum++;
    } while(eoflg);

    fprintf(tfp, "\r\n");
    fclose(xfp);
    attempts = 0;
    sendbyte(EOT);
    while(readbyte(5) != ACK && attempts++ < RETRYMAX)
        sendbyte(EOT);
    if (attempts >= RETRYMAX) {
	fprintf(tfp,"Receiver not responding to completion.\r\n");
        return(ERROR);
    }

    fprintf(tfp,"Transmission complete.\r\n");
    return(OK);
}

/*
    wcgetsec() inputs an XMODEM "sector".
    This routine returns the sector number encountered, or ERROR if a valid
    sector is not received or CAN received; or WCEOT if EOT sector.

    Maxtime is the timeout for the first character, set to 6 seconds for
    retries.  No ACK is sent if the sector is received ok.  This must be
    done by the caller when it is ready to receive the next sector.
*/

static wcgetsec(maxtime)
int maxtime;
{
    register checksum, j, c;
    register unsigned short oldcrc;
    int sectcurr, sectcomp, attempts;

    for(attempts = 0; attempts < RETRYMAX; attempts++) {
        do {
            c = readbyte(maxtime);
        } while(c != SOH && c != EOT && c != CAN && c != TIMEOUT);

        switch(c) {
        case SOH:
            sectcurr = readbyte(3);
            sectcomp = readbyte(3);
            if ((sectcurr + sectcomp) == 0xff) {
                oldcrc = checksum = 0;
                for(j = 0; j < SECSIZ; j++) {
                    if ((c = readbyte(3)) < 0)
                        goto timeout;
                    wcbuff[j] = c;
                    if (crcheck)
                        oldcrc = updcrc(c, oldcrc);
                    else
                        checksum += c;
                }
                if ((c = readbyte(3)) < 0)
                    goto timeout;
                if (crcheck) {
                    oldcrc = updcrc(c, oldcrc);
                    if ((c = readbyte(3)) < 0)
                        goto timeout;
                    if (updcrc(c, oldcrc)) {
			fprintf(tfp,"\r\nCRC error\r\n");
                        break;
                    }
                }
                else if (((checksum - c) & 0xff) != 0) {
		    fprintf(tfp,"\r\nChecksum error\r\n");
                    break;
                }
                firstsec = FALSE;
                return(sectcurr);
            }
            else 
                fprintf(tfp,"\r\nSector number garbled 0%03o 0%03o\r\n",
			sectcurr, sectcomp);
            break;
        case EOT:
            if (readbyte(3) == TIMEOUT)
                return(WCEOT);
            break;
        case CAN:
	    fprintf(tfp,"\r\nSender CANcelled.\r\n");
            return(ERROR);
        case TIMEOUT:
            if (firstsec)
                break;
timeout:
	    fprintf(tfp,"\r\nTimeout\r\n");
            break;
        }
	fprintf(tfp,"\r\nTrying again on ths sector.\r\n");
        purge();
        if (firstsec)
            sendbyte(crcheck ? WANTCRC : NAK);
        else {
            maxtime = 6;
            sendbyte(NAK);
        }
    }
    fprintf(tfp,"\r\nRetry count exceeded.\r\n");
    canit();
    return(ERROR);
}

/* wcputsec outputs a Ward Christensen type sector.
 * it returns OK or ERROR
 */
static wcputsec(sectnum)
int sectnum;
{
    register checksum, j, c, attempts;
    register unsigned short oldcrc;

    oldcrc = checksum = 0;
    for(j = 0; j < SECSIZ; j++) {
        c = wcbuff[j];
        oldcrc = updcrc(c, oldcrc);
        checksum += c;
    }
    oldcrc = updcrc(0, updcrc(0, oldcrc));

    for(attempts = 0; attempts < RETRYMAX; attempts++) {
        sendbyte(SOH);
        sendbyte(sectnum);
        sendbyte(-sectnum - 1);
        for(j = 0; j < SECSIZ; j++)
            sendbyte(wcbuff[j]);
        if (badline)
            purge();
        if (crcheck) {
            sendbyte((int) (oldcrc >> 8));
            sendbyte((int) oldcrc);
        }
        else
            sendbyte(checksum);
        switch(c = readbyte(10)) {
        case CAN:
	    fprintf(tfp,"\r\nReceiver CANcelled.\r\n");
            return(ERROR);
        case ACK:
            firstsec = FALSE;
            return(OK);
        case NAK:
	    fprintf(tfp,"\r\nGot a NAK on sector acknowledge.\r\n");
            break;
        case TIMEOUT:
	    fprintf(tfp,"\r\nTimeout on sector acknowledge.\r\n");
            break;
        default:
	    fprintf(tfp,"\r\nGot 0%03o for sector acknowledge.\r\n", c);
            do {
                if ((c = readbyte(3)) == CAN) {
		    fprintf(tfp,"\r\nReceiver CANcelled.\r\n");
                    return(ERROR);
                }
            } while(c != TIMEOUT);
            if (firstsec)
                crcheck = (c == WANTCRC);
            break;
        }
    }
    fprintf(tfp,"\r\nRetry count exceeded.\r\n");
    return(ERROR);
}

/* update the cyclic redundantcy check value
 */
static unsigned short updcrc(c, crc)
register c;
register unsigned crc;
{
    register int i;

    for(i = 0; i < 8; i++) {
        if (crc & 0x8000) {
            crc <<= 1;
            crc += (((c <<= 1) & 0400) != 0);
            crc ^= 0x1021;
        }
        else {
            crc <<= 1;
            crc += (((c <<= 1) & 0400) != 0);
        }
    }
    return(crc);
}

static zerobuff()
{
    register int i;

    for(i = 0; i < SECSIZ; i++)
        wcbuff[i] = '\0';
}

/* send 10 CAN's to try to get the other end to shut up */
static canit()
{
    register int i;

    for(i = 0; i < 20; i++)
        sendbyte(CAN);
}

static setmode(c)
int c;
{
    intdel(1);

    switch(mklow(c)) {
    case 't':
        textmode = TRUE;
        break;
    case 'b':
        textmode = FALSE;
    case ' ':
        break;

    default:
        return(1);
    }

    fprintf(tfp,"%s file transfer mode.\r\n", textmode ? "Text" : "Binary");

    return(0);
}

/* fill the CP/M sector buffer from the UNIX file
 * do text adjustments if necessary
 * return 1 if more sectors are to be read, or 0 if this is the last
 */
static getsec()
{
    register int i, c;

    i = 0;
    while(i < SECSIZ && (c = getc(xfp)) != EOF) {
        if (textmode && c == '\n') {
            wcbuff[i++] = '\r';
            if (i >= SECSIZ) {          /* handle a newline on the last byte */
                ungetc(c, xfp);         /* of the sector                     */
                return(1);
            }
        }
        wcbuff[i++] = c;
    }
    /* make sure that an extra blank sector is not sent */
    if (c != EOF && (c = getc(xfp)) != EOF) {
        ungetc(c, xfp);
        return(1);
    }
    /* fill up the last sector with ^Z's if text mode or 0's if binary mode */
    while(i < SECSIZ)
        wcbuff[i++] = textmode ? CPMEOF : '\0';
    return(0);
}

/* Put received WC sector into a UNIX file
 * using text translations in nessesary.
 */

static putsec()
{
    register int i, c;

    for(i = 0; i < SECSIZ; i++) {
        c = wcbuff[i];
        if (textmode) {
            if (c == CPMEOF)
                return(1);
            if (c == '\r')
                continue;
        }
        putc(c, xfp);
    }
    return(0);
}

