/*
    Program	xcscrpt.c	Script handler for xcomm
    Author	larry gensch, ESQ  December 4, 1987

    This code is released to the public domain
*/

#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <signal.h>

#include "xcomm.h"

#define	MAX_PATH	256
#define	MAX_LINE	128

extern char *getenv();
extern FILE *tfp;

static FILE *script_file, *cf;
static int waitfor_time = 0;
static int tty_flag = 1;
static int echo_flag = 0;
static int if_flag = 0;
static int waitflag = 0;
static int captflag = 0;
static jmp_buf here;

static void k_waitfor(), k_transmit(), k_pause(), k_exit(), k_quit();
static void k_if(), k_goto(), k_else(), k_endif(), k_redial();
static void k_dial(), s_set(), hangup(), k_echo(), k_tty(), k_capture();

/* globals */

int linkflag = 0;
int scriptflag = 0;

static struct kw kw[] = {
	{ "waitfor",    k_waitfor },
	{ "transmit",	k_transmit },
	{ "pause",	k_pause },
	{ "exit",	k_exit },
	{ "if",		k_if },
	{ "else",	k_else },
	{ "endif",	k_endif },
	{ "hangup",	hangup },
	{ "goto",	k_goto },
	{ "dial",	k_dial },
	{ "redial",	k_redial },
	{ "quit",	k_quit },
	{ "set",	s_set },
	{ "echo",	k_echo },
	{ "capture",	k_capture },
	{ "tty",	k_tty },
	{ NULL,		NULL }};

static void newsigint();

do_script(file)
 char *file;
{
    int quiet = 0;

    if (open_script(file)) {
	fprintf(tfp, "Cannot find script file %s\r\n", file);
	return;
    }

    if (linkflag == 2) {
	quiet = 1;
	linkflag = 0;
    }

    if (!quiet)
	fprintf(tfp, "<<< RUNNING SCRIPT >>>\r\n");

    if_flag = 0;
    echo_flag = 0;
    captflag = 0;
    tty_flag = 1;
    scriptflag = 1;

    if (setjmp(here) == 0) {
	signal(SIGINT, newsigint);
	intdel(1);

	for (eof_flag = 0; !eof_flag; )
	    get_line();
    }

    fclose(script_file);
    if (captflag)
	fclose(cf);

    if (!quiet)
	fprintf(tfp, "<< SCRIPT COMPLETE >>\r\n");

    linkflag = 0;
    scriptflag = 0;

    signal(SIGINT, SIG_IGN);
    intdel(0);

    return;
}

static void newsigint()
{
    signal(SIGINT, SIG_IGN);

    eof_flag = 1;
    fprintf(tfp, "\r\nUser Abort...\r\n");
    longjmp(here);
}

static char path[MAX_PATH];

static open_script(file)
 char *file;
{
    char *home;

    eof_flag = 0;

    if (script_file = fopen(file, "r"))
	return 0;
    
    if ((home = getenv("HOME")) == NULL)
	return 1;

    sprintf(path, "%s/%s", home, file);

    if (script_file = fopen(path, "r"))
	return 0;

    return 1;
}
    
static char wf[MAX_LINE];

static get_line()
{
    int i;

    getline(script_file);

    if (eof_flag)
	return;

    getword();
    lc_word(word);

    if (echo_flag) {
	if (strcmp(word, "transmit"))
	    fprintf(tfp, "%s\r\n", line);
	else
	    fprintf(tfp, "TRANSMIT...\r\n");
    }
	
    if (word[0] == '\0')		/* Ignore blank lines */
	return;
    if (word[0] == '#')			/* Ignore comments */
	return;

    if (word[strlen(word) - 1] == ':')	/* Ignore labels */
	return;

    if (if_flag == -1) {
	if (strcmp(word, "else") && strcmp(word, "endif"))
	    return;
    }

    for (i = 0; kw[i].keyword != NULL; i++)
	if (strcmp(kw[i].keyword, word) == 0) {
	    (*kw[i].rtn)();
	    return;
	}

    fprintf(tfp, "UNDEFINED COMMAND: \"%s\"\r\n", word);
    eof_flag = 1;
    return;
}

static void k_waitfor()
{
    long t;
    int c;
    char *ptr = wf;

    quote_flag = 1;
    getword();
    quote_flag = 0;
    if (eof_flag)
	return;

    if (word[0] == '\0') {
	fprintf(tfp, "No argument to WAITFOR command\r\n");
	eof_flag = 1;
	return;
    }

    strcpy(wf, word);
    lc_word(wf);

    getword();
    if (eof_flag)
	return;

    if (word[0])
	waitfor_time = atoi(word);
    else
	waitfor_time = 30;

    t = time(NULL) + waitfor_time;

    while (t != time(NULL) && !eof_flag) {
	if ((c = read_mbyte(1)) == -1)
	    continue;

	if (cismode && c == ENQ) {
	    fprintf(tfp,"B transfer in middle of script\r\n");
	    cismode = 1;
	    B_Transfer();

	    t = time(NULL) + waitfor_time;

	    if ((c = read_mbyte(1)) == -1)
	      continue;
	}

	if (tty_flag)
	    fputc(c, tfp);

	if (captflag)
	    fputc(c, cf);

	if (tolower(c) != *ptr) {
	    ptr = wf;
	    continue;
	}

	if (*++ptr == '\0') {
	    waitflag = 1;
	    return;
	}
    }

    waitflag = 0;
}

static void k_transmit()
{
    char *ptr;

    quote_flag = 1;
    getword();
    quote_flag = 0;
    if (eof_flag)
	return;

    if (word[0] == '\0') {
	fprintf(tfp, "No argument to TRANSMIT command\r\n");
	eof_flag = 1;
	return;
    }

    send_slowly(word);
}

static void k_pause()
{
    int pause_time;

    getword();
    if (eof_flag)
	return;

    if (word[0] == '\0')
	pause_time = 5;
    else
	pause_time = atoi(word);

    sleep(pause_time);
}

static void k_quit()
{
    do_exit(0);		/* Terminate XCOMM */
}

static void k_exit()
{
    eof_flag = 1;
}

static char label[WBSIZE];

static void k_goto()
{
    int found = 0, i;

    getword();
    if (word[0] == '\0') {
	fprintf(tfp, "No argument for GOTO: %s\r\n", line);
	eof_flag++;
	return;
    }

    strcpy(label, word);
    lc_word(label);

    rewind(script_file);
    while (!found) {
	getline(script_file);
	if (eof_flag)
	    break;

	getword();
	if (word[0] == '\0' || word[0] == '#')
	    continue;

	if (word[i = (strlen(word)-1)] != ':')
	    continue;

	word[i] = '\0';

	lc_word(word);
	found = (strcmp(word, label) == 0);
    }

    if (eof_flag) {
	fprintf(tfp, "Label %s not found\r\n", label);
	eof_flag++;
	return;
    }

    if_flag = 0;			/* reset IF flag */
}

static if_negate = 0;

static void k_if()
{
    char *ptr;

    if (if_flag) {
	fprintf(tfp, "Nested IF statements not allowed\r\n");
	eof_flag++;
	return;
    }

    if_negate = 0;
    getword();
    if (word[0] == '\0') {
	fprintf(tfp, "No condition on IF statement\r\n");
	eof_flag++;
	return;
    }

    lc_word(word);
    if (strcmp(word, "not") == 0) {
	if_negate = 1;
	getword();
	if (word[0] == '\0') {
	    fprintf(tfp, "No condition on IF statement\r\n");
	    eof_flag++;
	    return;
	}
    }

    if (word[0] == '!') {
	if_negate = 1;
	ptr = word + 1;
    } else
	ptr = word;

    if (strcmp(ptr, "waitfor") == 0) {
	if_flag = if_test(waitflag);
	return;
    }

    if (strcmp(ptr, "linked") == 0) {
	if_flag = if_test(linkflag);
	return;
    }

    fprintf(tfp, "Undefined IF condition %s\r\n", ptr);
    eof_flag++;
    return;
}

static if_test(cond)
 int cond;
{
    if (if_negate)
	cond = !cond;

    if (cond)
	return 1;
    else
	return -1;
}

static void k_else()
{
    if (!if_flag) {
	fprintf(tfp, "ELSE not within IF\r\n");
	eof_flag++;
	return;
    }

    if_flag = -if_flag;
}

static void k_endif()
{
    if (!if_flag) {
	fprintf(tfp, "ENDIF not wihtin IF\r\n");
	eof_flag++;
	return;
    }

    if_flag = 0;
}

static void k_dial()
{
    getword();

    if (word[0] == '\0') {
	fprintf(tfp, "DIAL command must have an argument\r\n");
	eof_flag++;
	return;
    }

    dial(word);
}

static void k_redial()
{
    if (redial()) {
	eof_flag++;
	return;
    }
}

static void k_echo()
{
    set_onoff(&echo_flag);
    return;
}

static void k_capture()
{
    int val = captflag;

    set_onoff(&captflag);
    if (eof_flag)
	return;

    if (val == captflag)
	return;
	
    if (captflag == 0)
	fclose(cf);
    else {
	if ((cf = fopen(captfile, "a")) == NULL) {
	    fprintf(tfp, "Cannot open capture file %s\r\n");
	    eof_flag++;
	    return;
	}
    }
}

static void k_tty()
{
    set_onoff(&tty_flag);
    return;
}

