/*
    Module:	xcport.c	XCOMM Modem Interface Routines

    This code is purely public domain!
*/

#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <termio.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>

#include "xcomm.h"

/*
    The dial() routine uses these two defines.

    DIALSTR is a sprintf format string that assumes a HAYES-compatible
    modem.

    HAYSATT is the HAYES "attention" signal (used if DTR_DROPS_CARRIER
    is not set to 1).

    HAYSHUP is the HAYES "hangup" command (used if DTR_DROPS_CARRIER is not
    set to 1).

    MDELAY is the delay in the output because (on my modem) the command
    would be ignored if sent at full speed.  Change for other setups.
    (This setting is for U.S. Robotics Password Modem).
*/

#define DIALSTR "\rATDT %s\r"   /* format string for Hayes-type modem */
#define HAYSATT "+++"		/* Hayes "attention" signal */
#define	HAYSHUP	"ATH\r"		/* Hayes "hang up" command */
#define MDELAY  20000           /* delay for output to modem itself */

/* globals */

extern int tfp;			/* TTY file pointer */
int bitmask = 0xFF;             /* modem port i/o data mask */
int flowflag = 0;               /* modem port i/o data mask */

static int mfd = -1;            /* modem port file descriptor */
static struct termio pmode;	/* modem device control string */
static char port[NMSIZE];       /* modem port device file string */
static int baudrate = B1200;	/* baud rate */

int cpmode()
{
    pmode.c_iflag |= IGNBRK;

    pmode.c_lflag = 0;
#ifdef XCLUDE
    pmode.c_lflag |= XCLUDE;
#endif
    pmode.c_oflag = 0;	/* Transparent output */

    pmode.c_cflag = baudrate | CS8 | CREAD | CLOCAL;

    pmode.c_cc[VMIN] = 1; 	/* This many chars satisfies reads */
    pmode.c_cc[VTIME] = 1;	/* or in this many tenths of seconds */

    xc_setflow();
}

int xc_setxon(mode)
{
    cpmode();

    if (mode) {
	pmode.c_iflag |= IXON | IXOFF;
	pmode.c_iflag &= ~IXANY;
    } else
	pmode.c_iflag &= ~(IXON | IXOFF | IXANY);

    if(mfd != -1)
	ioctl(mfd, TCSETAW, &pmode);
}

int xc_flowtoggle()
{
    flowflag = 1 - flowflag;

    xc_setflow();
}

xc_setflow()
{
    if (flowflag) {
	pmode.c_iflag |= IXON | IXOFF;
	pmode.c_iflag &= ~IXANY;
    } else
	pmode.c_iflag &=~(IXON | IXOFF | IXANY);

    if (mfd != -1)
	ioctl(mfd, TCSETAW, &pmode);
}

char *mport(s)  /* get/set port string */
char *s;
{
    if(s != NULL && mfd == -1)
        strcpy(port, s);
    return(port);
}

/* Get/set the baud rate of the modem port. If the port hasn't been opened yet,
 * just store in pmode for mopen() to use when it opens the port.
 */
int mbaud(s)
char *s;
{
    cpmode();

    if(s != NULL){
        /* this gives a more limited, realistic range than in sgtty.h */
        switch (atoi(s)) {
        case  300: baudrate = B300;  break;
        case 1200: baudrate = B1200; break;
        case 2400: baudrate = B2400; break;
        case 9600: baudrate = B9600; break;
        default:   return(-1);
        }
	pmode.c_cflag &= ~CBAUD;
	pmode.c_cflag |= baudrate;

        if(mfd != -1)
            ioctl(mfd, TCSETAW, &pmode);
    }

    switch(pmode.c_cflag & CBAUD){
    case  B300: return(300);
    case B1200: return(1200);
    case B2400: return(2400);
    case B9600: return(9600);
    }

    fprintf(tfp,"Impossible error in baud rate.\n");
    return(0);
}

/*
    The following routine is used to hang up the modem.  This is
    accomplished by setting the baud rate to 0.  According to my
    documentation on termio, setting the baud rate to zero will
    result in DTR not being asserted.  This hangs up some (most?)
    modems.  If not, the second part of the routine sends the Hayes
    modem "escape" and then a hangup command.
*/

void hangup()
{
    if (mfd == -1)
	return;

    fprintf(tfp,"<< HANGUP >>\r\n");
    fflush(stdout);

#if DTR_DROPS_CARRIER
    pmode.c_cflag &= ~CBAUD;
    pmode.c_cflag |= B0;	/* set baud 0 (drop DTR) */
    ioctl(mfd, TCSETAW, &pmode);

    sleep(1);			/* wait a minute */

    pmode.c_cflag &= ~CBAUD;	/* reset baud rate */
    pmode.c_cflag |= baudrate;
    ioctl(mfd, TCSETAW, &pmode);
#else	/* use Hayes command */
    sleep(2);			/* Allow for "escape guard time" */
    send_slowly(HAYSATT);	/* Send modem escape command */

    sleep(3);			/* More "escape guard time" */
    send_slowly(HAYSHUP);	/* Send hangup command */
#endif
}

/* Opens the modem port and configures it. If the port string is
 * already defined it will use that as the modem port;  otherwise it
 * gets the environment variable MODEM. Returns 0 for success
 * or -1 on error.
 */

static int	  opened = 0;

int mopen()
{
    char *p, *getenv();
    int oldflags;

    cpmode();

    if(port[0] == '\0'){
        if((p = getenv("MODEM")) == NULL)
            return(-1);
        strcpy(port, p);
    }

    if (lock_tty()) {
	fprintf(stderr, "Modem device is locked -- try later.\r\n");
	return -1;
    }

    /* Need O_NDELAY to get the file open before we have carrier */
    if((mfd = open(port, O_RDWR | O_NDELAY)) < 0) {
	perror("mopen()");
	unlock_tty();
        return -1;
	}

    /* Now, we must reset the O_NDELAY mode so that read() works correctly */
    if ( ((oldflags = fcntl(mfd, F_GETFL, 0)) == -1) ||
	 (fcntl(mfd, F_SETFL, oldflags & ~O_NDELAY) == -1) ) {
		perror("mopen can not reset O_NDELAY");
		unlock_tty();
		return -1;
		}

    ioctl(mfd, TCFLSH, 2);
    ioctl(mfd, TCSETAW, &pmode);

    opened++;

    return(0);
}

#if !HAVE_DUP2		/* For those that do not have dup2()... */
int dup2(oldfd, newfd)
int oldfd, newfd;
{
        if (fcntl(oldfd, F_GETFL, 0) == -1)     /* Valid file descriptor? */
                return (-1);                    /* No, return an error. */
        close(newfd);                           /* Ensure newfd is closed */
        return (fcntl(oldfd, F_DUPFD, newfd));  /* Dup oldfd into newfd */
}
#endif /* !HAVE_DUP2  Thanks to Bill Allie CIS: 76703,2061 */

/* Attach standard input and output to the modem port. This only gets called
 * after a fork by the child process; which then exec's a program that uses
 * standard i/o for some data transfer protocol. (To put this here is actually
 * a kludge, but I wanted to keep the modem-specific stuff in a black box.)
 */
mattach()       /* attach standard i/o to port */
{
    dup2(mfd, 0);       /* close local stdin and connect to port */
    dup2(mfd, 1);       /* close local stdout and connect to port */

    close(mfd);         /* close the old port descriptor */
}

int readbyte(seconds)
int seconds;
{
    return trminp(mfd, seconds);
}

/*
    Read a byte using bitmask
*/

read_mbyte(secs)
{
    int c;

    return (c = readbyte(secs)) == -1 ? -1 : c & bitmask;
}

/* Output a byte to the modem port.
 * All data sent to the modem is output through this routine.
 */
sendbyte(ch)
int ch;
{
    char c = ch & 0xff;

    write(mfd, &c, 1);
}

send_mbyte(ch)
{
    sendbyte(ch & bitmask);
}

/* Dial a phone number, using proper format and delay.
 */

static char *last_nbr = NULL;
char *strdup();

dial(s)
char *s;
{
    char buffer[WBSIZE];

    if (last_nbr) 
	free(last_nbr);

    last_nbr = strdup(s);

    sprintf(buffer, DIALSTR, s);
    send_slowly(buffer);
}

redial(last_nbr)
 char *last_nbr;
{
    char *s;

    if (last_nbr == NULL) {
	fprintf(tfp,"REDIAL FAILURE\r\n");
	return 1;
    }

    s = strdup(last_nbr);
    dial(s);
    free(s);
    return 0;
}

send_slowly(s)
 char *s;
{
    int i;

/*
    This busy-waiting, normally a bad idea on a multi-tasking system,
    was used because sleep(1) is way too much of a delay.
*/
    while(*s){
        send_mbyte(*s++);
        for(i = 0; i < MDELAY; i++)
            ;
    }
}

/*
    I have had requests to support the LCK..ttyxx files that CU and UUCP
    tend to support.  I do not have any need for such code, and am not
    exactly sure how the code should work.  I have therefore placed these
    two entry points for some enterprising coder to code so that the lock
    files may be created and destroyed.

    lock_tty() returns non-zero if the lock file exists (prevents XCOMM from
    running).

    unlock_tty() deletes the lock file.

    Simple, eh?
*/

char lckf[40] = "/usr/spool/locks/LCK..";
char ltmp[40] = "/usr/spool/locks/LTMP.";

lock_tty()
{
    int pid, lckpid, lfd;
    char pidstr[20];
    char lckpidstr[20];
    char *modemname;
    extern int errno;

    /*
     * Get our PID, and initialize the filename strings.
     */
    pid = getpid();
    sprintf(pidstr, "%d", pid);
    modemname = strrchr(port, '/');
    strcat(lckf, (modemname ? (modemname+1) : port));
    strcat(ltmp, pidstr);

    /*
     * Create the LTMP.<pid> file and scribble our PID in it.
     */
    unlink(ltmp);
    if ((lfd = creat(ltmp, 0444)) == -1) {
	fprintf(stderr, "Can't creat(2) LTMP?\r\n");
	return -1;
    }
    sprintf(pidstr, "%10d\n", pid);
    write(lfd, pidstr, 11);
    close(lfd);

    /*
     * Attempt to link directly - if it works, we're done.
     */
  relink:
    if (link(ltmp, lckf) == 0) {
	unlink(ltmp);
        return 0;
    }

    /*
     * Oh, bother, there's a LCK..* file already; we must
     * now expend effort to learn if it's stale or not.
     */
    if ((lfd = open(lckf, O_RDONLY)) != -1) {
	if (read(lfd, lckpidstr, 11) == 11) {
	    lckpid = atol(lckpidstr);
	    if (kill(lckpid, 0) == 0) {
		fprintf(stderr, "%s locked by process %d.\r\n", port, lckpid);
		unlink(ltmp);
		return -1;
	    }
	}
    }

    /*
     * The LCK..* file was stale.  Remove and retry.
     */
    if (unlink(lckf)) {
	fprintf(stderr, "Can't unlink(2) stale LCK?\r\n");
	unlink(ltmp);
	return -1;
    }
    goto relink;
    /*NOTREACHED*/
}

unlock_tty()
{
    unlink(lckf);
}
