/*
	xcomm		External (and XMODEM) Communications
			By Eric E. Coe

			Version 2.0 and above by larry gensch, ESQ

   This program is in the public domain, and anyone can use it for
   personal or non-profit use.  Feel free to study the code and borrow
   from it (I borrowed from other programs to create it).

   I am supporting versions 2.0 and higher of XCOMM.  Information (bug
   fixes, requests, etc.) may be sent to any one of the following electronic
   addresses:

   CompuServe				[72236,3516]	UNIXFORUM
   Delphi				LARRYG
   Bix					lar3ry
   Andover CNode (BBS) [617-470-2548]	larry gensch

   -lar3ry gensch
    programmer extraordinaire

*/

#define  VERSION   "2.2"

/*
		R E V I S I O N     H I S T O R Y

Version 2.2	larry gensch	12/8/87		Major code restructure
    Reduced program and command options
    Added SET command for setting various parameters
    Added HANGUP command for disconnecting modem
    Modified xccisb.c code to support CIS "Quick B" Protocol
    Added xcscrpt.c code for processing script files
    Added SCRIPT= parameter to phonebook processing (auto-logon scripts)
    Added CIS parameter for CIS <ENQ> auto transfers
    Added NL parameter for newline translation
    Added HANGUP command from terminal mode
    Added SCRIPT command from terminal mode

Version 2.1b	larry gensch	11/11/87	Bug Fix release
    (no bugs in the code, just some portability fixes)
    Changed "sigset" in xccisb to "signal"
    Removed #include <setjmp.h> from xcxmdm

Version 2.1a	larry gensch	10/28/87	General Update
    Added CIS "B" Protocol (c, ct commands)
    Added BAUD= & BITS= parameters to phonelist file.
    Switched some command letters
    Revised command line parsing to use getopt()

Version 2.0	larry gensch	10/19/87	Revisions for System V.3
    Basically version 1.1 with minor modifications

Version 1.1	Eric E Coe	7/21/85
    Autodial for Hayes-compatible modem.
    General rearranging of the code

Version 1.0	Eric E Coe	4/12/85
    Program created
*/

#include <stdio.h>
#include <signal.h>
#include <ctype.h>
#include <termio.h>

#include "xcomm.h"

/* globals */

jmp_buf erret;			/* non-local error return */
int mungmode = FALSE;		/* ok to overwrite files? */
int nlmode = TRUE;		/* Map newlines to carriage returns */
int stat_flag = FALSE;		/* Flag for status display */

/* locals */
struct termio newmode, oldmode, sigmode;

static int termflag = 0;
static int reterm = 0;

int tfd;
FILE *tfp;

void s_cis(), s_script(), s_xmodem(), s_set(), s_term(), s_exit();
void s_shell(), s_help(), hangup();

#ifdef PUT_TAKE
void puttake();
#endif

/*
    Command table
*/

static struct kw cmds[] = {
	{ "c",		s_cis },
	{ "g",		s_script },
	{ "hangup",	hangup },
	{ "rb",		s_xmodem },
	{ "rt",		s_xmodem },
	{ "sb",		s_xmodem },
	{ "st",		s_xmodem },
	{ "set",	s_set },
	{ "t",		s_term },
	{ "x",		s_exit },
	{ "exit",	s_exit },
	{ "!",		s_shell },
	{ "!!",		s_shell },
	{ "$",		s_shell },
#ifdef PUT_TAKE
	{ "%p",		puttake },
	{ "%t",		puttake },
#endif
	{ "help",	s_help },
	{ "?",		s_help },
	{ NULL,		NULL }};

static char oldshell[WBSIZE];

main(argc, argv)
int argc;
char *argv[];
{
    extern char *optarg;
    extern int optind, opterr;

    struct kw *ptr;
    void catch();
    int c;
    char *script = NULL;

    if ((tfp = fopen("/dev/tty", "r+")) == NULL) {
	fprintf(stderr, "Cannot access local terminal!\n");
	exit(1);
    }

    tfd = fileno(tfp);

    ioctl(tfd, TCGETA, &oldmode);	/* get current console tty mode	*/
    newmode = oldmode;                  /* copy (structure) to newmode  */

    newmode.c_oflag = 0;
    newmode.c_iflag &= ~(IXON | IXOFF);
    newmode.c_lflag &= ~(ICANON | ISIG | ECHO);
    newmode.c_cc[VMIN] = 1;
    newmode.c_cc[VTIME] = 1;

    sigmode = newmode;
    sigmode.c_lflag |= ISIG;

    nlmode = TRUE;

    oldshell[0] = '\0';		/* set last command to blank    */
    reterm = FALSE;		/* no direct jump into terminal mode */
    if(setjmp(erret))		/* set error handler to exit    */
        exit(0);		/* while parsing command line   */
    signal(SIGINT, catch);	/* catch break & quit signals/keys */
    signal(SIGQUIT, catch);

    opterr = 0;				/* Disable getopt error messages */
    while ((c = getopt(argc, argv, "g:l:t")) != -1)
	switch (c) {
	case 'l':			/* set modem port name          */
	    mport(optarg);
	    break;
	case 'g':			/* Execute SCRIPT file		*/
	    script = optarg;
	    break;
	case 't':			/* jump into terminal mode	*/
	    reterm = TRUE;
	    break;
	default:			/* Bad command .. print help	*/
	    usage();
	    exit(1);
	}

    if(mopen() < 0) {			/* open modem port and configure */
        fprintf(stderr, "Error in modem initialization routine.\r\n");
        fprintf(stderr, "Check environment variable MODEM for modem port.\r\n");
        exit(1);
    }

    setuid(getuid());			/* Return to being oneself. */

    intdel(0);

    do_startup();

    status();       /* display all parameters */

    fprintf(tfp, "Type \"help\" or ? for help\r\n");

    for (;;) {
        setjmp(erret);
	ioctl(tfd, TCSETAW, &newmode);
	eof_flag = 0;

	if (script) {
	    do_script(script);
	    script = NULL;
	    reterm = 1;
	}

	if (reterm) {
	    reterm = 0;
	    s_term();
	    continue;
	}

	intdel(1);
        fprintf(tfp, "\r\nXCOMM> ");	/* command prompt               */

	getsome(line);
	intdel(0);

	fprintf(tfp, "\r\n");

	getword();
	if (word[0] == '\0')		/* If blank line... reprompt	*/
	    continue;

	for (ptr = cmds; ptr->keyword != NULL; ptr++) 
	    if (strcmp(word, ptr->keyword) == 0)
		break;

	if (ptr->keyword)
	    (*ptr->rtn)();
	else 
	    fprintf(tfp, "Unrecognized keyword: %s\r\n", word);
    }

}

static char *babble[] = {
"Usage:  xcomm [-l device] [-t] [-g file]",
"",
"-l device      Use 'device' as the modem port (eg, -l /dev/tty00)",
"-t             Enter terminal mode immediately",
"-g file        Execute script file 'file' immediately",
"",
NULL };

usage()
{
    char **ptr;

    for (ptr = babble; *ptr != NULL; ptr++)
	fprintf(stderr, "%s\r\n", *ptr);
}
    
void s_script()
{
    getword();

    if (word[0] == '\0') {
	fprintf(tfp, "Script file not specified\r\n");
	return;
    }

    do_script(word);
    reterm = 1;
}

void s_xmodem()
{
    char d = word[0];
    char c = word[1];

    getword();
    if (word[0] == '\0')
	fprintf(tfp, "No transfer file specified\r\n");
    else if (d == 's')
	xsend(c);
    else
	xreceive(c);

    reterm = termflag;
}

void s_exit()
{
    signal(SIGINT, SIG_IGN);
    signal(SIGQUIT, SIG_IGN);

    do_exit(0);
}

void s_term()
{
    for (;;) {
	terminal();
	if (cismode != 2)
	    return;
	cismode = 1;
	B_Transfer();
    }
}

void s_cis()
{
    B_Transfer();
    reterm = termflag;
}

void s_shell()
{
    char c = word[0];
    char *ptr;
    static char *shell = NULL;
    char *getenv();
    int child;

    if (word[0] == word[1])
	strcpy(wptr=word, oldshell);
    else {
	getword();
	if (*wptr)
	    strcpy(oldshell, wptr);
    }
	
    if (!shell) {
	shell = getenv("SHELL");
	if (!shell)
	    shell = "/bin/sh";
    }

    signal(SIGCLD, SIG_DFL);

    ioctl(tfd, TCSETAW, &oldmode);

    if ((child = forkem()) == 0) {
	if (c == '$')	/* Attach modem to stdin, stdout */
	    mattach();
	if (word[0] == '\0')
	    execl(shell, shell, "-i", (char *) NULL);
	else
	    execl(shell, shell, "-c", wptr, (char *) NULL);
	fprintf(stderr, "Exec failed!");
	exit(1);
    }

    signal(SIGINT, SIG_IGN);
    signal(SIGQUIT, SIG_IGN);

    while (wait((int *) 0) >= 0)
	;

    intdel(0);

    strcpy(oldshell, wptr);

    signal(SIGINT, catch);	/* catch break & quit signals/keys */
    signal(SIGQUIT, catch);
}

static char *cmdlist[] = {
"XCOMM Command Summary",
" ",
"c               <-- Initiate CIS QuickB File Transfer (Upload and Download)",
"g file          <-- Execute XCOMM Script file",
"rb file         <-- XMODEM receive (binary mode)",
"rt file         <-- XMODEM receive (Ascii mode)",
"sb file...      <-- XMODEM send (binary mode)",
"st file...      <-- XMODEM send (Ascii mode)",
"set ...         <-- XCOMM Parameter set",
"t               <-- Enter terminal emulation",
"x               <-- Exit program",
" ",
"!               <-- Execute a local interactive shell",
"! cmd           <-- Execute shell command string on the local system",
"!!              <-- Re-execute the last shell command string",
"$ cmd           <-- Shell command with stdin and stdout redirected to modem",
"%p file         <-- Put file (transmit a file to a UNIX system)",
"%t file         <-- Take file (receive a file from a UNIX system)",
"?               <-- Print this help message",
" ",
" ",
"",
"SET Keywords:",
" ",
"set                     Display current XCOMM status",
"set 7bit  on|off        Set/Reset 7-bit data mask (ignore high bits)",
"set crc   on|off        Enable/Disable Xmodem CRC",
"set term  on|off        Enable/Disable terminal mode after file transfer",
"set cis   on|off        Set/Reset CIS <ENQ> mode (Auto up/download)",
"set mung  on|off        Set/Reset file overwrite",
"set nl    on|off        Set/Reset Terminal mode newline translation",
"set purge on|off        Set/Reset bad phone line mode",
"set xoff  on|off        Set/Reset XON/XOFF flow control",
"set baud  <value>       Set baud rate to <value>",
"set cfile name          Change name of capture file",
"set pfile name          Change name of phonelist file",
NULL };

void s_help()
{
    char **ptr = cmdlist;

    for ( ; *ptr != NULL; ptr++)
	if (**ptr)
	    fprintf(tfp, "%s\r\n", *ptr);
	else {
	    fprintf(tfp, "PRESS ENTER: ");
	    getsome(line);
	    erasln();
	}
}

void s_set_7bit(), s_set_crc(), s_set_term(), s_set_cis(), s_set_mung();
void s_set_purge(), s_set_xoff(), s_set_baud(), s_set_cfile(), s_set_pfile();
void s_set_cr();

static struct kw setlist[] = {
	{ "7bit",	s_set_7bit },
	{ "crc",	s_set_crc },
	{ "term",	s_set_term },
	{ "nl",		s_set_cr },
	{ "cis",	s_set_cis },
	{ "mung",	s_set_mung },
	{ "purge",	s_set_purge },
	{ "xoff",	s_set_xoff },
	{ "baud",	s_set_baud },
	{ "cfile",	s_set_cfile },
	{ "pfile",	s_set_pfile },
	{ NULL,		NULL }};

void s_set()
{
    struct kw *ptr;

    getword();

    if (word[0] == '\0' && !scriptflag) {
	status();
	return;
    } else if (word[0] == '\0') {
	fprintf(tfp, "SET keyword requires an argument\r\n");
	eof_flag++;
	return;
    }

    lc_word(word);

    for (ptr = setlist; ptr->keyword != NULL; ptr++)
	if (strcmp(ptr->keyword, word) == 0) {
	    (*ptr->rtn)();
	    return;
	}

    fprintf(tfp, "Invalid SET keyword: %s\r\n", word);
    eof_flag++;
}

static char *statfmt = "%-8s %30s is %s\r\n";

void s_set_7bit()
{
    int val;

    if (stat_flag) {
	fprintf(tfp, statfmt, "7bit", "Seven-bit Mask",
		bitmask==0x7f?"ON":"OFF");
	return;
    }

    set_onoff(&val);
    bitmask = val?0x7f:0xff;

    if (!scriptflag)
	fprintf(tfp, "<<%s-bit mask enabled>>\r\n", val?"Seven":"Eight");
}

void s_set_crc()
{
    if (stat_flag) {
	fprintf(tfp, statfmt, "crc", "Xmodem CRC",
		crcheck?"ON":"OFF");
	return;
    }

    set_onoff(&crcheck);

    if (!scriptflag)
	fprintf(tfp, "<<XMODEM %s enabled>>\r\n", 
		crcheck?"CRC Check":"Checksum");
}

void s_set_cr()
{
    if (stat_flag) {
	fprintf(tfp, statfmt, "nl", "Terminal NL Translation",
		nlmode?"ON":"OFF");
	return;
    }

    set_onoff(&nlmode);

    if (!scriptflag)
	fprintf(tfp, "<<%s mode enabled>>\r\n", 
		nlmode?"Newline translation":"Carriage return");
}

void s_set_term()
{
    if (stat_flag) {
	fprintf(tfp, statfmt, "term", "Auto Term Mode",
		termflag?"ON":"OFF");
	return;
    }

    set_onoff(&termflag);

    if (!scriptflag)
	fprintf(tfp, "<<File transfers return to %s mode>>\r\n",
		termflag?"terminal":"command");
}

void s_set_cis()
{
    if (stat_flag) {
	fprintf(tfp, statfmt, "cis", "CIS <ENQ> Auto Download",
		cismode?"ON":"OFF");
	return;
    }

    set_onoff(&cismode);

    if (!scriptflag)
	fprintf(tfp, "<<CIS <ENQ> Auto Download is %s>>\r\n", 
		cismode?"ON":"OFF");
}

void s_set_mung()
{
    if (stat_flag) {
	fprintf(tfp, statfmt, "mung", "File Overwrite", mungmode?"ON":"OFF");
	return;
    }

    set_onoff(&mungmode);

    if (!scriptflag)
	fprintf(tfp, "<<File Overwrite is %s>>\r\n", mungmode?"ON":"OFF");
}

void s_set_purge()
{
    if (stat_flag) {
	fprintf(tfp, statfmt, "purge", "Bad Telephone line",
		badline?"ON":"OFF");
	return;
    }

    set_onoff(&badline);

    if (!scriptflag)
	fprintf(tfp,"<<Bad telephone line purging is %s\r\n",
		badline?"ON":"OFF");
}

void s_set_xoff()
{
    if (stat_flag) {
	fprintf(tfp, statfmt, "xoff", "Terminal mode XON/XOFF",
		flowflag?"ON":"OFF");
	return;
    }

    set_onoff(&flowflag);
    xc_setflow();

    if (!scriptflag)
	fprintf(tfp, "<<XON/XOFF Flow control is %s>>\r\n",flowflag?"ON":"OFF");
}

set_onoff(flag)
 int *flag;
{
    char *strdup();
    char *ptr = strdup(word);

    getword();
    lc_word(word);

    if (strcmp(word, "on") == 0) {
	*flag = 1;
    } else if (strcmp(word, "off") == 0) {
	*flag = 0;
    } else {
	fprintf(tfp, "Set '%s' value must be 'on' or 'off'.\r\n", ptr);
	eof_flag++;
    }
    
    if (ptr != NULL)
	    free(ptr);
}

void s_set_baud()
{
    if (stat_flag) {
	char br[10];

	sprintf(br, "%d", mbaud((char *) 0));
	fprintf(tfp, statfmt, "baud", "Baud Rate", br);
	return;
    }

    getword();
    if (word[0] == '\0') {
	fprintf(tfp, "Set Baud must have a baud rate.\r\n");
	eof_flag++;
	return;
    }

    if (mbaud(word)<0) {
	fprintf(tfp, "Unsupported baud rate %s\r\n", word);
	eof_flag++;
    } else if (!scriptflag)
	fprintf(tfp, "<<Baud rate set to %d>>\r\n", mbaud((char *) 0));
}

void s_set_cfile()
{
    if (stat_flag) {
	fprintf(tfp, statfmt, "cfile", "Capture file", captfile);
	return;
    }

    getword();
    if (word[0] == '\0') {
	fprintf(tfp, "Set CFILE must have file name.\r\n");
	eof_flag++;
	return;
    }

    strcpy(captfile, word);

    if (!scriptflag)
	fprintf(tfp, "<<Capture file set to '%s'>>\r\n", captfile);
}

void s_set_pfile()
{
    if (stat_flag) {
	fprintf(tfp, statfmt, "pfile", "Phone number file", phonefile);
	return;
    }

    getword();
    if (word[0] == '\0') {
	fprintf(tfp, "Set PFILE must have file name.\r\n");
	eof_flag++;
	return;
    }

    strcpy(phonefile, word);

    if (!scriptflag)
	fprintf(tfp, "<<Phone number file set to '%s'>>\r\n", phonefile);
}

#ifdef PUT_TAKE
/*
    Put and Take a file to/from a UNIX-type "cu" system.  Unfortunately,
    the stty command is one of those commands that always gets changed
    with different UNIX systems, so you will get (at least) a file full of
    ^M on the take command for systems later than V7 or work-alikes.

    Additionally, the Take command takes a bit too much!

    Note:  This code was taken from the original XCOMM; I personally have
	   not tested it.
*/

void puttake()
{
    FILE *fp;
    char wrkbuff[WBSIZE];
    register int i;
    char c = word[1];

    getword();
    if (word[0] == '\0') {
        fprintf(tfp, "Must give a filename with the put and take options.\r\n");
        return;
    }

    switch(c) {
    case 'p':
        if((fp = fopen(word, "r")) == NULL){
            fprintf(tfp,"Can't read file '%s'.\r\n", word);
            break;
        }

	fprintf(tfp,"Putting file '%s' to UNIX system.\r\n", word);
        sprintf(wrkbuff, "stty -echo;cat >%s;stty echo\r\n", word);
        sendstr(wrkbuff);       /* send command string to remote shell */
        i = 0;
        while((c = getc(fp)) != EOF){
            if(++i > 64){       /* this prevents an overload on the */
                i = 0;          /* receiver's input buffer (64=kludge) */
                sleep(DRIBBLE);
            }
            send_mbyte(c);        /* send characters to cat command */
        }
        fclose(fp);
        send_mbyte('D' - '@');    /* send a ^D to cat */
        purge();                /* get rid of whatever was sent back */
        break;

    case 't':
        if(!mungmode && !access(word, 0)){
            fprintf(tfp,"Can't over write existing file '%s'.\r\n", word);
            break;
        }
        if((fp = fopen(word, "w")) == NULL){
            fprintf(tfp,"Can't write file '%s'.\r\n", word);
            break;
        }

	fprintf(tfp,"Taking file '%s' from UNIX system.\r\n", word);
        sprintf(wrkbuff, "stty -echo -onlcr tabs; sleep 3; cat %s; sleep 5; stty echo onlcr -tabs\n", word);
        sendstr(wrkbuff);       /* send command string to remote shell */
	sleep(2);		/* wait for remote system to consume command */
        purge();                /* get rid of the echo */
        while((c = read_mbyte(3)) != -1)  /* while more chars are being sent */
            putc(c, fp);
        fclose(fp);
        break;

    }
}
#endif /* PUT_TAKE */

/*
    Catch a signal and jump to main.  Reset signal and do a longjmp
*/

void catch()
{
    fprintf(tfp,"\r\nXCOMM: Interrupt\r\n");

    signal(SIGINT, catch);
    signal(SIGQUIT, catch);
    longjmp(erret, 1);
}

/*
    Print the status of the program
*/

status()
{
    struct kw *ptr;
    void (*fct)() = NULL;

    stat_flag = 1;

    fprintf(tfp, "\fXCOMM Version %s  by larry gensch - modem port: %s\r\n\n", 
	    VERSION, mport((char *) NULL));
    fprintf(tfp, "Keyword              Description and Status\r\n");
    fprintf(tfp, "-------  ----------------------------------------------\r\n");

    for (ptr = setlist; ptr->keyword != NULL; ptr++)
	if (ptr->rtn != fct) {
	    fct = ptr->rtn;
	    (*fct)();
	}

    fprintf(tfp, "\r\n");
    stat_flag = 0;
}

do_exit(rc)
{
    unlock_tty();

    fprintf(tfp, "\r\n");

    ioctl(tfd, TCSETAW, &oldmode);
    ioctl(tfd, TCFLSH, 2);

    fclose(tfp);

    exit(rc);
}

do_startup()
{
    char *home, *getenv();
    char file[WBSIZE];

    if (access(STARTUP, 0) == 0) 
	strcpy(file, STARTUP);
    else {
	if ((home = getenv("HOME")) == NULL)
	    return;
	sprintf(file, "%s/%s", home, STARTUP);
	if (access(file, 0))
	    return;
    }

    linkflag = 2;
    do_script(file);
}
