/* xcterm.c             XCOMM terminal mode
 */

#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "xcomm.h"

/* globals */
char captfile[NMSIZE]  = CAPTFILE;	/* capture file's name */
char phonefile[NMSIZE] = PHFILE;	/* phone number file's name */

int cismode = CIS_INIT;			/* Respond to CIS "ENQ" */
extern FILE *tfp;			/* Local terminal pointer */

/* locals */

static FILE *cfp;			/* capture file pointer */
static int ctop;			/* index top of capture buffer */
static int child_pid = 0;		/* ID of child process */
static int capture = FALSE;		/* are we capturing or not ? */
static char *cbuffp = NULL;		/* capture buffer pointer */
static jmp_buf  rtm;

/* forward declarations */
void toggle(), cleanup(), newbmask(), cisbmode();

static int script_flag = 0;
static char *script_file;
static char script_buf[100];

terminal()
{
    int c;

reterm:
    setjmp(rtm);

    if (cismode > 1)
	return;

    script_flag = 0;		/* reset scripting flag */

    fprintf(tfp,"Entering Terminal Mode\r\n");
    /* split into read and write processes */
    if((child_pid = forkem()) == 0){
	/* child, read proc: read from port and write to tty */
        cfp = NULL;
        signal(SIGALRM, toggle);
        signal(SIGTERM, cleanup);
        signal(SIGUSR1, newbmask);
        while(1){
            while((c = read_mbyte(0)) == -1)
                ;
	    if (cismode && c == ENQ) {
		cismode = 2;
		cleanup();
	    }
            fputc(c, tfp);
	    fflush(tfp);
            if(capture && c != '\r')
		fputc(c, cfp);
        }
	/*NOTREACHED*/
    }
    /* parent, write proc: read from tty and write to port */
    signal(SIGCLD, cisbmode);

    do {
        switch(c = getconchr()) {
        case TOGCHAR:		/* signal child to toggle buffer */
	    kill(child_pid, SIGALRM);
	    break;

        case DIVCHAR:		/* divert a file through modem port */
	    divert();
	    break;

	case SCRPCHR:		/* execute a script file */
	    if (get_script())
		break;
	    script_flag = 1;
	    goto fillicide;

        case PSELECT:		/* select and dial a phone number */
	    pselect();
	    if (!script_flag) 
		break;

        case ENDCHAR:		/* signal child to cleanup and exit */
fillicide:
	    c = ENDCHAR;
	    signal(SIGCLD, SIG_IGN);
	    kill(child_pid, SIGTERM);
	    break;

	case HUPCHAR:		/* Hangup */
	    hangup();
	    break;

	case '\n':		/* See if NL translation in effect */
	    if (nlmode)
		c = '\r';

        default:		/* just send the character to the port */
	    send_mbyte(c);
	    break;
        }
    } while(c != ENDCHAR);

    while(wait((int *) 0) >= 0)		/* wait for the read process to die */
        ;

    if (script_flag) {
	do_script(script_file);
	goto reterm;
    }

    fprintf(tfp,"\r\nLeaving Terminal Mode\r\n");
}

static void cisbmode()
{
    cismode = 2;
    signal(SIGCLD, SIG_IGN);

    longjmp(rtm);
}

/* The next three functions are only run by the port read process (child).
 * They handle the capture buffer.
 *
 * toggle capture status
 */
static void toggle()
{
    char *malloc();

    if(capture) {
	fclose(cfp);
        capture = FALSE;
	fprintf(tfp, "\r\n<<CAPTURE BUFF CLOSED>>\r\n");
    } else {
	if ((cfp = fopen(captfile, "a")) == NULL) {
	    fprintf(tfp,"\r\n<<CAN'T OPEN CAPTURE FILE>\r\n");
	} else {
	    capture = TRUE;
	    fprintf(tfp,"\r\n<<CAPTURE BUFF OPEN>>\r\n");
	}
    }

    signal(SIGALRM, toggle);	/* set signal for next toggle */
}

/* cleanup, flush and exit
 */
static void cleanup()
{
    if(capture){
        fclose(cfp);
	fprintf(tfp,"\r\n<<CAPTURE BUFF SAVED>>\r\n");
    }

    if (cismode == 2)
	fprintf(tfp,"\r\n<<FILE TRANSFER REQUEST>>\r\n");

    fflush(tfp);
    exit(0);
}

static void newbmask()
{
    if (bitmask == 0xff)
	bitmask = 0x7f;
    else
	bitmask = 0xff;
    
    if (child_pid)
	kill(child_pid, SIGUSR1);
    else
	fprintf(tfp,"<<SET TO %d BIT MASK>>\r\n", bitmask == 0xff ? 8 : 7);
}

static FILE *openfile(name)
 char *name;
{
    FILE *fp;
    char *home, fullname[NMSIZE];

    if (fp = fopen(name, "r"))
	return fp;

    if (home = (char *) getenv("HOME")) {
	sprintf(fullname, "%s/%s", home, name);
	return fopen(fullname, "r");
    } else
	return NULL;
}

/*
    Select a script file.  If the file exists, execute it, otherwise
    exit with a non-zero return.
*/

get_script()
{
    fprintf(tfp,"\r\nEnter script file: ");
    getsome(script_buf);
    if (script_buf[0] == '\0') {
	fprintf(tfp,"\r\n<<ABORTED>>\r\n");
	return 1;
    }
    if (access(script_buf, 04)) {
	fprintf(tfp," <<- NOT FOUND\r\n");
	return 1;
    }
    fprintf(tfp, "\r\n");
    linkflag = FALSE;
    script_file = script_buf;
    return 0;
}

/* pselect() and divert() are used by the port write process (parent).
 * They produce other sources of input for the port than the keayboard.
 *
 * Select phone number and dial it. (Hayes - type modem)
 * Phone number is stored in a file, one number per line, with the phone
 * number first, followed by some space and then any comments.
 * VERSION 2.1 - If line contains BAUD=nnnn, the baud rate will be changed
 *		 to nnn.  If lines contains BITS=7 or BITS=8, the bitmask
 *		 will be changed to the specified value.
 * VERSION 2.2 - If line contains SCRIPT=nnnnn, the designated script file
 *		 will be started.
 */
pselect()
{
    FILE *fp;
    char buffer[WBSIZE + 1];
    int c, i;

    if ((fp = openfile(phonefile)) == NULL) {
	fprintf(tfp,"Phonelist file '%s' not in current or home directory\r\n",
		phonefile);
	return;
    }

    fprintf(tfp,"Type D-Dial   X-Exit  Any other key for next line.\r\n");
    do {
        if(fgets(buffer, WBSIZE, fp) == NULL){
            rewind(fp);
            if(fgets(buffer, WBSIZE, fp) == NULL){
                fprintf(tfp,"Empty phonelist file!\r\n");
                return;
            }
        }
        erasln();
        for(i = 0; i < WBSIZE && (c = buffer[i]) != '\n'; i++)
            fputc(c, tfp);
        fputc('\r', tfp);
    } while((c = mklow(getc(tfp))) != 'd' && c != 'x');
    erasln();
    fclose(fp);
    if(c == 'd')
	dialbuf(buffer);
}

#if	!HAVE_STRSTR
/*
    Find first occurance of str2 in str1 and return a pointer to it
*/

char *strstr(str1, str2)
 char *str1, *str2;
{
    char *Sptr, *Tptr;
    int len = strlen(str1) - strlen(str2) + 1;

    if (*str2)
	for (; len > 0; len--, str1++) {
	    if (*str1 != *str2)
		continue;

	    for (Sptr = str1, Tptr = str2; *Tptr != '\0'; Sptr++, Tptr++)
		if (*Sptr != *Tptr)
		    break;
	    
	    if (*Tptr == '\0')
		return str1;
	}

    return NULL;
}
#endif

dialbuf(buf)
 char *buf;
{
    char *nbr, *ptr, *ptr1, c;

    if (ptr = strstr(buf, "BAUD=")) {
	ptr += 5;
	if (mbaud(ptr) == -1) 
	    fprintf(tfp,"Invalid BAUD= value\r\n");
	else
	    fprintf(tfp,"<<SET TO %d BAUD>>\r\n", mbaud(NULL));
    }

    if (ptr = strstr(buf, "BITS=")) {
	ptr += 5;
	switch (*ptr) {
	case '7':
	    if (bitmask == 0xff)
		newbmask();
	    break;
	case '8':
	    if (bitmask == 0x7f)
		newbmask();
	    break;
	default:
	    fprintf(tfp,"Invalid BITS= value\r\n");
	}
    }

    while (isspace(*buf) && *buf)
	    buf++;

    if (!(*buf))
	    return;

    for (nbr = buf; !isspace(*buf) && *buf; buf++)
	    ;
    
    c = *buf;
    *buf = '\0';
    dial (nbr);

    *buf = c;
    if (ptr = strstr(buf, "SCRIPT=")) {
	ptr += 7;
	ptr1 = ptr;
	while (!isspace(*ptr1) && *ptr1 != '\0')
	    ptr1++;
	*ptr1 = '\0';
	script_file = ptr;
	script_flag = 1;
	linkflag = TRUE;
    }
}

/* Divert file into input stream, with optional delay after each newline.
 */
divert()
{
    FILE *fp;
    char buffer[NMSIZE];
    int c, i;

    fprintf(tfp,"\r\nFile? ");
    getsome(buffer);
    if((fp = fopen(buffer, "r")) == NULL){
        fprintf(tfp,"\r\nCan't open %s for input\r\n", buffer);
        return;
    }
    fprintf(tfp,"\r\nThe default delay of %d can be ", DRIBBLE);
    fprintf(tfp,"changed by typing the new delay after the 'y'\r\n");
    fprintf(tfp,"Delay after every newline (y/n)? ");

    getsome(buffer);
    fprintf(tfp,"\r\n");

    i = 0;
    if(mklow(buffer[0]) == 'y' && (i = atoi(buffer + 1)) == 0)
        i = DRIBBLE;
    while((c = getc(fp)) != EOF)
        if(c != '\n')
            send_mbyte(c);
        else {
            send_mbyte('\r');
            if(i)
                sleep((unsigned) i);
        }
    fclose(fp);
}

getconchr()
{
    int c, tc;

    if ((c = coninp()) == ESCAPE_CHR) {
	switch (c = ((tc = coninp()) & 0x1f) + 128) {
	case ENDCHAR:
	case TOGCHAR:
	case DIVCHAR:
	case PSELECT:
	case HUPCHAR:
	case SCRPCHR:
	    break;

	default:
	    c = tc;
	}
    }
    return c;
}

